package com.payu.india.Model;

import android.util.Log;

import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;

import org.json.JSONException;
import org.json.JSONObject;

public class LookupRequest {

    private final String merchantAccessKey;
    private final String amount;
    private final String currency;
    private final String cardBin;
    private final String merchantOrderId;
    private final LookupRequest.ProductType productType;
    private final String signature;

    public String getMerchantAccessKey() {
        return merchantAccessKey;
    }

    public String getAmount() {
        return amount;
    }

    public String getCurrency() {
        return currency;
    }

    public String getCardBin() {
        return cardBin;
    }

    public String getMerchantOrderId() {
        return merchantOrderId;
    }

    public LookupRequest.ProductType getProductType() {
        return productType;
    }

    public String getSignature() {
        return signature;
    }

    private LookupRequest(LookupApiRequestBuilder lookupApiRequestBuilder) {
        this.merchantAccessKey = lookupApiRequestBuilder.merchantAccessKey;
        this.amount = lookupApiRequestBuilder.amount;
        this.currency = lookupApiRequestBuilder.currency;
        this.cardBin = lookupApiRequestBuilder.cardBin;
        this.merchantOrderId = lookupApiRequestBuilder.merchantOrderId;
        this.productType = lookupApiRequestBuilder.productType;
        this.signature = lookupApiRequestBuilder.signature;
    }

    public String prepareJSON() {
        JSONObject jsonObject = new JSONObject();

        if (!isValid(merchantAccessKey))
            throw new RuntimeException(PayuErrors.ERROR_INVALID_MERCHANT_ACCESS_KEY);
        else if (!isValid(amount))
            throw new RuntimeException(PayuErrors.ERROR_INVALID_AMOUNT);
        else if (!isValid(currency))
            throw new RuntimeException(PayuErrors.ERROR_INVALID_CURRENCY);
        else if (!isValid(cardBin) && productType != null && productType == ProductType.DCC)
            throw new RuntimeException(PayuErrors.ERROR_INVALID_CARD_BIN);
        else if (!isValid(merchantOrderId))
            throw new RuntimeException(PayuErrors.ERROR_INVALID_MERCHANT_ORDER_ID);
        else if (productType == null)
            throw new RuntimeException(PayuErrors.ERROR_INVALID_PRODUCT_TYPE);
        else if (!isValid(signature))
            throw new RuntimeException(PayuErrors.ERROR_INVALID_SIGNATURE);
        else {
            try {
                jsonObject.put(PayuConstants.PAYU_MERCHANT_ACCESS_KEY, merchantAccessKey);

                JSONObject baseAmountObject = new JSONObject();
                baseAmountObject.put(PayuConstants.PAYU_VALUE, amount);
                baseAmountObject.put(PayuConstants.PAYU_CURRENCY, currency);
                jsonObject.put(PayuConstants.PAYU_BASE_AMOUNT, baseAmountObject);

                jsonObject.put(PayuConstants.PAYU_CARD_BIN, cardBin);
                jsonObject.put(PayuConstants.PAYU_MERCHANT_ORDER_ID, merchantOrderId);
                jsonObject.put(PayuConstants.PAYU_PRODUCT_TYPE, productType.name());
                jsonObject.put(PayuConstants.PAYU_SIGNATURE, signature);

            } catch (JSONException e) {
                Log.d(getClass().getSimpleName(), "JSONException " + e.getMessage());
                throw new RuntimeException(PayuErrors.ERROR_SOME_ERROR_OCCURRED);
            }
        }
        return jsonObject.toString();
    }

    private boolean isValid(String value) {
        return value != null && !value.isEmpty();
    }

    public static class LookupApiRequestBuilder {

        private String merchantAccessKey;
        private String amount;
        private String currency;
        private String cardBin;
        private String merchantOrderId;
        private ProductType productType;
        private String signature;

        public LookupApiRequestBuilder setMerchantAccessKey(String merchantAccessKey) {
            this.merchantAccessKey = merchantAccessKey;
            return this;
        }

        public LookupApiRequestBuilder setAmount(String amount) {
            this.amount = amount;
            return this;
        }

        public LookupApiRequestBuilder setCurrency(String currency) {
            this.currency = currency;
            return this;
        }

        public LookupApiRequestBuilder setCardBin(String cardBin) {
            this.cardBin = cardBin;
            return this;
        }

        public LookupApiRequestBuilder setMerchantOrderId(String merchantOrderId) {
            this.merchantOrderId = merchantOrderId;
            return this;
        }

        public LookupApiRequestBuilder setProductType(ProductType productType) {
            this.productType = productType;
            return this;
        }

        public LookupApiRequestBuilder setSignature(String signature) {
            this.signature = signature;
            return this;
        }

        public LookupRequest build() {
            return new LookupRequest(this);
        }
    }

    public enum ProductType {
        DCC,
        MCP
    }
}
