package com.payu.india.Model.binBaseDetails;

import android.util.Log;

import com.payu.india.Model.validateOffer.ValidateOfferRequest;
import com.payu.india.Payu.PayuConstants;
import com.payu.paymentparamhelper.V2ApiBase;

import org.json.JSONException;
import org.json.JSONObject;



/**
 * Represents a request to fetch details for a particular BIN (Bank Identification Number) based transaction.
 * This class contains information about the transaction such as the card number, amount, additional charges,
 * and merchant details. It uses a Builder pattern to facilitate object construction. it is used as request class for BinBaseDetails Api Task class{@link com.payu.india.Tasks.BinBaseDetailsTask}
 *
 * <p>This class extends {@link V2ApiBase} and provides a method {@link #getJson()} to get the
 * JSON representation of the request for use in an API call.</p>
 *
 * <p>Example usage:</p>
 * <pre>
 * BinBaseDetailsRequest request = new BinBaseDetailsRequest.Builder("1234567890123456")
 *      .setAmount("1000")
 *      .setAdditionalCharges("20")
 *      .setPercentageAdditionalCharges("2")
 *      .setTxnId("txn123")
 *      .setMerchantKey("merchantXYZ")
 *      .setValidateOfferRequest(validateOfferRequest)
 *      .setPricingCFEnable(true)
 *      .setSource("Android_sdk")
 *      .setIsSITxn(true)
 *      .build();
 * </pre>
 */
public class BinBaseDetailsRequest extends V2ApiBase {

    /**
     * The card number associated with the BinBaseDetailsRequest.
     */
    private final String cardNumber;

    /**
     * The amount to be processed in the BinBaseDetailsRequest.
     */
    private final String amount;

    /**
     * Any additional charges that apply to the BinBaseDetailsRequest.
     */
    private final String additionalCharges;

    /**
     * The percentage of additional charges applied to the BinBaseDetailsRequest.
     */
    private final String percentageAdditionalCharges;

    /**
     * The unique identifier for the BinBaseDetailsRequest.
     */
    private final String txnId;

    /**
     * The merchant key associated with the BinBaseDetailsRequest.
     */
    private final String merchantKey;

    /**
     * The offer validation request containing offer details.
     */
    private final ValidateOfferRequest validateOfferRequest;

    /**
     * Flag indicating whether custom feature pricing is enabled.
     */
    private final boolean isPricingCFEnable;

    /**
     * The source of the transaction (e.g., website, mobile).
     */
    private final String source;

    /**
     * Flag indicating whether the transaction is part of a subscription installment.
     */
    private final boolean isSiTxn;

    /**
     * Constructs a new {@link BinBaseDetailsRequest} with the specified transaction details.
     *
     * @param builder The builder containing the transaction details to create the {@link BinBaseDetailsRequest}.
     */
    private BinBaseDetailsRequest(Builder builder) {
        this.cardNumber = builder.cardNumber;
        this.amount = builder.amount;
        this.additionalCharges = builder.additionalCharges;
        this.percentageAdditionalCharges = builder.percentageAdditionalCharges;
        this.txnId = builder.txnId;
        this.merchantKey = builder.merchantKey;
        this.validateOfferRequest = builder.validateOfferRequest;
        this.isPricingCFEnable = builder.isPricingCFEnable;
        this.source = builder.source;
        this.isSiTxn = builder.isSiTxn;
    }

    /**
     * Returns the JSON representation of this request.
     *
     * @return A JSON string representing the request.
     */
    @Override
    public String getJson() {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put(PayuConstants.P_CARD_NUMBER, cardNumber);
            jsonObject.put(PayuConstants.AMOUNT, amount);
            jsonObject.put(PayuConstants.P_ADDITIONAL_CHARGE, additionalCharges);
            jsonObject.put(PayuConstants.PAYU_PERCENTAGE_ADDITIONAL_CHARGES, percentageAdditionalCharges);
            jsonObject.put(PayuConstants.PAYU_TXN_ID, txnId);
            jsonObject.put(PayuConstants.PAYU_MERCHANT_KEY, merchantKey);
            if (validateOfferRequest != null)
                jsonObject.put(PayuConstants.PAYU_VALIDATE_OFFER_REQUEST_JSON, validateOfferRequest.getJsonValue());
            jsonObject.put(PayuConstants.PAYU_IS_PRICING_CF_ENABLED, isPricingCFEnable);
            jsonObject.put(PayuConstants.PAYU_SOURCE, source.toUpperCase());
            jsonObject.put(PayuConstants.PAYU_IS_SI_TXN, isSiTxn);
        } catch (JSONException e) {
            Log.d("BinBaseDetailsRequest", "Exception " + e.getMessage());
        }
        return jsonObject.toString();
    }

    /**
     * Builder class to construct a {@link BinBaseDetailsRequest}.
     */
    public static class Builder {
        private final String cardNumber;
        private String amount;
        private String additionalCharges;
        private String percentageAdditionalCharges;
        private String txnId;
        private String merchantKey;
        private ValidateOfferRequest validateOfferRequest;
        private boolean isPricingCFEnable;
        private String source;
        private boolean isSiTxn;

        /**
         * Constructs a Builder with the specified card number.
         *
         * @param cardNumber The card number associated with the request.
         */
        public Builder(String cardNumber) {
            this.cardNumber = cardNumber;
        }

        /**
         * Sets the amount to be processed in the request.
         *
         * @param amount The amount to be processed.
         * @return The Builder instance.
         */
        public Builder setAmount(String amount) {
            this.amount = amount;
            return this;
        }

        /**
         * Sets any additional charges that apply to the request.
         *
         * @param additionalCharges The additional charges.
         * @return The Builder instance.
         */
        public Builder setAdditionalCharges(String additionalCharges) {
            this.additionalCharges = additionalCharges;
            return this;
        }

        /**
         * Sets the percentage of additional charges applied to the request.
         *
         * @param percentageAdditionalCharges The percentage additional charges.
         * @return The Builder instance.
         */
        public Builder setPercentageAdditionalCharges(String percentageAdditionalCharges) {
            this.percentageAdditionalCharges = percentageAdditionalCharges;
            return this;
        }

        /**
         * Sets the unique transaction ID for the request.
         *
         * @param txnId The transaction ID.
         * @return The Builder instance.
         */
        public Builder setTxnId(String txnId) {
            this.txnId = txnId;
            return this;
        }

        /**
         * Sets the merchant key associated with the request.
         *
         * @param merchantKey The merchant key.
         * @return The Builder instance.
         */
        public Builder setMerchantKey(String merchantKey) {
            this.merchantKey = merchantKey;
            return this;
        }

        /**
         * Sets the offer validation request for the transaction.
         *
         * @param validateOfferRequest The offer validation request.
         * @return The Builder instance.
         */
        public Builder setValidateOfferRequest(ValidateOfferRequest validateOfferRequest) {
            this.validateOfferRequest = validateOfferRequest;
            return this;
        }

        /**
         * Sets whether custom feature pricing is enabled.
         *
         * @param pricingCFEnable The flag for custom feature pricing.
         * @return The Builder instance.
         */
        public Builder setPricingCFEnable(boolean pricingCFEnable) {
            isPricingCFEnable = pricingCFEnable;
            return this;
        }

        /**
         * Sets the source of the transaction (e.g., website, mobile).
         *
         * @param source The source of the transaction.
         * @return The Builder instance.
         */
        public Builder setSource(String source) {
            this.source = source;
            return this;
        }

        /**
         * Sets whether the transaction is part of a subscription installment.
         *
         * @param isSiTxn The flag indicating subscription installment.
         * @return The Builder instance.
         */
        public Builder setIsSITxn(boolean isSiTxn) {
            this.isSiTxn = isSiTxn;
            return this;
        }

        /**
         * Builds a new {@link BinBaseDetailsRequest} object with the specified details.
         *
         * @return A new {@link BinBaseDetailsRequest} instance.
         */
        public BinBaseDetailsRequest build() {
            return new BinBaseDetailsRequest(this);
        }
    }
}
