// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser.util;

import static com.payu.custombrowser.bean.CustomBrowserData.SINGLETON;
import static com.payu.custombrowser.util.SharedPreferenceUtil.getSharedPref;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkInfo;
import android.net.Uri;
import android.net.http.SslError;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.webkit.CookieManager;
import android.webkit.CookieSyncManager;
import android.webkit.WebView;
import android.widget.ImageView;

import androidx.vectordrawable.graphics.drawable.Animatable2Compat;
import androidx.vectordrawable.graphics.drawable.AnimatedVectorDrawableCompat;

import com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant;
import com.payu.custombrowser.Bank;
import com.payu.custombrowser.BuildConfig;
import com.payu.custombrowser.R;
import com.payu.custombrowser.analytics.AnalyticsHandler;
import com.payu.custombrowser.bean.CustomBrowserConfig;
import com.payu.custombrowser.bean.CustomBrowserData;
import com.payu.payuanalytics.analytics.model.AnalyticsConfig;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;

import javax.net.ssl.HttpsURLConnection;

/**
 * Created by minie on 29/6/15.
 * Contains cb's utility functions.
 */
public class CBUtil {

    public final static String CB_PREFERENCE = "com.payu.custombrowser.payucustombrowser";
    public final static String CB_JS_PREFERENCE = "com.payu.custombrowser.payucustombrowser.js";
    private static SharedPreferences sharedPreferences;
    private static final String TAG = "PAYU CBUtil";
    public static String currentUrl;
    public static String currentUrlStatus;
    private static List<String> descList = new ArrayList<String>() {{
        add(CBConstant.ERR_CONNECTION_REFUSED);
        add(CBConstant.ERR_CONNECTION_ABORTED);
        add(CBConstant.ERR_INTERNET_DISCONNECTED);
        add(CBConstant.ERR_ADDRESS_UNREACHABLE);
        add(CBConstant.ERR_CONNECTION_RESET);
        add(CBConstant.ERR_FAILED);
    }};


    public static WeakReference<AnimatedVectorDrawableCompat> animRef;

    public static WeakReference<Animatable2Compat.AnimationCallback> animationCallbackRef;

    /**
     * Return system current time of system
     *
     * @return system current time in string
     */
    public static String getSystemCurrentTime() {
        try {
            Date currentDate = new Date(System.currentTimeMillis());
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String date = df.format(currentDate);
            return date;
        } catch (Exception e) {
        }
        return "";
    }

    /**
     * Provide JSONObject for analytics
     *
     * @param context        application context
     * @param key            analytics event ket
     * @param value          event value
     * @param bank           name of the bank
     * @param sdkMerchantKey merchant key
     * @param trnxID         transaction  ID
     * @return
     */
    public static String getLogMessage(Context context, String key, String value, String bank, String sdkMerchantKey, String trnxID, String pageType) {
        try {
            JSONObject eventAnalytics = new JSONObject();
            eventAnalytics.put(CBAnalyticsConstant.PAYU_ID, getCookie(CBConstant.PAYUID, context));
            eventAnalytics.put(CBAnalyticsConstant.TRANSACTION_ID, trnxID);
            eventAnalytics.put(CBAnalyticsConstant.MERCHANT_KEY, sdkMerchantKey);
            eventAnalytics.put(CBAnalyticsConstant.PAGE_TYPE, (pageType == null ? "" : pageType));
            eventAnalytics.put(CBAnalyticsConstant.KEY, key);
            eventAnalytics.put(CBAnalyticsConstant.VALUE, URLEncoder.encode(value, "UTF-8"));
            eventAnalytics.put(CBAnalyticsConstant.BANK_NAME, (bank == null ? "" : bank));
            eventAnalytics.put(CBAnalyticsConstant.PAKAGE_NAME, context.getPackageName());
            eventAnalytics.put(CBAnalyticsConstant.TIMESTAMP, getSystemCurrentTime());
            eventAnalytics.put(CBAnalyticsConstant.VERSION_CODE, getVersionCode(context));
            eventAnalytics.put(CBAnalyticsConstant.CB_VERSION_NAME, BuildConfig.VERSION_NAME + "");
            return eventAnalytics.toString();
        } catch (Exception e) {
            return "{}";
        }
    }

    /**
     * should be used to decode initialize.js and bank specific js.
     *
     * @param fileInputStream reference of file input stream
     * @return decoded string.
     */

    public static String decodeContents(FileInputStream fileInputStream) {
        StringBuilder decoded = new StringBuilder();
        try {
            int c;
            int i = 0;
            while ((c = fileInputStream.read()) != -1) {
                if (i % 2 == 0) {
                    decoded.append((char) (c - ((i % 5) + 1)));
                } else {
                    decoded.append((char) (c + ((i % 5) + 1)));
                }
                i++;
            }
            fileInputStream.close();
        } catch (IOException e) {
        }
        return decoded.toString();
    }

    public static void setAlpha(float alpha, View view) {
        view.setAlpha(alpha);
    }

    /**
     * Update the last Url
     *
     * @param lastUrl last url in webview before transaction terminate
     * @return s:last stated url||f last finished url
     */
    public static String updateLastUrl(String lastUrl) {
        try {
            if (!(lastUrl.contains("||"))) {
                if (lastUrl.length() > 128)
                    return lastUrl.substring(0, 127);
                else
                    return lastUrl;
            } else {
                StringTokenizer st = new StringTokenizer(lastUrl, "||");
                String firstURl = st.nextToken();
                String secondUrl = st.nextToken();
                if (firstURl.length() > 128)
                    firstURl = firstURl.substring(0, 125);
                if (secondUrl.length() > 128)
                    secondUrl = secondUrl.substring(0, 125);
                return firstURl + "||" + secondUrl;
            }
        } catch (Exception e) {
            return "";

        }
    }

    /**
     * set the analytics key for the analytics of events and device
     *
     * @param value analytics key
     */
    public static void setVariableReflection(String className, String value, String varName) {
        try {

            if (value != null && !value.trim().equals("")) {
                Class aClass = Class.forName(className);
                Field field = aClass.getDeclaredField(varName);
                field.setAccessible(true);
                field.set(null, value);
                field.setAccessible(false);
            }
        } catch (Exception e) {
        }
    }

    /**
     * Filter the OTP from sms
     *
     * @param mBankJS bank JS
     * @param msgBody message received
     * @return filter OTP
     */
    public static String filterSMS(JSONObject mBankJS, String msgBody, Context context) {
        String mPassword = null;
        try {
            Matcher match;
            if (msgBody != null) {
                match = Pattern.compile(mBankJS.getString(context.getString(R.string.cb_detect_otp)), Pattern.CASE_INSENSITIVE).matcher(msgBody);
                if (match.find()) {
                    String[] regexValues = mBankJS.getString(context.getString(R.string.cb_find_new_otp)).split("::");
                    for (String regex : regexValues) {
                        //Check for empty regex
                        if (!TextUtils.isEmpty(regex)) {
                            // we have otp sms
                            match = Pattern.compile(regex, Pattern.CASE_INSENSITIVE).matcher(msgBody);
                            if (match.find()) {
                                //Remove space and dot from string
                                mPassword = match.group().trim().replaceAll("\\.", "");
                                if (!TextUtils.isEmpty(mPassword))
                                    break;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
        }
        return mPassword;
    }

    /***
     * Return HttpsURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @return connection object
     */
    public HttpsURLConnection getHttpsConn(String strURL, String postData) {
        try {
            return getHttpsConn(strURL, postData, -1);
        } catch (Exception e) {
            return null;
        }
    }


    /***
     * Return HttpsURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @param timeout timeout
     * @param cookiesList list of cookie
     * @return connection object
     */
    public HttpsURLConnection getHttpsConn(String strURL, String postData, int timeout, String cookiesList) {
        try {
            URL url = new URL(strURL);
            HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
            conn.setRequestMethod("POST");
            if (timeout != -1) {
                conn.setConnectTimeout(timeout);
            }
            conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            if (postData != null) {
                conn.setRequestProperty("Content-Length", String.valueOf(postData.length()));
            }
            if (cookiesList != null) {
                conn.setRequestProperty("Cookie", cookiesList);
            }
            conn.setSSLSocketFactory(new TLSSocketFactory());
            conn.setDoOutput(true);
            if (postData != null) {
                byte[] postParamsByte = postData.getBytes();
                conn.getOutputStream().write(postParamsByte);
            }
            return conn;
        } catch (Exception e) {
            return null;
        }
    }

    /***
     * Return HttpsURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @param timeout timeout
     * @return connection object
     */
    public HttpsURLConnection getHttpsConn(String strURL, String postData, int timeout) {
        return getHttpsConn(strURL, postData, timeout, null);
    }

    /**
     * Return the connection object
     *
     * @param strURL url
     * @return HttpsURLConnection object
     */
    public static HttpsURLConnection getHttpsConn(String strURL) {
        try {
            URL url = new URL(strURL);
            HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setSSLSocketFactory(new TLSSocketFactory());
            conn.setRequestProperty("Accept-Charset", "UTF-8");
            return conn;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Read InputStream and return string buffer
     *
     * @param responseInputStream input stream
     * @return StringBuffer
     */
    public static StringBuffer getStringBufferFromInputStream(InputStream responseInputStream) {
        try {
            StringBuffer responseStringBuffer = new StringBuffer();
            byte[] byteContainer = new byte[1024];
            for (int i; (i = responseInputStream.read(byteContainer)) != -1; ) {
                responseStringBuffer.append(new String(byteContainer, 0, i));
            }
            L.v("CbUtil Cb Response:  " + String.valueOf(responseStringBuffer));
            return responseStringBuffer;
        } catch (Exception e) {
        }
        return null;
    }

    /**
     * Function to find whether network is available or not.
     *
     * @param context application context
     * @return true if network available else false
     */
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
        return (networkInfo != null && networkInfo.isAvailable() && networkInfo.isConnected());
    }

    /**
     * Method to update retry urls
     *
     * @param data    URL List
     * @param context Application context
     * @return
     */
//    public static List<String> updateRetryData(String data, Context context) {
//
//        // Update data into SP
//        setRetryData(data, context);
//        // Then, update whitelistedRetryURl list
//        return processAndAddWhiteListedUrls(data);
//
//    }

    /**
     * Save retry urls in shared preference
     *
     * @param data
     * @param context
     */
//    private static void setRetryData(String data, Context context) {
//        if (data == null) {
//            // Data is null, clear whitelisted urls
//            SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SP_RETRY_FILE_NAME, CBConstant.SP_RETRY_WHITELISTED_URLS, "");
//        } else {
//            // Update whitelisted urls in SP
//            SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SP_RETRY_FILE_NAME, CBConstant.SP_RETRY_WHITELISTED_URLS, data);
//        }
//
//        L.v("#### PAYU", "DATA UPDATED IN SHARED PREFERENCES");
//    }

    /**
     * Clear Cookie(with n expiry date)
     */
    public void clearCookie(CustomBrowserConfig customBrowserConfig) {
        CookieManager cookieManager = CookieManager.getInstance();
        if (Build.VERSION.SDK_INT >= 21) { // for icici white page fix.
            if (null != SINGLETON.getDomainListofSession() && SINGLETON.getDomainListofSession().size() > 0
                    && null != customBrowserConfig.getDomainUrlListToUnclear() && customBrowserConfig.getDomainUrlListToUnclear().size() > 0) {

                Iterator<String> iterator = CustomBrowserData.SINGLETON.getDomainListofSession().iterator();
                while (iterator.hasNext()) {
                    String domainName = iterator.next();
                    if (customBrowserConfig.getDomainUrlListToUnclear().isEmpty() || !customBrowserConfig.getDomainUrlListToUnclear().contains(domainName)) {
                        if (null != domainName)
                            deleteCookies(domainName);
                    }
                }
            } else if (null != customBrowserConfig && null != customBrowserConfig.getDomainUrlListToUnclear() && customBrowserConfig.getDomainUrlListToUnclear().isEmpty()) {
                cookieManager.removeSessionCookies(null);
            }
        } else {
            cookieManager.removeSessionCookie();
        }
    }

    /**
     * Return a list of whitelisted urls
     *
     * @param data
     * @return
     */
//    public static List<String> processAndAddWhiteListedUrls(String data) {
//        if (data != null && !data.equalsIgnoreCase("")) {
//            // Explode at |
//            String[] urls = data.split("\\|");
//            for (String url : urls) {
//                L.v("#### PAYU", "Split Url: " + url);
//            }
//            // For each of the urls in this exploded array, add to whitelisted url list
//            if (urls != null && urls.length > 0) {
//                List<String> whiteListedUrls = Arrays.asList(urls);
//                return whiteListedUrls;
//            }
//            L.v("#### PAYU", "Whitelisted URLs from JS: " + data);
//        }
//
//        return new ArrayList<String>();
//    }

    /**
     * @param key     shared preference key
     * @param context application context
     * @return
     */

    public boolean getBooleanSharedPreference(String key, Context context) {
        try {
            return SharedPreferenceUtil.getBooleanFromSharedPreference(context, CB_PREFERENCE, key, false);
        } catch (Exception e) {
            SharedPreferenceUtil.removeAllFromSharedPref(context, CB_PREFERENCE);
            return false;
        }
    }

    /**
     * @param key     shared preference key
     * @param context application context
     * @return
     */

    public boolean getBooleanSharedPreferenceDefaultTrue(String key, Context context) {
        try {
            return SharedPreferenceUtil.getBooleanFromSharedPreference(context, CB_PREFERENCE, key, true);
        } catch (Exception e) {
            SharedPreferenceUtil.removeAllFromSharedPref(context, CB_PREFERENCE);
            return true;
        }


    }

    /**
     * @param key     shared preference key
     * @param value   value of shared preference
     * @param context application context
     */
    public void setBooleanSharedPreference(String key, boolean value, Context context) {
        SharedPreferenceUtil.addBooleanToSharedPreference(context, CB_PREFERENCE, key, value);
    }

    /**
     * Calculate Device Density
     *
     * @param activity base activity
     * @return device density
     */
    public String getDeviceDensity(Activity activity) {
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.densityDpi + "";
    }

//    /**
//     * Calculate download speed
//     */
//    private void getDownloadSpeed() {
//        String[] testing;
//        testing = new String[2];
//        long BeforeTime = System.currentTimeMillis();
//        long TotalTxBeforeTest = TrafficStats.getTotalTxBytes();
//        long TotalRxBeforeTest = TrafficStats.getTotalRxBytes();
//
//        long TotalTxAfterTest = TrafficStats.getTotalTxBytes();
//        long TotalRxAfterTest = TrafficStats.getTotalRxBytes();
//        long AfterTime = System.currentTimeMillis();
//
//        double TimeDifference = AfterTime - BeforeTime;
//
//        double rxDiff = TotalRxAfterTest - TotalRxBeforeTest;
//        double txDiff = TotalTxAfterTest - TotalTxBeforeTest;
//
//        if ((rxDiff != 0) && (txDiff != 0)) {
//            double rxBPS = (rxDiff / (TimeDifference / 1000)); // total rx bytes per second.
//            double txBPS = (txDiff / (TimeDifference / 1000)); // total tx bytes per second.
//            testing[0] = String.valueOf(rxBPS) + "bps. Total rx = " + rxDiff;
//            testing[1] = String.valueOf(txBPS) + "bps. Total tx = " + txDiff;
//        } else {
//            testing[0] = "No uploaded or downloaded bytes.";
//        }
//    }

    /**
     * Calculate network status
     *
     * @param context
     * @return
     */

    public String getNetworkStatus(Context context) {
        try {
            if (null != context) {
                ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info == null || !info.isConnected())
                    return "Not connected"; //not connected
                if (info.getType() == ConnectivityManager.TYPE_WIFI)
                    return "WIFI";
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    int networkType = info.getSubtype();
                    switch (networkType) {
                        case TelephonyManager.NETWORK_TYPE_GPRS:
                            return "GPRS";
                        case TelephonyManager.NETWORK_TYPE_EDGE:
                            return "EDGE";
                        case TelephonyManager.NETWORK_TYPE_CDMA:
                            return "CDMA";
                        case TelephonyManager.NETWORK_TYPE_1xRTT:
                        case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
                            return "2G";
                        case TelephonyManager.NETWORK_TYPE_UMTS:
                        case TelephonyManager.NETWORK_TYPE_EVDO_0:
                        case TelephonyManager.NETWORK_TYPE_EVDO_A:
                        case TelephonyManager.NETWORK_TYPE_HSDPA:
                        case TelephonyManager.NETWORK_TYPE_HSUPA:
                        case TelephonyManager.NETWORK_TYPE_HSPA:
                            return "HSPA";
                        case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
                        case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
                        case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
                            return "3G";
                        case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
                            return "4G";
                        default:
                            return "?";
                    }
                }
            }
        } catch (Exception e) {
            return "?";
        }
        return "?";
    }

    public NetworkInfo getNetWorkInfo(Context mContext) {
        ConnectivityManager connectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo network = null;
        if (Build.VERSION.SDK_INT >= 21) {
            Network[] networks = connectivityManager.getAllNetworks();
            for (Network mNetwork : networks) {
                NetworkInfo networkInfo = connectivityManager.getNetworkInfo(mNetwork);
                if (networkInfo.getState().equals(NetworkInfo.State.CONNECTED)) {
                    network = networkInfo;
                }
            }
        } else {
            NetworkInfo[] info = connectivityManager.getAllNetworkInfo();
            if (info != null) {
                for (NetworkInfo anInfo : info) {
                    if (anInfo.getState() == NetworkInfo.State.CONNECTED) {
                        network = anInfo;
                    }
                }
            }

        }
        return network;
    }

    /**
     * Checking for all possible internet providers
     **/

    public int getNetworkStrength(Context mContext) {
        NetworkInfo network = getNetWorkInfo(mContext);
        //  Toast.makeText(mContext, mContext.getString(R.string.please_connect_to_internet), Toast.LENGTH_SHORT).show();
        if (network != null) {
            if (network.getTypeName().equalsIgnoreCase("MOBILE")) {
                return getMobileStrength(mContext, network);
            } else if (network.getTypeName().equalsIgnoreCase("wifi") && hasPermission(mContext, "android.permission.ACCESS_WIFI_STATE")) {
                final WifiManager wifiManager = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);
                try {
                    final WifiInfo connectionInfo = wifiManager.getConnectionInfo();
                    if (connectionInfo != null) {
                        // connectionInfo.getRssi() - According to IEEE 802.11 documentation: Lesser negative values denotes higher signal strength.The range is between -100 to 0 dBm, closer to 0 is higher strength and vice-versa.
                        return WifiManager.calculateSignalLevel(connectionInfo.getRssi(), 5); //returning a number between 0 and 4 (i.e. numLevel-1) : the number of bars you see in toolbar.
                    }
                } catch (Exception e) {
                }
            }
            return 0;
        }
        return 0;
    }

    /**
     * Determines if the context calling has the required permission
     *
     * @param context    - the IPC context
     * @param permission - The permissions to check
     * @return true if the IPC has the granted permission
     */
    private boolean hasPermission(Context context, String permission) {
        int res = context.checkCallingOrSelfPermission(permission);
        return res == PackageManager.PERMISSION_GRANTED;

    }
    /**
     * @param originalColor color, without alpha
     * @param alpha         from 0.0 to 1.0
     * @return
     */
/*
  public static String addAlpha(String originalColor, Double alpha) {
      long alphaFixed = Math.round(alpha * 255);

        String alphaHex = java.lang.Long.toHexString(alphaFixed);
        if (alphaHex.length() == 1) {
            alphaHex = "0"+alphaHex;
        }
        originalColor = originalColor.replace("#", "#"+alphaHex);
        return originalColor;
    }
*/


    /**
     * Function to check if App has entry of permission in Manifest
     *
     * @param context    Context
     * @param permission Permission
     * @return True if permission in Manifest else False
     */
    public boolean hasRequestedPermission(Context context, String permission) {
        try {
            PackageInfo packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
            String[] permissionInfos = packageInfo.requestedPermissions;
            for (String permissionInfo : permissionInfos) {
                L.d("hasPermission: " + permissionInfo);
                if (permission.equalsIgnoreCase(permissionInfo)) {
                    return true;
                }
            }
        } catch (PackageManager.NameNotFoundException e) {
            e.getMessage();
        }
        return false;
    }

    private int getMobileStrength(Context context, NetworkInfo networkInfo) {
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            int strength = 0;
            // we are interested only in mobile.
            // okay now its mobile network
            // lets find the strength.
            for (CellInfo info : telephonyManager.getAllCellInfo()) {
                if (info.isRegistered()) { // connected
                    if (info instanceof CellInfoGsm) {
                        CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                        strength = gsm.getDbm();
                    } else if (info instanceof CellInfoCdma) {
                        CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                        strength = cdma.getDbm();
                    } else if (info instanceof CellInfoLte) {
                        CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                        strength = lte.getDbm();
                    } else if (info instanceof CellInfoWcdma) {
                        CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength(); // jelly bean mr2
                        strength = wcdma.getDbm();
                    }
                }
            }

            return strength;
        } catch (Exception e) {
            return 0;
        }

    }

    /**
     * Update the last Url in shared preference
     *
     * @param context base activity context
     * @param key     shared preference key
     * @param url     lastUrl stored
     */
    public void setStringSharedPreferenceLastURL(Context context, String key, String url) {
        String str = getStringSharedPreference(context, key);
        if (str.equalsIgnoreCase("")) {
            str = url;
        } else if (!str.contains("||")) {
            str = str + "||" + url;
        } else {
            StringTokenizer st = new StringTokenizer(str, "||");
            st.nextToken();
            str = st.nextToken() + "||" + url;
        }
        storeInSharedPreferences(context, key, str);
    }

    /**
     * @param context base activity context
     * @param key     shared preferance key
     * @return value for the provided key
     */
    public String getStringSharedPreference(Context context, String key) {
        try {
            return SharedPreferenceUtil.getStringFromSharedPreference(context, CB_PREFERENCE, key, "");
        } catch (Exception e) {
            SharedPreferenceUtil.removeAllFromSharedPref(context, CB_PREFERENCE);
            return "";
        }
    }


    /**
     * @param context base activity context
     * @param key     shared preferance key
     * @return value for the provided key
     */
    public String getJSStringSharedPreference(Context context, String key, boolean isPersistent) {
        try {
            if (isPersistent) {
                return SharedPreferenceUtil.getStringFromSharedPreference(context, CB_PREFERENCE, key, CBConstant.UNDEFINED);
            } else {
                return SharedPreferenceUtil.getStringFromSharedPreference(context, CB_JS_PREFERENCE, key, CBConstant.UNDEFINED);
            }
        } catch (Exception e) {
            return CBConstant.UNDEFINED;
        }

    }

    /**
     * Function to remove key from CB JS preference
     *
     * @param context base Activity Context
     * @param key     to be removed from CB JS preference
     */
    public void removeJSStringSharedPreference(Context context, String key) {
        SharedPreferences sharedPreferences = getSharedPref(context, CB_JS_PREFERENCE);
        if (sharedPreferences != null) {
            SharedPreferences.Editor editor = sharedPreferences.edit();
            editor.remove(key);
            editor.apply();
        }

    }

    /**
     * Function to clear CB JS preference
     *
     * @param context
     */
    public void clearJSStringSharedPreference(Context context) {
        SharedPreferences sharedPreferences = getSharedPref(context, CB_JS_PREFERENCE);
        if (sharedPreferences != null) {
            SharedPreferences.Editor editor = sharedPreferences.edit();
            editor.clear();
            editor.apply();
        }
    }


    /**
     * @param context base activity context
     * @param key     shared preference key
     * @param value   for the provided key
     */
    public void setStringSharedPreference(Context context, String key, String value) {
        SharedPreferenceUtil.addStringToSharedPreference(context, CB_PREFERENCE, key, value);
    }

    /**
     * @param context base activity context
     * @param key     shared preference key
     * @param value   for the provided key
     */
    public void setJSStringSharedPreference(Context context, String key, String value) {
        SharedPreferenceUtil.addStringToSharedPreference(context, CB_JS_PREFERENCE, key, value);
    }


    /**
     * delete the given key from shared preference
     *
     * @param context base activity context
     * @param key     shared preference key
     */
    public void deleteSharedPrefKey(Context context, String key) {
        try {
            SharedPreferences.Editor sharedPreferencesEditor = (getSharedPref(context, CB_PREFERENCE)).edit();
            sharedPreferencesEditor.remove(key);
            sharedPreferencesEditor.apply();
        } catch (Exception e) {
        }
    }

    /**
     * store the value for given key in shared Preference
     *
     * @param context base activity context
     * @param key     shared preference key
     * @param value   value for the given key
     */
    public void storeInSharedPreferences(Context context, String key, String value) {
        SharedPreferenceUtil.addStringToSharedPreference(context, CB_PREFERENCE, key, value);
    }

    /**
     * @param context of base activity
     * @param key     key of share preference
     */
    public void removeFromSharedPreferences(Context context, String key) {
        try {
            SharedPreferences sharedPreferences = getSharedPref(context, CB_PREFERENCE);
            SharedPreferences.Editor editor = sharedPreferences.edit();
            editor.remove(key);
            editor.apply();
        } catch (Exception e) {
        }

    }

    /**
     * **
     * Return the drawble for the given resourse ID
     *
     * @param context context of base activity
     * @param resID   resourse ID of the drawable
     * @return bank icon drawable
     */
    public Drawable getDrawableCB(Context context, int resID) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            return context.getResources().getDrawable(resID);
        } else {
            return context.getResources().getDrawable(resID, context.getTheme());
        }
    }

    /**
     * Cancel timer
     *
     * @param timer timer object
     */
    public void cancelTimer(Timer timer) {
        if (timer != null) {
            timer.cancel();
            timer.purge();
        }
    }

    /**
     * Read input stream from given file
     * base activity context
     *
     * @param fileName    file name
     * @param contextMode define access to file
     * @return data read from file
     */
    public String readFileInputStream(Context mContext, String fileName, int contextMode) {//how to handle first parameter as it can be both context or activity
        int c;
        String temp = "";
        try {
            File file = new File(mContext.getFilesDir(), fileName);
            if (!file.exists()) { // create the file if not created yet.
                mContext.openFileOutput(fileName, contextMode);
            }
            FileInputStream fileInputStream = mContext.openFileInput(fileName);
            while ((c = fileInputStream.read()) != -1) {
                temp = temp + Character.toString((char) c);
            }
            fileInputStream.close();
        } catch (FileNotFoundException e) {
        } catch (IOException e) {
        } catch (Exception e) {
        }
        return temp;


    }

    /**
     * Write the input stream to file
     *
     * @param inputStream InputStream
     * @param context     base activity
     * @param fileName    file to which data to be written
     * @param contextMode define access to file
     */
    public void writeFileOutputStream(InputStream inputStream, Context context, String fileName, int contextMode) {

        try {
            GZIPInputStream responseInputStream = new GZIPInputStream(inputStream);
            byte[] buf = new byte[1024];
            int len;
            FileOutputStream outputStream = context.openFileOutput(fileName, contextMode);
            while ((len = responseInputStream.read(buf)) > 0) {
                outputStream.write(buf, 0, len);
            }
            responseInputStream.close();
            outputStream.close();
        } catch (IOException e) {
        } catch (Exception e) {
        }

    }


    /**
     * reset payuID in cookie
     */
    public void resetPayuID(CustomBrowserConfig customBrowserConfig) {
        // Reset PAYUID Cookie
        clearCookie(customBrowserConfig);
        // set PAYUID empty(when OLD payuID is picked)
        /*CookieManager cookieManager = CookieManager.getInstance();
        cookieManager.setCookie(CBConstant.PAYU_DOMAIN, CBConstant.PAYUID+"="); // setting to empty string*/
    }

    /**
     * Return list of cookie of the given domain
     *
     * @param appContext application context
     * @param domainName name of domain
     * @return cookie string
     */
    public String getCookieList(Context appContext, String domainName) {
        String cookieString = "";

        try {
            CookieManager cookieManager = CookieManager.getInstance();
            if (Build.VERSION.SDK_INT < 21) {
                CookieSyncManager.createInstance(appContext);
                CookieSyncManager.getInstance().sync();
            }
            String cookies = cookieManager.getCookie(domainName); // return string of all cookie for the given domain
            if (cookies != null) {
                String[] temp = cookies.split(";");
                for (String ar1 : temp) {
                    String[] cookiesNameValue = ar1.split("=");
                    cookieString += (cookiesNameValue[0] + "=" + cookiesNameValue[1] + ";");
                }
            }
            if (cookieString.length() > 0) {
                cookieString = cookieString.substring(0, cookieString.length() - 1);
            }
        } catch (Exception e) {
        }
        return cookieString;
    }

    /**
     * Return Value of given cookie name
     *
     * @param cookieName name of cookie
     * @return
     */
    public static String getCookie(String cookieName, Context context) {

        String cookieValue = "";
        try {

            String siteName = CBConstant.PAYU_DOMAIN; // domain name for cookie
            CookieManager cookieManager = CookieManager.getInstance();
            if (Build.VERSION.SDK_INT < 21) {
                CookieSyncManager.createInstance(context);
                CookieSyncManager.getInstance().sync();
            }
            String cookies = cookieManager.getCookie(siteName); // return string of all cookie for the given domain
            if (cookies != null) {
                String[] temp = cookies.split(";");
                for (String ar1 : temp) {
                    if (ar1.contains(cookieName)) {
                        String[] temp1 = ar1.split("=");
                        cookieValue = temp1[1];
                    }
                }
            }
        } catch (Exception e) {
        }
        return cookieValue;
    }

    // Todo: Remove this method use  public HashMap getDataFromPostData(String postData) instead
    @Deprecated
    public String getDataFromPostData(String postData, String key) {
        String[] list = postData.split("&");
        for (String item : list) {
            String[] items = item.split("=");
            if (items.length >= 2) {
                String id = items[0];
                if (id.equalsIgnoreCase(key)) {
                    return items[1];
                }
            }
        }
        return "";
    }

    /**
     * Using Hash map to get all the post data key and value.
     * Later we can use this methods to parase non payu's post data.
     *
     * @param postData
     * @return
     */

    public HashMap<String, String> getDataFromPostData(String postData) {
        HashMap<String, String> postParamsMap = new HashMap<>();
        if (null != postData) {
            StringTokenizer tokens = new StringTokenizer(postData, "&");
            while (tokens.hasMoreTokens()) {
                String[] keyValue = tokens.nextToken().split("=");
                if (null != keyValue && keyValue.length > 0 && null != keyValue[0]) {
                    postParamsMap.put(keyValue[0], keyValue.length > 1 ? keyValue[1] : "");
                }
            }
        }
        return postParamsMap;
    }

    public PaymentOption getPaymentOptionFromPostData(String postData) {
        if (!TextUtils.isEmpty(postData)) {
            String bankCode = getDataFromPostData(postData, CBConstant.BANK_CODE);
            if (!TextUtils.isEmpty(bankCode)) {
                for (PaymentOption option : PaymentOption.values()) {
                    if (option.getPaymentName().equalsIgnoreCase(bankCode))
                        return option;
                }
            }
        }
        return null;
    }

    public static String getRandomString(int length) {
        // create a string of all characters
        String alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

        // create random string builder
        StringBuilder sb = new StringBuilder();

        // create an object of Random class
        Random random = new Random();


        for (int i = 0; i < length; i++) {

            // generate random index number
            int index = random.nextInt(alphabet.length());

            // get character specified by index
            // from the string
            char randomChar = alphabet.charAt(index);

            // append the character to string builder
            sb.append(randomChar);
        }

        String randomString = sb.toString();
        return randomString;

    }

    public static String getBankCodeFromPostData(HashMap<String, String> map) {
        return map.get(CBConstant.BANK_CODE);
    }

    /**
     * Helper method to parse snooze config and keep it in shard pref.
     * This method should be called from {@link com.payu.custombrowser.Bank#setSnoozeConfig(String)}
     * key - url
     * value - percentage||timeout
     * sample value of snoozeConfig {"default":[{"url":"default_payment_url","time_out":"5","progress_percent":"20"}],"0MQaQP":[{"url":"url21||url22","time_out":"5","progress_percent":"50"},{"url":"url23","time_out":"5","progress_percent":"20"}]}
     *
     * @param context      Activity or application context
     * @param snoozeConfig actual json string given by initialize js.
     */
    public SnoozeConfigMap storeSnoozeConfigInSharedPref(Context context, String snoozeConfig) {
        SnoozeConfigMap snoozeConfigMap = new SnoozeConfigMap();
        try {
            JSONObject snoozeConfigObject = new JSONObject(snoozeConfig);

            // Deleting the existing config.
            SharedPreferenceUtil.removeAllFromSharedPref(context, CBConstant.SNOOZE_SHARED_PREF);
            // storing the default values first.
            snoozeConfigMap = storeSnoozeConfigInSharedPref(context, snoozeConfigObject.getJSONArray("default"), snoozeConfigMap);
            // Removing default from snoozeConfig.
            snoozeConfigObject.remove("default");
            Iterator<String> snoozeConfigIterator = snoozeConfigObject.keys();
            if (snoozeConfigIterator.hasNext()) {
                snoozeConfigMap = storeSnoozeConfigInSharedPref(context, snoozeConfigObject.getJSONArray(snoozeConfigIterator.next()), snoozeConfigMap);
            }

        } catch (JSONException e) {
        }
        return snoozeConfigMap;
    }

    private SnoozeConfigMap storeSnoozeConfigInSharedPref(Context context, JSONArray configArray, SnoozeConfigMap snoozeConfigMap) {
        String urlCollections;
        String progressPercent;
        String timeOut;
        StringTokenizer snoozeTokenizer;
        String url;
        int surePayEnableMode = CBConstant.ENABLE_WARN_AND_FAIL_MODE;
        try {
            for (int i = 0, snoozeDefaultArrayLength = configArray.length(); i < snoozeDefaultArrayLength; i++) {
                JSONObject currentSnoozeConfigObject = configArray.getJSONObject(i);
                // more than one url can have same config like: "url":"url21||url22"
                urlCollections = currentSnoozeConfigObject.get(CBConstant.URL).toString();
                progressPercent = currentSnoozeConfigObject.get(CBConstant.PROGRESS_PERCENT).toString();
                timeOut = currentSnoozeConfigObject.get(CBConstant.TIME_OUT).toString();
                if (currentSnoozeConfigObject.has(CBConstant.DISABLE_SP_FOR)) {
                    JSONObject surePayEnableModeObject = currentSnoozeConfigObject.getJSONObject(CBConstant.DISABLE_SP_FOR);
                    surePayEnableMode = getSurePayEnableMode(surePayEnableModeObject);
                }
                // Using tokenizer because split requires to escape pipe: using pipe because url might have , . ; # etc.
                snoozeTokenizer = new StringTokenizer(urlCollections, CBConstant.CB_DELIMITER);
                while (snoozeTokenizer.hasMoreTokens()) {
                    url = snoozeTokenizer.nextToken();
                    // key : url
                    // value : progressPercentage||timeout||surepayEnableMode
                    SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SNOOZE_SHARED_PREF, url.contentEquals("*") ? CBConstant.DEFAULT_PAYMENT_URLS : url.trim(), progressPercent.trim() + CBConstant.CB_DELIMITER + timeOut.trim() + CBConstant.CB_DELIMITER + surePayEnableMode);
                    snoozeConfigMap.put(url.contentEquals("*") ? CBConstant.DEFAULT_PAYMENT_URLS : url.trim(), progressPercent.trim() + CBConstant.CB_DELIMITER + timeOut.trim() + CBConstant.CB_DELIMITER + surePayEnableMode);
                }
            }
        } catch (JSONException e) {
        }
        return snoozeConfigMap;

    }

    /**
     * Method to convert the shared pref's hash map value to SnoozeConfigMap value.
     * Using this method because {@link SharedPreferenceUtil#getSharedPrefMap(Context, String)} is a generic method which returns Map.
     *
     * @param snoozeMap map of shardperf
     * @return snoozeConfigMap
     */
    public SnoozeConfigMap convertToSnoozeConfigMap(Map<String, ?> snoozeMap) {
        SnoozeConfigMap snoozeConfigMap = new SnoozeConfigMap();
        for (Map.Entry<String, ?> entry : snoozeMap.entrySet()) {
            snoozeConfigMap.put(entry.getKey(), entry.getValue());
        }
        return snoozeConfigMap;
    }

//    /**
//     * Method to add all the Error codes. - We can have these config from JS in future.
//     * Contains list of error codes where we should show sure pay dialog in fail mode,
//     * more info at https://developer.android.com/reference/android/webkit/WebViewClient.html#ERROR_AUTHENTICATION
//     * @return set of error codes
//     */
//    public Set<String> getSurePayErrorCodes(){
//        Set<String> showSurePayErrorCodesSet  = new HashSet<>();
//        // showSurePayErrorCodesSet.add("-6"); // ERROR_CONNECT
//        showSurePayErrorCodesSet.add("-7"); // ERROR_IO
//        showSurePayErrorCodesSet.add("-8"); // ERROR_TIMEOUT
//        showSurePayErrorCodesSet.add("-15"); // ERROR_TOO_MANY_REQUESTS
//        return showSurePayErrorCodesSet;
//    }


    /**
     * Method to get sure pay disable config.
     * 0 - don't disable sure pay.
     * 1 - only warn mode disabled.
     * 2 - only fail mode disabled
     * 3 - both warn and fail are disabled.
     *
     * @param jsonObject
     * @return {'warn' : 'true', 'fail' : 'false'}
     */
    private int getSurePayEnableMode(JSONObject jsonObject) {
        try {
            if (jsonObject.has(CBConstant.WARN) && jsonObject.getBoolean(CBConstant.WARN)
                    && jsonObject.has(CBConstant.FAIL) && jsonObject.getBoolean(CBConstant.FAIL)) {
                return CBConstant.DISABLE_WARN_AND_FAIL_MODE;
            } else if (jsonObject.has(CBConstant.FAIL) && jsonObject.getBoolean(CBConstant.FAIL)) {
                return CBConstant.DISABLE_FAIL_MODE;
            } else if (jsonObject.has(CBConstant.WARN) && jsonObject.getBoolean(CBConstant.WARN)) {
                return CBConstant.DISABLE_WARN_MODE;
            }
        } catch (Exception e) { // oops something went wrong, dont disable sure pay
            return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
        }
        // By default dont disable sure pay
        return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
    }


    /**
     * Method returns sure pay disable status, by parsing the snooze config map.
     *
     * @param snoozeConfigMap given snooze config map from js or shared pref
     * @param webViewUrl      current url.
     * @return sure pay disable status
     */
    public int getSurePayDisableStatus(SnoozeConfigMap snoozeConfigMap, String webViewUrl) {
        if (null != snoozeConfigMap && webViewUrl != null) {
            int[] snoozeConfig;

            // loop through snooze config map and check given url is found
            for (Object key : snoozeConfigMap.keySet()) {
                if (webViewUrl.startsWith(key.toString())) { // given url is in snoozeConfig,
                    snoozeConfig = snoozeConfigMap.getPercentageAndTimeout(key.toString());
                    return snoozeConfig[2];
                }
            }

            // We have not found the given url, lets return the default url value from the js.
            // This takes care of * as well!
            snoozeConfig = snoozeConfigMap.getPercentageAndTimeout(CBConstant.DEFAULT_PAYMENT_URLS);
            return snoozeConfig[2];
        }

        // we don't have snooze config map value lets return default snooze mode.
        return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
    }

    /**
     * Function to get value of a key from json string.
     *
     * @param data - input string .
     * @param key  - input key
     * @return - value of the key.
     * @throws JSONException
     * @author: Minie Sahni.
     */
    public String getValueOfJSONKey(String data, String key) throws JSONException {
        final JSONObject jsonObject = new JSONObject(data);
        if (jsonObject.has(key)) {
            return jsonObject.get(key).toString();
        } else {
            throw new JSONException("Key not found");
        }
    }

//    /**
//     * Method to launch play store app
//     * @param context
//     */
//    public static void launchPlayStore(Context context){
//        launchPlayStore(context, null, null);
//    }

    /**
     * Method to launch play store with a specific app name.
     *
     * @param context
     * @param url
     * @param webViewVersion
     */
    public static void launchPlayStore(Context context, String url, String webViewVersion) {

        String packageName = getPackageNameFromPlayStoreLink(url);

        if (null == packageName) {
            packageName = "";
        } else {
            packageName = "details?id=" + packageName;
        }

        setWebViewVersionInSP(context, webViewVersion);

        try {
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("market://" + packageName)));
        } catch (ActivityNotFoundException e) {
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/" + packageName)));
        }
    }

    /**
     * Method to check given url is a play store url
     *
     * @param url
     * @return
     */
    public static boolean isPlayStoreUrl(String url) {
        return url.startsWith(CBConstant.PLAY_STORE_URL) || url.startsWith(CBConstant.PLAY_STORE_MARKET_URI);
    }


    /**
     * Method to get the package name from play store url
     * Using regex (?<=[?&]id=)[^&]+ to do so
     * the extra bracket in the regex is for making it as a group. ((?<=[?&]id=)[^&]+)
     *
     * @param url
     * @return
     */
    public static String getPackageNameFromPlayStoreLink(String url) {
        Matcher m = Pattern.compile("((?<=[?&]id=)[^&]+)").matcher(url);
        return m.find() ? m.group(1) : null;
    }

    /**
     * method to find the given webview version
     *
     * @param view
     * @return
     */
    public static String getWebViewVersion(WebView view) {
        String userAgent = view.getSettings().getUserAgentString();
        Matcher m = Pattern.compile("(Chrome\\/(.*?)\\s)").matcher(userAgent);
        return m.find() ? m.group(2) : null;
    }

    /**
     * method to set the webview version in SP
     *
     * @param context
     * @param webViewVersion
     */
    public static void setWebViewVersionInSP(Context context, String webViewVersion) {
        if (null != webViewVersion) {
            SharedPreferenceUtil.addStringToSharedPreference(context, CB_PREFERENCE, CBConstant.WEBVIEW_VERSION, webViewVersion);
        }
    }

    /**
     * Method to get the previously stored webview/chrome version from sp
     *
     * @param context
     * @return
     */
    public static String getWebViewVersionFromSP(Context context) {
        return SharedPreferenceUtil.getStringFromSharedPreference(context, CB_PREFERENCE, CBConstant.WEBVIEW_VERSION, "");
    }
//
//    /**
//     * Method to get the Base64 Decoded String value
//     *
//     * @param data - Base64 encoded String
//     * @return Base64 decoded String
//     */
//    public static String getBase64DecodedString(String data) {
//        return new String(Base64.decode(data, Base64.DEFAULT));
//    }
//    /**
//     * Get Phones IMEI number, will be required in every communication with payu server.
//     *
//     * @return iemei
//     */
//    public static String getImei(Context applicationContex) {
//        String imei;
//        try {
//            TelephonyManager mTelephonyMgr = (TelephonyManager) applicationContex.getSystemService(Context.TELEPHONY_SERVICE);
//            imei = mTelephonyMgr.getDeviceId();
//        } catch (Exception e) {
//            imei = CBConstant.DEFAULT_VALUE;
//        }
//        return imei;
//    }
//
//    /**
//     * Phone's UDID, will be required in every communication with payu server.
//     *
//     * @return UDID.
//     */
//    public  static String getUdid(Context applicationContex) {
//        String udid;
//        try {
//            udid = Settings.Secure.getString(applicationContex.getContentResolver(), Settings.Secure.ANDROID_ID);
//        } catch (Exception e) {
//            udid = CBConstant.DEFAULT_VALUE;
//        }
//
//        return udid;
//    }

//    public static boolean isSPModuleAvailable(){
//        try {
//            Class myObjectClass = CBUtil.class.getClassLoader().loadClass("com.payu.samsungpay.SamsungWrapper");
//            return true;
//        } catch (ClassNotFoundException e) {
//            Log.e(TAG, "Please import com.payu.samsungpay to make Payment by Samsung Pay");
//            return false;
//        }
//    }

    /**
     * Checks if the module for payment option passed in argument is available
     *
     * @param paymentOption paymentOption like, TEZAPP,etc
     * @return true, if module is available else false
     */
    public static boolean isPaymentModuleAvailable(PaymentOption paymentOption) {
        try {
            Class myObjectClass = CBUtil.class.getClassLoader().loadClass(paymentOption.getClassName());
            return true;
        } catch (ClassNotFoundException e) {
            Log.e(TAG, "Please import " + paymentOption.getPackageName().toLowerCase() + " to make Payment by " + paymentOption.toString());
            return false;
        }
    }

    private static String getVersionCode(Context context) {
        try {
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            return pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
        }
        return "";
    }

    public static void deleteCookies(String url) {

        CookieManager cookieManager = CookieManager.getInstance();
//            String domainName = getDomain(url);
        String cookiesString = cookieManager.getCookie(url);

        if (!TextUtils.isEmpty(cookiesString)) {
            String[] cookies = cookiesString.split("; ");
            for (String cookie : cookies) {
                if (TextUtils.isEmpty(cookie))
                    continue;
                int equalCharIndex = cookie.indexOf('=');
                if (equalCharIndex == -1)
                    continue;
                String cookieString = cookie.substring(0, equalCharIndex) + '='
                        + "; ";
                cookieManager.setCookie(url, cookieString);
                SINGLETON.removeDomain(url);
            }
        } else {
            SINGLETON.removeDomain(url);
        }

        L.v("List Saved: " + SINGLETON.getDomainListofSession());
    }

    public static String getDomain(String urlString) {
        URL url;
        try {
            if (null != urlString) {
                url = new URL(urlString);
            } else
                return null;
        } catch (MalformedURLException e) {
            return null;
        }
        String protocol = url.getProtocol();
        String host = url.getHost();
        int port = url.getPort();

// if the port is not explicitly specified in the input, it will be -1.
        if (port == -1) {
            return String.format("%s://%s", protocol, host);
        } else {
            return String.format("%s://%s:%d", protocol, host, port);
        }
    }

    /**
     * Add analytics data to SharedPreference to be used by all SDKs before pushing to server.
     *
     * @param postData Merchant's PostData
     */
    public String getAnalyticsAddedPostData(String postData) {
        JSONArray jsonArray;
        String jsonArrayValue = "";
        HashMap<String, String> hashMap = getDataFromPostData(postData);
        if (hashMap.containsKey(CBConstant.SDK_PLATFORM_KEY)) {
            jsonArrayValue = hashMap.get(CBConstant.SDK_PLATFORM_KEY);
        }
        try {
            if (!TextUtils.isEmpty(jsonArrayValue)) {
                jsonArray = new JSONArray(jsonArrayValue);
            } else {
                jsonArray = new JSONArray();
            }
            JSONObject object = new JSONObject();
            object.put(CBConstant.PLATFORM_KEY, CBConstant.PLATFORM_VALUE);
            object.put(CBConstant.NAME_KEY, CBConstant.NAME_VALUE);
            object.put(CBConstant.VERSION_KEY, BuildConfig.VERSION_NAME);
            jsonArray.put(object);
        } catch (JSONException e) {
            return convertHashMapToPostData(hashMap);
        }
        hashMap.put(CBConstant.SDK_PLATFORM_KEY, jsonArray.toString());
        return convertHashMapToPostData(hashMap);
    }

    private String convertHashMapToPostData(HashMap<String, String> hashMap) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String key : hashMap.keySet()) {
            stringBuilder.append(key);
            stringBuilder.append("=");
            stringBuilder.append(hashMap.get(key));
            stringBuilder.append("&");
        }
        return stringBuilder.toString().substring(0, stringBuilder.lastIndexOf("&"));
    }

    public static AlertDialog.Builder getAlertDialog(Activity activity, DialogInterface.OnClickListener onClickDialog, String positiveText, String negativeText, String title, String message) {
        AlertDialog.Builder alertbox = new AlertDialog.Builder(activity, R.style.cb_dialog);
        if (onClickDialog != null) {
            if (title != null)
                alertbox.setTitle(title);
            if (message != null)
                alertbox.setMessage(message);
            if (positiveText != null)
                alertbox.setPositiveButton(positiveText, onClickDialog);
            if (negativeText != null)
                alertbox.setNegativeButton(negativeText, onClickDialog);
        }
        return alertbox;

    }

    public static String getSslErrorReason(int primaryError) {
        String errorReason;
        switch (primaryError) {
            case SslError.SSL_UNTRUSTED:
                errorReason = "The certificate authority is not trusted.";
                break;
            case SslError.SSL_DATE_INVALID:
                errorReason = "The date of the certificate is invalid.";
                break;
            case SslError.SSL_EXPIRED:
                errorReason = "The certificate has expired.";
                break;
            case SslError.SSL_IDMISMATCH:
                errorReason = "Hostname mismatch.";
                break;
            case SslError.SSL_INVALID:
                errorReason = "A generic error occurred.";
                break;
            case SslError.SSL_MAX_ERROR:

                errorReason = "The number of different SSL errors.";
                break;
            case SslError.SSL_NOTYETVALID:
                errorReason = "The certificate is not yet valid.";

                break;
            default:
                errorReason = "Something went wrong.";
                break;
        }
        return errorReason;
    }


    public static void hideKeyboard(Context context, View view) {
        InputMethodManager imm =
                (InputMethodManager) context.getSystemService(Activity.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
    }

    public static String addAlpha(String originalColor, Double alpha) {
        String origColor;
        long alphaFixed = Math.round(alpha * 255);
        String alphaHex = java.lang.Long.toHexString(alphaFixed);
        if (alphaHex.length() == 1) {
            alphaHex = "0" + alphaHex;
        }
        origColor = originalColor.replace("#", "#" + alphaHex);
        return origColor;
    }

    public static void logAnalytics(Bank bank, String key, String value) {
        if (null != bank) {
            bank.addEventAnalytics(key, value);
        }
    }

    public static String getPaymenttype(HashMap<String, String> map) {
        String pg = map.get("pg");
        if (pg != null && (pg.equalsIgnoreCase(CBConstant.CC) || pg.equalsIgnoreCase(CBConstant.DC)))
            return CBConstant.CARDS;
        else if (pg != null && pg.equalsIgnoreCase(CBConstant.NB))
            return CBConstant.NETBANKING;
        else return pg;
    }

    public static boolean hasKeyValue(String params, String cbRegenerate) {
        JSONObject p;
        try {
            p = new JSONObject(params);
            final boolean regenerate = p.has(cbRegenerate) && p.getBoolean(cbRegenerate);
            if (regenerate) {
                return true;
            }
        } catch (Exception e) {
            return false;
        }
        return false;
    }

    public static void logData(Context context, String ctaName, String ctaPage, String eventName, String amount, String bankcode, String flowType, String screentype, String time) {
        Map<String, Object> eventData = new HashMap<>();
        eventData.put(AnalyticsConstant.UI_CTA_NAME, ctaName);
        eventData.put(AnalyticsConstant.UI_CTA_PAGE, ctaPage);
        eventData.put(AnalyticsConstant.UI_TIME, time);
        eventData.put(AnalyticsConstant.CB_TIMEIN_MILISECONDS, Double.parseDouble(time) / 1000);
        eventData.put(AnalyticsConstant.UI_FLOW_TYPE, flowType);
        eventData.put(AnalyticsConstant.UI_BANKCODE, bankcode);
        eventData.put(AnalyticsConstant.UI_SCREEN_TYPE, screentype);
        AnalyticsHandler analyticsHandler = new AnalyticsHandler();
        analyticsHandler.logAnalytics(context, eventName, eventData, amount);
    }

    public static void logUserCancellation(Context context, String ctaName, String ctaPage, String eventName, String amount, String bankcode, String flowType, String screentype, String time, String msg) {
        Map<String, Object> eventData = new HashMap<>();
        eventData.put(AnalyticsConstant.UI_CTA_NAME, ctaName);
        eventData.put(AnalyticsConstant.UI_CTA_PAGE, ctaPage);
        eventData.put(AnalyticsConstant.UI_TIME, time);
        eventData.put(AnalyticsConstant.CB_TIMEIN_MILISECONDS, Double.parseDouble(time) / 1000);
        eventData.put(AnalyticsConstant.UI_FLOW_TYPE, flowType);
        eventData.put(AnalyticsConstant.UI_BANKCODE, bankcode);
        eventData.put(AnalyticsConstant.UI_SCREEN_TYPE, screentype);
        eventData.put(CBAnalyticsConstant.URL, (currentUrl == null ? "" : currentUrl));
        eventData.put(CBAnalyticsConstant.URL_STATE, (currentUrlStatus == null ? "" : currentUrlStatus));
        eventData.put(CBAnalyticsConstant.MSG, (msg == null ? "" : msg));
        AnalyticsHandler analyticsHandler = new AnalyticsHandler();
        analyticsHandler.logAnalytics(context, eventName, eventData, amount);
    }

    public static void logBackButton(Context context, String ctaName, String ctaPage, boolean sdkClosed, String amount) {
        Map<String, Object> eventData = new HashMap<>();
        long time = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
        eventData.put(AnalyticsConstant.UI_TIME, time);
        eventData.put(AnalyticsConstant.CB_TIMEIN_MILISECONDS, time / 1000);
        eventData.put(AnalyticsConstant.UI_CTA_NAME, ctaName);
        eventData.put(AnalyticsConstant.UI_CTA_PAGE, ctaPage);
        eventData.put(AnalyticsConstant.UI_CTA_TYPE, AnalyticsConstant.CB_CTA_TYPE_ACTION);
        eventData.put(AnalyticsConstant.CB_SECTION_NAME, AnalyticsConstant.CB_LEFT_SIDEBAR);
        if (sdkClosed) eventData.put(AnalyticsConstant.CB_SDK_CLOSED, "Yes");
        else eventData.put(AnalyticsConstant.CB_SDK_CLOSED, "No");
        AnalyticsHandler analyticsHandler = new AnalyticsHandler();
        analyticsHandler.logAnalytics(context, AnalyticsConstant.CB_BACK_BUTTON, eventData, amount);
    }

    public static AnalyticsConfig getAnalyticsConfig(String type) {
        AnalyticsConfig analyticsConfig = new AnalyticsConfig();
//        analyticsConfig.setEnabled(false);
        analyticsConfig.setInitiatorIdentifier(BuildConfig.LIBRARY_PACKAGE_NAME.concat(type));
        return analyticsConfig;
    }

    public static int parseInt(String s) {
        try {
            if (s == null)
                return 0;
            return Integer.parseInt(s);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    public static String valueOf(Object o) {
        String value = String.valueOf(o);
        if (value.equals("null")) {
            return null;
        }
        return value;
    }

    synchronized public static void longLog(String str) {

        if (str.length() > 4000) {
            Log.d(
                    TAG,
                    str.substring(0, 4000)
            );
            longLog(str.substring(4000));
        } else Log.d(TAG, str);

    }

    public static void applyLoopingAnimatedVectorDrawable(ImageView ivProgress, Context context) {
        animRef = new WeakReference<>(AnimatedVectorDrawableCompat.create(context, R.drawable.payu_loader));

        if (animRef.get() != null) {
            ivProgress.setImageDrawable(animRef.get());
            animationCallbackRef = new WeakReference<>(new Animatable2Compat.AnimationCallback() {
                @Override
                public void onAnimationEnd(Drawable drawable) {
                    if(animRef != null)
                        animRef.get().start();
                }
            });
            animRef.get().registerAnimationCallback(animationCallbackRef.get());
            animRef.get().start();
        }
    }

    public static void stopLoopingAnimatedVectorDrawable() {
        if (animRef != null && animRef.get() != null) {
            animRef.get().stop();
            if (animationCallbackRef != null && animationCallbackRef.get() != null) {
                animRef.get().unregisterAnimationCallback(animationCallbackRef.get());
                animationCallbackRef.clear();
            }
        }
    }

    public static boolean errorDescList(String desc) {
        return descList.contains(desc);
    }


    public static ArrayList<String> getWizRocketID() {
        ArrayList<String> ctId = new ArrayList<>();
        ctId.add(CBConstant.CTID1);
        ctId.add(CBConstant.CTID2);
        ctId.add(CBConstant.CTID3);
        return ctId;
    }

    public static ArrayList<String> getWizRocketPass() {
        ArrayList<String> ctPass = new ArrayList<>();
        ctPass.add(CBConstant.CTID1);
        ctPass.add(CBConstant.CTID2);
        ctPass.add(CBConstant.CTID3);
        return ctPass;
    }

    public static void safeClose(InputStream inputStream) {
        try {
            inputStream.close();
        } catch (IOException e) {
            Log.d(TAG, "Exception: " + e.getMessage());
        }
    }

}
