// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.os.Message;

import androidx.annotation.NonNull;

import android.util.Log;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.payu.custombrowser.analytics.AnalyticsHandler;
import com.payu.custombrowser.util.CBConstant;

import java.util.HashMap;
import java.util.concurrent.TimeUnit;


/**
 * Created by amitkumar on 12/12/14.
 * Custom chrome client for payu.
 */
public class PayUWebChromeClient extends WebChromeClient {
    private Bank mBank;
    private boolean mPageDone = false;

    public PayUWebChromeClient(@NonNull Bank bank) {
        mBank = bank;
    }

    @Override
    public boolean onCreateWindow(WebView view, boolean isDialog, boolean isUserGesture, Message resultMsg) {
        WebView oldWebView = view;
        WebView newWebView = new WebView(view.getContext());
        newWebView.getSettings().setJavaScriptEnabled(true);
        newWebView.getSettings().setSupportZoom(true);
        newWebView.getSettings().setBuiltInZoomControls(true);
        newWebView.getSettings().setSupportMultipleWindows(true);
        view.addView(newWebView);
        WebView.WebViewTransport transport = (WebView.WebViewTransport) resultMsg.obj;
        transport.setWebView(newWebView);
        resultMsg.sendToTarget();

        newWebView.setWebViewClient(new WebViewClient() {
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url) {
                try {
                    if (url.startsWith(CBConstant.DEEP_LINK_INTENT_URI)) {
                        Context context = view.getContext();
                        Intent intent = Intent.parseUri(url, Intent.URI_INTENT_SCHEME);

                        if (intent != null) {
                            view.stopLoading();

                            PackageManager packageManager = context.getPackageManager();
                            ResolveInfo info = packageManager.resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY);
                            if (info != null) {
                                context.startActivity(intent);
                            } else {
                                String fallbackUrl = intent.getStringExtra(CBConstant.BROWSER_FALLBACK_URL);
                                if (fallbackUrl != null && !fallbackUrl.isEmpty()) {
                                    oldWebView.loadUrl(fallbackUrl);
                                } else {
                                    sendAnalyticEvent(CBConstant.FALLBACK_URL_NOT_FOUND, view.getContext());
                                }
                            }
                            return true;
                        }
                    } else if (url.startsWith(CBConstant.DEEP_LINK_GENERIC_INTENT_URI)) {
                        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                        view.getContext().startActivity(intent);
                        return true;
                    }
                } catch (Exception e) {
                    sendAnalyticEvent(e.getLocalizedMessage(), view.getContext());
                    Log.e("intent", "Can't resolve url "+e.getLocalizedMessage(), e);
                }
                return false;
            }
        });

        return true;
    }

    private void sendAnalyticEvent(String message, Context context) {
        HashMap<String, Object> eventData = new HashMap<>();
        eventData.put(com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant.UI_MERCHANT_NAME, mBank.customBrowserConfig.getMerchantName());
        eventData.put(com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant.UI_EVENT_NAME, mBank.customBrowserConfig.getMerchantName());
        String time = String.valueOf(((TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()))));
        eventData.put(com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant.CB_TIMEIN_MILISECONDS, Double.parseDouble(time) / 1000);
        eventData.put(com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant.UI_TIME, time);
        eventData.put(com.payu.custombrowser.AnalyticsHandler.AnalyticsConstant.UI_ERROR, message);
        new AnalyticsHandler().logAnalytics(context, CBConstant.CB_PAYMENT_INITIATED, eventData, null);
    }

    @Override
    public void onProgressChanged(WebView view, int newProgress) {
        if (mBank != null) {
            if (!mPageDone && newProgress < 100) {
                mPageDone = true;
                mBank.onPageStarted();
            } else if (newProgress == 100) {
                mBank.onPageStarted();
                mPageDone = false;
                mBank.onPageFinished();
            }
            mBank.onProgressChanged(newProgress);
        }

    }
}
