package com.payu.checkoutpro.utils

import android.content.Context
import android.util.Log
import com.payu.assetprovider.enums.ImageType
import com.payu.base.models.*
import com.payu.base.models.ApiResponse
import com.payu.checkoutpro.R
import com.payu.checkoutpro.layers.PayUbizApiLayer
import com.payu.checkoutpro.reConverter.PaymentModeConverter
import com.payu.checkoutpro.utils.AndroidUtils.isUPIAppInstalled
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_TIME_FORMAT
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.SODEXO
import com.payu.india.Model.*
import com.payu.india.Model.QuickPay.QuickPaySavedOption
import com.payu.india.Model.QuickPay.RecommendedOptions
import com.payu.india.Payu.PayuConstants
import com.payu.paymentparamhelper.PaymentParams
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.*
import kotlin.collections.ArrayList
import kotlin.collections.HashMap


/**
 * This class should have Utility methods related to parsing APU objects
 * */
object ParserUtils {
    private val TAG = PayUCheckoutProConstants.PARSERUTILS
    internal var emiList: ArrayList<PaymentOption>? = null
    internal var bnplList: ArrayList<PaymentOption>? = null
    internal var quickOptionsList: ArrayList<PaymentMode>? = null
    internal var moreOptionsList: ArrayList<PaymentMode>? = null
    internal var convenienceFeeResponse: PayuResponse? = null
    internal var lookupDetails: LookupDetails? = null
    internal var isNBOfferFetched = false
    internal var isSIMode: Boolean = false
    internal var isProdEnvironment: Boolean = true
    internal var issuingBankDownList: ArrayList<String>? = null
    internal var payuBizParams: PaymentParams? = null
    internal var sodexoSourceId: String? = null
    internal var enforcePaymentList: ArrayList<HashMap<String, String>>? = null
    internal var retryCount: Int = -1
    internal var paymentOptionMap = HashMap<PaymentType, HashMap<String, Any>>()
    internal var closedLoopWalletCardOption: SodexoCardOption = SodexoCardOption()


    /* in case of retry txn enforced payment is not used we will show all the options  */
    internal var isRetryTxn: Boolean = false
    internal var apiLayer: PayUbizApiLayer? = null
    internal var isCheckOutCompleted: Boolean = false
    internal var gvResponse: PayuResponse? = null
    internal var onGVQuickPayListener: ((QuickOptionsModel?) -> Unit?)? = null


    internal fun getMoreOptions(
        payUbizApiLayer: PayUbizApiLayer,
        payuResponse: PayuResponse?,
        quickOptionsListener: (ArrayList<PaymentMode>?) -> Unit?
    ) {
        isCheckOutCompleted = true
        val moreOptionsList = prepareMoreOptions(payUbizApiLayer, payuResponse)

        ApiResponseRepo.setL1MoreOptionsResponse(moreOptionsList)
        gvResponse?.let {
            onGVQuickPayListener?.let { it1 ->
                prepareGvResponse(payUbizApiLayer.context, it.savedOption,
                    it.recommendedOptions, it1
                )
            }
        }

    }

    internal fun getMoreOptionsForRetry(
        payUbizApiLayer: PayUbizApiLayer,
        moreOptionListener: (ArrayList<PaymentMode>?) -> Unit?,
        onGVQuickPayListener: (QuickOptionsModel?) -> Unit?
    ) {
        isCheckOutCompleted = true
        val moreOptionsList = prepareMoreOptions(payUbizApiLayer, convenienceFeeResponse)

        val payUApiResponse = PayUApiResponse(moreOptionListener, true, moreOptionsList)
        ApiResponseRepo.moreOptionsResponse = payUApiResponse

        ApiResponseRepo.setL1MoreOptionsResponse(moreOptionsList)

        convenienceFeeResponse?.savedOption?.let {
            prepareGvResponse(payUbizApiLayer.context, it,
                convenienceFeeResponse?.recommendedOptions, onGVQuickPayListener
            )
        }

    }


    /**
     * Function to prepare MoreOptions(L1 screen Options) List
     */

    internal fun prepareMoreOptions(
        payUbizApiLayer: PayUbizApiLayer,
        checkoutDetailsResponse: PayuResponse?
    ): ArrayList<PaymentMode> {
        val paymentModesList = ArrayList<PaymentMode>()
        val context = payUbizApiLayer.context.applicationContext
        if (checkoutDetailsResponse == null) {
            sendErrorMessage(payUbizApiLayer, context.getString(R.string.empty_response_message))
            return paymentModesList
        }
        val checkoutOptionsList = payUbizApiLayer.payUCheckoutProConfig.paymentModesOrder

        if (!isRetryTxn)
            apiLayer = payUbizApiLayer
        convenienceFeeResponse = checkoutDetailsResponse
        val orderedList = getFinalCheckoutOrder(
            context,
            if (!isSIMode) checkoutOptionsList else null,
            convenienceFeeResponse
        )
        /*if (!isSIMode)
            addLazyPayOption(checkoutDetailsResponse, orderedList)
        else*/ if (!CommonUtils.isValidAmountForSI(
                payuBizParams?.amount?.toDoubleOrNull(),
                checkoutDetailsResponse
            )
        ) {
            val baseTransactionListener = payUbizApiLayer.baseTransactionListener
            baseTransactionListener?.showProgressDialog(false)
            baseTransactionListener?.onError(ErrorResponse().apply {
                errorMessage = context.getString(R.string.payu_please_check_the_transaction_amount)
            })
        }
        if (checkoutDetailsResponse.isSodexoAvailable && !isSIMode) {
            addSodexoOption(checkoutDetailsResponse, orderedList)
        } else if (!CommonUtils.isValidAmountForSI(
                payuBizParams?.amount?.toDoubleOrNull(),
                checkoutDetailsResponse
            )
        )
            sendErrorMessage(
                payUbizApiLayer,
                context.getString(R.string.payu_please_check_the_transaction_amount)
            )

        for (item in orderedList) {
            when (item.type) {
                PaymentType.L1_OPTION -> {

                    item.l1OptionSubText =
                        CommonUtils.getL1OptionText(context, item, checkoutDetailsResponse)

                    paymentModesList.add(item)
                }
                PaymentType.CARD -> addCardOption(
                    context,
                    checkoutDetailsResponse,
                    paymentModesList
                )
                PaymentType.UPI -> addUPIOption(checkoutDetailsResponse, context, paymentModesList)
                PaymentType.NB -> addNBOption(context, checkoutDetailsResponse, paymentModesList)
                PaymentType.WALLET -> if (!isSIMode) addWalletOption(
                    context,
                    checkoutDetailsResponse,
                    paymentModesList
                )
                PaymentType.EMI -> if (!isSIMode) addEMIOption(
                    context,
                    checkoutDetailsResponse,
                    paymentModesList
                )
                PaymentType.NEFTRTGS -> if (!isSIMode) addNEFTRTGSOption(
                    context,
                    checkoutDetailsResponse,
                    paymentModesList
                )
                PaymentType.BNPL -> if (!isSIMode) addBNPLOption(
                    context,
                    checkoutDetailsResponse,
                    paymentModesList
                )
            }
        }

        moreOptionsList = paymentModesList
        if (moreOptionsList.isNullOrEmpty() || moreOptionsList!![0].optionDetail.isNullOrEmpty()) {
            sendErrorMessage(
                payUbizApiLayer,
                context.getString(R.string.payu_payment_mode_not_enabled_on_merchant_key)
            )
        } else {
            fetchConvenienceFees()
        }
        return paymentModesList
    }

    private fun fetchConvenienceFees() {
        if (SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams == null && InternalConfig.isPricingCFCall == true) {
            val variableDynamicList: ArrayList<VariableDynamics> = ArrayList()
            val upiPaymentOptions = moreOptionsList?.filter { it.type == PaymentType.UPI }
            val isUpiEnforced = enforcePaymentList?.filter { it.get(PayUCheckoutProConstants.CP_PAYMENT_TYPE) == "UPI"  }
            if (upiPaymentOptions != null && (enforcePaymentList == null || isUpiEnforced?.isNotEmpty() == true)) {
                for (option in moreOptionsList ?: listOf()) {
                    if (option.type == PaymentType.UPI) {
                        val list = option.optionDetail
                        list?.forEach { paymentOption ->
                            (paymentOption.otherParams as? HashMap<String, Any>)?.let { it1 ->
                                val bankCode = Utils.getBankCodeFromMap(
                                    it1
                                )
                                val variableDynamics = VariableDynamics(
                                    SdkUiConstants.CP_UPI.plus("_$bankCode"),
                                    bankCode
                                )
                                variableDynamicList.add(variableDynamics)
                            }
                        }
                    } else if (option.paymentId?.equals(SdkUiConstants.CP_TEZ) == true) {
                        val variableDynamics = VariableDynamics(
                            SdkUiConstants.CP_UPI.plus("_${SdkUiConstants.CP_TEZ}"),
                            SdkUiConstants.CP_TEZ
                        )
                        variableDynamicList.add(variableDynamics)
                    }
                }
                if (variableDynamicList.isNotEmpty())
                    SdkUiInitializer.apiLayer?.fetchConvenienceFee(
                        SdkUiConstants.CP_UPI,
                        variableDynamicList
                    ) {}
            }
        }
    }

    private fun sendErrorMessage(
        payUbizApiLayer: PayUbizApiLayer,
        message: String
    ) {
        val baseTransactionListener = payUbizApiLayer.baseTransactionListener
        baseTransactionListener?.showProgressDialog(false)
        baseTransactionListener?.onError(ErrorResponse().apply {
            errorMessage = message
        })
    }

    internal fun getFinalCheckoutOrder(
        context: Context?,
        checkoutOptionsList: ArrayList<PaymentMode>?,
        checkoutDetailsResponse: PayuResponse?
    ): ArrayList<PaymentMode> {

        val defaultOrderedList = ArrayList<PaymentMode>()
        var orderedList = ArrayList<PaymentMode>()
        var enforcedList = ArrayList<PaymentMode>()
        val uniqueList = ArrayList<PaymentMode>()

        for (item in CommonUtils.getDefaultCheckoutOrder()) {
            defaultOrderedList.add(item)
        }

        if (!checkoutOptionsList.isNullOrEmpty()) {
            //Payment Mode list prepared based on the order provided by merchant
            for (item in checkoutOptionsList) {

                //Do not add item in list if null is passed as payment type
                if (item.type == null)
                    continue

                if (!item.paymentId.isNullOrEmpty())
                    addL1Option(
                        context,
                        item.paymentId!!,
                        item.type!!,
                        orderedList,
                        checkoutDetailsResponse
                    )
                else orderedList.add(item)
            }

            for (item in defaultOrderedList) {
                if (!orderedList.contains(item))
                    orderedList.add(item)
            }
        } else {
            //Default order list if merchant does not provides any order
            orderedList = defaultOrderedList
        }

        //List updated based on whether or not merchant provides enforced list
        if (!enforcePaymentList.isNullOrEmpty()) {
            for (item in orderedList) {
                val keyMap: HashMap<String, Any?> = HashMap()
                keyMap[PayUCheckoutProConstants.CP_PAYMENT_TYPE] = item.type!!.name
                if (item.type != null && !CommonUtils.isKeyEnforced(keyMap) && !isRetryTxn) continue
                enforcedList.add(item)
            }
        } else {
            enforcedList = orderedList
        }

        //Remove duplicates from the list
        uniqueList.addAll(enforcedList.distinctBy { Pair(it.type, it.paymentId) })
        return uniqueList
    }

    internal fun addCardOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {
        if (isSIMode && (checkoutDetailsResponse.standingInstructions == null || (!checkoutDetailsResponse.isCreditCardAvailableFoSI && !checkoutDetailsResponse.isDebitCardAvailableFoSI)))
            return

        if (!isSIMode && (!checkoutDetailsResponse.isCreditCardAvailable && !checkoutDetailsResponse.isDebitCardAvailable))
            return

        payuBizParams?.amount?.toDoubleOrNull()?.let {
            //For UPISI, min Rs 1 txn is allowed
            if (isSIMode && it < 1.0) return
        }

        val cardMode = PaymentMode()
        cardMode.type = PaymentType.CARD

        cardMode.l1OptionSubText =
            CommonUtils.getL1OptionText(context, cardMode, checkoutDetailsResponse)

        cardMode.name =
            CommonUtils.getL1OptionHeaderText(context, cardMode, checkoutDetailsResponse)

        val paymentOptionsList = ArrayList<PaymentOption>()
        val map = HashMap<String, Any>()
        if (isSIMode) {
            if (checkoutDetailsResponse.isCreditCardAvailableFoSI) {
                for (card in checkoutDetailsResponse.standingInstructions) {
                    val cardOption: CardOption =
                        prepareCardOption(card, CardType.CC, PayuConstants.CC)
                    paymentOptionsList.add(cardOption)
                    map[card.bankCode] = cardOption
                }
            }

            if (checkoutDetailsResponse.isDebitCardAvailableFoSI) {
                for (card in checkoutDetailsResponse.standingInstructions) {
                    val cardOption: CardOption =
                        prepareCardOption(card, CardType.DC, CardType.DC.name)
                    paymentOptionsList.add(cardOption)
                    map[card.bankCode] = cardOption
                }
            }
        } else {
            if (checkoutDetailsResponse.isCreditCardAvailable) {
                for (card in checkoutDetailsResponse.creditCard) {
                    val cardOption: CardOption =
                        prepareCardOption(card, CardType.CC, PayuConstants.CC)
                    paymentOptionsList.add(cardOption)
                    map[card.bankCode] = cardOption
                }
            }

            if (checkoutDetailsResponse.isDebitCardAvailable) {
                for (card in checkoutDetailsResponse.debitCard) {
                    val cardOption: CardOption =
                        prepareCardOption(card, CardType.DC, PayuConstants.CC)
                    paymentOptionsList.add(cardOption)
                    map[card.bankCode] = cardOption
                }
            }
            paymentOptionMap[PaymentType.CARD] = map
        }
        cardMode.optionDetail = paymentOptionsList
        paymentModesList.add(cardMode)
    }

    private fun prepareCardOption(
        card: PaymentDetails,
        cardType: CardType,
        pg: String
    ): CardOption {
        val cardOption: CardOption =
            preparePaymentOption(
                CardOption().apply { paymentType = PaymentType.CARD },
                card.bankName,
                card.bankCode,
                pg,
                card.imageURL,
                card.category,
                card.imageUpdatedOn,
            )

        val cardBinInfo = CardBinInfo()
        cardBinInfo.cardType = cardType
        cardOption.cardBinInfo = cardBinInfo
        return cardOption
    }

    internal fun addUPIOption(
        checkoutDetailsResponse: PayuResponse,
        context: Context?,
        paymentModesList: ArrayList<PaymentMode>
    ) {

        val map = HashMap<String, Any>()
        //Add UPI if available on merchant credentials
        if (isSIMode && (checkoutDetailsResponse.isUPIAvailableFoSI || checkoutDetailsResponse.isUPIIntentAvailableForSI || checkoutDetailsResponse.isUpiAvailableForOtm || checkoutDetailsResponse.isUpiIntentAvailableForOtm)) {
            payuBizParams?.amount?.toDoubleOrNull()?.let {
                //For UPISI, min Rs 1 txn is allowed
                if (it < 1.0) return
            }
            val upiMode = PaymentMode()
            upiMode.name = context?.getString(R.string.upi_header_l1_list).toString()
            upiMode.type = PaymentType.UPI
            val paymentOptionsList = ArrayList<PaymentOption>()

            val upiSupportedHandles =
                checkoutDetailsResponse.upiSISupportedHandles ?: DataConstant.supportedUpiHandles() //TODO will remove default list after BE changes
            if ((checkoutDetailsResponse.isUPIAvailableFoSI || checkoutDetailsResponse.isUpiAvailableForOtm) && upiSupportedHandles != null) {
                val upiCollectOption: UPIOption = prepareUPIPaymentOption()
                map[PayuConstants.UPI.uppercase()] = upiCollectOption
                upiCollectOption.upiHandles = upiSupportedHandles
                map[PayuConstants.UPI.uppercase()] = upiCollectOption
                paymentOptionsList.add(upiCollectOption)
            }

            val upiAppsList: ArrayList<PaymentOption>? =
                AndroidUtils.getInstalledUpiAppsList(context)

            if ((checkoutDetailsResponse.isUPIIntentAvailableForSI || checkoutDetailsResponse.isUpiIntentAvailableForOtm) && (upiAppsList != null) && (upiAppsList.size > 0)) {
                val upiIntentOption = prepareUPIGenericIntentPaymentOption(
                    imageURL = checkoutDetailsResponse.upi?.imageURL,
                    isBankDown = false,
                    additionalCharges = 0.0,
                    gst = 0.0
                )
                val upiIntentAppsList = ArrayList<PaymentOption>()

                val upiAppsMap: HashMap<String, PaymentOption> = HashMap()
                for (upiOption in upiAppsList) {
                    upiAppsMap[upiOption.bankName.lowercase()] = upiOption
                }

                for (upiSISupportedApp in InternalConfig.supportedUpiSiIntentAppList
                    ?: DataConstant.supportedUpiSiIntentAppList()) {
                    if (upiAppsMap.containsKey(upiSISupportedApp.lowercase())) {
                        val upiApp: UPIOption = prepareUPIGenericIntentPaymentOption(
                            paymentOption = (upiAppsMap[upiSISupportedApp.lowercase()] as UPIOption).apply {
                                paymentType = PaymentType.UPI_INTENT
                            },
                            bankName = upiAppsMap[upiSISupportedApp.lowercase()]?.bankName!!,
                        )
                        InternalConfig.userAccountInfoList?.let {
                            upiApp.ifscCode = CommonUtils.getIfscCodesForPayment(it)
                            upiApp.accountNumber = CommonUtils.getAccountsForPayment(it)
                        }
                        upiIntentAppsList.add(upiApp)
                    }
                }
                val upiTopApps = DataConstant.upiTopAppsDescending()

                Collections.sort(upiIntentAppsList, object : Comparator<PaymentOption?> {
                    override fun compare(left: PaymentOption?, right: PaymentOption?): Int {
                        return upiTopApps.indexOf(right?.bankName).compareTo(upiTopApps.indexOf(left?.bankName))
                    }
                })
                upiIntentOption.optionList = upiIntentAppsList
                map[PayuConstants.UPI_INTENT.uppercase()] = upiIntentOption
                paymentOptionsList.add(upiIntentOption)
            }

            upiMode.optionDetail = paymentOptionsList
            if (!checkoutDetailsResponse.isUPIAvailableFoSI && checkoutDetailsResponse.isUPIIntentAvailableForSI && upiAppsList.isNullOrEmpty()) {
                upiMode.isPaymentModeDown = true
            }
            paymentModesList.add(upiMode)
        } else if (!isSIMode && (checkoutDetailsResponse.isUpiAvailable || checkoutDetailsResponse.isGenericIntentAvailable)) {
            val upiMode = PaymentMode()
            upiMode.name = context?.getString(R.string.upi_header_l1_list).toString()
            upiMode.type = PaymentType.UPI
            val paymentOptionsList = ArrayList<PaymentOption>()
            if (checkoutDetailsResponse.isGoogleTezOmniAvailable) {
                val upiCollectOption: UPIOption = prepareUPIPaymentOption(
                    bankCode = PayuConstants.TEZOMNI,
                    isBankDown = checkoutDetailsResponse.googleTezOmni.isBankDown,
                    additionalCharges = checkoutDetailsResponse.googleTezOmni.additionalCharge.toDoubleOrNull()
                )
                map[PayuConstants.TEZOMNI.uppercase()] = upiCollectOption
                paymentOptionsList.add(upiCollectOption)
            }
            if (checkoutDetailsResponse.isUpiAvailable) {
                val upiCollectOption: UPIOption = prepareUPIPaymentOption()
                upiCollectOption.upiHandles = DataConstant.upihandles()
                paymentOptionsList.add(upiCollectOption)
                map[PayuConstants.UPI.uppercase()] = upiCollectOption
            }

            var upiAppsList: ArrayList<PaymentOption>? = null
            if (checkoutDetailsResponse.isGenericIntentAvailable) {

                upiAppsList = AndroidUtils.getInstalledUpiAppsList(context)

                val upiApps = checkoutDetailsResponse.merchantInfo.merchantParamInfo.upiApps
                upiAppsList?.forEachIndexed { index, paymentOption ->
                    if (paymentOption is UPIOption) {
                        val upiOption = paymentOption as UPIOption

                        run breaking@{
                            upiApps?.forEach { upiApp ->
                                if (upiApp.androidBundleIdentifier.isNotEmpty() && upiOption.packageName.contains(upiApp.androidBundleIdentifier)) {
                                    upiOption.appName = upiApp.appName
                                    upiAppsList[index] = upiOption
                                    return@breaking
                                }
                            }
                        }
                    }
                }

                if (upiAppsList != null && upiAppsList.size > 0) {

                    val upiIntentOption: UPIOption = prepareUPIGenericIntentPaymentOption(
                        imageURL = checkoutDetailsResponse.upi?.imageURL,
                        isBankDown = false,
                        additionalCharges = 0.0,
                        gst = 0.0
                    )

                    val upiIntentAppsList = ArrayList<PaymentOption>()

                    for (upiOption in upiAppsList) {
                        val upiApp: UPIOption = prepareUPIGenericIntentPaymentOption(
                            paymentOption = (upiOption as UPIOption).apply {
                                paymentType = PaymentType.UPI_INTENT
                            },
                            bankName = upiOption.bankName,
                            isBankDown = false
                        )
                        InternalConfig.userAccountInfoList?.let {
                            upiApp.ifscCode = CommonUtils.getIfscCodesForPayment(it)
                            upiApp.accountNumber = CommonUtils.getAccountsForPayment(it)
                        }
                        upiIntentAppsList.add(upiApp)
                    }

                    //sort list
                    upiIntentAppsList.sortWith { o1, o2 ->
                        o1.bankName.compareTo(
                            o2.bankName,
                            ignoreCase = true
                        )
                    }

                    val sdkInitUPIOrderList = SdkUiInitializer.apiLayer?.config?.upiAppsOrder
                    val upiAppOrderList = if (!sdkInitUPIOrderList.isNullOrEmpty()) sdkInitUPIOrderList else ""
                    val upiTopApps = DataConstant.upiTopAppsDescending(upiAppOrderList.lowercase())

                    Collections.sort(upiIntentAppsList, object : Comparator<PaymentOption?> {
                        override fun compare(left: PaymentOption?, right: PaymentOption?): Int {
                            return upiTopApps.indexOf(right?.bankName?.lowercase()).compareTo(upiTopApps.indexOf(left?.bankName?.lowercase()))
                        }
                    })

                    upiIntentOption.optionList = upiIntentAppsList
                    paymentOptionsList.add(upiIntentOption)
                    map[PayuConstants.UPI_INTENT.uppercase()] = upiIntentOption
                }
            }
            upiMode.optionDetail = paymentOptionsList


            //set UPI option down if only intent is enabled but no upi apps are installed on device
            if (!checkoutDetailsResponse.isUpiAvailable && checkoutDetailsResponse.isGenericIntentAvailable && upiAppsList.isNullOrEmpty()) {
                upiMode.isPaymentModeDown = true
            }

            paymentModesList.add(upiMode)
        }

        paymentOptionMap[PaymentType.UPI]?.forEach { (key, value) ->
            map[key]  = value
        }
        paymentOptionMap[PaymentType.UPI] = map
    }

    internal fun prepareUPIPaymentOption(
        paymentOption: UPIOption = UPIOption().apply {
            paymentType = PaymentType.UPI
            InternalConfig.userAccountInfoList?.let {
                ifscCode = CommonUtils.getIfscCodesForPayment(it)
                accountNumber = CommonUtils.getAccountsForPayment(it)
            }
        },
        bankName: String = PayuConstants.UPI,
        bankCode: String = PayuConstants.UPI,
        pg: String = PayuConstants.UPI,
        imageURL: String? = convenienceFeeResponse?.upi?.imageURL,
        category: String? = convenienceFeeResponse?.upi?.category,
        imageUpdatedOn: Long? = convenienceFeeResponse?.upi?.imageUpdatedOn,
        isBankDown: Boolean = convenienceFeeResponse?.upi?.isBankDown == true,
        additionalCharges: Double? = convenienceFeeResponse?.upi?.additionalCharge?.toDoubleOrNull(),
        gst: Double? = convenienceFeeResponse?.taxSpecification?.upiTaxValue?.toDoubleOrNull(),
    ): UPIOption {
        val upiOption: UPIOption = preparePaymentOption(
            paymentOption,
            bankName,
            bankCode,
            pg,
            imageURL,
            category,
            imageUpdatedOn,
            isBankDown,
            additionalCharges,
            gst
        )
        return upiOption
    }

    private fun prepareUPIGenericIntentPaymentOption(
        paymentOption: UPIOption = UPIOption().apply {
            paymentType = PaymentType.UPI_INTENT
        },
        bankName: String = PayuConstants.UPI,
        bankCode: String = PayuConstants.UPI_INTENT,
        pg: String = PayuConstants.UPI,
        imageURL: String? = convenienceFeeResponse?.genericIntent?.imageURL,
        category: String? = convenienceFeeResponse?.genericIntent?.category,
        imageUpdatedOn: Long? = convenienceFeeResponse?.genericIntent?.imageUpdatedOn,
        isBankDown: Boolean = convenienceFeeResponse?.genericIntent?.isBankDown == true,
        additionalCharges: Double? = convenienceFeeResponse?.genericIntent?.additionalCharge?.toDoubleOrNull(),
        gst: Double? = convenienceFeeResponse?.taxSpecification?.upiTaxValue?.toDoubleOrNull(),
    ): UPIOption {
        val upiOption: UPIOption = preparePaymentOption(
            paymentOption,
            bankName,
            bankCode,
            pg,
            imageURL,
            category,
            imageUpdatedOn,
            isBankDown,
            additionalCharges,
            gst
        )
        return upiOption
    }

    internal fun addNBOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {

        if (isSIMode && !checkoutDetailsResponse.isNBAvailableFoSI)
            return

        if (!isSIMode && !checkoutDetailsResponse.isNetBanksAvailable)
            return

        //Add Net banking, if available on merchant credentials
        val nbMode = PaymentMode()
        nbMode.name = context?.getString(R.string.cp_netbanking).toString()
        nbMode.type = PaymentType.NB
        val optionsList =
            if (isSIMode) checkoutDetailsResponse.siBankList else checkoutDetailsResponse.netBanks
        val NBOptionList = getEnforcementListForPaymentOption(optionsList, PaymentType.NB)
        val paymentOptionsList = ArrayList<PaymentOption>()
        val map = HashMap<String, Any>()
        if (!NBOptionList.isNullOrEmpty()) {
            for (option in NBOptionList) {
                val gst: Double? =
                    if (isSIMode) checkoutDetailsResponse.taxSpecification?.enachTaxValue?.toDoubleOrNull() else checkoutDetailsResponse.taxSpecification?.nbTaxValue?.toDoubleOrNull()
                var priority: Int = 0
                priority = priorityGenerator(option.bankCode)
                val nbOption: NBOptions = preparePaymentOption(
                    NBOptions().apply {
                        paymentType = PaymentType.NB
                    },
                    option.bankName,
                    option.bankCode,
                    PayuConstants.NB,
                    option.imageURL,
                    option.category,
                    option.imageUpdatedOn,
                    option.isBankDown,
                    option.additionalCharge?.toDoubleOrNull(),
                    gst,
                    priority,
                    option.isSecureWebview
                )
                map[option.bankCode] = nbOption
                nbOption.verificationModeList = getVerificationModeMapping(option.verificationModeList)
                if (!InternalConfig.userAccountInfoList.isNullOrEmpty()) {
                    InternalConfig.userAccountInfoList?.forEach {
                        if (it.accountDetails?.beneficiaryIfsc?.startsWith(
                                option.ifscCode, ignoreCase = true
                            ) == true
                        ) {
                            val nbUpdatedOption = nbOption.clone() as NBOptions
                            /*
                                its already NB + TPV and if it isSIMode also then it is Enach +TPV and for Enach + TPV,
                                beneficiaryName and beneficiaryAccountType are mandatory
                            */
                            if (isSIMode) {
                                nbUpdatedOption.imageURL = it.imageUrl
                                if (it.accountDetails?.beneficiaryName != null && it.accountDetails?.beneficiaryAccountType != null) {
                                    nbUpdatedOption.beneficiaryDetails = it.accountDetails
                                    it.accountDetails?.verificationMode?.let { mode ->
                                        if (nbOption.verificationModeList?.contains(mode) != true) {
                                            nbUpdatedOption.beneficiaryDetails?.verificationMode =
                                                null
                                        }
                                    }
                                    paymentOptionsList.add(nbUpdatedOption)
                                }
                            } else {
                                nbUpdatedOption.accountNumber = it.accountDetails?.beneficiaryAccountNumber
                                nbUpdatedOption.ifscCode = it.accountDetails?.beneficiaryIfsc?.uppercase()
                                paymentOptionsList.add(nbUpdatedOption)
                            }
                        }
                    }
                } else {
                    paymentOptionsList.add(nbOption)

                }
            }
            paymentOptionMap[PaymentType.NB] = map
        }

        val sortedList = sortPaymentOptionListBasedOnPriority(paymentOptionsList)
        nbMode.optionDetail = ArrayList(sortedList)
        nbMode.l1OptionSubText =
            CommonUtils.getL1OptionText(context, nbMode, checkoutDetailsResponse)
        if (!nbMode.optionDetail.isNullOrEmpty())
            paymentModesList.add(nbMode)
    }

    private fun priorityGenerator(bankCode: String): Int {
        when (bankCode) {
            PayUCheckoutProConstants.SBIB, PayUCheckoutProConstants.SBINENCC -> return 6
            PayUCheckoutProConstants.HDFB, PayUCheckoutProConstants.HDFCENCC -> return 5
            PayUCheckoutProConstants.ICIB, PayUCheckoutProConstants.ICICENCC -> return 4
            PayUCheckoutProConstants.AXIB, PayUCheckoutProConstants.UTIBENCC -> return 3
            PayUCheckoutProConstants.PNBB, PayUCheckoutProConstants.PUNBENCC -> return 2
            PayUCheckoutProConstants.PAYU_162B, PayUCheckoutProConstants.KKBKENCC -> return 1
            else -> return 0
        }
    }


    internal fun addWalletOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {
        //Add Cash Cards if available on merchant credentials
        if (checkoutDetailsResponse.isCashCardAvailable) {
            val cashCardMode = PaymentMode()
            cashCardMode.name = PayUCheckoutProConstants.CP_WALLETS
            cashCardMode.type = PaymentType.WALLET
            var paymentOptionsList = ArrayList<PaymentOption>()
            val unSupportedBankCodesList = CommonUtils.getUnSupportedBankCodesList()
            val walletList = checkoutDetailsResponse.cashCard
            val walletOptionList =
                getEnforcementListForPaymentOption(walletList, PaymentType.WALLET)
            val map = HashMap<String, Any>()
            for (wallet in walletOptionList) {

                if (unSupportedBankCodesList.contains(wallet.bankCode) || ((android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.N) && (wallet.bankCode.equals(
                        PayUCheckoutProConstants.PHONEPE
                    )))
                ) continue

                val pgAppVersion = AndroidUtils.doesPhonePeAppExist(context)
                if (wallet.bankCode.equals(PayUCheckoutProConstants.CP_PPSDKLES) && pgAppVersion.isNullOrEmpty()) continue
                val paymentOption =
                    if (wallet.category.equals(PaymentState.OLW.name, ignoreCase = true))
                        OlwPaymentOption().apply { paymentType = PaymentType.OPEN_LOOP_WALLET }
                    else WalletOption().apply { paymentType = PaymentType.WALLET }
                val walletOption = preparePaymentOption(
                    paymentOption,
                    wallet.bankName,
                    wallet.bankCode,
                    PayuConstants.CASH,
                    wallet.imageURL,
                    wallet.category,
                    wallet.imageUpdatedOn,
                    CommonUtils.getBankDownStatusForBankCode(
                        wallet.bankCode,
                        checkoutDetailsResponse.cashCard
                    ),
                    CommonUtils.getAdditionalChargesForBankCode(
                        wallet.bankCode,
                        checkoutDetailsResponse.cashCard
                    ),
                    checkoutDetailsResponse.taxSpecification?.cashTaxValue?.toDoubleOrNull(),
                    0,
                    wallet.isSecureWebview,
                    pgAppVersion
                )
                map[wallet.bankCode] = walletOption
                paymentOptionsList.add(walletOption)
            }
            if (map.contains(PayUCheckoutProConstants.CP_PPSDKLES) && map.contains(
                    PayUCheckoutProConstants.PHONEPE
                )
            ) {
                paymentOptionsList = paymentOptionsList.filter {
                    it.bankName != PayUCheckoutProConstants.CP_PHONEPE
                } as ArrayList<PaymentOption>
            }
            paymentOptionMap[PaymentType.WALLET]?.forEach { (key, value) ->
                map[key] = value
            }
            paymentOptionMap[PaymentType.WALLET] = map
            cashCardMode.optionDetail = paymentOptionsList
            cashCardMode.l1OptionSubText =
                CommonUtils.getL1OptionText(context, cashCardMode, checkoutDetailsResponse)
            if (!walletOptionList.isNullOrEmpty())
                paymentModesList.add(cashCardMode)
        }
    }



    /**
     * Function to prepare BNPL payment mode list
     */

    internal fun addBNPLOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {
        if (checkoutDetailsResponse.isBnplAvailable) {
            val bnplMode = PaymentMode()
            bnplMode.name = PayUCheckoutProConstants.CP_BNPL
            bnplMode.type = PaymentType.BNPL
            val paymentOptionsList = ArrayList<PaymentOption>()

            val bnplList = checkoutDetailsResponse.bnpl
            val map = HashMap<String, Any>()
            for (bnpl in bnplList!!) {
                val bnplOption = preparePaymentOption(
                    BnplOption().apply { paymentType = PaymentType.BNPL },
                    bnpl.bankName,
                    bnpl.bankCode,
                    PayuConstants.BNPL,
                    bnpl.imageURL,
                    bnpl.category,
                    bnpl.imageUpdatedOn,
                    false,
                    bnpl.additionalCharge.toDoubleOrNull(),
                    checkoutDetailsResponse.taxSpecification?.bnplTaxValue?.toDoubleOrNull()
                )
                val bankDownText = bnpl.additionalCharge.toDoubleOrNull()?.let {
                    getBankDownText(
                        context, bnpl.minAmount.toDoubleOrNull(), bnpl.maxAmount.toDoubleOrNull(),
                        it
                    )
                }
                if (!bankDownText.isNullOrEmpty()) {
                    bnplOption.subText = bankDownText
                    bnplOption.isBankDown = true
                }
                bnplOption.isEligible = bnpl.status
                map[bnpl.bankCode] = bnplOption
                paymentOptionsList.add(bnplOption)
            }
            paymentOptionMap[PaymentType.BNPL] = map
            this.bnplList = paymentOptionsList
            bnplMode.optionDetail = paymentOptionsList
            bnplMode.l1OptionSubText =
                CommonUtils.getL1OptionText(context, bnplMode, checkoutDetailsResponse)
            if (!bnplList.isNullOrEmpty())
                paymentModesList.add(bnplMode)
        }
    }

    internal fun addEMIOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {
        //Add EMI, if available on merchant credentials
        if (checkoutDetailsResponse.isCCEmiAvailable
            || checkoutDetailsResponse.isDCEmiAvailable
            || checkoutDetailsResponse.isCardLessEmiAvailable
        ) {
            val emiMode = PaymentMode()
            emiMode.name = "EMI"
            emiMode.type = PaymentType.EMI
            val paymentOptionsList = ArrayList<PaymentOption>()
            if (checkoutDetailsResponse.isCCEmiAvailable)
                prepareEmiOption(
                    context,
                    checkoutDetailsResponse.ccemi,
                    EmiType.CC,
                    checkoutDetailsResponse.taxSpecification?.emiTaxValue?.toDoubleOrNull(),
                    paymentOptionsList
                )

            if (checkoutDetailsResponse.isDCEmiAvailable)
                prepareEmiOption(
                    context,
                    checkoutDetailsResponse.dcemi,
                    EmiType.DC,
                    checkoutDetailsResponse.taxSpecification?.emiTaxValue?.toDoubleOrNull(),
                    paymentOptionsList
                )

            if (checkoutDetailsResponse.isCardLessEmiAvailable)
                prepareEmiOption(
                    context,
                    checkoutDetailsResponse.cardlessemi,
                    EmiType.CARD_LESS,
                    checkoutDetailsResponse.taxSpecification?.emiTaxValue?.toDoubleOrNull(),
                    paymentOptionsList
                )

            emiList = paymentOptionsList
            emiMode.optionDetail = paymentOptionsList
            emiMode.l1OptionSubText =
                CommonUtils.getL1OptionText(context, emiMode, checkoutDetailsResponse)
            if (!emiMode.optionDetail.isNullOrEmpty())
                paymentModesList.add(emiMode)
        }
    }

    internal fun addNEFTRTGSOption(
        context: Context?,
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {

        if (checkoutDetailsResponse.isNEFTRTGSAvailable) {
            val neftRtgsMode = PaymentMode()
            neftRtgsMode.name = "NEFT/RTGS"
            neftRtgsMode.type = PaymentType.NEFTRTGS
            val paymentOptionsList = ArrayList<PaymentOption>()
            val neftRtgsList = getEnforcementListForPaymentOption(
                checkoutDetailsResponse.neftRtgs,
                PaymentType.NEFTRTGS
            )
            for (neftlist in neftRtgsList) {
                val neftRtgsOption = preparePaymentOption(
                    PaymentOption().apply { paymentType = PaymentType.NEFTRTGS },
                    neftlist.bankName,
                    neftlist.bankCode,
                    PayuConstants.NEFT_RTGS,
                    neftlist.imageURL,
                    neftlist.category,
                    neftlist.imageUpdatedOn,
                    false,
                    neftlist.additionalCharge.toDoubleOrNull(),
                    checkoutDetailsResponse.taxSpecification?.neftRtgsTaxValue?.toDoubleOrNull(),
                    0,
                    neftlist.isSecureWebview
                )
                paymentOptionsList.add(neftRtgsOption)
            }
            neftRtgsMode.optionDetail = paymentOptionsList
            neftRtgsMode.l1OptionSubText =
                CommonUtils.getL1OptionText(context, neftRtgsMode, checkoutDetailsResponse)
            if (!neftRtgsMode.optionDetail.isNullOrEmpty())
                paymentModesList.add(neftRtgsMode)
            paymentOptionsList
        }
    }

    /**
     * This method prepares EMI option and sets in paymentOptionList passed in argument to the function
     * EMI Option will have below structure -
     * |-- Credit Card |-- Bank1      |-- Tenure1
     *                                |-- Tenure2
     *                                |-- Tenure3
     *                                ...
     *
     *                  |-- Bank2      |-- Tenure1
     *                                 |-- Tenure2
     *                                 |-- Tenure3
     *                                 ...
     * |-- Debit Card   |-- Bank1      |-- Tenure1
     *                                 |-- Tenure2
     *                                 |-- Tenure3
     *                                 ...
     *
     * @param context Context
     * @param emiList list of EMIs i.e ccEmi, dcEmi, etc
     * @param emiType emiType like, EmiType.CC, etc
     * @param tax tax info for EMI
     * @param paymentOptionsList paymentOptions list to be updated
     * */
    internal fun prepareEmiOption(
        context: Context?,
        emiList: ArrayList<Emi>,
        emiType: EmiType,
        tax: Double?,
        paymentOptionsList: ArrayList<PaymentOption>
    ) {
        var paymentOptionsList1 = ArrayList<PaymentOption>()
        var tenureList: ArrayList<PaymentOption>? = null
        val emiOption = EMIOption()
        when (emiType) {
            EmiType.CC -> {
                emiOption.bankShortName = PayUCheckoutProConstants.CP_CC
                emiOption.bankName = PayUCheckoutProConstants.CP_CREDIT_CARD
            }
            EmiType.DC -> {
                emiOption.bankShortName = PayUCheckoutProConstants.CP_DC
                emiOption.bankName = PayUCheckoutProConstants.CP_DEBIT_CARD
            }
            EmiType.CARD_LESS -> {
                emiOption.bankShortName = PayUCheckoutProConstants.CP_OTHER
                emiOption.bankName = PayUCheckoutProConstants.CP_CARDLESS
            }
        }

        emiOption.emiType = emiType
        emiOption.paymentType = PaymentType.EMI

        for (item in emiList) {
            if (item.bankTitle.isEmpty()) continue
            val bankEmiOption = preparePaymentOption(
                EMIOption().apply {
                    bankShortName = when (emiType) {
                        EmiType.CC -> CommonUtils.getBankShortNameForCC(item.shortTitle)
                        EmiType.DC -> CommonUtils.getBankShortNameForDC(item.shortTitle)
                        EmiType.CARD_LESS -> CommonUtils.getBankShortNameForCardless(item.shortTitle)
                    }
                    paymentType = PaymentType.EMI
                },
                item.bankTitle,
                item.bankCode,
                PayuConstants.EMI,
                item.imageURL,
                item.category,
                item.imageUpdatedOn
            )
            bankEmiOption.priority = priorityGeneratorForEMI(bankEmiOption.bankShortName)
            bankEmiOption.emiType = emiType
            bankEmiOption.gstPercentageValue = tax
            bankEmiOption.isBankOption = true
            val bankDownText = getBankDownText(
                context,
                item.minAmount.toDoubleOrNull(),
                item.maxAmount.toDoubleOrNull(),
                item.payUEmiTenuresList
            )
            if (!bankDownText.isNullOrEmpty()) {
                bankEmiOption.isBankDown = true
                bankEmiOption.subText = bankDownText
            }
            if (emiType == EmiType.CARD_LESS) {
                emiOption.bankShortName = CommonUtils.getBankEmiTypeForCardLess(item.bankName)
            }

            //This returns a pair  which contains tenure list and the lowest interest rate among all the tenures in the particular bank
            val tenureListWithLowestInterestRate = prepareTenureList(
                item.payUEmiTenuresList,
                item.bankName,
                emiOption.bankShortName,
                bankEmiOption
            )
            tenureList = tenureListWithLowestInterestRate.first
            bankEmiOption.lowestInterestRate = tenureListWithLowestInterestRate.second
            updateEnforcedTenureList(bankEmiOption, tenureList)
            if (!bankEmiOption.optionList.isNullOrEmpty())
                paymentOptionsList1.add(bankEmiOption)
        }

        val sortedList = sortPaymentOptionListBasedOnPriority(paymentOptionsList1)
        val listOfPaymentOption: ArrayList<PaymentOption> = ArrayList(sortedList.size)
        listOfPaymentOption.addAll(sortedList)
        emiOption.optionList = listOfPaymentOption
        if (!emiOption.optionList.isNullOrEmpty())
            paymentOptionsList.add(emiOption)
    }

    /**
     * This function sorts the payment option arrayList based on the priority and if same priority occurs, sorting is done based on alphabetical order
     */
    private fun sortPaymentOptionListBasedOnPriority(paymentOptionsList1: ArrayList<PaymentOption>): List<PaymentOption> {
        return paymentOptionsList1.sortedWith { p0, p1 ->
            if (p0.priority == p1.priority) {
                p0.bankName.compareTo(p1.bankName)
            } else if (p0.priority > p1.priority) {
                -1
            } else {
                1
            }
        }
    }

    /**
     *Function to generate priority for top banks in EMI
     **/
    private fun priorityGeneratorForEMI(bankCode: String?): Int {
        return when (bankCode) {
            PayUCheckoutProConstants.SBI_CC, PayUCheckoutProConstants.SBI_DC -> 5
            PayUCheckoutProConstants.HDFC_CC, PayUCheckoutProConstants.HDFC_DC -> 4
            PayUCheckoutProConstants.AXIS_CC, PayUCheckoutProConstants.AXIS_DC -> 3
            PayUCheckoutProConstants.ICICI_CC, PayUCheckoutProConstants.ICICI_DC -> 2
            PayUCheckoutProConstants.KOTAK_CC, PayUCheckoutProConstants.KOTAK_DC -> 1
            else -> 0
        }
    }


    /**
     * This function check if the bank is eligible based on txn amount.
     * @param context Context
     * @param minAmount Minumum EMI Amount for Bank
     * @param maxAmount Maximum EMI Amount for Bank
     * @param payUEmiTenuresList Tenures list for Bank
     * @return "", if eligible else error message to show as described below -
     * If (txn amount + max tenure additionalCharges) < minAmount, then return minimum emi amount error
     * else If (txn amount + max tenure additionalCharges) > minAmount, then return maximum emi amount error
     * */
    internal fun getBankDownText(
        context: Context?,
        minAmount: Double?,
        maxAmount: Double?,
        payUEmiTenuresList: ArrayList<PayUEmiTenures>
    ): String? {
        if (context == null || payuBizParams == null || payuBizParams!!.amount.toDoubleOrNull() == null) return null
        var maxAdditionalCharges = 0.0
        var bankDownText = ""
        for (item in payUEmiTenuresList) {
            item.additionalCharge?.toDoubleOrNull()?.let {
                if (it > maxAdditionalCharges)
                    maxAdditionalCharges = it
            }
        }

        val totalTxnAmount = payuBizParams!!.amount.toDouble() + maxAdditionalCharges
        if (minAmount != null && totalTxnAmount < minAmount)
            bankDownText = context.getString(
                R.string.payu_minimum_emi_amount_error,
                CommonUtils.getFormattedAmount(minAmount.toInt().toString())
            )
        else if (maxAmount != null && totalTxnAmount > maxAmount)
            bankDownText = context.getString(
                R.string.payu_maximum_emi_amount_error,
                CommonUtils.getFormattedAmount(maxAmount.toInt().toString())
            )

        return bankDownText
    }

    internal fun getBankDownText(
        context: Context?,
        minAmount: Double?,
        maxAmount: Double?,
        additionalCharges: Double = 0.0
    ): String? {
        if (context == null || payuBizParams == null || payuBizParams!!.amount.toDoubleOrNull() == null) return null
        var bankDownText = ""

        val totalTxnAmount = payuBizParams!!.amount.toDouble() + additionalCharges
        if (minAmount != null && totalTxnAmount < minAmount)
            bankDownText = context.getString(
                R.string.payu_minimum_emi_amount_error,
                CommonUtils.getFormattedAmount(minAmount.toInt().toString())
            )
        else if (maxAmount != null && totalTxnAmount > maxAmount)
            bankDownText = context.getString(
                R.string.payu_maximum_emi_amount_error,
                CommonUtils.getFormattedAmount(maxAmount.toInt().toString())
            )

        return bankDownText
    }

    internal fun prepareTenureList(
        payUEmiTenuresList: ArrayList<PayUEmiTenures>,
        emiCode: String,
        subType: String?,
        emiOption: EMIOption
    ): Pair<ArrayList<PaymentOption>, Double> {
        var tenureList = ArrayList<PaymentOption>()
        var lowestInterestRate = -1.0
        for (emi in payUEmiTenuresList) {
            val ccemiOption = preparePaymentOption(
                EMIOption().apply {
                    bankShortName = emiOption.bankShortName
                    paymentType = PaymentType.EMI
                },
                emiOption.bankName,
                emi.bankCode,
                PayuConstants.EMI,
                emi.imageURL,
                emi.category,
                emi.imageUpdatedOn,
                false,
                emi.additionalCharge.toDoubleOrNull(),
                emiOption.gstPercentageValue
            )
            ccemiOption.emiType = emiOption.emiType
            ccemiOption.isEligible =
                isEmiEligible(emi.bankCode, emi.status, emi.minAmount.toDoubleOrNull())
            emi.minAmount.toIntOrNull()?.let { ccemiOption.minimumTxnAmount = it }
            emi.maxAmount.toIntOrNull()?.let { ccemiOption.maximumTxnAmount = it }
            emi.monthlyEmi.toDoubleOrNull()?.let { ccemiOption.emiValue = it }
            emi.tenure.toIntOrNull()?.let { ccemiOption.months = it }
            //This is because in ZestMoney we use ZESTMON as bankcode and does not show tenures
            if (emiCode == SdkUiConstants.ZESTMON) {
                lowestInterestRate = SdkUiConstants.VALUE_ZERO_INT
            }
            emi.interestRate.toDoubleOrNull()?.let {
                ccemiOption.interestRate = it
                if (lowestInterestRate == -1.0) {
                    lowestInterestRate = it
                } else {
                    if (lowestInterestRate > it)
                        lowestInterestRate = it
                }
            }
            emi.interestCharged.toDoubleOrNull()?.let { ccemiOption.interestCharged = it }
            CommonUtils.putKeyValueInOtherParams(
                ccemiOption,
                PayUCheckoutProConstants.CP_KEY_EMI_CODE,
                emiCode
            )
            CommonUtils.putKeyValueInOtherParams(
                ccemiOption,
                PayUCheckoutProConstants.CP_KEY_SUB_TYPE,
                subType
            )

            tenureList.add(ccemiOption)
        }
        tenureList.sortWith(compareBy({ (it as EMIOption).months }))
        return Pair(tenureList, lowestInterestRate)
    }

    /**
     * This function checks if EMI is eligible for bankCode after computing total txn amount i.e txn amount + additional charges
     * and compares it with minimumTxnAmount and maximumTxnAmount
     * @param bankCode bankcode
     * @param status current eligibility status as returned in API
     * @param minimumTxnAmount minimum txn amount
     * @param maximumTxnAmount maximum txn amount
     * @param additionalCharges additional charges applicable on bank code
     * @return true, if EMI is eligible else false
     * */
    internal fun isEmiEligible(
        bankCode: String,
        status: Boolean,
        additionalCharges: Double?
    ): Boolean {
        if (bankCode.isEmpty() || additionalCharges == null || payuBizParams == null || payuBizParams!!.amount.toDoubleOrNull() == null) return false
        return status
    }

    /**
     * @param payuCommandResponse, Response from UPI SDK.
     */
    internal fun parseVpaValidation(payuCommandResponse: String): ApiResponse {
        val apiResponse = ApiResponse()
        if (payuCommandResponse.isNotEmpty()) {
            try {
                val result = JSONObject(payuCommandResponse)
                if (result.has(PayUCheckoutProConstants.CP_STATUS)
                    && result.getString(PayUCheckoutProConstants.CP_STATUS)
                        .equals(PayUCheckoutProConstants.CP_SUCCESS, true)
                ) {
                    if (result.has(PayUCheckoutProConstants.CP_IS_VPA_VALID) &&
                        payuBizParams?.siParams != null && result.has(PayUCheckoutProConstants.CP_IS_AUTO_PAY_VALID) &&
                        result.getInt(PayUCheckoutProConstants.CP_IS_VPA_VALID) == 1
                    ) {
                        if (result.getInt(PayUCheckoutProConstants.CP_IS_AUTO_PAY_VALID) == 1) {
                            getPayerAccountName(apiResponse, result)
                        } else {
                            upiErrorMessage(
                                apiResponse,
                                PayUCheckoutProConstants.CP_AUTO_PAY_VALIDATE_ERROR_MESSAGE
                            )
                        }
                    } else if (result.has(PayUCheckoutProConstants.CP_IS_VPA_VALID) &&
                        result.getInt(PayUCheckoutProConstants.CP_IS_VPA_VALID) == 1
                    ) {
                        getPayerAccountName(apiResponse, result)
                    } else {
                        upiErrorMessage(
                            apiResponse,
                            PayUCheckoutProConstants.CP_VALIDATE_VPA_ERROR_MESSAGE
                        )
                    }
                } else {
                    upiErrorMessage(
                        apiResponse,
                        PayUCheckoutProConstants.CP_VALIDATE_VPA_ERROR_MESSAGE
                    )
                }
            } catch (e: JSONException) {
                upiErrorMessage(apiResponse, PayUCheckoutProConstants.CP_VALIDATE_VPA_ERROR_MESSAGE)
            }
        } else {
            upiErrorMessage(apiResponse, PayUCheckoutProConstants.CP_VALIDATE_VPA_ERROR_MESSAGE)
        }
        return apiResponse
    }

    private fun getPayerAccountName(apiResponse: ApiResponse, result: JSONObject): ApiResponse {
        apiResponse.status = true
        if (result.has(PayUCheckoutProConstants.CP_PAYER_ACCOUNT_NAME) && !result.getString(
                PayUCheckoutProConstants.CP_PAYER_ACCOUNT_NAME
            ).equals("null", true)
        ) {
            apiResponse.successMessage =
                result.optString(PayUCheckoutProConstants.CP_PAYER_ACCOUNT_NAME)
        } else {
            apiResponse.successMessage = ""
        }
        return apiResponse

    }

    private fun upiErrorMessage(apiResponse: ApiResponse, message: String): ApiResponse {
        apiResponse.status = false
        apiResponse.errorMessage = message
        return apiResponse

    }

    internal fun getEmiWithEligibleBins(
        payuResponse: PayuResponse?
    ): ArrayList<PaymentOption>? {
        if (emiList == null) return null //return null if full emi list if not stored
        if (payuResponse != null && payuResponse.isEligibleEmiBinsAvailable) {
            val eligibleBinsList = payuResponse.eligibleEmiBins
            for (list in emiList!!) {
                val bankList = list.optionList
                if (bankList.isNullOrEmpty()) return null
                for (item in bankList) {
                    val tenureList = item.optionList
                    if (tenureList.isNullOrEmpty()) break
                    for (tenureItem in tenureList) {
                        val emiOption = tenureItem as? EMIOption
                        emiOption?.let {
                            val eligibleEmiBins =
                                CommonUtils.getEligibleEmiBinsForBank(
                                    it.bankShortName,
                                    eligibleBinsList
                                )
                            if (eligibleEmiBins != null) {
                                it.supportedBins = eligibleEmiBins.cardBins
//                        tenureList[tenureList.indexOf(tenureItem)] = emiOption
//                        emiList!![index].optionList!![emiList!![index].optionList?.indexOf(item)!!] =
//                            emiOption
                            }
                        }
                    }
                }
            }
            return emiList
        } else return null
    }

    internal fun addGooglePayOption(
        checkoutDetailsResponse: PayuResponse,
        context: Context?,
        paymentModesList: ArrayList<PaymentMode>
    ) {

        //If Google Pay is not enabled on merchant then no need to add Google Pay in list
        if (!checkoutDetailsResponse.isGoogleTezAvailable)
            return

        var googlePayMode: PaymentMode? = null
        val paymentMap = HashMap<String, Any>()

        when {
            //In-App should be checked first
            CommonUtils.isSdkAvailable(PayUCheckoutProConstants.CP_GOOGLE_PAY_CLASS_NAME) -> {
                googlePayMode = prepareL1Mode(
                    PayUCheckoutProConstants.CP_GOOGLE_PAY,
                    UPIOption().apply { paymentType = PaymentType.UPI },
                    PayuConstants.UPI,
                    PayuConstants.TEZ,
                    false,
                    checkoutDetailsResponse.googleTez.additionalCharge.toDoubleOrNull(),
                    checkoutDetailsResponse.taxSpecification?.upiTaxValue?.toDoubleOrNull(),
                    paymentMap
                )
            }

            checkoutDetailsResponse.isGenericIntentAvailable && isUPIAppInstalled(
                context,
                PayUCheckoutProConstants.CP_GOOGLE_PAY_PACKAGE_NAME
            ) -> {
                val upiOption = UPIOption().also {
                    it.paymentType = PaymentType.UPI_INTENT
                    it.packageName = PayUCheckoutProConstants.CP_GOOGLE_PAY_PACKAGE_NAME

                    /**
                     * For a UPI_INTENT type payment option icon is fetched from the package manager
                     * whereas for this L1 option, it is not necessary that app is installed on device. Hence, its icon
                     * should be fetched from Asset Library. This flag will fetch icon for this L1 option from the
                     * Asset Library instead of package manager.
                     * */
                    val map = HashMap<String, Any?>()
                    map[PayUCheckoutProConstants.CP_IS_L1_OPTION] = true
                    map[PayUCheckoutProConstants.CP_UPI_APP_NAME] =
                        PayUCheckoutProConstants.CP_KEY_ASSET_GOOGLE_PAY
                    it.otherParams = map
                }

                googlePayMode = prepareL1Mode(
                    PayUCheckoutProConstants.CP_GOOGLE_PAY,
                    upiOption,
                    PayuConstants.UPI,
                    PayuConstants.UPI_INTENT,
                    false,
                    checkoutDetailsResponse.googleTez.additionalCharge.toDoubleOrNull(),
                    checkoutDetailsResponse.taxSpecification?.upiTaxValue?.toDoubleOrNull(),
                    paymentMap
                )
            }

            else -> return
        }
        paymentOptionMap[PaymentType.UPI]?.forEach { (key, value) ->
            paymentMap[key]  = value
        }
        paymentOptionMap[PaymentType.UPI] = paymentMap
        paymentModesList.add(googlePayMode)
    }

    internal fun addPhonePeOption(
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>,
        context: Context?
    ) {
        //If PhonePe is not enabled on merchant then no need to add PhonePe in list
        if (!checkoutDetailsResponse.isPhonePeIntentAvailable)
            return
        if ((android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.N) && (!isUPIAppInstalled(
                context,
                PayUCheckoutProConstants.CP_PHONEPE_PACKAGE_NAME
            ))
        ) {
            return
        }

        val bankCode = when {
            CommonUtils.isSdkAvailable(PayUCheckoutProConstants.CP_PHONEPE_CLASS_NAME) -> PayUCheckoutProConstants.CP_PPINTENT
            else -> PayUCheckoutProConstants.CP_PHONEPE_BANKCODE
        }
        val map = HashMap<String, Any>()
        val phonePeMode = prepareL1Mode(
            PayUCheckoutProConstants.CP_PHONEPE,
            WalletOption().apply { paymentType = PaymentType.WALLET },
            PayuConstants.CASH,
            bankCode,
            CommonUtils.getBankDownStatusForBankCode(bankCode, checkoutDetailsResponse.cashCard),
            CommonUtils.getAdditionalChargesForBankCode(
                PayUCheckoutProConstants.CP_PHONEPE,
                checkoutDetailsResponse.cashCard
            ),
            checkoutDetailsResponse.taxSpecification?.cashTaxValue?.toDoubleOrNull(),
            map
        )
        paymentModesList.add(phonePeMode)
        paymentOptionMap[PaymentType.WALLET]?.forEach { (key, value) ->
            map[key]  = value
        }
        paymentOptionMap[PaymentType.WALLET] = map
    }

    private fun addL1Option(
        context: Context?,
        bankCode: String,
        paymentType: PaymentType,
        paymentModesList: ArrayList<PaymentMode>,
        checkoutDetailsResponse: PayuResponse?
    ) {
/*        var list: ArrayList<PaymentDetails>? = null
        var pg:String? = null
        var isBankDown:Boolean = false*/
        when (paymentType) {
/*        PaymentType.NB -> {
            if (payuResponse.isNetBanksAvailable) {
                pg = PayuConstants.NB
                list = payuResponse.netBanks
                if (netBankingStatusMapping != null) isBankDown = isBankDown(netBankingStatusMapping, bankCode)
            }
        }*/
            PaymentType.WALLET -> {
                when (bankCode) {
                    PayUCheckoutProConstants.CP_PHONEPE -> {
                        checkoutDetailsResponse?.let {
                            addPhonePeOption(
                                it,
                                paymentModesList,
                                context
                            )
                        } ?: kotlin.run { return }
                    }
                    PayUCheckoutProConstants.CP_PAYTM -> {
                        checkoutDetailsResponse?.let {
                            addPaytmOption(
                                it,
                                paymentModesList
                            )
                        } ?: kotlin.run { return }
                    }
                    else -> return
                }
/*            if (bankCode.equals(PayUCheckoutProConstants.BL_PHONEPE, ignoreCase = true)) {
                addPhonePeOption(payuResponse, context, paymentModesList)
                return
            }
            else if (payuResponse.isCashCardAvailable){
                pg = PayuConstants.CASH
                list = payuResponse.cashCard
                isBankDown = false
            }*/
            }
            PaymentType.UPI -> {
                if (bankCode.equals(PayUCheckoutProConstants.CP_GOOGLE_PAY, ignoreCase = true)) {
                    checkoutDetailsResponse?.let {
                        addGooglePayOption(
                            it,
                            context,
                            paymentModesList
                        )
                    }
                    return
                }
            }
        }

/*        //Return if list is null
        if (list == null)
            return

        for ( item in list) {
            if ( item.bankCode.equals(bankCode,ignoreCase = true)) {
                val paymentMode = prepareMode(
                    item.bankName,
                    PaymentType.L1_OPTION,
                    paymentType,
                    getPaymentOption(paymentType),
                    pg!!,
                    bankCode,
                    isBankDown
                )
                paymentModesList.add(paymentMode)
            }
        }*/
    }

    internal fun addPaytmOption(
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {

        //If no wallets are activated return
        if (!checkoutDetailsResponse.isCashCardAvailable) return

        //If Paytm is not enabled on merchant then don't add it in list
        if (!CommonUtils.isBankCodeAvailable(
                PayUCheckoutProConstants.CP_PAYTM,
                checkoutDetailsResponse.cashCard
            )
        ) return
        val map = HashMap<String, Any>()
        val paytmMode = prepareL1Mode(
            PayUCheckoutProConstants.CP_PAYTM_NAME,
            WalletOption().apply { paymentType = PaymentType.WALLET },
            PayuConstants.CASH,
            PayUCheckoutProConstants.CP_PAYTM,
            CommonUtils.getBankDownStatusForBankCode(
                PayUCheckoutProConstants.CP_PAYTM,
                checkoutDetailsResponse.cashCard
            ),
            CommonUtils.getAdditionalChargesForBankCode(
                PayUCheckoutProConstants.CP_PAYTM,
                checkoutDetailsResponse.cashCard
            ),
            checkoutDetailsResponse.taxSpecification?.cashTaxValue?.toDoubleOrNull(),
            map
        )
        paymentModesList.add(paytmMode)
        paymentOptionMap[PaymentType.WALLET]?.forEach { (key, value) ->
            map[key]  = value
        }
        paymentOptionMap[PaymentType.WALLET] = map
    }


    internal fun addSodexoOption(
        checkoutDetailsResponse: PayuResponse,
        paymentModesList: ArrayList<PaymentMode>
    ) {
        val map = HashMap<String, Any>()
        val sodexoMode = prepareL1Mode(
            PayUCheckoutProConstants.CP_SODEXO_NAME,
            SodexoCardOption().apply {
                bankName = PayUCheckoutProConstants.CP_SODEXO_NAME
                paymentType = PaymentType.SODEXO
            },
            PayuConstants.SODEXO,
            SODEXO,
            isSodexoBankDown(checkoutDetailsResponse),
            CommonUtils.getAdditionalChargesForBankCode(
                SODEXO,
                checkoutDetailsResponse.mealCard
            ), checkoutDetailsResponse.taxSpecification?.mealCardTaxValue?.toDoubleOrNull(),
            map
        )

        if (!enforcePaymentList.isNullOrEmpty()) {
            val map: HashMap<String, Any?> = HashMap()
            map[PayUCheckoutProConstants.CP_PAYMENT_TYPE] = PaymentType.SODEXO.name
            map[PayUCheckoutProConstants.ENFORCED_IBIBOCODE] = SODEXO
            if (CommonUtils.isKeyEnforced(map))
                paymentModesList.add(sodexoMode)
        } else
            paymentModesList.add(sodexoMode)
        paymentOptionMap[PaymentType.SODEXO] = map

    }

    internal fun prepareL1Mode(
        bankeName: String,
        paymentOption: PaymentOption,
        pg: String,
        bankCode: String,
        isBankDown: Boolean,
        additionalCharges: Double? = 0.0,
        gst: Double? = 0.0,
        map: HashMap<String, Any>
    ): PaymentMode {
        val paymentMode = PaymentMode()
        paymentMode.name = bankeName
        paymentMode.type = PaymentType.L1_OPTION
        paymentMode.paymentId = bankCode
        val paymentOptionsList = ArrayList<PaymentOption>()
        paymentOption.isBankDown = isBankDown
        paymentOption.bankName = bankeName
        paymentOption.additionalCharge = additionalCharges
        paymentOption.gstPercentageValue = gst

        var otherParamsMap: HashMap<String, Any?>? = null
        if (paymentOption.otherParams != null) {
            otherParamsMap = paymentOption.otherParams as? HashMap<String, Any?>
            otherParamsMap!![PayUCheckoutProConstants.CP_PG] = pg
            otherParamsMap[PayUCheckoutProConstants.CP_BANK_CODE] = bankCode
        } else paymentOption.otherParams = CommonUtils.getOtherParamsMap(pg, bankCode)

        paymentOptionsList.add(paymentOption)
        map[bankCode] = paymentOption
        paymentMode.optionDetail = paymentOptionsList
        return paymentMode
    }

    internal fun <T> preparePaymentOption(
        paymentOption: T,
        bankName: String,
        bankCode: String,
        pg: String,
        imageURL: String?,
        category: String?,
        imageUpdatedOn: Long?,
        isBankDown: Boolean = false,
        additionalCharges: Double? = 0.0,
        gst: Double? = 0.0,
        priority: Int = 0,
        isSecureWebview: Boolean = false,
        pgAppVersion: String? = null
    ): T {
        paymentOption as PaymentOption
        paymentOption.bankName = CommonUtils.getFormattedBankName(bankName, bankCode)
        paymentOption.isBankDown = isBankDown
        paymentOption.additionalCharge = additionalCharges
        paymentOption.gstPercentageValue = gst
        paymentOption.otherParams = CommonUtils.getOtherParamsMap(pg, bankCode)
        val timeStamp = SimpleDateFormat(CP_TIME_FORMAT, Locale.UK).format(
            Calendar.getInstance().time
        )
        paymentOption.imageUpdatedOn = imageUpdatedOn ?: timeStamp.toLongOrNull()

        if (!imageURL.isNullOrEmpty() && !imageURL.equals("null"))
            paymentOption.imageURL =
                imageURL.plus(bankCode).plus(".").plus(ImageType.SVG.name.lowercase())
        paymentOption.category = category
        paymentOption.priority = priority
        paymentOption.isSecureWebview = isSecureWebview
        paymentOption.pgAppVersion = pgAppVersion
        return paymentOption
    }

    /**
     * This method returns the filtered list as received in CheckoutDetails API
     * Currently, it is for EMI mode only but can be extended for other modes in future
     * */
    internal fun getPaymentOptionFilteredList(
        payuResponse: PayuResponse?,
        paymentOptionList: ArrayList<PaymentOption>?
    ): ArrayList<PaymentOption>? {
        if (payuResponse == null) return null
        var updatedList: ArrayList<PaymentOption>? = null
        val paymentOption = paymentOptionList?.get(0)
        when (paymentOption?.paymentType) {
            PaymentType.EMI -> {
                val emiOption = paymentOption as EMIOption
                when (emiOption.emiType) {
                    EmiType.DC -> {
                        if (payuResponse.isDCEmiAvailable) {
                            updatedList =
                                updateEmiListWithEligibility(paymentOptionList, payuResponse.dcemi)
                        }
                    }
                    EmiType.CARD_LESS -> {
                        if (payuResponse.isCardLessEmiAvailable) {
                            updatedList = updateEmiListWithEligibility(
                                paymentOptionList,
                                payuResponse.cardlessemi
                            )
                        }
                    }
                }
            }
            PaymentType.BNPL -> {
                if (payuResponse.isBnplAvailable) {
                    updatedList =
                        updateBnplListWithEligibility(paymentOptionList, payuResponse.bnpl)
                }
            }
        }
        return updatedList
    }

    private fun updateEmiListWithEligibility(
        paymentOptionList: ArrayList<PaymentOption>,
        emiList: ArrayList<Emi>?
    ): ArrayList<PaymentOption> {
        if (emiList.isNullOrEmpty()) return paymentOptionList
        val newList = ArrayList<PaymentOption>()
        for (item in paymentOptionList) {
            val bankCode = CommonUtils.getValueFromPaymentOption<String>(
                PayUCheckoutProConstants.CP_BANK_CODE,
                item.otherParams as? java.util.HashMap<String, Any?>
            )
            for (emiItem in emiList) {
                val tenures = emiItem.payUEmiTenuresList
                if (tenures.isNullOrEmpty()) continue
                for (tenureItem in tenures) {
                    if (bankCode.equals(tenureItem.bankCode, ignoreCase = true)) {
                        val currentEmiOption = item as EMIOption
                        val newEmiOption = prepareEmiOptionCopy(currentEmiOption, tenureItem)
                        newList.add(newEmiOption)
                        break
                    }
                }
            }
        }
        return newList
    }

    private fun prepareEmiOptionCopy(
        currentEmiOption: EMIOption,
        emiItem: PayUEmiTenures
    ): EMIOption {
        return EMIOption().apply {
            bankName = currentEmiOption.bankName
            bankShortName = currentEmiOption.bankShortName
            paymentType = currentEmiOption.paymentType
            months = currentEmiOption.months
            minimumTxnAmount = currentEmiOption.minimumTxnAmount
            supportedBins = currentEmiOption.supportedBins
            emiType = currentEmiOption.emiType
            emiValue = currentEmiOption.emiValue
            interestRate = currentEmiOption.interestRate
            interestCharged = currentEmiOption.interestCharged
            phoneNumber = currentEmiOption.phoneNumber
            isEligible = emiItem.status
            additionalCharge = currentEmiOption.additionalCharge
            gstPercentageValue = currentEmiOption.gstPercentageValue
            otherParams = currentEmiOption.otherParams
        }
    }

    /**
     * Function to update BNPL List with new eligibility status
     */

    private fun updateBnplListWithEligibility(
        paymentOptionList: ArrayList<PaymentOption>,
        bnplList: ArrayList<Bnpl>?
    ): ArrayList<PaymentOption> {
        Log.d(TAG, "updateBnplListWithEligibility is called in ParserUtils")

        if (bnplList.isNullOrEmpty()) return paymentOptionList
        val bnplListWithEligibility = ArrayList<PaymentOption>()
        for (item in paymentOptionList) {
            val bankCode = CommonUtils.getValueFromPaymentOption<String>(
                PayUCheckoutProConstants.CP_BANK_CODE,
                item.otherParams as? java.util.HashMap<String, Any?>
            )
            if (bankCode.equals(bnplList.get(0).bankCode, ignoreCase = true)) {
                val bnpl: Bnpl = bnplList.get(0)
                val bnplOption = item as BnplOption
                bnplOption.isEligible = bnpl.status
                bnplOption.additionalCharge = bnpl.additionalCharge?.toDoubleOrNull()
                bnplListWithEligibility.add(bnplOption)
                break
            }
        }
        return bnplListWithEligibility
    }

    internal fun getEnforcementListForPaymentOption(
        paymentOptionlist: ArrayList<PaymentDetails>,
        paymentType: PaymentType
    ): ArrayList<PaymentDetails> {
        if (!enforcePaymentList.isNullOrEmpty() && !isRetryTxn) {
            val pOptionList = ArrayList<PaymentDetails>()
            for (enforce in enforcePaymentList!!) {
                if (!enforce.get(PayUCheckoutProConstants.ENFORCED_IBIBOCODE).isNullOrEmpty()) {
                    val list = enforce.get(PayUCheckoutProConstants.ENFORCED_IBIBOCODE)
                        ?.filter { !it.isWhitespace() }?.split("|")
                    for (option in paymentOptionlist) {
                        if (enforce[PayUCheckoutProConstants.CP_PAYMENT_TYPE].equals(paymentType.name) && (list.isNullOrEmpty() || list.contains(
                                option.bankCode
                            ))
                        )
                            pOptionList.add(option)
                    }
                } else if (pOptionList.isEmpty())
                    return paymentOptionlist
            }
            return pOptionList
        } else
            return paymentOptionlist

    }

    internal fun updateEnforcedTenureList(
        bankEmiOption: EMIOption,
        paymentOptionList: ArrayList<PaymentOption>
    ) {
        val emilist = ArrayList<PaymentOption>()
        if (!enforcePaymentList.isNullOrEmpty() && !isRetryTxn) {
            for (enforce in enforcePaymentList!!) {
                val list = enforce.get(PayUCheckoutProConstants.ENFORCED_IBIBOCODE)?.split("|")
                if (enforce[PayUCheckoutProConstants.CP_PAYMENT_TYPE].equals(PaymentType.EMI.name)) {
                    if (!list.isNullOrEmpty()) {
                        for (tenure in paymentOptionList) {
                            val tenureemi: HashMap<String, String> =
                                tenure.otherParams as HashMap<String, String>
                            if (!list.isNullOrEmpty() && list.contains(tenureemi[PayUCheckoutProConstants.CP_BANK_CODE]))
                                emilist.add(tenure)
                        }
                        if (paymentOptionList.isNotEmpty()) {
                            if (emilist.isNotEmpty()) {
                                bankEmiOption.optionList = emilist
                            }
                        }
                    } else
                        bankEmiOption.optionList = paymentOptionList
                }
            }
        } else bankEmiOption.optionList = paymentOptionList
    }

    fun isSodexoBankDown(payuResponse: PayuResponse?): Boolean {
        if (null != payuResponse && !payuResponse.mealCard.isNullOrEmpty()) {
            for (mealCard in payuResponse.mealCard) {
                if (mealCard.bankCode.equals(SODEXO, true) && mealCard.isBankDown)
                    return true
            }

        }
        return false
    }

    fun getSodexoMode(payuResponse: PayuResponse?): PaymentMode {
        val paymentMode = PaymentMode()
        paymentMode.type = PaymentType.SODEXO
        paymentMode.name = PayUCheckoutProConstants.SODEXO_CARD
        paymentMode.isBankDown = isSodexoBankDown(convenienceFeeResponse)
        val sodexoCardOption = SodexoCardOption()
        val map = HashMap<String, Any?>()
        map[PayUCheckoutProConstants.CP_BANK_CODE] = SODEXO
        map[PayUCheckoutProConstants.CP_PG] = PayUCheckoutProConstants.MC
        sodexoCardOption.otherParams = map
        sodexoCardOption.paymentType = PaymentType.SODEXO
        sodexoCardOption.bankName = SODEXO
        val cardBinInfo = CardBinInfo()
        cardBinInfo.cardScheme = CardScheme.SODEXO
        convenienceFeeResponse?.mealCard?.let {
            cardBinInfo.additionalCharge = CommonUtils.getAdditionalChargesForBankCode(SODEXO, it)
        }
        convenienceFeeResponse?.taxSpecification?.mealCardTaxValue?.toDoubleOrNull()
            ?.let { cardBinInfo.gstPercentageValue = it }
        sodexoCardOption.cardBinInfo = cardBinInfo
        if (null != payuResponse && null != payuResponse.sodexoCardInfo) {

            val sodexoCardInfo = payuResponse.sodexoCardInfo

            if (null != sodexoCardInfo?.cardNo)
                sodexoCardOption.cardNumber = sodexoCardInfo.cardNo

            if (null != sodexoCardInfo?.cardName)
                sodexoCardOption.nameOnCard = sodexoCardInfo.cardName

            if (null != sodexoCardInfo?.cardBalance)
                sodexoCardOption.balance = sodexoCardInfo.cardBalance

            sodexoCardOption.fetchedStatus = 1

        } else {

            sodexoCardOption.fetchedStatus = 0

        }
        val paymentOptionsList = ArrayList<PaymentOption>()
        paymentOptionsList.add(sodexoCardOption)
        paymentMode.optionDetail = paymentOptionsList

        return paymentMode
    }

    internal fun getClosedLoopWalletMode(payuResponse: PayuResponse?): PaymentMode {
        val closedLoopWalletOption = SodexoCardOption()
        val map = HashMap<String, Any?>()
        val paymentMode = PaymentMode()

        var walletName = payuResponse?.merchantInfo?.walletIdentifier?.lowercase();

        walletName = if (walletName.isNullOrEmpty() || walletName == "null") {
            PayUCheckoutProConstants.CP_WALLET
        } else {
            walletName.capitalize() + " " + PayUCheckoutProConstants.CP_WALLET
        }

        paymentMode.type = PaymentType.CLOSED_LOOP_WALLET
        paymentMode.name = walletName
        paymentMode.isBankDown = false

        if(!convenienceFeeResponse?.closedLoopWallet.isNullOrEmpty()) {
            map[PayUCheckoutProConstants.CP_BANK_CODE] =
                convenienceFeeResponse?.closedLoopWallet?.firstOrNull()?.bankCode
            map[PayUCheckoutProConstants.CP_PG] = PayUCheckoutProConstants.CLW
            convenienceFeeResponse?.closedLoopWallet?.let {
                closedLoopWalletOption.additionalCharge =
                    CommonUtils.getAdditionalChargesForBankCode(it.firstOrNull()?.bankCode, it)
            }
        }
        closedLoopWalletOption.otherParams = map
        closedLoopWalletOption.paymentType = PaymentType.CLOSED_LOOP_WALLET
        closedLoopWalletOption.bankName = walletName

        convenienceFeeResponse?.taxSpecification?.clwTaxValue?.toDoubleOrNull()?.let {
            closedLoopWalletOption.gstPercentageValue = it
        }

        if ((apiLayer?.payUPaymentParams?.additionalParams?.containsKey(SdkUiConstants.WALLET_URN) == true) && apiLayer?.payUPaymentParams?.additionalParams!![SdkUiConstants.WALLET_URN].toString()
                .isNotBlank()
        ) {
            closedLoopWalletOption.walletUrn =
                apiLayer?.payUPaymentParams?.additionalParams!![SdkUiConstants.WALLET_URN].toString()
        }
        closedLoopWalletOption.fetchedStatus = 0
        if (payuResponse?.sodexoCardInfo?.cardBalance != null) {
            closedLoopWalletOption.balance = payuResponse.sodexoCardInfo.cardBalance
            closedLoopWalletOption.fetchedStatus = 1
        }
        val paymentOptionsList = ArrayList<PaymentOption>()
        if(!payuResponse?.closedLoopWallet.isNullOrEmpty()) {
            closedLoopWalletOption.category = payuResponse?.closedLoopWallet?.firstOrNull()?.category
        }
        paymentOptionsList.add(closedLoopWalletOption)
        paymentMode.optionDetail = paymentOptionsList

        return paymentMode
    }

    fun updateClosedLoopWallet(paymentMode: PaymentMode, payuResponse: PayuResponse?): PaymentMode {

        closedLoopWalletCardOption =
            if(!paymentMode.optionDetail.isNullOrEmpty()) {
                paymentMode.optionDetail?.get(0) as? SodexoCardOption ?: SodexoCardOption()
            } else {
                SodexoCardOption()
            }

        closedLoopWalletCardOption.fetchedStatus = 0
        if (!payuResponse?.sodexoCardInfo?.cardBalance.isNullOrEmpty() && payuResponse?.sodexoCardInfo?.cardBalance?.toDoubleOrNull() != null) {
            closedLoopWalletCardOption.balance = payuResponse.sodexoCardInfo.cardBalance
            closedLoopWalletCardOption.fetchedStatus = 1
        }

        val paymentOptionsList = ArrayList<PaymentOption>()
        paymentOptionsList.add(closedLoopWalletCardOption)
        paymentMode.optionDetail = paymentOptionsList

        return paymentMode
    }

    fun getSodexoSourceId(additionalParams: HashMap<String, Any?>?): String {
        if (additionalParams?.containsKey(
                PayUCheckoutProConstants.SODEXO_SOURCE_ID
            ) == true && !additionalParams[PayUCheckoutProConstants.SODEXO_SOURCE_ID]?.toString()
                .isNullOrBlank()
        ) {
            return additionalParams[PayUCheckoutProConstants.SODEXO_SOURCE_ID]?.toString()!!
        }
        return ""
    }

    fun getTransactionResponseJson(transactionRespObj: JSONObject): JSONObject? {
        return try {
            val transactionResponse = JSONObject()
            val keysIterator = transactionRespObj.keys()
            while (keysIterator.hasNext()) {
                when (keysIterator.next()) {
                    PayuConstants.ID, PayuConstants.MIHPAY_ID -> if (transactionRespObj.has(
                            PayuConstants.ID
                        )
                    ) {
                        transactionResponse.put(
                            PayuConstants.ID,
                            transactionRespObj[PayuConstants.ID].toString()
                        )
                    } else if (transactionRespObj.has(PayuConstants.MIHPAY_ID)) {
                        transactionResponse.put(
                            PayuConstants.ID,
                            transactionRespObj[PayuConstants.MIHPAY_ID].toString()
                        )
                    }

                    PayuConstants.MODE -> if (transactionRespObj.has(PayuConstants.MODE)) {
                        transactionResponse.put(
                            PayuConstants.MODE,
                            transactionRespObj[PayuConstants.MODE].toString()
                        )
                    }

                    PayuConstants.AMT, PayuConstants.AMOUNT -> if (transactionRespObj.has(
                            PayuConstants.AMT
                        )
                    ) {
                        transactionResponse.put(
                            PayuConstants.AMOUNT,
                            transactionRespObj[PayuConstants.AMT].toString()
                        )
                    } else if (transactionRespObj.has(PayuConstants.TRANSACTION_AMOUNT)) {
                        transactionResponse.put(
                            PayuConstants.AMOUNT,
                            transactionRespObj[PayuConstants.TRANSACTION_AMOUNT].toString()
                        )
                    }

                    PayuConstants.STATUS -> if (transactionRespObj.has(PayuConstants.STATUS)) {
                        transactionResponse.put(
                            PayuConstants.STATUS,
                            transactionRespObj[PayuConstants.STATUS].toString()
                        )
                    }

                    PayuConstants.UNMAPPED_STATUS -> if (transactionRespObj.has(PayuConstants.UNMAPPED_STATUS)) {
                        transactionResponse.put(
                            PayuConstants.UNMAPPED_STATUS,
                            transactionRespObj[PayuConstants.UNMAPPED_STATUS].toString()
                        )
                    }

                    PayuConstants.KEY -> if (transactionRespObj.has(PayuConstants.KEY)) {
                        transactionResponse.put(
                            PayuConstants.KEY,
                            transactionRespObj[PayuConstants.KEY].toString()
                        )
                    }

                    PayuConstants.TXNID -> if (transactionRespObj.has(PayuConstants.TXNID)) {
                        transactionResponse.put(
                            PayuConstants.TXNID,
                            transactionRespObj[PayuConstants.TXNID].toString()
                        )
                    }

                    PayuConstants.TRANSACTION_FEE -> if (transactionRespObj.has(PayuConstants.TRANSACTION_FEE)) {
                        transactionResponse.put(
                            PayuConstants.TRANSACTION_FEE,
                            transactionRespObj[PayuConstants.TRANSACTION_FEE].toString()
                        )
                    }

                    PayuConstants.CARDCATEGORY -> if (transactionRespObj.has(PayuConstants.CARDCATEGORY)) {
                        transactionResponse.put(
                            PayuConstants.CARDCATEGORY,
                            transactionRespObj[PayuConstants.CARDCATEGORY].toString()
                        )
                    }

                    PayuConstants.DISCOUNT -> if (transactionRespObj.has(PayuConstants.DISCOUNT)) {
                        transactionResponse.put(
                            PayuConstants.DISCOUNT,
                            transactionRespObj[PayuConstants.DISCOUNT].toString()
                        )
                    }

                    PayuConstants.ADDITIONAL_CHARGES -> if (transactionRespObj.has(PayuConstants.ADDITIONAL_CHARGES)) {
                        transactionResponse.put(
                            PayuConstants.ADDITIONAL_CHARGES,
                            transactionRespObj[PayuConstants.ADDITIONAL_CHARGES].toString()
                        )
                    }

                    PayuConstants.ADDED_ON -> if (transactionRespObj.has(PayuConstants.ADDED_ON)) {
                        transactionResponse.put(
                            PayuConstants.ADDED_ON,
                            transactionRespObj[PayuConstants.ADDED_ON].toString()
                        )
                    }

                    PayuConstants.PRODUCT_INFO -> if (transactionRespObj.has(PayuConstants.PRODUCT_INFO)) {
                        transactionResponse.put(
                            PayuConstants.PRODUCT_INFO,
                            transactionRespObj[PayuConstants.PRODUCT_INFO].toString()
                        )
                    }

                    PayuConstants.FIRST_NAME -> if (transactionRespObj.has(PayuConstants.FIRST_NAME)) {
                        transactionResponse.put(
                            PayuConstants.FIRST_NAME,
                            transactionRespObj[PayuConstants.FIRST_NAME].toString()
                        )
                    }

                    PayuConstants.EMAIL -> if (transactionRespObj.has(PayuConstants.EMAIL)) {
                        transactionResponse.put(
                            PayuConstants.EMAIL,
                            transactionRespObj[PayuConstants.EMAIL].toString()
                        )
                    }

                    PayuConstants.UDF1 -> if (transactionRespObj.has(PayuConstants.UDF1)) {
                        transactionResponse.put(
                            PayuConstants.UDF1,
                            transactionRespObj[PayuConstants.UDF1].toString()
                        )
                    }

                    PayuConstants.UDF2 -> if (transactionRespObj.has(PayuConstants.UDF2)) {
                        transactionResponse.put(
                            PayuConstants.UDF2,
                            transactionRespObj[PayuConstants.UDF2].toString()
                        )
                    }

                    PayuConstants.UDF3 -> if (transactionRespObj.has(PayuConstants.UDF3)) {
                        transactionResponse.put(
                            PayuConstants.UDF3,
                            transactionRespObj[PayuConstants.UDF3].toString()
                        )
                    }

                    PayuConstants.UDF4 -> if (transactionRespObj.has(PayuConstants.UDF4)) {
                        transactionResponse.put(
                            PayuConstants.UDF4,
                            transactionRespObj[PayuConstants.UDF4].toString()
                        )
                    }

                    PayuConstants.UDF5 -> if (transactionRespObj.has(PayuConstants.UDF5)) {
                        transactionResponse.put(
                            PayuConstants.UDF5,
                            transactionRespObj[PayuConstants.UDF5].toString()
                        )
                    }

                    PayuConstants.HASH -> if (transactionRespObj.has(PayuConstants.HASH)) {
                        transactionResponse.put(
                            PayuConstants.HASH,
                            transactionRespObj[PayuConstants.HASH].toString()
                        )
                    }

                    PayuConstants.FIELD1 -> if (transactionRespObj.has(PayuConstants.FIELD1)) {
                        transactionResponse.put(
                            PayuConstants.FIELD1,
                            transactionRespObj[PayuConstants.FIELD1].toString()
                        )
                    }

                    PayuConstants.FIELD2 -> if (transactionRespObj.has(PayuConstants.FIELD2)) {
                        transactionResponse.put(
                            PayuConstants.FIELD2,
                            transactionRespObj[PayuConstants.FIELD2].toString()
                        )
                    }

                    PayuConstants.FIELD3 -> if (transactionRespObj.has(PayuConstants.FIELD3)) {
                        transactionResponse.put(
                            PayuConstants.FIELD3,
                            transactionRespObj[PayuConstants.FIELD3].toString()
                        )
                    }

                    PayuConstants.FIELD4 -> if (transactionRespObj.has(PayuConstants.FIELD4)) {
                        transactionResponse.put(
                            PayuConstants.FIELD4,
                            transactionRespObj[PayuConstants.FIELD4].toString()
                        )
                    }

                    PayuConstants.FIELD9 -> if (transactionRespObj.has(PayuConstants.FIELD9)) {
                        transactionResponse.put(
                            PayuConstants.FIELD9,
                            transactionRespObj[PayuConstants.FIELD9].toString()
                        )
                    }

                    PayuConstants.PAYMENT_SOURCE -> if (transactionRespObj.has(PayuConstants.PAYMENT_SOURCE)) {
                        transactionResponse.put(
                            PayuConstants.PAYMENT_SOURCE,
                            transactionRespObj[PayuConstants.PAYMENT_SOURCE].toString()
                        )
                    }

                    PayuConstants.PG_TYPE -> if (transactionRespObj.has(PayuConstants.PG_TYPE)) {
                        transactionResponse.put(
                            PayuConstants.PG_TYPE,
                            transactionRespObj[PayuConstants.PG_TYPE].toString()
                        )
                    }

                    PayuConstants.BANK_REF_NUM -> if (transactionRespObj.has(PayuConstants.BANK_REF_NUM)) {
                        transactionResponse.put(
                            PayuConstants.BANK_REF_NUM,
                            transactionRespObj[PayuConstants.BANK_REF_NUM].toString()
                        )
                    }

                    PayuConstants.IBIBO_CODE -> if (transactionRespObj.has(PayuConstants.IBIBO_CODE)) {
                        transactionResponse.put(
                            PayuConstants.IBIBO_CODE,
                            transactionRespObj[PayuConstants.IBIBO_CODE].toString()
                        )
                    }

                    PayuConstants.ERROR_CODE -> if (transactionRespObj.has(PayuConstants.ERROR_CODE)) {
                        transactionResponse.put(
                            PayuConstants.ERROR_CODE,
                            transactionRespObj[PayuConstants.ERROR_CODE].toString()
                        )
                    }

                    PayuConstants.ERROR_MESSAGE2 -> if (transactionRespObj.has(PayuConstants.ERROR_MESSAGE2)) {
                        transactionResponse.put(
                            PayuConstants.ERROR_MESSAGE2,
                            transactionRespObj[PayuConstants.ERROR_MESSAGE2].toString()
                        )
                    }

                    PayuConstants.CARD_TOKEN -> if (transactionRespObj.has(PayuConstants.CARD_TOKEN)) {
                        transactionResponse.put(
                            PayuConstants.CARD_TOKEN,
                            transactionRespObj[PayuConstants.CARD_TOKEN].toString()
                        )
                    }

                    PayuConstants.NAME_ON_CARD -> if (transactionRespObj.has(PayuConstants.NAME_ON_CARD)) {
                        transactionResponse.put(
                            PayuConstants.NAME_ON_CARD,
                            transactionRespObj[PayuConstants.NAME_ON_CARD].toString()
                        )
                    }

                    PayuConstants.CARD_NO -> if (transactionRespObj.has(PayuConstants.CARD_NO)) {
                        transactionResponse.put(
                            PayuConstants.CARD_NO,
                            transactionRespObj[PayuConstants.CARD_NO].toString()
                        )
                    }

                    PayuConstants.ISSUINGBANK -> if (transactionRespObj.has(PayuConstants.ISSUINGBANK)) {
                        transactionResponse.put(
                            PayuConstants.ISSUINGBANK,
                            transactionRespObj[PayuConstants.ISSUINGBANK].toString()
                        )
                    }

                    PayuConstants.CARD_TYPE -> if (transactionRespObj.has(PayuConstants.CARD_TYPE)) {
                        transactionResponse.put(
                            PayuConstants.CARD_TYPE,
                            transactionRespObj[PayuConstants.CARD_TYPE].toString()
                        )
                    }

                    PayuConstants.IS_SEAMLESS -> if (transactionRespObj.has(PayuConstants.IS_SEAMLESS)) {
                        transactionResponse.put(
                            PayuConstants.IS_SEAMLESS,
                            transactionRespObj[PayuConstants.IS_SEAMLESS].toString()
                        )
                    }

                    PayuConstants.SURL -> if (transactionRespObj.has(PayuConstants.SURL)) {
                        transactionResponse.put(
                            PayuConstants.SURL,
                            transactionRespObj[PayuConstants.SURL].toString()
                        )
                    }

                    PayuConstants.FURL -> if (transactionRespObj.has(PayuConstants.FURL)) {
                        transactionResponse.put(
                            PayuConstants.FURL,
                            transactionRespObj[PayuConstants.FURL].toString()
                        )
                    }

                    PayuConstants.MERCHANT_HASH -> if (transactionRespObj.has(PayuConstants.MERCHANT_HASH)) {
                        transactionResponse.put(
                            PayuConstants.MERCHANT_HASH,
                            transactionRespObj[PayuConstants.MERCHANT_HASH].toString()
                        )
                    }
                }
            }
            transactionResponse
        } catch (e: JSONException) {
            Log.d("PayuUtils", "Exception: " + e.message)
            null
        }
    }

    internal fun isSufficientBalanceForLoadAndPay(balance: String): Boolean {
        payuBizParams?.amount?.let {
            if (balance.toDoubleOrNull() != null && it.toDoubleOrNull() != null) {
                return (balance.toDouble() < it.toDouble())
            }
        }
        return false
    }

    internal fun resetApiRepoResponse() {
        ApiResponseRepo.gvQuickOptionsListener = null
        ApiResponseRepo.moreOptionsListener = null
        ApiResponseRepo.payuFetchOfferListener = null
        ApiResponseRepo.payuApiFetchOffer = null
        ApiResponseRepo.moreOptionsResponse = null
        ApiResponseRepo.gvOptionsResponse = null
        InternalConfig.isEnforced = false
        gvResponse = null
        isCheckOutCompleted = false
        onGVQuickPayListener = null
        apiLayer = null
    }

    internal fun prepareGvResponse(
        context: Context,
        savedOptionList: ArrayList<QuickPaySavedOption>?,
        recommendedOptions: ArrayList<RecommendedOptions>?,
        onGVQuickPayListener: (QuickOptionsModel?) -> Unit?
    ) {
        val quickSavedList=
            if (!InternalConfig.isQuickPayEnabled)
                savedOptionList?.filter { it.userCredential == apiLayer?.payUPaymentParams?.userCredential }
            else savedOptionList
        val quickOptions =
            PaymentModeConverter.globalVaultQuickOptions(context, quickSavedList ?: arrayListOf())
        convenienceFeeResponse?.savedOption = savedOptionList
        convenienceFeeResponse?.recommendedOptions = recommendedOptions
        quickOptionsList = quickOptions
        val quickOptionsModel = QuickOptionsModel()
        quickOptionsModel.quickOptionsList = quickOptions
        quickOptionsModel.recommendedOptionsList =
            PaymentModeConverter.getRecommendedOptionsResponse(recommendedOptions)
        val payUGvResponse = PayUApiResponse(onGVQuickPayListener, false, quickOptionsModel)
        ApiResponseRepo.setGvResponse(payUGvResponse)
    }

    internal fun isClosedLoopWalletToAdd(): Boolean {
        return convenienceFeeResponse?.isClosedLoopWalletPayAvailable == true && !convenienceFeeResponse?.merchantInfo?.walletIdentifier.isNullOrEmpty() && convenienceFeeResponse?.merchantInfo?.walletIdentifier != "null" && (!apiLayer?.payUPaymentParams?.additionalParams?.get(
            SdkUiConstants.WALLET_URN
        )?.toString().isNullOrEmpty())
    }

    fun getSkuJson(payuPaymentParams: PayUPaymentParams, isAutoApply: Boolean): Pair<JSONObject?,Boolean> {
        var totalSkuAmount = 0.0
        var isAutoApplyLocal = isAutoApply
        val jsonArray = JSONArray()
        payuPaymentParams.skuDetails?.skus?.let {
            for (sku in it) {
                val skuJsonObject = JSONObject()
                val selectedOfferInfo = InternalConfig.userSelectedOfferInfo
                if (selectedOfferInfo != null) {
                    if ((selectedOfferInfo.sku?.skuId == sku.skuId)) {
                        isAutoApplyLocal = false
                        val jsonOfferKey = JSONArray()
                        jsonOfferKey.put(selectedOfferInfo.offerKey)
                        skuJsonObject.put(PayuConstants.OFFER_KEY, jsonOfferKey)
                        skuJsonObject.put(PayuConstants.P_AUTO_APPLY_M_OFFER, false)
                    } else {
                        skuJsonObject.put(PayuConstants.P_AUTO_APPLY_M_OFFER, true)

                    }
                } else {
                    skuJsonObject.put(PayuConstants.P_AUTO_APPLY_M_OFFER, true)
                }
                skuJsonObject.put(PayuConstants.P_SKU_M_ID, sku.skuId)
                totalSkuAmount += (sku.skuAmount.toDouble()) * (sku.quantity.toDouble()
                    ?: 0.0)
                skuJsonObject.put(PayuConstants.P_SKU_M_AMOUNT, sku.skuAmount)
                skuJsonObject.put(PayuConstants.P_QUANTITY, sku.quantity)
                skuJsonObject.put(PayuConstants.P_SKU_NAME, sku.skuName)
                jsonArray.put(skuJsonObject)
            }
        }

        if (totalSkuAmount == (payuPaymentParams.amount?.toDouble()
                ?: 0.0) && (InternalConfig.selectedOfferInfo == null ||
                    InternalConfig.selectedOfferInfo?.isSkuOffer == true)
        ) {
            val jsonObject = JSONObject()
            jsonObject.put(PayuConstants.P_SKU_DETAILS, jsonArray)
            return Pair(jsonObject,isAutoApplyLocal)
        }

        return Pair(null,isAutoApplyLocal)
    }

    private fun getVerificationModeMapping(verificationModeList: ArrayList<String>?): ArrayList<PayUBeneficiaryVerificationMode>? {
        if (verificationModeList == null) return null
        val result = ArrayList<PayUBeneficiaryVerificationMode>()
        verificationModeList.forEach {
            PayUBeneficiaryVerificationMode.typeFrom(it)?.let { it1 -> result.add(it1) }
        }
        return result
    }
}