package com.payu.checkoutpro.utils

import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.util.Log
import com.payu.base.models.*
import com.payu.base.models.PaymentOptionOfferinfo
import com.payu.checkoutpro.BuildConfig
import com.payu.checkoutpro.R
import com.payu.checkoutpro.factory.HashGenerationHelper
import com.payu.checkoutpro.listeners.PayUInternalHashGenerationListener
import com.payu.checkoutpro.models.PaymentRenderer
import com.payu.checkoutpro.models.V1BaseApiObject
import com.payu.checkoutpro.models.V2BaseApiObject
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_AMAZON_PAY
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_AUBANK
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_AUSF
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_AXIS
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_AXISD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJAJ_12_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJAJ_2_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJAJ_3_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJAJ_6_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJAJ_9_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJFIN
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BAJFINSERV
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BANK_CODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BOB
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BOBD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_FED
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_FEDD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_GOOGLE_PAY
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_GPAY
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_HASH_STRING
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_HDFC
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_HDFCD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ICICI
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ICICID
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_INDUS
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_INDUSIND
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_KOTAK
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_KOTAKD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_LAZYPAY_NAME
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_OLAM
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_OLAMONEY
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ONEC
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ONECARD
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PAYTM_NAME
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PAYZAPP
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PAYZAPP_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PG
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PHONEPE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PHONEPE_BANKCODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PPINAPP
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PPINTENT
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_SODEXO_NAME
import com.payu.india.Model.*
import com.payu.india.Payu.PayuConstants
import com.payu.india.Payu.PayuErrors
import com.payu.paymentparamhelper.PaymentParams
import com.payu.paymentparamhelper.Products
import com.payu.paymentparamhelper.PayuErrors.FIRST_NAME_LENGTH_LIMIT
import com.payu.paymentparamhelper.PayuErrors.PRODUCT_INFO_LENGTH_LIMIT
import com.payu.paymentparamhelper.PayuErrors.TRANSACTION_ID_LENGTH_LIMIT
import com.payu.paymentparamhelper.siparams.SIParams
import com.payu.paymentparamhelper.siparams.SIParamsDetails
import com.payu.paymentparamhelper.siparams.enums.BeneficiaryAccountType
import com.payu.paymentparamhelper.siparams.enums.BillingCycle
import com.payu.paymentparamhelper.siparams.enums.BillingLimit
import com.payu.paymentparamhelper.siparams.enums.BillingRule
import com.payu.ui.model.listeners.PayUCheckoutProListener
import com.payu.ui.model.listeners.PayUHashGenerationListener
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.CPCallbackType
import com.payu.ui.model.utils.SdkUiConstants
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.net.URL
import java.security.SecureRandom
import java.util.*
import java.util.regex.Pattern

/**
 * This class should have common utility methods
 * */
object CommonUtils {

    private val TAG = PayUCheckoutProConstants.COMMONUTILS

    private const val VALIDATE_PHONE_NUMBER_REGEX = "^\\d{10}\$"
    private const val VALIDATE_SODEXO_CARD = "^6375[\\d]+"
    private var hashGenerationParamsFactory: HashGenerationHelper? = null
    private var listenerMap: HashMap<String?,PayUInternalHashGenerationListener> = HashMap()
    internal var checkoutProListener: PayUCheckoutProListener ? = null;
    private var startTime: HashMap<String, Long> = HashMap()


    @Throws(RuntimeException::class)
    internal fun preparePayUbizPaymentParams(
        payUPaymentParams: PayUPaymentParams,
        context: Context,
        isQRScan: Boolean
    ): PaymentParams {
        val payuBizparams = PaymentParams()

        /**
         * For Test Environment, merchantKey = please contact mobile.integration@payu.in with your app name and registered email id
         *
         */
        ParserUtils.isProdEnvironment = payUPaymentParams.isProduction

        if (isValid(payUPaymentParams.key))
            payuBizparams.key = payUPaymentParams.key
        else
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_KEY_IS_MISSING)

        if (isValidAmount(payUPaymentParams.amount, payUPaymentParams.payUSIParams != null, isQRScan))
            payuBizparams.amount = payUPaymentParams.amount
        else
            throw RuntimeException(context.getString(R.string.payu_invalid_amount_error))

        if (!isValid(payUPaymentParams.productInfo))
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_PRODUCT_INFO_IS_MISSING)
        if ((payUPaymentParams.productInfo?.length ?: 0) > PRODUCT_INFO_LENGTH_LIMIT)
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_PRODUCT_INFO_LENGTH_EXCEEDING)
        payuBizparams.productInfo = payUPaymentParams.productInfo

        if (!isValid(payUPaymentParams.firstName))
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_FIRST_NAME_IS_MISSING)
        if ((payUPaymentParams.firstName?.length ?: 0) > FIRST_NAME_LENGTH_LIMIT)
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_FIRST_NAME_LENGTH_EXCEEDING)
        payuBizparams.firstName = payUPaymentParams.firstName

        if (payUPaymentParams.email != null)
            payuBizparams.email = payUPaymentParams.email
        else
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_EMAIL_IS_MISSING)
        if (payUPaymentParams.userToken != null)
            payuBizparams.userToken = payUPaymentParams.userToken

        /**
         * Surl --> Success url is where the transaction response is posted by PayU on successful transaction
         * Furl --> Failre url is where the transaction response is posted by PayU on failed transaction
         */

        val surl = payUPaymentParams.surl
        if (!isValid(surl))
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_SURL_IS_MISSING)
        else if (!isValidUrl(surl))
            throw RuntimeException(PayuConstants.SURL + PayuErrors.INVALID_URL)
        else
            payuBizparams.surl = surl


        val furl = payUPaymentParams.furl
        if (!isValid(furl))
            throw RuntimeException(PayuErrors.MANDATORY_PARAM_FURL_IS_MISSING)
        else if (!isValidUrl(furl))
            throw RuntimeException(PayuConstants.FURL + PayuErrors.INVALID_URL)
        else
            payuBizparams.furl = furl

        var additionalParams: HashMap<String, Any?>? = payUPaymentParams.additionalParams
        if (additionalParams.isNullOrEmpty()) additionalParams = HashMap()

        /*
         * udf1 to udf5 are options params where you can pass additional information related to transaction.
         * If you don't want to use it, then send them as empty string like, udf1=""
         * */
        payuBizparams.udf1 = getParamValue(additionalParams, PayUCheckoutProConstants.CP_UDF1)
        payuBizparams.udf2 = getParamValue(additionalParams, PayUCheckoutProConstants.CP_UDF2)
        payuBizparams.udf3 = getParamValue(additionalParams, PayUCheckoutProConstants.CP_UDF3)
        payuBizparams.udf4 = getParamValue(additionalParams, PayUCheckoutProConstants.CP_UDF4)
        payuBizparams.udf5 = getParamValue(additionalParams, PayUCheckoutProConstants.CP_UDF5)
        payuBizparams.sodexoSourceId =
            getParamValue(additionalParams, PayUCheckoutProConstants.SODEXO_SOURCE_ID)
        payUPaymentParams.additionalParams = additionalParams

        if (payUPaymentParams.phone != null) {
            if (isValidPhone(payUPaymentParams.phone)) payuBizparams.phone =
                payUPaymentParams.phone
            else throw java.lang.RuntimeException(context.getString(R.string.payu_invalid_phone_number))
        }

        /*
         * Transaction Id should be kept unique for each transaction.
         * */
        if (isValidTxnId(payUPaymentParams.transactionId).not())
            throw RuntimeException("InValid transactionId")
        payuBizparams.txnId = payUPaymentParams.transactionId


        payuBizparams.notifyURL = payuBizparams.surl //for lazy pay
        if (null != payUPaymentParams.payUSIParams)
            payuBizparams.siParams = setSIDetails(payUPaymentParams.payUSIParams)

        if (payUPaymentParams.skuDetails != null) {
            val skuCartDetails = prepareSkuCartDetails(payUPaymentParams)
            skuCartDetails?.let { payuBizparams.skuCartDetails = it.toString() }
        }

        /**
         * These are used for store card feature. If you are not using it then user_credentials = "default"
         * user_credentials takes of the form like user_credentials = "merchant_key : user_id"
         * here merchant_key = your merchant key,
         * user_id = unique id related to user like, email, phone number, etc.
         */
        payuBizparams.userCredentials =
            payUPaymentParams.userCredential
        payuBizparams.sdkPlatformData =
            getAnalyticsString(additionalParams[PayUCheckoutProConstants.CP_ANALYTICS_DATA] as? String)
        ParserUtils.payuBizParams = payuBizparams
        payuBizparams.additionalCharges = payUPaymentParams.additionalCharges
        payuBizparams.percentageAdditionalCharges = payUPaymentParams.percentageAdditionalCharges

        if (payUPaymentParams.payUWealthProducts.isNullOrEmpty().not()) {
            val cpTpvProductDetails = payUPaymentParams.payUWealthProducts
            val tpvProductDetails: ArrayList<Products> = ArrayList()
            cpTpvProductDetails?.forEach { cpTpvProductDetail ->
                val tpvProductDetail = Products()
                tpvProductDetail.amount = cpTpvProductDetail.amount
                tpvProductDetail.type = cpTpvProductDetail.type
                tpvProductDetail.plan = cpTpvProductDetail.plan
                tpvProductDetail.scheme = cpTpvProductDetail.scheme
                tpvProductDetail.folio = cpTpvProductDetail.folio
                tpvProductDetail.mfAmcCode = cpTpvProductDetail.mfAmcCode
                tpvProductDetail.mfInvestmentType = cpTpvProductDetail.mfInvestmentType
                tpvProductDetail.mfMemberId = cpTpvProductDetail.mfMemberID
                tpvProductDetail.mfPartner = cpTpvProductDetail.mfPartner
                tpvProductDetail.mfUserId = cpTpvProductDetail.mfUserID
                tpvProductDetail.option = cpTpvProductDetail.option
                tpvProductDetail.receipt = cpTpvProductDetail.receipt
                tpvProductDetails.add(tpvProductDetail)
            }
            payuBizparams.productsList = tpvProductDetails

        }
        payUPaymentParams.addressDetails?.let {
            payuBizparams.lastName = it.lastName
            payuBizparams.address1 = it.address1
            payuBizparams.address2 = it.address2
            payuBizparams.city = it.city
            payuBizparams.state = it.state
            payuBizparams.country = it.country
            payuBizparams.zipCode = it.zipcode
        }

        return payuBizparams
    }

    internal fun getParamValue(additionalParams: HashMap<String, Any?>, key: String): String {
        val paramValue: String
        if (additionalParams[key] as? String != null)
            paramValue = additionalParams[key] as String
        else {
            paramValue = ""
            additionalParams[key] = paramValue
        }
        return paramValue
    }

    /**
     * Checks if the phone number is valid or not
     *
     * @param phone phone number
     * @return true, if phone number is valid else false
     */
    internal fun isValidPhone(phone: String): Boolean {
        val pattern = Pattern.compile(VALIDATE_PHONE_NUMBER_REGEX)
        val matcher = pattern.matcher(phone)
        return matcher.matches()
    }

    internal fun preparePayUConfig(): PayuConfig {

        val payuConfig = PayuConfig()

        if (ParserUtils.isProdEnvironment)
            payuConfig.environment = PayuConstants.PRODUCTION_ENV
        else
            payuConfig.environment = PayuConstants.STAGING_ENV

        return payuConfig
    }

    internal fun getCardScheme(cardBin: String?) = when {
        cardBin.equals("MASTERCARD", ignoreCase = true) -> CardScheme.MAST
        cardBin.equals("MAST", ignoreCase = true) -> CardScheme.MAST
        cardBin.equals("MAES", ignoreCase = true) -> CardScheme.MAES
        cardBin.equals("SMAE", ignoreCase = true) -> CardScheme.SMAE
        cardBin.equals("VISA", ignoreCase = true) -> CardScheme.VISA
        cardBin.equals("AMEX", ignoreCase = true) -> CardScheme.AMEX
        cardBin.equals("MAES", ignoreCase = true) -> CardScheme.MAES
        cardBin.equals("JCB", ignoreCase = true) -> CardScheme.JCB
        cardBin.equals("RUPAY", ignoreCase = true) -> CardScheme.RUPAY
        cardBin.equals("RUPAYCC", ignoreCase = true) -> CardScheme.RUPAYCC
        cardBin.equals("DINR", ignoreCase = true) -> CardScheme.DINR
        cardBin.equals("DINERS", ignoreCase = true) -> CardScheme.DINR
        cardBin.equals("DISCOVER", ignoreCase = true) -> CardScheme.DISCOVER
        else -> CardScheme.UNKNOWN
    }

    internal fun getCardType(cardType: String) = when {
        cardType.equals("CC", ignoreCase = true) -> CardType.CC
        else -> CardType.DC
    }

    internal fun isBankCodeAvailable(bankCode: String, list: ArrayList<PaymentDetails>): Boolean {

        var isBankCodeAvailable = false
        if (bankCode.isEmpty() || list.isEmpty()) return isBankCodeAvailable

        for (item in list) {
            if (item.bankCode.equals(bankCode, ignoreCase = true)) isBankCodeAvailable = true
        }

        return isBankCodeAvailable
    }

    internal fun getDefaultCheckoutOrder(): ArrayList<PaymentMode> {
        val checkoutOrderList = ArrayList<PaymentMode>()
        checkoutOrderList.add(PaymentMode(PaymentType.CARD))
        checkoutOrderList.add(PaymentMode(PaymentType.NB))
        checkoutOrderList.add(PaymentMode(PaymentType.UPI))
        checkoutOrderList.add(PaymentMode(PaymentType.WALLET))
        checkoutOrderList.add(PaymentMode(PaymentType.EMI))
        checkoutOrderList.add(PaymentMode(PaymentType.NEFTRTGS))
        checkoutOrderList.add(PaymentMode(PaymentType.SODEXO))
        checkoutOrderList.add(PaymentMode(PaymentType.BNPL))
        return checkoutOrderList
    }

    /**
     * List of Unsupported bank codes. This list is dynamic and can change with time.
     * */
    internal fun getUnSupportedBankCodesList(): ArrayList<String> {
        val unSupportedBankCodesList = ArrayList<String>()
        unSupportedBankCodesList.add(CP_PPINTENT)
        unSupportedBankCodesList.add(CP_PPINAPP)
        return unSupportedBankCodesList
    }

    internal fun getL1OptionHeaderText(
        context: Context?,
        paymentMode: PaymentMode,
        paymentResponse: PayuResponse,
    ): String {
        when (paymentMode.type) {
            PaymentType.CARD -> {
                if (paymentResponse.isDebitCardAvailable && paymentResponse.isCreditCardAvailable) {
                    return context?.getString(R.string.payu_cards_pay_using_credit_debit_text)!!
                } else if (paymentResponse.isDebitCardAvailable) {
                    return context?.getString(R.string.payu_cards_pay_using_debit_text)!!
                } else if (paymentResponse.isCreditCardAvailable) {
                    return context?.getString(R.string.payu_cards_pay_using_credit_text)!!
                }
            }
        }
        return paymentMode.name
    }

// TODO need to remove after analysis
    internal fun getL1OptionText(
        context: Context?,
        paymentMode: PaymentMode,
        paymentResponse: PayuResponse,
        isGenericIntentSupported: Boolean = false
    ): String {

        var l1String = ""
        val isSIMode = ParserUtils.isSIMode

        Log.d(TAG, "PaymentType =" + paymentMode.type)

        when (paymentMode.type) {
            PaymentType.EMI -> {
                if (context == null) return l1String
                return when {
                    paymentResponse.isCCEmiAvailable -> {
                        if (paymentResponse.isDCEmiAvailable) {
                            if (paymentResponse.isCardLessEmiAvailable) {
                                context?.getString(R.string.credit_debit_cardless)
                            } else {
                                context?.getString(R.string.credit_debit)
                            }
                        } else context?.getString(R.string.credit)
                    }

                    paymentResponse.isDCEmiAvailable -> context?.getString(R.string.debit)
                    else -> context?.getString(R.string.cardless)
                }
            }

            PaymentType.WALLET -> {
                when {

                    paymentMode.optionDetail?.size!! > 3 -> {
                        val randomThreeWallets = getRandomElement(
                            paymentMode.optionDetail!!,
                            arrayListOf(CP_PAYTM_NAME, CP_PHONEPE, CP_AMAZON_PAY)
                        )

                        return randomThreeWallets[0].bankName + ", " + randomThreeWallets[1].bankName + ", " + randomThreeWallets[2].bankName + context?.getString(
                            R.string.and_more
                        )
                    }

                    else -> {
                        for (i in 0 until paymentMode.optionDetail?.size!!) {
                            if (i == paymentMode.optionDetail?.size!! - 1) {
                                l1String = if (paymentMode.optionDetail?.size == 1)
                                    paymentMode.optionDetail?.get(i)?.bankName!!
                                else
                                    l1String + context?.getString(R.string.and) + paymentMode.optionDetail?.get(
                                        i
                                    )?.bankName
                            } else {
                                l1String += if (i == paymentMode.optionDetail?.size!! - 2) paymentMode.optionDetail?.get(
                                    i
                                )?.bankName else
                                    l1String + paymentMode.optionDetail?.get(i)?.bankName + ", "
                            }
                        }
                        return l1String

                    }
                }
            }

            PaymentType.BNPL -> {
                when {
                    paymentMode.optionDetail?.size!! > 3 -> {
                        return BNPLStringForAboveThreeOption(paymentMode, context)
                    }

                    else -> {
                        for (i in 0 until paymentMode.optionDetail?.size!!) {
                            if (i == paymentMode.optionDetail?.size!! - 1) {
                                l1String = if (paymentMode.optionDetail?.size == 1)
                                    paymentMode.optionDetail?.get(i)?.bankName!!
                                else
                                    l1String + context?.getString(R.string.and) + paymentMode.optionDetail?.get(
                                        i
                                    )?.bankName
                            } else {
                                l1String += if (i == paymentMode.optionDetail?.size!! - 2) paymentMode.optionDetail?.get(
                                    i
                                )?.bankName else
                                    l1String + paymentMode.optionDetail?.get(i)?.bankName + ", "
                            }
                        }
                        return l1String

                    }
                }
            }

            PaymentType.NB -> {

                var upBankCounts = 0
                for (option in paymentMode.optionDetail!!) {
                    if (!option.isBankDown) {
                        upBankCounts++
                    }
                }
                val bankCountString: String
                if (upBankCounts > 10) {
                    upBankCounts -= (upBankCounts % 10)
                    bankCountString = "$upBankCounts+"
                } else {
                    bankCountString = upBankCounts.toString()
                }
                return "" + bankCountString + context?.getString(
                    R.string.banks_supported
                )
            }

            PaymentType.L1_OPTION -> {
                when (paymentMode.name) {
                    CP_PHONEPE -> return context?.getString(R.string.pay_directly_from_your_phonepe_account)!!
                    CP_PAYTM_NAME -> return context?.getString(R.string.pay_using_paytm_wallet_or_upi)!!
                    CP_GOOGLE_PAY -> return context?.getString(R.string.pay_directly_from_google_pay)!!
                    CP_LAZYPAY_NAME -> return context?.getString(R.string.pay_directly_from_lazy_pay)!!
                    CP_SODEXO_NAME -> return ""
                }
            }

            PaymentType.CARD -> {
                if ((!isSIMode && paymentResponse.isDebitCardAvailable && paymentResponse.isCreditCardAvailable)
                    || (isSIMode && paymentResponse.isDebitCardAvailableFoSI && paymentResponse.isCreditCardAvailableFoSI)
                ) {
                    return context?.getString(R.string.pay_using_debit_card_or_credit_card)!!
                } else if ((!isSIMode && paymentResponse.isDebitCardAvailable)
                    || (isSIMode && paymentResponse.isDebitCardAvailableFoSI)
                ) {
                    return context?.getString(R.string.pay_using_debit_card)!!
                } else if ((!isSIMode && paymentResponse.isCreditCardAvailable)
                    || (isSIMode && paymentResponse.isCreditCardAvailableFoSI)
                ) {
                    return context?.getString(R.string.pay_using_credit_card)!!
                }

            }

            PaymentType.UPI -> {
                if (isSIMode && ((paymentResponse.isUPIAvailableFoSI) || (isGenericIntentSupported && paymentResponse.isUPIAvailableFoSI))) {
                    var L1Text: String = context?.getString(R.string.payu_using_your_vpa_si)!!
                    val suportedUpiApps =
                        paymentResponse.upiSISupportedApps ?: DataConstant.supportedUpiSiApps() //TODO will remove default list after BE changes
                    if (suportedUpiApps == null) {
                        L1Text = context.getString(R.string.payu_using_your_vpa_app)
                    } else if (suportedUpiApps.size > 4) {
                        L1Text = context.getString(R.string.payu_using_your_vpa_app)
                    } else {
                        for (i in 0 until suportedUpiApps.size) {
                            if (i == suportedUpiApps.size - 1)
                                L1Text += " " + suportedUpiApps[i]
                            else if (i == suportedUpiApps.size - 2)
                                L1Text += " " + suportedUpiApps[i] + " or"
                            else L1Text += " " + suportedUpiApps[i] + ","
                        }
                        return L1Text
                    }
                    return L1Text
                } else if (!isSIMode && isGenericIntentSupported && paymentResponse.isUpiAvailable && paymentResponse.isGoogleTezOmniAvailable) {
                    when {
                        paymentMode.optionDetail?.get(2)?.optionList?.size!! > 3 -> {

                            val randomThreeUpiApps =
                                getRandomElement(
                                    paymentMode.optionDetail?.get(2)?.optionList!!,
                                    arrayListOf(CP_GPAY, CP_PHONEPE, CP_PAYTM_NAME)
                                )
                            return randomThreeUpiApps[0].bankName + ", " +
                                    randomThreeUpiApps[1].bankName + ", " + randomThreeUpiApps[2].bankName +
                                    context?.getString(R.string.or_any_other_upi_app)
                        }

                        else -> {
                            for (i in 0 until paymentMode.optionDetail?.get(2)?.optionList?.size!!) {
                                l1String =
                                    if (i == paymentMode.optionDetail?.get(2)?.optionList?.size!! - 1) {
                                        if (paymentMode.optionDetail?.get(2)?.optionList?.size == 1) {
                                            paymentMode.optionDetail?.get(2)?.optionList?.get(i)?.bankName + context?.getString(
                                                R.string.upi_app
                                            )
                                        } else
                                            l1String + context?.getString(R.string.or) + paymentMode.optionDetail?.get(
                                                2
                                            )?.optionList?.get(i)?.bankName + context?.getString(R.string.upi_app)
                                    } else {
                                        l1String + paymentMode.optionDetail?.get(2)?.optionList?.get(
                                            i
                                        )?.bankName + ", "
                                    }
                            }
                            return l1String
                        }
                    }
                } else if (!isSIMode && isGenericIntentSupported && paymentResponse.isUpiAvailable) {
                    when {
                        paymentMode.optionDetail?.get(1)?.optionList?.size!! > 3 -> {

                            val randomThreeUpiApps =
                                getRandomElement(
                                    paymentMode.optionDetail?.get(1)?.optionList!!,
                                    arrayListOf(CP_GPAY, CP_PHONEPE, CP_PAYTM_NAME)
                                )
                            return randomThreeUpiApps[0].bankName + ", " +
                                    randomThreeUpiApps[1].bankName + ", " + randomThreeUpiApps[2].bankName +
                                    context?.getString(R.string.or_any_other_upi_app)
                        }

                        else -> {
                            for (i in 0 until paymentMode.optionDetail?.get(1)?.optionList?.size!!) {
                                l1String =
                                    if (i == paymentMode.optionDetail?.get(1)?.optionList?.size!! - 1) {
                                        if (paymentMode.optionDetail?.get(1)?.optionList?.size == 1) {
                                            paymentMode.optionDetail?.get(1)?.optionList?.get(i)?.bankName + context?.getString(
                                                R.string.upi_app
                                            )
                                        } else
                                            l1String + context?.getString(R.string.or) + paymentMode.optionDetail?.get(
                                                1
                                            )?.optionList?.get(i)?.bankName + context?.getString(R.string.upi_app)
                                    } else {
                                        l1String + paymentMode.optionDetail?.get(1)?.optionList?.get(
                                            i
                                        )?.bankName + ", "
                                    }
                            }
                            return l1String
                        }
                    }
                } else if ((!isSIMode && isGenericIntentSupported && !paymentResponse.isUpiAvailable) || (isSIMode && isGenericIntentSupported)) {
                    when {
                        paymentMode.optionDetail?.get(0)?.optionList?.size!! > 3 -> return paymentMode.optionDetail?.get(
                            0
                        )?.optionList?.get(0)?.bankName + ", " +
                                paymentMode.optionDetail?.get(0)?.optionList?.get(1)?.bankName + ", " + paymentMode.optionDetail?.get(
                            0
                        )?.optionList?.get(2)?.bankName +
                                context?.getString(R.string.or_any_other_upi_app)

                        else -> {
                            val upiAppsList = paymentMode.optionDetail?.get(0)?.optionList
                            for (i in 0 until upiAppsList?.size!!) {
                                l1String = if (i == upiAppsList.size - 1) {
                                    if (upiAppsList.size == 1) {
                                        upiAppsList.get(i).bankName + context?.getString(R.string.upi_app)
                                    } else
                                        l1String + context?.getString(R.string.or) + upiAppsList.get(
                                            i
                                        ).bankName + context?.getString(R.string.upi_app)
                                } else {
                                    l1String + upiAppsList.get(i).bankName + ", "

                                }
                            }
                            return l1String
                        }
                    }
                } else if (!isSIMode && !isGenericIntentSupported && paymentResponse.isUpiAvailable) {
                    return context?.getString(R.string.payu_using_your_upi_id_or_vpa)!!
                }

            }

            PaymentType.NEFTRTGS -> {
                if (context != null)
                    return context.getString(R.string.payu_pay_with_neft_rtgs)
            }

        }

//    else if(paymentMode?.optionDetail?.size!!)
        return ""
    }

    /**
     * This function gets random 3 payment options, by prioritizing the payment options available in specificList
     */

    private fun getRandomElement(
        list: ArrayList<PaymentOption>, specificList: ArrayList<String>
    ): List<PaymentOption> {
        val rand = SecureRandom()
        val paymentOptionList = list.clone() as ArrayList<PaymentOption>
        val map = HashMap<String, PaymentOption>()

        for (listItem in list) {
            map[listItem.bankName] = listItem

        }

        val newList = ArrayList<PaymentOption>()

        for (item in specificList) {
            if (map.containsKey(item)) {
                val idx = paymentOptionList.indexOf(map[item])
                newList.add(map[item]!!)
                paymentOptionList.removeAt(idx)
            }
        }

        for (i in 0 until (3 - newList.size)) {
            val randomIndex = rand.nextInt(paymentOptionList.size)
            newList.add(paymentOptionList[randomIndex])
            paymentOptionList.removeAt(randomIndex)
        }
        return newList
    }

    private fun BNPLStringForAboveThreeOption(paymentMode: PaymentMode, context: Context?): String {
        val bankName = CP_LAZYPAY_NAME
        val randomThreeWallets =
            getRandomElement(paymentMode.optionDetail!!, arrayListOf(CP_LAZYPAY_NAME))

        var returnString =
            randomThreeWallets[0].bankName + ", " + randomThreeWallets[1].bankName + ", " + randomThreeWallets[2].bankName + context?.getString(
                R.string.and_more
            )
        for (i in 0 until 3) {
            if (randomThreeWallets[i].bankName == bankName) {
                returnString = bankName + "," + returnString.subSequence(
                    returnString.indexOf(',') + 1,
                    returnString.length
                ).toString()
            }
        }
        return returnString
    }

    internal fun getPaymentOption(paymentType: PaymentType) = when (paymentType) {
        PaymentType.UPI -> UPIOption()
        PaymentType.WALLET -> WalletOption()
        else -> PaymentOption()
    }

    internal fun getFormattedBankName(bankName: String, bankCode: String): String {
        val newTitleMap = getNewTitleMap()
        return if (bankCode.isNotEmpty() && newTitleMap.containsKey(bankCode))
            newTitleMap[bankCode]!!
        else bankName
    }

    internal fun getNewTitleMap(): HashMap<String, String> {
        val map = HashMap<String, String>()
        map[CP_OLAM] = CP_OLAMONEY
        map[CP_PHONEPE_BANKCODE] = CP_PHONEPE
        map[CP_PAYZAPP_BANKCODE] = CP_PAYZAPP
        map[CP_BAJAJ_2_BANKCODE] = CP_BAJFINSERV
        map[CP_BAJAJ_3_BANKCODE] = CP_BAJFINSERV
        map[CP_BAJAJ_6_BANKCODE] = CP_BAJFINSERV
        map[CP_BAJAJ_9_BANKCODE] = CP_BAJFINSERV
        map[CP_BAJAJ_12_BANKCODE] = CP_BAJFINSERV
        return map
    }

    internal fun getOtherParamsMap(pg: String, bankCode: String): HashMap<String, Any?> {
        val hashMap = HashMap<String, Any?>()
        hashMap[CP_PG] = pg
        hashMap[CP_BANK_CODE] = bankCode
        return hashMap
    }

    internal fun putKeyValueInOtherParams(
        paymentOption: PaymentOption,
        key: String,
        value: String?
    ) {
        val hashMap = paymentOption.otherParams as? HashMap<String, Any?>
        hashMap?.put(key, value)
    }

    internal fun generateHash(
        context: Context,
        payUPaymentParams: PayUPaymentParams,
        checkoutListener: PayUCheckoutProListener?,
        baseApiObject: V1BaseApiObject
    ) {
        listenerMap[baseApiObject.getHashName()] = baseApiObject
        val hashGenerationParamsFactory =
            HashGenerationHelper(
                payUPaymentParams = payUPaymentParams
            )
        val map = hashGenerationParamsFactory.getHashData(baseApiObject.getHashName())
        val hashString = map[CP_HASH_STRING]
        startTime[baseApiObject.getHashName()] = System.currentTimeMillis()
        CPAnalyticsUtils.logGenerateHashEvent(baseApiObject.getHashName(), hashString, context)
        checkoutListener?.generateHash(
            map,
            object : PayUHashGenerationListener{
                override fun onHashGenerated(map: HashMap<String, String?>) {
                    val timeTaken = System.currentTimeMillis() - (startTime[baseApiObject.getHashName()] ?: 0)
                    if (map.isEmpty()) {
                        val errorResponse = ErrorResponse()
                        errorResponse.errorMessage =
                            "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}"
                        CPAnalyticsUtils.logHashGeneratedEventError(baseApiObject.getHashName(), timeTaken, "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}", context)
                        AnalyticsUtils.logCPCallbackEventKibana(context, CPCallbackType.Error)
                        checkoutListener.onError(errorResponse)
                        return
                    }
                    val iterator = map.iterator()
                    var key = ""
                    while (iterator.hasNext()) {
                        val obj = iterator.next()
                        key = obj.key
                        if (obj.value.isNullOrEmpty()) {
                            val errorResponse = ErrorResponse()
                            errorResponse.errorMessage =
                                "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}"
                            CPAnalyticsUtils.logHashGeneratedEventError(baseApiObject.getHashName(), timeTaken, "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}", context)
                            AnalyticsUtils.logCPCallbackEventKibana(context, CPCallbackType.Error)
                            checkoutListener.onError(errorResponse)

                        }
                    }
                    CPAnalyticsUtils.logHashGeneratedEventSuccess(baseApiObject.getHashName(), timeTaken, map[baseApiObject.getHashName()], context)
                    listenerMap[key]?.onHashGenerated(map)
                    listenerMap.remove(key)


                }

            }
        )

    }

    internal fun generateHash(
        paymentOption: PaymentOption?,
        payUPaymentParams: PayUPaymentParams,
        paymentParams: PaymentParams? = null,
        checkoutListener: PayUCheckoutProListener?,
        renderer: PaymentRenderer,
        hashListener: PayUHashGenerationListener
    ) {

        with(HashGenerationHelper(payUPaymentParams,paymentParams,  paymentOption)) {
            checkoutListener?.generateHash(
                getHashData(renderer.getHashName()),
                hashListener
            )
        }

    }

    internal fun generateV2Hash(
        baseApiObject: V2BaseApiObject) {
        listenerMap[baseApiObject.getHashName()] = baseApiObject

        val map = hashGenerationParamsFactory?.getV2Hashdata(baseApiObject.getHashName())?: hashMapOf()
        val hashString = map[CP_HASH_STRING] ?: ""
        startTime[baseApiObject.getHashName()] = System.currentTimeMillis()
        CPAnalyticsUtils.logGenerateHashEvent(baseApiObject.getHashName(), hashString)

        checkoutProListener?.generateHash(
            map, object : PayUHashGenerationListener {
                override fun onHashGenerated(map: HashMap<String, String?>) {
                    val timeTaken = System.currentTimeMillis() - (startTime[baseApiObject.getHashName()] ?: 0)
                    if (map.isEmpty()) {
                        val errorResponse = ErrorResponse()
                        errorResponse.errorMessage =
                            "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}"
                        CPAnalyticsUtils.logHashGeneratedEventError(baseApiObject.getHashName(), timeTaken, "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}")
                        checkoutProListener?.onError(errorResponse)
                        return
                    }
                    val iterator = map.iterator()
                    var key = ""
                    while (iterator.hasNext()) {
                        val obj = iterator.next()
                        key = obj.key
                        if (obj.value.isNullOrEmpty()) {
                            val errorResponse = ErrorResponse()
                            errorResponse.errorMessage =
                                "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}"
                            CPAnalyticsUtils.logHashGeneratedEventError(baseApiObject.getHashName(), timeTaken, "${baseApiObject.getHashName()} ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}")
                            checkoutProListener?.onError(errorResponse)

                        }
                    }
                    CPAnalyticsUtils.logHashGeneratedEventSuccess(baseApiObject.getHashName(), timeTaken, map[baseApiObject.getHashName()])

                    listenerMap[key]?.onHashGenerated(map)
                    listenerMap.remove(key)
                }

            }
        )
    }

    internal fun setV2hashData(signingString: String, payUPaymentParams: PayUPaymentParams) {
        hashGenerationParamsFactory = HashGenerationHelper(
            payUPaymentParams = payUPaymentParams,
        )
        hashGenerationParamsFactory?.setV2HashData(signingString)
    }

    internal fun getV2HashData(baseApiObject: V2BaseApiObject): HashMap<String, String?> {
        return hashGenerationParamsFactory!!.getV2Hashdata(baseApiObject.getHashName())
    }

    internal fun isValid(key: String?): Boolean {
        return !(key == null || key.isEmpty())
    }

    internal fun isValidAmount(amount: String?, isSiPayment: Boolean, isQRPayment: Boolean = false): Boolean {
        return if (isQRPayment) {
            if (amount != null) {
                amount.toDoubleOrNull() != null
            } else {
                true
            }
        } else {
            amount != null && amount.toDoubleOrNull() != null && (isSiPayment || amount.toDouble() >= 1.0)
        }
    }

    internal fun isValidUrl(url: String?): Boolean {

        val isValid: Boolean

        isValid = if (url != null && url.isNotEmpty()) {
            try {
                URL(url).toURI()
                true
            } catch (e: Exception) {
                false
            }

        } else
            false

        return isValid
    }

    internal fun getHash(hashName: String, payUPaymentParams: PayUPaymentParams): String? {
        var hash: String? = null

        val additionalParams = payUPaymentParams.additionalParams
        if (additionalParams.isNullOrEmpty())
            return hash

        if (additionalParams.containsKey(hashName))
            hash = additionalParams[hashName] as? String

        return hash
    }


    internal fun getEmiTenureListForBank(emiOption: EMIOption): ArrayList<PaymentOption> {
        var emiList = ArrayList<PaymentOption>()
        val categoryList = getEmiCategoryList(emiOption.emiType, ParserUtils.emiList)
        if (!categoryList.isNullOrEmpty()) {
            for (item in categoryList) {
                if (item.bankName.equals(emiOption.bankName, ignoreCase = true)) {
                    if (!item.optionList.isNullOrEmpty()) emiList = item.optionList!!
                }
            }
        }
        return emiList
    }

    internal fun getEmiCategoryList(
        emiType: EmiType?,
        fullList: ArrayList<PaymentOption>?
    ): ArrayList<PaymentOption>? {
        if (emiType == null || fullList.isNullOrEmpty()) return null
        val bankName = when (emiType) {
            EmiType.CC -> PayUCheckoutProConstants.CP_CREDIT_CARD
            EmiType.DC -> PayUCheckoutProConstants.CP_DEBIT_CARD
            EmiType.CARD_LESS -> PayUCheckoutProConstants.CP_CARDLESS
        }

        for (item in fullList) {
            if (item.bankName.equals(bankName, ignoreCase = true)) {
                return item.optionList
            }
        }
        return null
    }


    internal fun updatePaymentOptionWithMCPList(
        mcpConversionBeans: ArrayList<McpConversionBean>?,
        paymentOption: PaymentOption
    ): ArrayList<PaymentOption>? {
        if (mcpConversionBeans.isNullOrEmpty()) return null
        val paymentOptionList = ArrayList<PaymentOption>()
        for (item in mcpConversionBeans) {
            if (paymentOption is SavedCardOption) {
                val savedCardOption: SavedCardOption =
                    prepareCardOptionClone(
                        SavedCardOption(),
                        paymentOption,
                        item.lookupId,
                        item.offerAmount,
                        item.offerCurrency
                    ) as SavedCardOption
                savedCardOption.cardToken = paymentOption.cardToken
                paymentOptionList.add(savedCardOption)
            } else if (paymentOption is CardOption) {
                val cardOption: CardOption =
                    prepareCardOptionClone(
                        CardOption(),
                        paymentOption,
                        item.lookupId,
                        item.offerAmount,
                        item.offerCurrency
                    )
                paymentOptionList.add(cardOption)
            }
        }
        return paymentOptionList
    }

    internal fun prepareCardOptionClone(
        newCardOption: CardOption,
        oldCardOption: CardOption,
        lookupId: String,
        offerAmount: String,
        offerCurrency: String
    ): CardOption {
        newCardOption.paymentType = oldCardOption.paymentType
        newCardOption.cardBinInfo = oldCardOption.cardBinInfo
        newCardOption.cardNumber = oldCardOption.cardNumber
        newCardOption.additionalCharge = oldCardOption.cardBinInfo?.additionalCharge
        newCardOption.gstPercentageValue = oldCardOption.cardBinInfo?.gstPercentageValue
        newCardOption.nameOnCard = oldCardOption.nameOnCard
        newCardOption.cvv = oldCardOption.cvv
        newCardOption.expiryMonth = oldCardOption.expiryMonth
        newCardOption.expiryYear = oldCardOption.expiryYear
        newCardOption.shouldSaveCard = oldCardOption.shouldSaveCard
        newCardOption.cardAlias = oldCardOption.cardAlias
        newCardOption.lookupId = lookupId
        newCardOption.convertedAmount = offerAmount
        newCardOption.convertedCurrency = offerCurrency
        return newCardOption
    }

    internal fun isCardSchemeSupportedForMCP(
        supportedCardSchemes: ArrayList<String>?,
        cardBinInfo: CardBinInfo?
    ): Boolean {
        if (supportedCardSchemes.isNullOrEmpty() || cardBinInfo == null) return false
        for (cardScheme in supportedCardSchemes) {
            if (!cardBinInfo.isDomestic && cardScheme.equals(
                    cardBinInfo.cardScheme?.name,
                    ignoreCase = true
                )
            )
                return true
        }
        return false
    }

    internal fun isSdkAvailable(className: String?): Boolean {
        try {
            if (null != className && className.isNotEmpty()) {
                val wrapperClassLoader =
                    CommonUtils::class.java.classLoader
                val wrapperClass =
                    wrapperClassLoader?.loadClass(className)
                return true
            }
        } catch (e: Exception) {
            e.printStackTrace()

        }
        return false
    }

    internal fun generateVasForOlaMoney(paymentParams: PaymentParams?): String {
        if (null != paymentParams) {
            val jsonObject = JSONObject()
            jsonObject.put("amount", paymentParams.amount)
            jsonObject.put("txnid", paymentParams.txnId)
            jsonObject.put("mobile_number", paymentParams.phone.toString().trim { it <= ' ' })
            jsonObject.put("first_name", paymentParams.firstName)
            jsonObject.put("bankCode", "OLAM")
            jsonObject.put("email", "")
            jsonObject.put("last_name", "")
            return jsonObject.toString()
        }
        return ""
    }

    internal fun <T> getValueFromPaymentOption(key: String, map: HashMap<String, Any?>?): T? {
        var value: T? = null
        if (map != null
            && map.containsKey(key)
            && (map[key] as? T) != null
        ) {
            value = map[key] as T
        }
        return value
    }

    private fun getAnalyticsString(analyticsData: String?): String {
        val analyticsJsonArray = JSONArray()

        if (!analyticsData.isNullOrEmpty()) {
            try {
                val analyticsJsonObject = JSONObject(analyticsData)
                analyticsJsonArray.put(analyticsJsonObject)
            } catch (e: JSONException) {
                e.printStackTrace()
            }
        }
        val analyticsJsonObject = JSONObject()
        analyticsJsonObject.put(PayuConstants.NAME_KEY, PayUCheckoutProConstants.CP_NAME_VALUE)
        analyticsJsonObject.put(PayuConstants.PLATFORM_KEY, PayuConstants.PLATFORM_VALUE)
        analyticsJsonObject.put(PayuConstants.VERSION_KEY, BuildConfig.VERSION_NAME)
        analyticsJsonArray.put(analyticsJsonObject)
        return analyticsJsonArray.toString()
    }

    internal fun getEligibleEmiBinsForBank(
        bankShortName: String?,
        eligibleBinsList: ArrayList<EligibleEmiBins>
    ): EligibleEmiBins? {
        var eligibleEmiBins: EligibleEmiBins? = null
        if (bankShortName.isNullOrEmpty()) return eligibleEmiBins
        for (item in eligibleBinsList) {
            if (item.bankShortName.equals(bankShortName, ignoreCase = true)) {
                eligibleEmiBins = item
                break
            }
        }
        return eligibleEmiBins
    }


    internal fun getOfferIdentifier(paymentOption: PaymentOption) = when (paymentOption) {
        is SavedCardOption -> 2
        is CardOption -> 1
        else -> 3
    }

    internal fun getVar3ForOffersCall(
        paymentOption: PaymentOption,
        userCredentials: String
    ) = when (paymentOption) {
        is SavedCardOption -> userCredentials
        is CardOption -> paymentOption.cardNumber
        else -> paymentOption.paymentType!!.name
    }

    internal fun isOfferAvailableForBank(
        bankCode: String?,
        payuOfferList: ArrayList<PayuOffer>
    ): Boolean {
        if (payuOfferList.isNullOrEmpty()) return false
        for (offer in payuOfferList) {
            if (!offer.status.isNullOrEmpty()
                && !offer.status.equals("0", ignoreCase = true)
                && !offer.allowedOn.isNullOrEmpty()
                && offer.allowedOn.contains(bankCode)
            ) return true
        }
        return false
    }

    internal fun getOfferText(
        identifier: String,
        bestUserOffer: PayuOffer?
    ): String {
        var offerText = ""
        if (bestUserOffer != null) {
            val formattedAmount: String = getFormattedAmount(bestUserOffer.discount)
            when (identifier.toInt()) {
                2 -> offerText = "Offer applied: Save ₹$formattedAmount on this order"
                3, 1 -> offerText = "Save ₹$formattedAmount on this order"
            }
        }
        return offerText
    }

    /**
     * This function will format the amount as below -
     * 1000 -> 1,000
     * 100000 -> 1,00,000
     * */
    internal fun getFormattedAmount(amount: String?): String {
        if (amount.isNullOrEmpty()) return ""
        val stringBuilder = StringBuilder()
        var amountString: String = amount
        var decimalString = ""
        if (amount.contains(".")) {
            amountString = amount.split(".")[0]
            decimalString = amount.split(".")[1]
        }
        val amountArray: CharArray = amountString.toCharArray()
        var a = 0
        var b = 0
        for (i in amountArray.indices.reversed()) {
            if (a < 3) {
                stringBuilder.append(amountArray[i])
                a++
            } else if (b < 2) {
                if (b == 0) {
                    stringBuilder.append(",")
                    stringBuilder.append(amountArray[i])
                    b++
                } else {
                    stringBuilder.append(amountArray[i])
                    b = 0
                }
            }
        }
        return if (decimalString.isEmpty()) stringBuilder.reverse()
            .toString() else stringBuilder.reverse().toString() + ".$decimalString"
    }

    /**
     * This returns best offer for store cards
     * */
    internal fun getBestUserOffer(availableCardOffers: ArrayList<PayuOffer>): PayuOffer? {
        var bestOffer: PayuOffer? = null
        for (offer in availableCardOffers) {
            if (offer.discount.isNullOrEmpty() || offer.discount.toDoubleOrNull() == null || offer.discount.toDouble() == 0.0) continue
            if (bestOffer == null) bestOffer = offer
            else if (offer.discount.toDouble() > bestOffer.discount.toDouble()) bestOffer = offer
        }
        return bestOffer
    }

    /**
     * This returns best offer for payment modes other than store cards
     * */
    internal fun getBestOffer(
        availableCardOffers: ArrayList<PayuOffer>,
        bankCode: String?
    ): PayuOffer? {
        var bestOffer: PayuOffer? = null
        for (offer in availableCardOffers) {
            if (offer.discount.isNullOrEmpty() || offer.discount.toDoubleOrNull() == null || offer.discount.toDouble() == 0.0) continue
            if (!offer.allowedOn.isNullOrEmpty() && offer.allowedOn.contains(bankCode)) {
                if (bestOffer == null) bestOffer = offer
                else if (offer.discount.toDouble() > bestOffer.discount.toDouble()) bestOffer =
                    offer
            }
        }
        return bestOffer
    }

    internal fun getOfferDetailsForToken(
        cardToken: String,
        userOffersList: ArrayList<PayuUserOffer>
    ): PayuUserOffer? {
        if (cardToken.isEmpty() || userOffersList.isEmpty()) return null
        for (userOffer in userOffersList) {
            if (userOffer.storedCard.cardToken.equals(cardToken, ignoreCase = true)) {
                val validOffersList = getValidOffersList(userOffer.availableCardOffers)
                if (validOffersList.isNullOrEmpty()) continue
                else {
                    userOffer.availableCardOffers = validOffersList
                    return userOffer
                }
            } else continue
        }
        return null
    }

    internal fun getValidOffersList(availableCardOffers: ArrayList<PayuOffer>?): ArrayList<PayuOffer>? {
        if (availableCardOffers.isNullOrEmpty()) return null
        val validOffersList = ArrayList<PayuOffer>()
        for (offer in availableCardOffers) {
            if (offer.discount != null) validOffersList.add(offer) else continue
        }
        return validOffersList
    }

    internal fun isOfferDetailsAlreadyFetched(paymentType: PaymentType?): Boolean {
        var isOfferFetched = false
        if (paymentType != null) {
            isOfferFetched = when (paymentType) {
                PaymentType.NB -> ParserUtils.isNBOfferFetched
                else -> false
            }
        }
        return isOfferFetched
    }

    internal fun getPaymentModeFromList(
        moreOptionsList: ArrayList<PaymentMode>?,
        paymentType: PaymentType?
    ): PaymentMode? {
        if (moreOptionsList.isNullOrEmpty() || paymentType == null) return null
        var paymentMode: PaymentMode? = null
        for (mode in moreOptionsList) {
            if (mode.type == paymentType) paymentMode = mode
        }
        return paymentMode
    }

    internal fun getBankDownStatusForCards(bankname: String): Boolean {
        var bankDownStatus = false
        if (bankname.isEmpty() || ParserUtils.issuingBankDownList.isNullOrEmpty()) return bankDownStatus
        for (item in ParserUtils.issuingBankDownList!!)
            if (bankname.equals(item))
                bankDownStatus = true
        return bankDownStatus
    }

    internal fun getAdditionalChargesForCardBin(
        cardBinInfo: CardBinInfo?,
        convenienceFeeResponse: PayuResponse?
    ): Double {
        val additionalCharge = 0.0
        if (cardBinInfo?.cardType == null || cardBinInfo.cardScheme == null) return additionalCharge
        val list =
            if (cardBinInfo.cardType == CardType.CC) convenienceFeeResponse?.creditCard else convenienceFeeResponse?.debitCard
        return if (cardBinInfo.cardType == CardType.CC && (cardBinInfo.cardScheme?.name.equals(
                CardScheme.VISA.name
            ) || cardBinInfo.cardScheme?.name.equals(CardScheme.MAST.name))
        ) getAdditionalChargesForBankCode(
            CardType.CC.name,
            convenienceFeeResponse?.creditCard
        ) //BankCode for VISA and MASTER is CC for Credit cards
        else getAdditionalChargesForBankCode(cardBinInfo.cardScheme?.name, list)
    }

    internal fun getGSTForCardBin(
        cardBinInfo: CardBinInfo?,
        convenienceFeeResponse: PayuResponse?
    ): Double? {
        val gst = 0.0
        if (cardBinInfo?.cardType == null || convenienceFeeResponse == null) return gst
        return if (cardBinInfo.cardType == CardType.CC)
            convenienceFeeResponse.taxSpecification?.ccTaxValue?.toDouble() //BankCode for VISA and MASTER is CC for Credit cards
        else convenienceFeeResponse.taxSpecification?.dcTaxValue?.toDouble()
    }

    internal fun getAdditionalChargesForBankCode(
        bankCode: String?,
        list: ArrayList<PaymentDetails>?
    ): Double {
        var additionalCharge = 0.0
        if (list.isNullOrEmpty() || bankCode.isNullOrEmpty()) return additionalCharge
        for (item in list) {
            if (item.bankCode.equals(bankCode, ignoreCase = true)) {
                try {
                    additionalCharge =
                        String.format("%.2f", item.additionalCharge.toDouble()).toDouble()
                } catch (e: java.lang.Exception) {
                    e.printStackTrace()
                }
            }
        }
        return additionalCharge
    }

    internal fun getBankDownStatusForBankCode(
        bankCode: String?,
        list: ArrayList<PaymentDetails>?
    ): Boolean {
        var isBankDown = false
        if (list.isNullOrEmpty() || bankCode.isNullOrEmpty()) return isBankDown
        for (item in list) {
            if (item.bankCode.equals(bankCode, ignoreCase = true)) {
                try {
                    isBankDown = item.isBankDown
                } catch (e: java.lang.Exception) {
                    e.printStackTrace()
                }
            }
        }
        return isBankDown
    }

    internal fun getCardCategoryFromScheme(cardScheme: CardScheme) = when {
        cardScheme == CardScheme.MAST -> "MAST"
        cardScheme == CardScheme.MAES -> "MAES"
        cardScheme == CardScheme.SMAE -> "SMAE"
        cardScheme == CardScheme.VISA -> "VISA"
        cardScheme == CardScheme.AMEX -> "AMEX"
        cardScheme == CardScheme.JCB -> "JCB"
        cardScheme == CardScheme.RUPAY -> "RUPAY"
        cardScheme == CardScheme.RUPAYCC -> "RUPAYCC"
        cardScheme == CardScheme.DINR -> "DINR"
        cardScheme == CardScheme.DISCOVER -> "DISCOVER"
        else -> ""
    }

    internal fun getBeneficiaryAccountTypeMapping(beneficiaryAccountType: PayUBeneficiaryAccountType?) =
        when (beneficiaryAccountType) {
            PayUBeneficiaryAccountType.CURRENT -> {
                BeneficiaryAccountType.CURRENT
            }
            PayUBeneficiaryAccountType.SAVINGS -> {
                BeneficiaryAccountType.SAVINGS
            }
            else -> null
        }

    internal fun getVerificationModeMapping(verificationMode: PayUBeneficiaryVerificationMode?): String? =
        when(verificationMode) {
            PayUBeneficiaryVerificationMode.NET_BANKING -> SdkUiConstants.CP_NET__BANKING
            PayUBeneficiaryVerificationMode.DEBIT_CARD -> SdkUiConstants.CP_DEBIT__CARD
            PayUBeneficiaryVerificationMode.AADHAAR -> SdkUiConstants.CP_AADHAAR
            else -> null
        }

    internal fun getBillingCycleMapping(billingCycle: PayUBillingCycle?) = when (billingCycle) {
        PayUBillingCycle.ADHOC -> BillingCycle.ADHOC
        PayUBillingCycle.DAILY -> BillingCycle.DAILY
        PayUBillingCycle.MONTHLY -> BillingCycle.MONTHLY
        PayUBillingCycle.ONCE -> BillingCycle.ONCE
        PayUBillingCycle.WEEKLY -> BillingCycle.WEEKLY
        PayUBillingCycle.YEARLY -> BillingCycle.YEARLY
        else -> null
    }

    internal fun getBankShortNameForDC(bankName: String): String = when (bankName) {
        CP_HDFC -> CP_HDFCD
        CP_AXIS -> CP_AXISD
        CP_ICICI -> CP_ICICID
        CP_KOTAK -> CP_KOTAKD
        CP_FED -> CP_FEDD
        CP_BOB -> CP_BOBD
        else -> bankName.uppercase(Locale.getDefault())
    }

    internal fun getBankShortNameForCC(bankName: String): String = when (bankName) {
        CP_INDUSIND -> CP_INDUS
        CP_AUBANK -> CP_AUSF
        CP_ONECARD -> CP_ONEC
        else -> bankName.uppercase(Locale.getDefault())
    }

    internal fun getBankShortNameForCardless(bankName: String): String = when (bankName) {
        CP_BAJFINSERV -> CP_BAJFIN
        else -> bankName.uppercase(Locale.getDefault())
    }

    internal fun getIFSCDetails(payuResponse: PayuResponse?): IFSCDetails? {
        if (payuResponse == null || !payuResponse.isIFSCDetailsAvailable) return null
        val ifscDetails = payuResponse.ifscCodeDetails
        return IFSCDetails(
            ifscDetails.bank,
            ifscDetails.city,
            ifscDetails.ifsc,
            ifscDetails.micr,
            ifscDetails.state,
            ifscDetails.branch,
            ifscDetails.office,
            ifscDetails.address,
            ifscDetails.contact,
            ifscDetails.district
        )
    }

    internal fun getBillingLimitMapping(billingLimit: PayuBillingLimit?) = when (billingLimit) {
        PayuBillingLimit.ON -> BillingLimit.ON
        PayuBillingLimit.BEFORE -> BillingLimit.BEFORE
        PayuBillingLimit.AFTER -> BillingLimit.AFTER
        else -> null
    }

    internal fun getBillingRuleMapping(billingRule: PayuBillingRule?) = when (billingRule) {
        PayuBillingRule.EXACT -> BillingRule.EXACT
        PayuBillingRule.MAX -> BillingRule.MAX
        else -> null
    }

    internal fun isValidAmountForSI(
        txnAmount: Double?,
        checkoutDetailsResponse: PayuResponse?
    ): Boolean {
        if (txnAmount == null || checkoutDetailsResponse == null) return false
        return !(txnAmount < 1.0 && !checkoutDetailsResponse.isNBAvailableFoSI)
    }

    internal fun isKeyEnforced(map: HashMap<String, Any?>): Boolean {
        if (ParserUtils.enforcePaymentList.isNullOrEmpty()) return true
        val enforcelist = ParserUtils.enforcePaymentList
        var bool = false
        for ((key, value) in map) {
            for (item in enforcelist ?: listOf()) {
                if (!item[PayUCheckoutProConstants.CP_PAYMENT_TYPE].isNullOrEmpty()) {
                    val mapvalue = map[PayUCheckoutProConstants.CP_PAYMENT_TYPE]
                    val itemvalue = item[PayUCheckoutProConstants.CP_PAYMENT_TYPE]
                    if (mapvalue?.equals(itemvalue) == true) {
                        if (!key.equals(PayUCheckoutProConstants.CP_PAYMENT_TYPE) && item.contains(
                                key
                            )
                        ) { // key other than payment type
                            val list = item[key]?.filter { !it.isWhitespace() }?.split("|")
                            bool = list?.contains(value.toString()) == true || value == null
                            if(!bool)
                                return false
                        } else
                            bool = true // if key not present then means enforced
                    }
                }
            }
//            if (!bool) // if get result in first iteration
//                return bool
        }

        return bool
    }

    internal fun isSupportedVpaForSI(vpa: String): Boolean {
        var isVpaSupportedForSI = false
        val supportedVpa = ParserUtils.convenienceFeeResponse?.upiSISupportedHandles
            ?: DataConstant.supportedUpiHandles()
        if (supportedVpa != null) {
            for (upivpa in supportedVpa) {
                if (vpa.endsWith(upivpa)) {
                    isVpaSupportedForSI = true
                    break
                }
            }
        }
        return isVpaSupportedForSI
    }

    internal fun isSodexoCard(cardNumber: String): Boolean {
        return cardNumber.matches(Regex(VALIDATE_SODEXO_CARD))
    }

    internal fun getAppVersion(context: Context): String {
        try {
            val pInfo: PackageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            return pInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            e.printStackTrace()
        }
        return ""
    }

    private fun setSIDetails(payuSIParams: PayUSIParams?): SIParams {
        val siParams = SIParams()
        siParams.isFree_trial = payuSIParams?.isFreeTrial!!

        val siParamsDetails = SIParamsDetails()
        siParamsDetails.billingAmount = payuSIParams.billingAmount
        siParamsDetails.billingCurrency = payuSIParams.billingCurrency
        siParamsDetails.billingCycle = getBillingCycleMapping(payuSIParams.billingCycle)
        siParamsDetails.billingInterval = payuSIParams.billingInterval
        siParamsDetails.paymentStartDate = payuSIParams.paymentStartDate
        siParamsDetails.paymentEndDate = payuSIParams.paymentEndDate
        siParamsDetails.remarks = payuSIParams.remarks
        siParamsDetails.billingLimit = getBillingLimitMapping(payuSIParams.billingLimit)
        siParamsDetails.billingRule = getBillingRuleMapping(payuSIParams.billingRule)
        siParamsDetails.isPreAuthorize = payuSIParams.isPreAuthTxn
        siParamsDetails.billingDate = payuSIParams.billingDate
        siParams.si_details = siParamsDetails
        return siParams
    }

    fun getsListForCards(bankList: ArrayList<com.payu.india.Model.PaymentOptionOfferinfo>?): ArrayList<PaymentOptionOfferinfo> {
        val listForCards: ArrayList<PaymentOptionOfferinfo> = ArrayList()
        if (!bankList.isNullOrEmpty()) {
            for (payentOfferInfo in bankList) {
                val paymentOptionOfferinfo = PaymentOptionOfferinfo(
                    null,
                    payentOfferInfo.title,
                    payentOfferInfo.paymentCode,
                    payentOfferInfo.paymentOptionName
                )
                listForCards.add(paymentOptionOfferinfo)
            }
        }
        return listForCards
    }

    fun getListForEmiOptionOffers(
        emiOptionListForOffers: ArrayList<EMIOptionInOffers>?,
        isNoCostEmi: Boolean = false
    ): ArrayList<PaymentOptionOfferinfo> {
        val paymentOptionOfferinfoList: ArrayList<PaymentOptionOfferinfo> = ArrayList()
        if (!emiOptionListForOffers.isNullOrEmpty()) {
            for (emiOptionInfo in emiOptionListForOffers) {
                for (paymentInfo in emiOptionInfo.paymentOptionOfferinfoArrayList) {
                    val paymentOptionOfferinfo: PaymentOptionOfferinfo = PaymentOptionOfferinfo(
                        emiOptionInfo.bankCode,
                        paymentInfo.title,
                        paymentInfo.paymentCode,
                        paymentInfo.paymentOptionName
                    )
                    if (isNoCostEmi) paymentInfo?.paymentCode?.let {
                        InternalConfig.noCostEmi?.add(
                            it
                        )
                    }
                    paymentInfo?.paymentCode?.let { InternalConfig.offerBankListEmi?.add(it) }
                    paymentOptionOfferinfoList.add(paymentOptionOfferinfo)
                }
            }
        }
        return paymentOptionOfferinfoList
    }
    internal fun getBankEmiTypeForCardLess(bankName: String): String = when (bankName) {
        CP_BAJFIN -> PayUCheckoutProConstants.CP_OTHER
        else -> PayUCheckoutProConstants.CP_CARD_LESS
    }

    internal fun drawableTypeMapping(drawableType: com.payu.assetprovider.enums.DrawableType): DrawableType {
        return when (drawableType) {
            com.payu.assetprovider.enums.DrawableType.PictureDrawable -> DrawableType.PictureDrawable
            com.payu.assetprovider.enums.DrawableType.Bitmap -> DrawableType.Bitmap
        }
    }

    /**
     * Returns an enum entry with the specified name or `null` if no such entry was found.
     */
    inline fun <reified T : Enum<T>> enumValueOfOrNull(name: String): T? {
        return enumValues<T>().find { it.name == name }
    }

    internal fun isSITxn(payUPaymentParams: PayUPaymentParams) =
        (payUPaymentParams.payUSIParams != null)

    private fun prepareSkuCartDetails(payUPaymentParams: PayUPaymentParams): JSONObject? {
        return try {
            val skuJsonDetails = getSKUPaymentJson(payUPaymentParams)
            val jsonObject = skuJsonDetails?.second
            jsonObject?.put(PayuConstants.AMOUNT, payUPaymentParams.amount)
            jsonObject?.put(PayuConstants.P_ITEMS, skuJsonDetails.first)
            jsonObject
        } catch (e: JSONException) {
            Log.d(javaClass.simpleName, "JSONException " + e.message)
            null
        }
    }

    fun getSKUPaymentJson(payUPaymentParams: PayUPaymentParams): Pair<String,JSONObject?>? {
        val skuArray = JSONArray()
        var totalSKUAmount = 0.0
        var totalQuantity = 0
        for (sku in payUPaymentParams.skuDetails?.skus ?: listOf()) {
            val jsonObject = JSONObject()
            try {
                jsonObject.put(PayuConstants.P_SKU_M_ID, sku.skuId)
                jsonObject.put(PayuConstants.P_SKU_M_AMOUNT, sku.skuAmount)
                jsonObject.put(PayuConstants.P_QUANTITY, sku.quantity)
                jsonObject.put(PayuConstants.P_SKU_NAME, sku.skuName)
                jsonObject.put(
                    PayuConstants.OFFER_KEY,
                    sku.offerKeys
                )
                jsonObject.put(
                    PayuConstants.P_AUTO_APPLY_M_OFFER,
                    true
                )
            } catch (e: JSONException) {
                Log.d(javaClass.simpleName, "JSONException " + e.message)
            }
            skuArray.put(jsonObject)
            totalSKUAmount += (sku.skuAmount?.toDouble() ?: 0.0) * (sku.quantity?.toDouble() ?: 0.0)
            totalQuantity += sku.quantity
        }

        if (totalSKUAmount == payUPaymentParams.amount?.toDouble()){
            val jsonObject = JSONObject()
            jsonObject.put(PayuConstants.P_SKU_DETAILS, skuArray)

            return Pair(totalQuantity.toString(), jsonObject)
        }
        return null
    }

    internal fun getIfscCodesList(payUPaymentParams: PayUPaymentParams): ArrayList<String>? {
        val ifscCodeList = ArrayList<String>()
        payUPaymentParams.beneficiaryDetailList?.forEach {
            if (isValidBeneficiary(it)) it.beneficiaryIfsc?.let { it1 -> ifscCodeList.add(it1.uppercase()) }
        }
        return if (ifscCodeList.isEmpty().not()) ifscCodeList else null
    }

    internal fun getIfscCodesForPayment(beneficiaryDetail: List<UserAccountInfo>): String? {
        return beneficiaryDetail.joinToString(separator = "|") { it -> "${it.accountDetails?.beneficiaryIfsc ?: ""}" }
    }

    internal fun getAccountsForPayment(beneficiaryDetail: List<UserAccountInfo>): String? {
        return beneficiaryDetail.joinToString(separator = "|") { it -> "${it.accountDetails?.beneficiaryAccountNumber ?: ""}" }
    }

    internal fun isValidBeneficiary(payUBeneficiaryDetail: PayUBeneficiaryDetail): Boolean {
        return payUBeneficiaryDetail.beneficiaryIfsc.isNullOrEmpty()
            .not() && payUBeneficiaryDetail.beneficiaryAccountNumber.isNullOrEmpty()
            .not()
    }

    internal fun getPaymentTypeValue(key: String): PaymentType? {
        return when (key) {
            PayUCheckoutProConstants.CP_CC, PayUCheckoutProConstants.CP_DC -> PaymentType.CARD
            PayUCheckoutProConstants.CP_CASH_CARD -> PaymentType.WALLET
            else -> enumValueOfOrNull<PaymentType>(key.uppercase())
        }
    }

    internal fun clearReferences(){
        hashGenerationParamsFactory = null
        listenerMap.clear()
        checkoutProListener = null
    }

    private fun isValidTxnId(txnId: String?): Boolean {
        if (txnId.isNullOrBlank()) return false
        else if (txnId.length > TRANSACTION_ID_LENGTH_LIMIT) return false
        val pattern = Regex("^[a-zA-Z0-9]*$")
        return txnId.matches(pattern);
    }
}
