package com.payu.checkoutpro.parser.utils

import android.text.TextUtils
import com.payu.base.models.CustomNote
import com.payu.base.models.OrderDetails
import com.payu.base.models.PaymentMode
import com.payu.base.models.PaymentType
import com.payu.checkoutpro.parser.constants.PayUHybridKeys
import com.payu.checkoutpro.utils.PayUCheckoutProConstants

object CheckoutProConfigParserUtils {
    /**
     * Method to convert List of type Any to List of type OrderDetails which CheckoutPro SDK understand
     */
    internal fun getCartDetails(cartDetailArray: List<Any?>): ArrayList<OrderDetails> {
        val resultList: ArrayList<OrderDetails> = ArrayList(cartDetailArray.size)
        for (element in cartDetailArray) {
            val reviewOrderMap =
                element as? HashMap<String, Any?>
            if (reviewOrderMap != null) {
                val keyIterator = reviewOrderMap.keys.iterator()
                while (keyIterator.hasNext()) {
                    val key: String = keyIterator.next()
                    val details = OrderDetails(key, reviewOrderMap[key].toString())
                    resultList.add(details)
                }
            }
        }
        return resultList
    }

    /**
     * Method to convert List of type Any to List of type PaymentMode
     * which CheckoutPro SDK understand
     */
    internal fun getPaymentOrder(paymentOrderArray: List<Any>): ArrayList<PaymentMode> {
        val resultList: ArrayList<PaymentMode> = ArrayList(paymentOrderArray.size)

        for (element in paymentOrderArray) {
            val paymentOptionOrderMap =
                element as? HashMap<String, Any?>
            if (paymentOptionOrderMap != null) {
                val keyIterator = paymentOptionOrderMap.keys.iterator()
                while (keyIterator.hasNext()) {
                    val key: String = keyIterator.next()
                    val type: PaymentType? = getValidPaymentType(key)
                    if (null != type) {
                        var mode: PaymentMode
                        val bankCode: String = paymentOptionOrderMap[key].toString()
                        mode = if (TextUtils.isEmpty(bankCode)) {
                            PaymentMode(type)
                        } else {
                            PaymentMode(type, getValidPaymentCode(bankCode))
                        }
                        resultList.add(mode)
                    }
                }
            }
        }
        return resultList
    }

    /**
     * Method to map String value of PaymentType to CheckoutPro specific enum
     */
    private fun getValidPaymentType(typeValue: String): PaymentType? {
        when (typeValue) {
            PayUHybridKeys.Others.NetBanking -> return PaymentType.NB
            PayUHybridKeys.Others.UPI -> return PaymentType.UPI
            PayUHybridKeys.Others.Wallets -> return PaymentType.WALLET
            PayUHybridKeys.Others.EMI -> return PaymentType.EMI
            PayUHybridKeys.Others.Cards -> return PaymentType.CARD
        }
        return null
    }

    /**
     * Method to map String value of PaymentOption to CheckoutPro specific enum
     */
    private fun getValidPaymentCode(paymentOptionName: String): String {
        when (paymentOptionName) {
            PayUHybridKeys.Others.TEZ -> return PayUCheckoutProConstants.CP_GOOGLE_PAY
            PayUHybridKeys.Others.PAYTM -> return PayUCheckoutProConstants.CP_PAYTM
            PayUHybridKeys.Others.PHONEPE -> return PayUCheckoutProConstants.CP_PHONEPE
        }
        return paymentOptionName
    }

    /**
     * Method to map ArrayList of Hashmap to ArrayList of Custom note
     */
    internal fun getCustomerNoteDetails(notesList: ArrayList<HashMap<String, Any>>): ArrayList<CustomNote> {
        val customNoteList = ArrayList<CustomNote>()

        for (notesMap in notesList.toList()) {
            val noteText = notesMap[PayUHybridKeys.Others.custom_note] as? String

            if (noteText != null) {
                val noteCategoryList: ArrayList<PaymentType> = ArrayList()
                (notesMap[PayUHybridKeys.Others.custom_note_category] as? ArrayList<String>)?.let {
                    for (paymentType in it) {
                        noteCategoryList.add(PaymentType.valueOf(paymentType))
                    }
                }
                val customNote = CustomNote(noteText, noteCategoryList)
                customNoteList.add(customNote)
            }
        }
        return customNoteList
    }
}