package com.payu.checkoutpro.models

import com.payu.base.listeners.BaseTransactionListener
import com.payu.base.listeners.OnLookupApiListener
import com.payu.base.models.CardOption
import com.payu.base.models.ErrorResponse
import com.payu.checkoutpro.layers.PayUbizApiLayer
import com.payu.checkoutpro.utils.CPAnalyticsUtils
import com.payu.checkoutpro.utils.ParserUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.india.Interfaces.LookupApiListener
import com.payu.india.Model.LookupRequest
import com.payu.india.Model.PayuResponse
import com.payu.india.Payu.PayuConstants
import com.payu.india.Tasks.LookupTask
import com.payu.paymentparamhelper.PaymentParams
import com.payu.ui.SdkUiInitializer

class LookupApiObject(
    payuBizparams: PaymentParams,
    private val baseTransactionListener: BaseTransactionListener?,
    private val payUbizApiLayer: PayUbizApiLayer,
    private val merchantOrderId: String,
    private val cardOption: CardOption,
    private val merchantAccessKey: String,
    private val currency: String,
    _listener: Any?
) : V1BaseApiObject(payuBizparams), LookupApiListener {
    private var startTime: Long = 0
    private var onLookupApiListener = _listener as OnLookupApiListener

    override fun callApi(map: HashMap<String, String?>) {
        payUbizApiLayer.isLookupApiInProgress = true
        var transactionAmount =
            SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
        transactionAmount = transactionAmount?.plus(cardOption.cardBinInfo?.additionalCharge!!)
        try {
            val postData = LookupRequest.LookupApiRequestBuilder()
                .setAmount(transactionAmount.toString())
                .setCurrency(currency)
                .setMerchantAccessKey(merchantAccessKey)
                .setMerchantOrderId(merchantOrderId)
                .setProductType(LookupRequest.ProductType.MCP)
                .setSignature(map[getHashName()])
                .build().prepareJSON()

            payuConfig.data = postData
            val lookupTask =
                LookupTask(this)
            startTime = System.currentTimeMillis()
            lookupTask.execute(payuConfig)
        } catch (e: Exception) {
            payUbizApiLayer.resetMcpFlags()
            val errorResponse = ErrorResponse()
            errorResponse.errorMessage = e.message
            baseTransactionListener?.showProgressDialog(false)
            onLookupApiListener.onError(errorResponse)
        }
    }

    override fun getHashName() = PayUCheckoutProConstants.CP_LOOKUP_API_HASH

    override fun onLookupApiResponse(payuResponse: PayuResponse?) {
        val timeTaken = System.currentTimeMillis() - startTime
        payUbizApiLayer.isLookupApiInProgress = false
        if (payuResponse?.responseStatus?.status.equals(PayuConstants.SUCCESS, ignoreCase = true)) {
            CPAnalyticsUtils.logApiResponseSuccess(getHashName(), timeTaken, payuConfig.data, payUbizApiLayer.context.applicationContext)
            ParserUtils.lookupDetails = payuResponse?.lookupDetails
            baseTransactionListener?.showProgressDialog(false)
            onLookupApiListener.onLookupApiCalled()

            //If lookup API was in progress when make payment was called. Then call it after completion.
            // This is the case when in Add card screen after identifying if bin is not domestic silently lookup api is called.
            // If the API was in progress till the time all details were entered on screen and pay not button is clicked.
            // Loader is displayed and after API completes make payment is performed.
            if (payUbizApiLayer.isMakePaymentCalled && payUbizApiLayer.mcpPaymentModel != null && payUbizApiLayer.mcpToolbar != null) {
                payUbizApiLayer.updatePaymentState(
                    payUbizApiLayer.mcpPaymentModel!!,
                    payUbizApiLayer.mcpToolbar!!
                )
            }
        } else {
            payUbizApiLayer.resetMcpFlags()
            CPAnalyticsUtils.logApiResponseFailure(getHashName(), timeTaken, payuResponse?.responseStatus?.result, payuResponse?.responseStatus?.code, payUbizApiLayer.context)
            payuResponse?.responseStatus?.result?.let {
                val errorResponse = ErrorResponse()
                errorResponse.errorMessage = payuResponse.responseStatus?.result
                errorResponse.errorCode = payuResponse.responseStatus?.code

                //If Signature mismatched or Merchant Access Key is Invalid, then pass error response to merchant and close CheckoutPro SDK
                if (it.equals(
                        PayUCheckoutProConstants.CP_ERROR_SIGNATURE_MISMATCHED,
                        ignoreCase = true
                    ) || payuResponse.responseStatus?.code == PayUCheckoutProConstants.CP_ERROR_INVALID_MERCHANT_ACCESS_KEY_CODE
                ) {
                    baseTransactionListener?.showProgressDialog(false)
                    onLookupApiListener.onError(errorResponse)
                } else {
                    baseTransactionListener?.showProgressDialog(false)
                    onLookupApiListener.onError(errorResponse)
                }
            }
        }
    }
}