package com.payu.checkoutpro.models

import android.content.Context
import com.payu.assetprovider.enums.ImageType
import com.payu.base.listeners.BaseApiListener
import com.payu.base.models.ErrorResponse
import com.payu.base.models.InternalConfig
import com.payu.base.models.PayUAddressDetails
import com.payu.base.models.PaymentMode
import com.payu.base.models.UserAccountInfo
import com.payu.checkoutpro.R
import com.payu.checkoutpro.handlers.CategoryHandler
import com.payu.checkoutpro.layers.PayUbizApiLayer
import com.payu.checkoutpro.utils.ApiResponseRepo
import com.payu.checkoutpro.utils.CPAnalyticsUtils
import com.payu.checkoutpro.utils.CommonUtils
import com.payu.checkoutpro.utils.ConfigUtils
import com.payu.checkoutpro.utils.ParserUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_ADDRESS_DETAILS
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_ADDRESS_DETAILS_MESSAGE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_BENEFICIARY_DETAILS
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_BENEFICIARY_DETAILS_MESSAGE
import com.payu.india.Interfaces.CheckoutDetailsListener
import com.payu.india.Model.MerchantInfo
import com.payu.india.Model.MerchantWebService
import com.payu.india.Model.PayuResponse
import com.payu.india.Payu.PayuConstants
import com.payu.india.Payu.PayuErrors
import com.payu.india.PostParams.MerchantWebServicePostParams
import com.payu.india.Tasks.GetPaymentInstrumentTask
import com.payu.paymentparamhelper.PaymentParams

class GetCheckoutDetailsApiObject(
    private val payUbizApiLayer: PayUbizApiLayer,
    payuBizparams: PaymentParams,
    val var1: String,
    listener: Any?,
    private val l1OptionsListener: (ArrayList<PaymentMode>?) -> Unit?
    ) : V1BaseApiObject(payuBizparams), CheckoutDetailsListener {
    private val baseApiListener = listener as BaseApiListener

    private var startTime: Long = 0

    override fun callApi(map: HashMap<String, String?>) {
        val merchantWebService = MerchantWebService()
        merchantWebService.key = payuBizparams.key
        merchantWebService.command = PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS
        merchantWebService.var1 = var1
        merchantWebService.hash = map[getHashName()]

        val postData =
            MerchantWebServicePostParams(merchantWebService).merchantWebServicePostParams

        if (postData.code == PayuErrors.NO_ERROR) {
            payuConfig.data = postData.result

            val getCheckoutDetailsTask =
                GetPaymentInstrumentTask(this)
            startTime = System.currentTimeMillis()
            getCheckoutDetailsTask.execute(payuConfig)
        } else {
            val errorResponse = ErrorResponse()
            errorResponse.errorMessage = postData.result
            baseApiListener.showProgressDialog(false)
            ApiResponseRepo.setL1MoreOptionsResponse(ParserUtils.moreOptionsList)
        }
    }

    override fun getHashName() = PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS


    override fun onCheckoutDetailsResponse(payuResponse: PayuResponse?) {
        val timeTaken = System.currentTimeMillis() - startTime
        if (payuResponse?.responseStatus?.status.equals(PayuConstants.SUCCESS, ignoreCase = true)) {
            CPAnalyticsUtils.logApiResponseSuccess(getHashName(), timeTaken, payuConfig.data, payUbizApiLayer.context.applicationContext)
            if (payuResponse?.merchantInfo?.merchantParamInfo?.isOfferEnabled == false) payuBizparams.userToken =
                null
            if (payUbizApiLayer.payUPaymentParams.beneficiaryDetailList.isNullOrEmpty()
                    .not() && payuResponse?.merchantInfo?.ifscMappingDetails.isNullOrEmpty()
            ) {
                returnErrorResponse(CP_ERROR_INVALID_BENEFICIARY_DETAILS_MESSAGE, CP_ERROR_INVALID_BENEFICIARY_DETAILS)
            } else if (payuResponse != null) {

                // This is to check address values for OPGSP merchant
                payuResponse.merchantInfo?.merchantParamInfo?.let { paramInfo ->
                    if (paramInfo.isOpgspMerchant) {
                        val addressDetails = payUbizApiLayer.payUPaymentParams.addressDetails
                        var udf5: String? = null
                        payUbizApiLayer.payUPaymentParams.additionalParams?.let { additionalParams ->
                            udf5 =
                                additionalParams[PayUCheckoutProConstants.CP_UDF5] as? String
                        }
                        if (checkAddressDetailsForOpgsp(addressDetails) || udf5.isNullOrEmpty()) {
                            returnErrorResponse(CP_ERROR_INVALID_ADDRESS_DETAILS_MESSAGE, CP_ERROR_INVALID_ADDRESS_DETAILS)
                            return
                        }
                    }
                }

                payuResponse.merchantInfo?.let {
                    updateMerchantInfo(it)
                }
                ParserUtils.issuingBankDownList = payuResponse.downIssuingBanks
                ParserUtils.isRetryTxn = false
                ParserUtils.retryCount = payuResponse.merchantInfo?.retryCount ?: 0
                payUbizApiLayer.checkoutAPICompleted = true
                CategoryHandler.filterDataOnBasisOfCategory(payuResponse)
                if (!payUbizApiLayer.configAPICompleted) {
                    ConfigUtils.setCachedConfig(
                        payUbizApiLayer.context,
                        payUbizApiLayer.payUCheckoutProConfig
                    )
                }
                ParserUtils.getMoreOptions(
                    payUbizApiLayer,
                    payuResponse,
                    l1OptionsListener
                )
            } else {
                val context = payUbizApiLayer.context.applicationContext
                returnErrorResponse(context.getString(R.string.empty_response_message), payuResponse?.responseStatus?.code)
            }
        } else {
            returnErrorResponse(payuResponse?.responseStatus?.result, payuResponse?.responseStatus?.code)
        }
    }

    private fun returnErrorResponse(errorMessage: String?, errorCode: Int?) {
        val timeTaken = System.currentTimeMillis() - startTime
        val errorResponse = ErrorResponse()
        errorResponse.errorMessage = errorMessage
        errorResponse.errorCode = errorCode
        CPAnalyticsUtils.logApiResponseFailure(getHashName(), timeTaken, errorResponse.errorMessage, errorCode, payUbizApiLayer.context)
        baseApiListener.showProgressDialog(false)
        baseApiListener.onError(errorResponse)
    }

    private fun updateMerchantInfo(merchantInfo: MerchantInfo){
        merchantInfo.merchantParamInfo?.apply {
            isOfferEnabled.let {
                InternalConfig.isOfferEnabled =
                    it && payUbizApiLayer.payUPaymentParams.beneficiaryDetailList.isNullOrEmpty() && payUbizApiLayer.payUPaymentParams.payUSIParams?.isPreAuthTxn != true
            }
            isAdsEnabled.let {
                InternalConfig.isAdsEnabled = it
            }
            isQuickPayEnabled.let {
                InternalConfig.isQuickPayEnabled =
                    it && payUbizApiLayer.payUPaymentParams.beneficiaryDetailList.isNullOrEmpty() && payUbizApiLayer.payUPaymentParams.payUSIParams?.isPreAuthTxn != true
            }
            isEnableInternal3DSS.let {
                InternalConfig.enableInternal3DSS = it
            }
            isInternationalOn3DSS.let {
                InternalConfig.internationalOn3DSS = it
            }
            isEnable3dsSDK.let {
                InternalConfig.enable3dsSDK = it
            }
            isDeviceIdEnabled.let {
                InternalConfig.isDeviceIdApiEnabled = it
            }
            threedsBankList.let {
                InternalConfig.threeDSSupportedBankList = it
            }
            customerRevenueEnabled.let {
                InternalConfig.isPricingCFCall = it
            }
            InternalConfig.isQuickPayBottomSheetEnabled =
                isQuickPayBottomSheetEnabled && payUbizApiLayer.payUPaymentParams.beneficiaryDetailList.isNullOrEmpty() && payUbizApiLayer.payUPaymentParams.payUSIParams?.isPreAuthTxn != true

            upiSIApps?.let {
                InternalConfig.supportedUpiSiIntentAppList = it
            }
            customerRevenueEnabled.let {
                InternalConfig.isPricingCFCall = it
            }
            isInsuranceMerchant.let {
                InternalConfig.isInsuranceMerchant = it
            }
        }
        InternalConfig.logo = merchantInfo.logo
        InternalConfig.oneClickPayBankCodes = merchantInfo.oneClickPayBankCodes
        InternalConfig.walletIdentifier = merchantInfo.walletIdentifier
        InternalConfig.merchantDisplayName = merchantInfo.displayName
        updateUserAccountInfoList(merchantInfo)
    }

    private fun updateUserAccountInfoList(merchantInfo: MerchantInfo) {
        val userAccountInfoList = ArrayList<UserAccountInfo>()
        merchantInfo.ifscMappingDetails?.forEach { ifscDetails ->
            payUbizApiLayer.payUPaymentParams.beneficiaryDetailList?.forEach { payUBeneficiaryDetail ->
                if (CommonUtils.isValidBeneficiary(payUBeneficiaryDetail) && payUBeneficiaryDetail.beneficiaryIfsc
                        ?.startsWith(ifscDetails.ifscCode, ignoreCase = true) == true
                ) {
                    val userAccountInfo = UserAccountInfo(
                        ifscDetails.bankName,
                        ifscDetails.imageUrl.plus(ifscDetails.ibiboCode).plus(".")
                            .plus(ImageType.SVG.name.lowercase()),
                        payUBeneficiaryDetail
                    )
                    userAccountInfoList.add(userAccountInfo)
                }
            }
        }
        InternalConfig.userAccountInfoList = userAccountInfoList
    }

    private fun checkAddressDetailsForOpgsp(address: PayUAddressDetails?): Boolean {
        return address == null || listOf(
            address.lastName, address.address1, address.address2,
            address.city, address.state, address.country, address.zipcode
        ).any {
            it.isNullOrEmpty()
        }
    }
}