package com.payu.checkoutpro.reConverter

import android.content.Context
import com.payu.base.models.PaymentMode
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentType
import com.payu.base.models.SavedCardOption
import com.payu.base.models.SodexoCardOption
import com.payu.base.models.UPIOption
import com.payu.base.models.WalletOption
import com.payu.checkoutpro.utils.CommonUtils.enumValueOfOrNull
import com.payu.checkoutpro.utils.CommonUtils.getPaymentTypeValue
import com.payu.checkoutpro.utils.ParserUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_NA
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_NULL
import com.payu.india.Model.QuickPay.QuickPaySavedOption
import com.payu.india.Model.QuickPay.RecommendedOptions

object PaymentModeConverter {

    internal fun globalVaultQuickOptions(
        context: Context,
        savedOptionList: List<QuickPaySavedOption>
    ): ArrayList<PaymentMode> {

        val quickOptionsList: ArrayList<PaymentMode> = ArrayList()
        prepareClwSodexoOptions(quickOptionsList)

        if (!savedOptionList.isNullOrEmpty()) {
            val list = (savedOptionList.sortedBy { it.pgId.toInt() })
            for (item in list) {
                val mode = getPaymentMode(context, item)
                mode.type = getPaymentTypeValue(item.paymentType)
                if (!mode.optionDetail.isNullOrEmpty())
                    quickOptionsList.add(mode)
            }
        }
        return quickOptionsList

    }

    internal fun prepareClwSodexoOptions(quickOptionsList: ArrayList<PaymentMode>) {
        if (ParserUtils.isClosedLoopWalletToAdd() && !ParserUtils.isSIMode) {
            val paymentMode =
                ParserUtils.getClosedLoopWalletMode(ParserUtils.convenienceFeeResponse)
            val closedLoopWalletCardOption =
                paymentMode.optionDetail?.get(0) as? SodexoCardOption
            closedLoopWalletCardOption?.let {
                it.fetchedStatus =
                    -1 //set initial to -1 since API is fetching the details in parallel
            }
            quickOptionsList.add(paymentMode)
        }

        if (ParserUtils.convenienceFeeResponse?.isSodexoAvailable == true && !ParserUtils.sodexoSourceId.isNullOrEmpty() && !ParserUtils.isSIMode) {
            val paymentMode = ParserUtils.getSodexoMode(ParserUtils.convenienceFeeResponse)
            val sodexoCardOption = paymentMode.optionDetail?.get(0) as? SodexoCardOption
            sodexoCardOption?.let {
                it.fetchedStatus =
                    -1 //set initial to -1 since API is fetching the details in parallel
            }
            (sodexoCardOption)?.let { updateTaxValueForSodexo(it) }
            quickOptionsList.add(paymentMode)
        }
    }

    private fun getPaymentMode(
        context: Context,
        quickPaySavedOption: QuickPaySavedOption,
    ): PaymentMode {

        val paymentMode = PaymentMode()

        paymentMode.type = getPaymentTypeValue(quickPaySavedOption.paymentType)

        when (paymentMode.type) {
            PaymentType.CARD -> {
                paymentMode.optionDetail =
                    CardPaymentModeConverter.getSavedCardPaymentMode(quickPaySavedOption)
            }

            PaymentType.NB -> {
                paymentMode.optionDetail =
                    NBPaymentModeConverter.getSavedNBPaymentMode(quickPaySavedOption)
            }

            PaymentType.UPI -> {
                paymentMode.optionDetail =
                    UPIPaymentModeConverter.getSavedUPIPaymentMode(context, quickPaySavedOption)
            }

            PaymentType.BNPL -> {
                if (!ParserUtils.isSIMode) paymentMode.optionDetail =
                    BNPLPaymentModeConverter.getBNPLPaymentOption(quickPaySavedOption)
            }

            PaymentType.WALLET -> {
                if (!ParserUtils.isSIMode)
                    paymentMode.optionDetail =
                        WalletPaymentModeConverter.getSavedWalletPaymentMode(
                            context,
                            quickPaySavedOption
                        )
            }
        }
        paymentMode.isGlobalVaultPaymentMode = true

        return paymentMode
    }

    private fun getPaymentMode(
        recommendedOptions: RecommendedOptions
    ): PaymentMode {
        val paymentMode = PaymentMode()

        paymentMode.type = getPaymentTypeValue(recommendedOptions.paymode)
        if (!recommendedOptions.title.equals(CP_NA) && !recommendedOptions.title.equals(CP_NULL))
            paymentMode.name = recommendedOptions.title
        if (!recommendedOptions.msg.equals(CP_NA) && !recommendedOptions.msg.equals(CP_NULL))
            paymentMode.l1OptionSubText = recommendedOptions.msg
        when (paymentMode.type) {
            PaymentType.UPI -> {

                paymentMode.optionDetail =
                    UPIPaymentModeConverter.getREUPIPaymentOption(recommendedOptions)
            }

            PaymentType.WALLET -> {

                paymentMode.optionDetail =
                    WalletPaymentModeConverter.getREWalletPaymentOption(recommendedOptions)
            }

            PaymentType.NB -> {

                paymentMode.optionDetail =
                    NBPaymentModeConverter.getRENBPaymentOption(recommendedOptions)
            }

            PaymentType.BNPL -> {

                paymentMode.optionDetail =
                    BNPLPaymentModeConverter.getREBNPLPaymentOption(recommendedOptions)
            }

            PaymentType.CARD -> {
                paymentMode.optionDetail =
                    CardPaymentModeConverter.getRECardPaymentOption(recommendedOptions)
            }
        }
        paymentMode.isGlobalVaultPaymentMode = true
        return paymentMode
    }


    internal fun getRecommendedOptionsResponse(recommendedOptionsArrayList: ArrayList<RecommendedOptions>?): ArrayList<PaymentMode>? {
        val quickOptionsList: ArrayList<PaymentMode> = ArrayList()
        if (recommendedOptionsArrayList != null) {
            if (recommendedOptionsArrayList.isNotEmpty()) {
                for (item in recommendedOptionsArrayList) {
                    val paymentMode = getPaymentMode(item)
                    if (paymentMode.optionDetail?.isNotEmpty() == true)
                        quickOptionsList.add(paymentMode)
                }
            }
        } else return null
        return quickOptionsList
    }

    internal fun getDeleteSavedOptions(paymentOption: PaymentOption): QuickPaySavedOption? {
        return when (paymentOption.paymentType) {
            PaymentType.CARD -> (paymentOption as? SavedCardOption)?.let {
                CardPaymentModeConverter.getDeleteApiSavedPaymentMode(
                    it
                )
            }

            PaymentType.WALLET -> WalletPaymentModeConverter.getDeleteApiSavedPaymentMode(
                paymentOption as WalletOption
            )

            PaymentType.UPI, PaymentType.UPI_INTENT -> UPIPaymentModeConverter.getDeleteApiSavedPaymentMode(
                paymentOption as UPIOption
            )

            else -> {
                NBPaymentModeConverter.getDeleteApiSavedPaymentMode(paymentOption)
            }
        }
    }

    internal fun updateTaxValueForSodexo(
        savedCardOption: SodexoCardOption
    ) {
        for (paymentDetail in ParserUtils.convenienceFeeResponse?.mealCard
            ?: arrayListOf()) {
            if (paymentDetail.bankCode.equals(savedCardOption.bankCode)) {
                savedCardOption.additionalCharge =
                    paymentDetail.additionalCharge?.toDoubleOrNull() ?: 0.0
                savedCardOption.gstPercentageValue =
                    ParserUtils.convenienceFeeResponse?.taxSpecification?.mealCardTaxValue?.toDoubleOrNull()
                        ?: 0.0
            }
        }
    }

    internal fun updateTaxValue(
        savedOption: PaymentOption,
        paymentType: PaymentType

    ) {
        when (paymentType) {
            PaymentType.NB -> {
                for (paymentDetail in ParserUtils.convenienceFeeResponse?.netBanks
                    ?: arrayListOf()) {
                    if (savedOption.bankCode.equals(paymentDetail.bankCode)) {
                        savedOption.additionalCharge =
                            paymentDetail.additionalCharge?.toDoubleOrNull() ?: 0.0
                        savedOption.gstPercentageValue =
                            ParserUtils.convenienceFeeResponse?.taxSpecification?.nbTaxValue?.toDoubleOrNull()
                                ?: 0.0
                        break
                    }
                }
            }

            PaymentType.WALLET -> {
                for (paymentDetail in ParserUtils.convenienceFeeResponse?.cashCard
                    ?: arrayListOf()) {
                    if (savedOption.bankCode.equals(paymentDetail.bankCode)) {
                        savedOption.additionalCharge =
                            paymentDetail.additionalCharge?.toDoubleOrNull() ?: 0.0
                        savedOption.gstPercentageValue =
                            ParserUtils.convenienceFeeResponse?.taxSpecification?.cashTaxValue?.toDoubleOrNull()
                                ?: 0.0
                    }
                }
            }

            PaymentType.UPI, PaymentType.UPI_INTENT -> {
                savedOption.additionalCharge =
                    ParserUtils.convenienceFeeResponse?.upi?.additionalCharge?.toDoubleOrNull()
                        ?: 0.0
                savedOption.gstPercentageValue =
                    ParserUtils.convenienceFeeResponse?.taxSpecification?.upiTaxValue?.toDoubleOrNull()
                        ?: 0.0
            }

            PaymentType.BNPL -> {
                for (paymentDetail in ParserUtils.convenienceFeeResponse?.bnpl
                    ?: arrayListOf()) {
                    if (savedOption.bankCode.equals(paymentDetail.bankCode)) {
                        savedOption.additionalCharge =
                            paymentDetail.additionalCharge?.toDoubleOrNull() ?: 0.0
                        savedOption.gstPercentageValue =
                            ParserUtils.convenienceFeeResponse?.taxSpecification?.bnplTaxValue?.toDoubleOrNull()
                                ?: 0.0
                    }
                }
            }
        }
    }
}