package com.payu.checkoutpro.utils

import android.app.ActivityManager
import android.content.Context
import android.os.Build
import com.payu.checkoutpro.BuildConfig
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.utils.AnalyticsConstant
import com.payu.ui.model.utils.AnalyticsHandler
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.ViewUtils
import org.json.JSONObject

object CPAnalyticsUtils {

    enum class EventType{
        Info,
        Error
    }

    enum class EventStatus {
        Success,
        Failure
    }

    enum class EventSeverity {
        High,
        Low
    }

    fun logGenerateHashEvent(hashName: String, hashString: String?, context: Context? = null) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, "${AnalyticsConstant.EVENT_CP_HASH}:$hashName", BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        val json = JSONObject()
        json.put(AnalyticsConstant.KEY_HASH_STRING, hashString)
        data.put(AnalyticsConstant.EVENT_VALUE, json.toString())
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Info)
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.Low)
        if (context == null) {
            AnalyticsUtils.addLogInLogListWithoutContext(data)
        } else {
            AnalyticsHandler.logKibanaData(context, data)
        }
    }

    fun logHashGeneratedEventError(hashName: String, timeTaken: Long, errorMessage: String, context: Context? = null) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, "${AnalyticsConstant.EVENT_CP_HASH_GENERATED}:$hashName", BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        data.put(AnalyticsConstant.EVENT_VALUE, "failure {message: $errorMessage}")
        data.put(AnalyticsConstant.EVENT_TIME, timeTaken)
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Error)
        data.put(AnalyticsConstant.EVENT_STATUS, EventStatus.Failure)
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.High)
        if (context == null) {
            AnalyticsUtils.addLogInLogListWithoutContext(data)
        } else {
            AnalyticsHandler.logKibanaData(context, data)
        }
    }

    fun logHashGeneratedEventSuccess(hashName: String, timeTaken: Long, generatedHash: String?, context: Context? = null) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, "${AnalyticsConstant.EVENT_CP_HASH_GENERATED}:$hashName", BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        val jsonObject = JSONObject()
        jsonObject.put(AnalyticsConstant.KEY_GENERATED_HASH,generatedHash )
        data.put(AnalyticsConstant.EVENT_VALUE, jsonObject.toString())
        data.put(AnalyticsConstant.EVENT_TIME, timeTaken)
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Info)
        data.put(AnalyticsConstant.EVENT_STATUS, EventStatus.Success)
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.Low)
        if (context == null) {
            AnalyticsUtils.addLogInLogListWithoutContext(data)
        } else {
            AnalyticsHandler.logKibanaData(context, data)
        }
    }

    fun logSdkInitialization(context: Context, payUPaymentParams: String, payUCheckoutProConfig: String) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, AnalyticsConstant.EVENT_CP_INITIALISED, BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        val eventDataJson = JSONObject()
        eventDataJson.put(AnalyticsConstant.EVENT_CP_CONFIG, payUCheckoutProConfig)
        eventDataJson.put(AnalyticsConstant.EVENT_PAYMENT_PARAM, payUPaymentParams)
        data.put(AnalyticsConstant.EVENT_VALUE, eventDataJson.toString())
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Info)
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.Low)

        AnalyticsHandler.logKibanaData(context, data)

    }

    fun logApiResponseSuccess(apiName: String, timeTaken: Long, request: String?, context: Context? = null) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, "${AnalyticsConstant.CP_API} $apiName", BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        data.put(AnalyticsConstant.EVENT_TIME, timeTaken)
        val eventValue = JSONObject()
        eventValue.put(AnalyticsConstant.API_REQUEST, request)
        data.put(AnalyticsConstant.EVENT_VALUE, eventValue.toString())
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Info)
        data.put(AnalyticsConstant.EVENT_STATUS, EventStatus.Success)
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.Low)

        if (context != null) {
            AnalyticsHandler.logKibanaData(context, data)
        } else {
            AnalyticsUtils.addLogInLogListWithoutContext(data)
        }
    }

    fun logApiResponseFailure(apiName: String, timeTaken: Long, failureReason: String?, errorCode: Int?, context: Context? = null) {
        val data = AnalyticsUtils.getAnalyticsJsonWithBasicProperties(context, "${AnalyticsConstant.CP_API} $apiName", BuildConfig.LIBRARY_PACKAGE_NAME, PayUCheckoutProConstants.SDK_NAME_CHECKOUT_PRO_WRAPPER, BuildConfig.VERSION_NAME)
        data.put(AnalyticsConstant.EVENT_TIME, timeTaken)
        failureReason?.let {
            data.put(AnalyticsConstant.EVENT_VALUE, it)
        }
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Error)
        data.put(AnalyticsConstant.EVENT_STATUS, EventStatus.Failure)
        errorCode?.let {
            data.put(AnalyticsConstant.API_RESPONSE_CODE, it)
        }
        data.put(AnalyticsConstant.EVENT_SEVERITY, EventSeverity.High)

        if (context != null) {
            AnalyticsHandler.logKibanaData(context, data)
        } else {
            AnalyticsUtils.addLogInLogListWithoutContext(data)
        }
    }

    private fun getAnalyticsJsonWithBasicProperties(context: Context?, eventKey: String): JSONObject {
        val analyticsJson = JSONObject()
        analyticsJson.put(
            SdkUiConstants.CP_EVENT_KEY,
            eventKey
        )
        analyticsJson.put(SdkUiConstants.CP_TIMESTAMP, ViewUtils.getSystemCurrentTime())
        analyticsJson.put(SdkUiConstants.CP_SDK_VERSION, BuildConfig.VERSION_NAME)
        analyticsJson.put(
            AnalyticsConstant.CP_LOGGING_SDK,
            BuildConfig.LIBRARY_PACKAGE_NAME + " " + BuildConfig.VERSION_NAME
        )
        analyticsJson.put(
            SdkUiConstants.CP_TXN_ID,
            SdkUiInitializer.apiLayer?.payUPaymentParams?.transactionId
        )
        analyticsJson.put(
            SdkUiConstants.CP_MERCHANT_KEY,
            SdkUiInitializer.apiLayer?.payUPaymentParams?.key
        )
        val deviceName = Build.MANUFACTURER + " " + Build.MODEL
        val deviceOsVersion = Build.VERSION.SDK_INT
        if (context != null) {
            val deviceTotalMemory = getTotalMemoryInfo(context)
            val availMemory = getAvailableMemoryInfo(context)
            val memoryInfo = availMemory * 100 / deviceTotalMemory
            analyticsJson.put(
                AnalyticsConstant.CP_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
            )
        } else {
            analyticsJson.put(
                AnalyticsConstant.CP_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion
            )
        }
        return analyticsJson
    }

    private fun getTotalMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        val totalRam = memoryInfo.totalMem / (1024 * 1024)
        return totalRam
    }

    private fun getAvailableMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        val availmem = memoryInfo.availMem / (1024 * 1024)
        return availmem
    }
}