package com.payu.checkoutpro.factory

import com.payu.base.models.NBOptions
import com.payu.base.models.PayUPaymentParams
import com.payu.base.models.PayUSIParams
import com.payu.base.models.PaymentOption
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_HASH_NAME
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_HASH_STRING
import com.payu.india.Payu.PayuConstants
import com.payu.paymentparamhelper.PaymentParams
import com.payu.paymentparamhelper.PayuUtils

class HashGenerationHelper(
    private val payUPaymentParams: PayUPaymentParams,
    private val paymentParams: PaymentParams? = null,
    private val paymentOption: PaymentOption? = null
) {
    private var v2HashData: String? = null
    fun getHashData(
        hashName: String
    ): HashMap<String, String?> {

        return when (hashName) {
            PayUCheckoutProConstants.CP_PAYMENT_HASH -> getPaymentHashData(hashName)
            PayUCheckoutProConstants.CP_SI_PAYMENT_HASH -> getPaymentHashDataForSIModeOrOtm(hashName)
            PayUCheckoutProConstants.CP_TPV_PAYMENT_HASH -> getPaymentHashDataForTpv(
                hashName
            )

            PayUCheckoutProConstants.CP_WEALTH_TECH_PAYMENT_SOURCE -> getWealthHashString(hashName)

            PayUCheckoutProConstants.CP_LOOKUP_API_HASH -> getLookupApiData(hashName)
            else -> getWebServiceHashData(hashName)
        }
    }

    fun getV2Hashdata(hashName: String): HashMap<String, String?> {
        val hashMap = HashMap<String, String?>()
        hashMap.put(CP_HASH_STRING, v2HashData.toString())
        hashMap.put(CP_HASH_NAME, hashName)
        return hashMap
    }

    internal fun setV2HashData(hashData: String) {
        this.v2HashData = hashData
    }

    private fun getLookupApiData(hashName: String): HashMap<String, String?> {
        var hashString: String? = null
        if (!payUPaymentParams.additionalParams.isNullOrEmpty())
            hashString =
                payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_LOOKUP_API_POST_DATA] as String
        else return HashMap()

        val map = HashMap<String, String?>()
        map[CP_HASH_STRING] = hashString
        map[CP_HASH_NAME] = hashName
        return map
    }

    /**
     * Function to return hash data for generating payment hash. Hash data will be of below form
     *
     * key|txnid|amount|productinfo|firstname|email|udf1|udf2|udf3|udf4|udf5||||||SALT
     *
     * */
    private fun getPaymentHashData(hashName: String): HashMap<String, String?> {
        payUPaymentParams.additionalParams?.let { additionalParams ->
            val udf1 =
                additionalParams[PayUCheckoutProConstants.CP_UDF1] as? String
            val udf2 =
                additionalParams[PayUCheckoutProConstants.CP_UDF2] as? String
            val udf3 =
                additionalParams[PayUCheckoutProConstants.CP_UDF3] as? String
            val udf4 =
                additionalParams[PayUCheckoutProConstants.CP_UDF4] as? String
            val udf5 =
                additionalParams[PayUCheckoutProConstants.CP_UDF5] as? String

            val map = HashMap<String, String?>()
            map[CP_HASH_STRING] =
                "${payUPaymentParams.key}|${payUPaymentParams.transactionId}|${payUPaymentParams.amount}|${payUPaymentParams.productInfo}|${payUPaymentParams.firstName}|${payUPaymentParams.email}|$udf1|$udf2|$udf3|$udf4|$udf5||||||"
            map[CP_HASH_NAME] = hashName
            appendPostSalt(map)
            return map
        }
        return HashMap()
    }

    private fun getWebServiceHashData(command: String): HashMap<String, String?> {
        val map = HashMap<String, String?>()
        val var1 = getWebServiceVar1(command)
        if (var1 != null) {
            map[CP_HASH_STRING] = "${payUPaymentParams.key}|$command|$var1|"
            map[CP_HASH_NAME] = command
        }
        return map
    }

    private fun getWebServiceVar1(command: String): String? {
        var result: String? = null
        when (command) {
            PayUCheckoutProConstants.CP_VAS_FOR_MOBILE_SDK, PayUCheckoutProConstants.CP_ELIGIBLE_BINS_FOR_EMI -> {
                result = PayUCheckoutProConstants.CP_DEFAULT
            }

            PayUCheckoutProConstants.CP_PAYMENT_RELATED_DETAILS_FOR_MOBILE_SDK, PayuConstants.DELETE_TOKENISED_USER_CARD -> {
                result =
                    payUPaymentParams.userCredential
                if (result == null || result.isEmpty())
                    result = PayUCheckoutProConstants.CP_DEFAULT
            }

            PayuConstants.CHECK_IS_DOMESTIC -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayuConstants.CARD_BIN] as String
            }

            PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY] as String
//                payUPaymentParams.key + "|get_eligible_payment_options|{\"amount\":\"" + payUPaymentParams.amount + "\",\"txnid\":\"" + payUPaymentParams.transactionId + "\",\"mobile_number\":\"" + payUPaymentParams.phone.toString().trim { it <= ' ' } + "\",\"first_name\":\"" + payUPaymentParams.merchantName + "\",\"bankCode\":\"OLAM\",\"email\":\"\",\"last_name\":\"\"}"
            }

            PayUCheckoutProConstants.CP_VPA_VALIDATION -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_VPA_VALIDATION] as String
//                    payUPaymentParams.key + "|validateVPA|" + vpa
            }

            PayuConstants.GET_BIN_INFO -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayuConstants.VAR1] as String
            }

            PayUCheckoutProConstants.CP_GET_EMI_AMOUNT_ACCORDING_TO_INTEREST -> result =
                payUPaymentParams.amount

            PayUCheckoutProConstants.CP_CHECK_OFFER_DETAILS -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_OFFER_KEYS] as String
            }

            PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null

                val checkEligibilityVar1 =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_CHECK_ELIGIBILITY_VAR1] as? String
                val getCheckoutDetailsVar1 =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS_VAR1] as? String
                result = if (!checkEligibilityVar1.isNullOrEmpty()) {
                    checkEligibilityVar1
                } else if (!getCheckoutDetailsVar1.isNullOrEmpty()) {
                    getCheckoutDetailsVar1
                } else null

            }

            PayUCheckoutProConstants.CP_CHECK_BALANCE -> {
                if (payUPaymentParams.additionalParams.isNullOrEmpty()) return null
                result =
                    payUPaymentParams.additionalParams!![PayUCheckoutProConstants.CP_CHECK_BALANCE_VAR1] as String
            }

            PayUCheckoutProConstants.SDK_CONFIGURATION -> {
                result = PayUCheckoutProConstants.CP_GET
            }

            PayUCheckoutProConstants.VERIFY_PAYMENT -> {
                result = payUPaymentParams.transactionId
            }
        }
        return result!!
    }

    private fun getPaymentHashDataForTpv(
        hashName: String
    ): HashMap<String, String?> {
        payUPaymentParams.additionalParams?.let { additionalParams ->
            val udf1 =
                additionalParams[PayUCheckoutProConstants.CP_UDF1] as? String
            val udf2 =
                additionalParams[PayUCheckoutProConstants.CP_UDF2] as? String
            val udf3 =
                additionalParams[PayUCheckoutProConstants.CP_UDF3] as? String
            val udf4 =
                additionalParams[PayUCheckoutProConstants.CP_UDF4] as? String
            val udf5 =
                additionalParams[PayUCheckoutProConstants.CP_UDF5] as? String

            val map = HashMap<String, String?>()

            map[CP_HASH_STRING] =
                "${payUPaymentParams.key}|${payUPaymentParams.transactionId}|${payUPaymentParams.amount}|${payUPaymentParams.productInfo}|${payUPaymentParams.firstName}|${payUPaymentParams.email}|$udf1|$udf2|$udf3|$udf4|$udf5||||||${getBeneficiaryDetails()}"
            map[CP_HASH_NAME] = hashName

            map[CP_HASH_NAME] = hashName
            appendPostSalt(map)
            return map
        }
        return HashMap()
    }

    private fun appendPostSalt(map: HashMap<String, String?>) {
        var postsalt = ""

        payUPaymentParams.additionalCharges.takeIf { it.isNullOrBlank().not() }?.let { postsalt += "|$it" }
        payUPaymentParams.splitPaymentDetails.takeIf { it.isNullOrBlank().not() }?.let { postsalt += "|$it" }
        payUPaymentParams.percentageAdditionalCharges.takeIf { it.isNullOrBlank().not() }?.let { postsalt += "|$it" }

        map[PayUCheckoutProConstants.CP_POST_SALT] = postsalt
    }


    /**
        Most Important: Keep the order of values same as in PaymentPostParams
     * */
    private fun generateSiHashString(siDetails: PayUSIParams?): String {
        return if (siDetails != null) {
            val siDetailsStringForHash = StringBuilder()
            siDetailsStringForHash.append("{")
            siDetailsStringForHash.append(
                PayuUtils.appendKeyValueToStingBuilder(
                    PayuConstants.BILLING_START_DATE,
                    siDetails?.paymentStartDate
                )
            )
            siDetailsStringForHash.append(
                PayuUtils.appendKeyValueToStingBuilder(
                    PayuConstants.BILLING_END_DATE,
                    siDetails?.paymentEndDate
                )
            )

            if (siDetails?.isPreAuthTxn != true) {
                siDetailsStringForHash.append(
                    PayuUtils.appendKeyValueToStingBuilder(
                        PayuConstants.BILLING_AMOUNT,
                        siDetails?.billingAmount
                    )
                )
                siDetailsStringForHash.append(
                    PayuUtils.appendKeyValueToStingBuilder(
                        PayuConstants.BILLING_CURRENCY,
                        siDetails?.billingCurrency
                    )
                )
                siDetailsStringForHash.append(
                    PayuUtils.appendKeyValueToStingBuilder(
                        PayuConstants.BILLING_CYCLE,
                        siDetails?.billingCycle
                    )
                )
                siDetailsStringForHash.append(
                    PayuUtils.appendKeyValueToStingBuilder(
                        PayuConstants.BILLING_INTERVAL,
                        siDetails?.billingInterval
                    )
                )
                if (!siDetails?.remarks.isNullOrEmpty()) {
                    siDetailsStringForHash.append(
                        PayuUtils.appendKeyValueToStingBuilder(
                            PayuConstants.BILLING_REMARKS,
                            siDetails?.remarks
                        )
                    )
                }
                if (payUPaymentParams.additionalParams?.get(PayUCheckoutProConstants.CP_IS_UPISI) == true) {
                    if (!siDetails?.billingLimit?.name.isNullOrEmpty()) {
                        siDetailsStringForHash.append(
                            PayuUtils.appendKeyValueToStingBuilder(
                                PayuConstants.BILLING_LIMIT,
                                siDetails?.billingLimit
                            )
                        )
                    }
                    if (!siDetails?.billingRule?.name.isNullOrEmpty()) {
                        siDetailsStringForHash.append(
                            PayuUtils.appendKeyValueToStingBuilder(
                                PayuConstants.BILLING_RULE,
                                siDetails?.billingRule
                            )
                        )
                    }
                    if (!siDetails?.billingDate.isNullOrEmpty()) {
                        siDetailsStringForHash.append(
                            PayuUtils.appendKeyValueToStingBuilder(
                                PayuConstants.BILLING_DATE,
                                siDetails?.billingDate
                            )
                        )
                    }
                }
                siDetailsStringForHash.deleteCharAt(siDetailsStringForHash.length - 1)
                siDetailsStringForHash.append("}")
                return "${siDetailsStringForHash}|${getFreeTrialSuffix(siDetails)}"
            }
            siDetailsStringForHash.deleteCharAt(siDetailsStringForHash.length - 1)
            siDetailsStringForHash.append("}")
            "${siDetailsStringForHash}|"
        } else "|"
    }

    /**
     * Function to return hash data for generating payment hash. Hash data will be of below form
     *
     * HASH = SHA512(key|txnid|amount|productinfo|firstname|email|udf1|udf2|udf3|udf4|udf5||||||si_details|SALT)
     *
     * */
    internal fun getPaymentHashDataForSIModeOrOtm(hashName: String): HashMap<String, String?> {
        return payUPaymentParams.additionalParams?.let { additionalParams ->
            val siDetails = payUPaymentParams.payUSIParams
            val udfs = extractUdfs(additionalParams)
            val amount = determineAmount(additionalParams, siDetails)

            createHashMapWithPaymentData(hashName, udfs, amount, siDetails)
        } ?: HashMap()
    }

    private fun extractUdfs(additionalParams: Map<String, Any?>): List<String?> {
        return listOf(
            additionalParams[PayUCheckoutProConstants.CP_UDF1] as? String,
            additionalParams[PayUCheckoutProConstants.CP_UDF2] as? String,
            additionalParams[PayUCheckoutProConstants.CP_UDF3] as? String,
            additionalParams[PayUCheckoutProConstants.CP_UDF4] as? String,
            additionalParams[PayUCheckoutProConstants.CP_UDF5] as? String
        )
    }

    private fun determineAmount(additionalParams: Map<String, Any?>, siDetails: PayUSIParams?): String? {
        val enachAmount = additionalParams[PayUCheckoutProConstants.CP_ENACH_AMOUNT] as? String
        return if (!enachAmount.isNullOrEmpty() && siDetails?.isPreAuthTxn != true) {
            enachAmount
        } else {
            payUPaymentParams.amount
        }
    }

    private fun getFreeTrialSuffix(siDetails: PayUSIParams?): String {
        return if (siDetails != null && siDetails.isFreeTrial && !siDetails.isPreAuthTxn) {
            "1|"
        } else {
            ""
        }
    }

    private fun buildHashString(udfs: List<String?>, amount: String?, siDetails: PayUSIParams?): String {
        val (udf1, udf2, udf3, udf4, udf5) = udfs

        return "${payUPaymentParams.key}|${payUPaymentParams.transactionId}|" +
                "$amount|${payUPaymentParams.productInfo}|" +
                "${payUPaymentParams.firstName}|${payUPaymentParams.email}|" +
                "$udf1|$udf2|$udf3|$udf4|$udf5||||||" +
                generateSiHashString(siDetails)
    }

    private fun buildHashStringForWealthTech(
        udfs: List<String?>,
        amount: String?,
        siDetails: PayUSIParams?
    ): String {
        val (udf1, udf2, udf3, udf4, udf5) = udfs
        val products = getProductDetailsJsonArray(payUPaymentParams)
        val productHash = if (products.isBlank().not())
            "${if (paymentParams?.userToken.isNullOrBlank()) "" else paymentParams?.userToken}|${if (paymentParams?.offerKey.isNullOrBlank()) "" else paymentParams?.offerKey}||${if (paymentParams?.skuCartDetails.isNullOrBlank()) "" else paymentParams?.skuCartDetails}|$products|" else "|"
        return "${payUPaymentParams.key}|${payUPaymentParams.transactionId}|" +
                "$amount|${payUPaymentParams.productInfo}|" +
                "${payUPaymentParams.firstName}|${payUPaymentParams.email}|" +
                "$udf1|$udf2|$udf3|$udf4|$udf5||||||" +
                getBeneficiaryDetails() + generateSiHashString(siDetails)  + productHash
    }

    private fun createHashMapWithPaymentData(
        hashName: String,
        udfs: List<String?>,
        amount: String?,
        siDetails: PayUSIParams?
    ): HashMap<String, String?> {
        return HashMap<String, String?>().apply {
            this[CP_HASH_STRING] =
                if (payUPaymentParams.payUWealthProducts.isNullOrEmpty()) buildHashString(
                    udfs,
                    amount,
                    siDetails
                ) else buildHashStringForWealthTech(udfs, amount, siDetails)
            this[CP_HASH_NAME] = hashName
            appendPostSalt(this)
        }
    }

   private fun getProductDetailsJsonArray(mPaymentParams: PayUPaymentParams): String {
        val jsonBuilder = java.lang.StringBuilder()
        jsonBuilder.append("[")

       val productList = mPaymentParams.payUWealthProducts
       productList?.indices?.forEach { i ->
           val product = productList[i]
           jsonBuilder.append("{")

           jsonBuilder.append("\"").append(com.payu.paymentparamhelper.PayuConstants.PAYU_TYPE)
               .append("\":\"").append(product.type).append("\",")
           if (product.plan.isBlank().not()) {
               jsonBuilder.append("\"").append(com.payu.paymentparamhelper.PayuConstants.PAYU_PLAN)
                   .append("\":\"").append(product.plan).append("\",")
           }
           if (product.folio.isBlank().not()) {
               jsonBuilder.append("\"").append(com.payu.paymentparamhelper.PayuConstants.PAYU_FOLIO)
                   .append("\":\"").append(product.folio).append("\",")
           }
           jsonBuilder.append("\"").append(com.payu.paymentparamhelper.PayuConstants.AMOUNT)
               .append("\":\"").append(product.amount).append("\",")
           if (product.option.isBlank().not()) {
               jsonBuilder.append("\"")
                   .append(com.payu.paymentparamhelper.PayuConstants.PAYU_OPTION)
                   .append("\":\"").append(product.option).append("\",")
           }
           if (product.scheme.isBlank().not()) {
               jsonBuilder.append("\"")
                   .append(com.payu.paymentparamhelper.PayuConstants.PAYU_SCHEME)
                   .append("\":\"").append(product.scheme).append("\",")
           }
           jsonBuilder.append("\"").append(com.payu.paymentparamhelper.PayuConstants.PAYU_RECEIPT)
               .append("\":\"").append(product.receipt).append("\",")
           jsonBuilder.append("\"")
               .append(com.payu.paymentparamhelper.PayuConstants.PAYU_MF_MEMBER_ID).append("\":\"")
               .append(product.mfMemberID).append("\",")
           jsonBuilder.append("\"")
               .append(com.payu.paymentparamhelper.PayuConstants.PAYU_MF_USER_ID).append("\":\"")
               .append(product.mfUserID).append("\",")
           jsonBuilder.append("\"")
               .append(com.payu.paymentparamhelper.PayuConstants.PAYU_MF_PARTNER).append("\":\"")
               .append(product.mfPartner).append("\",")
           jsonBuilder.append("\"")
               .append(com.payu.paymentparamhelper.PayuConstants.PAYU_MF_INVESTMENT_TYPE)
               .append("\":\"").append(product.mfInvestmentType).append("\",")
           if (product.folio.isBlank().not()) {
               jsonBuilder.append("\"")
                   .append(com.payu.paymentparamhelper.PayuConstants.PAYU_MF_AMC_CODE)
                   .append("\":\"")
                   .append(product.mfAmcCode).append("\"")
           }

           jsonBuilder.append("}")

           if (i != productList.size - 1) {
               jsonBuilder.append(",")
           }
       }

       jsonBuilder.append("]")
       return jsonBuilder.toString()
   }

    private fun getWealthHashString(hashName: String): HashMap<String, String?> {
        return payUPaymentParams.additionalParams?.let { additionalParams ->
            val siDetails = payUPaymentParams.payUSIParams
            val udfs = extractUdfs(additionalParams)
            val amount = determineAmount(additionalParams, siDetails)

            createHashMapWithPaymentData(hashName, udfs, amount, siDetails)
        } ?: HashMap()
    }


    private fun getBeneficiaryDetails(): StringBuilder {
        val beneficiaryDetail = StringBuilder()
        val beneficiaryDetailOption = (paymentOption as? NBOptions)?.beneficiaryDetails
        if (paymentOption?.accountNumber.isNullOrBlank()
                .not() && paymentOption?.ifscCode.isNullOrBlank().not()
        ) {
            beneficiaryDetail.append("{").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.BENEFICIARY_ACCOUNT_NUMBER)
                .append("\"").append(":").append("\"").append(paymentOption?.accountNumber)
                .append("\"")
            beneficiaryDetail.append(",").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.IFSC_CODE).append("\"")
                .append(":").append("\"").append(paymentOption?.ifscCode).append("\"")
            beneficiaryDetail.append("}")
        } else if ((paymentOption as? NBOptions)?.beneficiaryDetails != null) {
            beneficiaryDetail.append("{").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.BENEFICIARY_NAME)
                .append("\"").append(":").append("\"")
                .append(beneficiaryDetailOption?.beneficiaryName)
                .append("\"")
            beneficiaryDetail.append(",").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.BENEFICIARY_ACCOUNT_NUMBER)
                .append("\"").append(":").append("\"")
                .append(beneficiaryDetailOption?.beneficiaryAccountNumber)
                .append("\"")
            beneficiaryDetail.append(",").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.IFSC_CODE).append("\"")
                .append(":").append("\"").append(beneficiaryDetailOption?.beneficiaryIfsc)
                .append("\"")
            beneficiaryDetail.append(",").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.BENEFICIARY_ACCOUNT_TYPE).append("\"")
                .append(":").append("\"").append(beneficiaryDetailOption?.beneficiaryAccountType)
                .append("\"")
            beneficiaryDetail.append(",").append("\"")
                .append(com.payu.paymentparamhelper.PayuConstants.VERTIFICATION_MODE).append("\"")
                .append(":").append("\"").append(beneficiaryDetailOption?.verificationMode?.name)
                .append("\"")
            beneficiaryDetail.append("}")
        }
        beneficiaryDetail.append(
            "|"
        )

        return beneficiaryDetail
    }



}