package com.payu.checkoutpro.utils

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64
import android.util.Log
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.GLOBAL_VAULT_USER_TOKEN_PREF
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.SHARED_PREFERENCE_KEY
import com.payu.ui.model.utils.SdkUiConstants
import java.security.KeyStore
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.SecretKey
import javax.crypto.spec.IvParameterSpec


object SharedPrefsUtils {

    private const val CIPHER_TRANSFORMATION = "AES/CBC/PKCS7Padding";
    private const val ENCRYPTION_ALGORITHM = "AES"
    private const val KEYSTORE_PROVIDER_ANDROID_KEYSTORE: String = "AndroidKeyStore"
    private const val mAlias: String = "PayUKeyAlias"

    private fun createKeys(context: Context?): SecretKey? {
        try {
            val kpGenerator = KeyGenerator
                .getInstance(
                    ENCRYPTION_ALGORITHM,
                    KEYSTORE_PROVIDER_ANDROID_KEYSTORE
                )

            val spec = if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
                KeyPairGeneratorSpec.Builder(context!!)
                    .setAlias(mAlias)
                    .build()
            } else {
                KeyGenParameterSpec.Builder(
                    mAlias,
                    KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
                )
                    .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                    .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_PKCS7)
                    .build()
            }
            kpGenerator.init(spec)
            return kpGenerator.generateKey()
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "createKey:Exception" + e.localizedMessage)
        }
        return null
    }

    private fun getKeys(context: Context?): SecretKey? {
        try {
            val ks = KeyStore.getInstance(KEYSTORE_PROVIDER_ANDROID_KEYSTORE)
            // Weird artifact of Java API.  If you don't have an InputStream to load, you still need
            // to call "load", or it'll crash.
            ks.load(null)
            // Load the key pair from the Android Key Store
            val existingKey = ks.getEntry(mAlias, null) as? KeyStore.SecretKeyEntry
            return existingKey?.secretKey ?: createKeys(context)
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "getKeys:Exception" + e.localizedMessage)
            createKeys(context)
        }
        return null
    }

    private fun encrypt(context: Context, data: String): String? {
        return try {
            getKeys(context)?.let {
                val cipher = Cipher.getInstance(CIPHER_TRANSFORMATION)
                cipher.init(Cipher.ENCRYPT_MODE, it)
                val cipherText = cipher.doFinal(data.toByteArray())
                val iv = cipher.iv
                val encryptedDataWithIV = ByteArray(iv.size + cipherText.size)
                System.arraycopy(iv, 0, encryptedDataWithIV, 0, iv.size)
                System.arraycopy(cipherText, 0, encryptedDataWithIV, iv.size, cipherText.size)
                Base64.encodeToString(encryptedDataWithIV, Base64.DEFAULT)
            }
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "encrypt:Exception" + e.localizedMessage)
            null
        }
    }

    private fun decrypt(context: Context, encryptedData: String): String? {
        return try {
            getKeys(context)?.let {
                val ex = encryptedData.replace("\n", "");
                val cipherText: ByteArray = Base64.decode(ex, Base64.DEFAULT)
                val cipher = Cipher.getInstance(CIPHER_TRANSFORMATION)
                val iv = cipherText.copyOfRange(0, cipher.blockSize)
                cipher.init(Cipher.DECRYPT_MODE, it, IvParameterSpec(iv))
                val data = cipherText.copyOfRange(cipher.blockSize, cipherText.size)
                String(cipher.doFinal(data))
            }
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "decrypt:Exception" + e.localizedMessage)
            null
        }
    }

    private fun getSharedPref(
        context: Context,
        fileName: String
    ): SharedPreferences {
        getKeys(context)
        return context.getSharedPreferences(fileName, MODE_PRIVATE)
    }


    fun storeInSharedPreferences(context: Context, key: String?, value: String?) {
        val sharedPreferences = getSharedPref(context, SHARED_PREFERENCE_KEY)
        val editor = sharedPreferences.edit()
        value?.let { nonNullValue ->
            editor?.putString(
                key,
                encrypt(
                    context,
                    nonNullValue
                )
            )
        }
        editor?.apply()

    }

    fun getFromSharedPreferences(context: Context, key: String?): String? {
        try {
            val sharedPreferences = getSharedPref(context, SHARED_PREFERENCE_KEY)
            var decodedString: String? = null
            key?.let { nonNullKey ->
                decodedString = sharedPreferences.getString(nonNullKey, "")
                decodedString = decodedString?.let { nonNullDecodedString ->
                    decrypt(
                        context,
                        nonNullDecodedString
                    )
                }
            }
            return decodedString
        } catch (e: Exception) {
            removeAllFromSharedPref(context, SHARED_PREFERENCE_KEY)
            return ""
        }
    }

    fun deleteFromSharedPreferences(context: Context, key: String?) {
        val sharedPreferences = getSharedPref(context, SHARED_PREFERENCE_KEY)
        val editor = sharedPreferences.edit()
        editor?.remove(key)
        editor?.apply()
    }

    fun getLoggedInPhoneNumber(context: Context): String {
        val sharedPreferences = getSharedPref(context, GLOBAL_VAULT_USER_TOKEN_PREF)
        sharedPreferences.let {
            try {
                return it.getString(SdkUiConstants.MOBILE_NUMBER, "") ?: ""
            } catch (e: Exception) {
                val editor = it.edit()
                editor?.remove(SdkUiConstants.MOBILE_NUMBER)
                editor?.apply()
            }
        }
        return ""
    }

    /**
     * Simple helper method to remove all the keys from given shared pref file.
     *
     * @param context
     * @param prefName
     */
    private fun removeAllFromSharedPref(context: Context, prefName: String) {
        try {
            val settings: SharedPreferences? = getSharedPref(context, prefName)
            settings?.edit()?.clear()?.apply()
        } catch (e: java.lang.Exception) {
        }
    }

}