package com.payu.checkoutpro.utils

import android.content.Context
import com.payu.base.models.PaymentMode
import com.payu.checkoutpro.models.PayUCheckoutProConfig
import com.payu.checkoutpro.utils.ConfigConstants.CP_MERCHANT_APP_VERSION
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CONFIG_RESPONSE_KEY
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.SP_CONFIG_KEY
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

object ConfigUtils {
    /**
     * Helper method to override configuration values passed by merchant during SDK
     * initialization with values received in config api response from server.
     */
    internal fun updateConfig(
        rawResponse: JSONObject?,
        payUCheckoutProConfig: PayUCheckoutProConfig?
    ) {
        if (rawResponse == null) return

        val response = rawResponse.optJSONArray(CONFIG_RESPONSE_KEY) ?: return

        for (i in 0 until response.length()) {
            val result = response.getJSONObject(i)
            payUCheckoutProConfig?.let {
                if (haveKey(ConfigConstants.CP_MERCHANT_NAME, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.merchantName = it.toString()
                    }

                if (haveKey(ConfigConstants.CP_MERCHANT_LOGO_URL, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.merchantLogoUrl = it.toString()
                    }

                if (haveKey(ConfigConstants.CP_AUTO_APPROVE, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.autoApprove = (it as String).toBoolean()
                    }

                if (haveKey(ConfigConstants.CP_AUTO_SELECT_OTP, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.autoSelectOtp = (it as String).toBoolean()
                    }

                if (haveKey(ConfigConstants.CP_SHOW_EXIT_CONFIRMATION_ON_PAYMENT_SCREEN, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.showExitConfirmationOnPaymentScreen =
                            (it as String).toBoolean()
                    }

                if (haveKey(
                        ConfigConstants.CP_MERCHANT_EXIT_CONFIRMATION_ON_CHECKOUT_SCREEN,
                        result
                    )
                )
                    extractResult(result).let {
                        payUCheckoutProConfig.showExitConfirmationOnCheckoutScreen =
                            (it as String).toBoolean()
                    }

                if (haveKey(ConfigConstants.CP_SHOW_TOOLBAR, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.showCbToolbar = (it as String).toBoolean()
                    }

                if (haveKey(ConfigConstants.CP_MERCHANT_SMS_PERMISSION, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.merchantSmsPermission = (it as String).toBoolean()
                    }

                if (haveKey(ConfigConstants.CP_MERCHANT_RESPONSE_TIMEOUT, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.merchantResponseTimeout = (it as String).toInt()
                    }

                if (haveKey(ConfigConstants.CP_MERCHANT_WAITING_TIMEOUT, result))
                    extractResult(result).let {
                        payUCheckoutProConfig.waitingTime = (it as String).toInt()
                    }

                if (haveKey(ConfigConstants.CP_PAYMENT_MODE_ORDER, result)) {
                    val paymentModesOrder = extractPaymentModesOrder(result)
                    payUCheckoutProConfig.paymentModesOrder = paymentModesOrder
                }


                if (haveKey(ConfigConstants.CP_PRIMARY_COLOR, result)) {
                    extractResult(result).let {
                        payUCheckoutProConfig.primaryColor = it.toString()
                    }
                }
                if (haveKey(ConfigConstants.CP_PRIMARY_COLOR_DARK, result)) {
                    extractResult(result).let {
                        payUCheckoutProConfig.primaryColorDark = it.toString()
                    }
                }
                if (haveKey(ConfigConstants.CP_PRIMARY_COLOR_ACCENT, result)) {
                    extractResult(result).let {
                        payUCheckoutProConfig.primaryColorAccent = it.toString()
                    }
                }
                if (haveKey(ConfigConstants.CP_BASE_TEXT_COLOR, result)) {
                    extractResult(result).let {
                        payUCheckoutProConfig.baseTextColor = it.toString()
                    }
                }
                if (haveKey(ConfigConstants.CP_UPI_APP_ORDER, result)) {
                    extractResult(result).let {
                        payUCheckoutProConfig.upiAppsOrder = it.toString()
                    }
                }
            }

        }
    }


    /**
     * Helper method to extract out enforced payment mode object from config response
     * and override current config value
     */
//    private fun extractEnforcePaymentList(result: JSONObject): java.util.ArrayList<HashMap<String, String>>? {
//        var enforcePayment = result!!.getJSONArray("enforcePaymentList")
//        var enforcePaymentList = ArrayList<HashMap<String, String>>()
//        for (j in 0 until enforcePayment.length()) {
//            val jsonObject = enforcePayment.getJSONObject(j)
//            var map = HashMap<String, String>()
//            map["payment_type"] = jsonObject.getString("payment_type")
//            map["card_type"] = jsonObject.getString("card_type")
//            map["card_scheme"] = jsonObject.getString("card_scheme")
//            enforcePaymentList.add(map)
//        }
//        return enforcePaymentList
//    }
//
//    /**
//     * Helper method to extract out custom note object from config response
//     * and override current config value
//     */
//    private fun extractCustomNote(result: JSONObject): java.util.ArrayList<CustomNote>? {
//        var customNote = result!!.getJSONArray(ConfigConstants.CP_CUSTOM_NOTE_DETAILS)
//        var customNoteDetailsList = ArrayList<CustomNote>()
//        for (j in 0 until customNote.length()) {
//            val jsonObject = customNote.getJSONObject(j)
//            var paymentTypeList = ArrayList<PaymentType>()
//
//            val jsonArray = jsonObject.getJSONArray("custom_note_category")
//            for (i in 0 until jsonArray.length()) {
//                paymentTypeList.add(enumValueOf(jsonArray.getString(i)))
//            }
//            customNoteDetailsList.add(
//                CustomNote(
//                    jsonObject.getString("custom_note"),
//                    paymentTypeList
//                )
//            )
//        }
//        return customNoteDetailsList
//
//    }
//
//    /**
//     * helper method to check if config response contains a particular key,
//     * this is done to override only those config values which we have received
//     * in response.
//     */
    private fun haveKey(key: String, response: JSONObject): Boolean {
        return response.get(ConfigConstants.CP_KEY) == key
    }

    private fun extractResult(response: JSONObject): Any {
        return response.get(ConfigConstants.CP_VALUE)
    }

    /**
     * Helper method to extract out payment modes order object from config response
     * and override current config value
     */
    private fun extractPaymentModesOrder(result: JSONObject): ArrayList<PaymentMode> {
        val extractResult = JSONArray((extractResult(result) as String))

        val paymentModesOrderList = ArrayList<PaymentMode>()
        for (j in 0 until extractResult.length()) {
            val jsonObject = extractResult.getJSONObject(j)

            paymentModesOrderList.add(
                PaymentMode(
                    enumValueOf(jsonObject.getString(ConfigConstants.CP_PAYMENT_TYPE))
                )
            )
        }
        return paymentModesOrderList
    }

    /**
     * Helper method to extract/fetch cached config data.
     * This method is responsible for two things:
     * 1. Fetches cached config data from server
     * 2. If cached data is not null, extracts app version value from cached data and compares with
     * current app version if same then return cached data and if not it removes/deletes
     * cached dat as it is outdated.
     */
    internal fun getCachedConfigForAppVersion(context: Context): JSONObject? {
        val fromSharedPreferences =
            SharedPrefsUtils.getFromSharedPreferences(context, SP_CONFIG_KEY) ?: return null

        if (fromSharedPreferences != null && fromSharedPreferences.isEmpty().not()) {
            try {
                val appVersion = JSONObject(fromSharedPreferences).getString(CP_MERCHANT_APP_VERSION)

                /**
                 * compare current app version with version stored in config SP
                 * This is do remove old version of config saved if config fetch api failed
                 * and we have a new version of app, if version are same return cached
                 * config data
                 */
                if (appVersion.equals(CommonUtils.getAppVersion(context), ignoreCase = true))
                    return JSONObject(fromSharedPreferences)
                else {
                    //Delete config SP as this is an outdated/for previous app version
                    SharedPrefsUtils.deleteFromSharedPreferences(context, SP_CONFIG_KEY)
                }
            }catch (e: JSONException){
                return null
            }

        }
        return null
    }

    /**
     * Method to cache config data, we are adding a new key value in response received
     * from server to make cached config versioned
     */
    internal fun cacheConfig(context: Context, rawResponse: JSONObject) {
        rawResponse.put(CP_MERCHANT_APP_VERSION, CommonUtils.getAppVersion(context))
        SharedPrefsUtils.storeInSharedPreferences(
            context,
            SP_CONFIG_KEY, rawResponse.toString()
        )
    }

    /**
     * Method to set cached config data to override merchant passed config
     */
    internal fun setCachedConfig(context: Context, config: PayUCheckoutProConfig) {
        val cachedConfigForAppVersion = getCachedConfigForAppVersion(context)
        if (cachedConfigForAppVersion != null)
            updateConfig(cachedConfigForAppVersion, config)
    }
}