package com.payu.checkoutpro.utils

import android.content.Context
import android.content.Intent
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.net.Uri
import android.util.Log
import androidx.core.graphics.drawable.toBitmap
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.google.android.gms.common.GooglePlayServicesNotAvailableException
import com.google.android.gms.common.GooglePlayServicesRepairableException
import com.payu.base.models.GoogleAdIdInfo
import com.payu.base.models.PaymentOption
import com.payu.base.models.UPIOption
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_PHONE_PE_MIN_VERSION
import java.io.IOException
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale

/**
 * This class should have utility methods that require context
 * */
object AndroidUtils {

    internal fun getInstalledUpiAppsList(context: Context?): ArrayList<PaymentOption>? {
        if (context == null)
            return null

        val paymentOptionList = ArrayList<PaymentOption>()
        val intent = Intent()
        intent.data = Uri.parse(PayUCheckoutProConstants.CP_UPI_INTENT_PREFIX)
        val resolveInfos = context.packageManager
            .queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY)
        for (resolveInfo in resolveInfos) {
            var packageInfo: PackageInfo? = null
            try {
                packageInfo = context.packageManager
                    .getPackageInfo(resolveInfo.activityInfo.packageName, 0)
                val packageDrawable =
                    context.packageManager.getApplicationIcon(packageInfo.applicationInfo)
                val name =
                    context.packageManager.getApplicationLabel(packageInfo.applicationInfo) as String

                val paymentOption = UPIOption()
                paymentOption.bankName = name
                paymentOption.packageName = packageInfo.packageName
                paymentOptionList.add(paymentOption)
            } catch (e: PackageManager.NameNotFoundException) {
                e.printStackTrace()
            }
        }
        return paymentOptionList
    }

    internal fun getUpiAppBitmap(context: Context?, packageName: String): Bitmap? {

        var upiAppBitmap: Bitmap? = null
        if (context == null)
            return upiAppBitmap

        upiAppBitmap = context.packageManager.getApplicationIcon(packageName).toBitmap()
        return upiAppBitmap
    }

    internal fun isUPIAppInstalled(context: Context?, packageName: String): Boolean {
        if (context == null)
            return false

        val installedApps = getInstalledUpiAppsList(context)
        if (installedApps == null || installedApps.size == 0) return false
        for (upiApp in installedApps) {
            val upiOption = upiApp as? UPIOption
            if (upiOption?.packageName.equals(packageName, ignoreCase = true))
                return true
        }
        return false
    }

    fun determineAdvertisingInfo(context: Context, response: (GoogleAdIdInfo?) -> Unit) {
        Thread {
            try {
                val idInfo: AdvertisingIdClient.Info =
                    AdvertisingIdClient.getAdvertisingIdInfo(context)
                response(GoogleAdIdInfo(idInfo, true, null))
            } catch (e: IOException) {
                response(GoogleAdIdInfo(null, false, e.message))
            } catch (e: GooglePlayServicesNotAvailableException) {
                response(GoogleAdIdInfo(null, false, e.message))
            } catch (e: GooglePlayServicesRepairableException) {
                response(GoogleAdIdInfo(null, false, e.message))
            } catch (t: Throwable) {
                response(GoogleAdIdInfo(null, false, t.message))
            }
        }.start()
    }

    internal fun doesPhonePeAppExist(context: Context?): String? {
        var packageInfo: PackageInfo? = null
        var phonePeVersionCode = -1L
        try {
            packageInfo = context?.packageManager?.getPackageInfo(
                PayUCheckoutProConstants.CP_PHONEPE_PACKAGE_NAME,
                PackageManager.GET_ACTIVITIES
            )
            phonePeVersionCode = packageInfo?.versionCode?.toLong()?:0
        } catch (e: PackageManager.NameNotFoundException) {
            Log.e(
                PayUCheckoutProConstants.CP_PHONEPE_PACKAGE_NAME,
                java.lang.String.format(
                    "failed to get package info for package name = {%s}, exception message = {%s}",
                    PayUCheckoutProConstants.CP_PHONEPE_PACKAGE_NAME,
                    e.message
                )
            )
        }
        if (packageInfo == null) {
            return null
        }
        return if (phonePeVersionCode > CP_PHONE_PE_MIN_VERSION)
            phonePeVersionCode.toString()
        else null
    }

    internal fun getRequestDateTimeFormat(): String {
        val dateTimeFormatter = SimpleDateFormat("yyyyMMddHHmmss", Locale.getDefault())
        val now = Date()
        return dateTimeFormatter.format(now)
    }
}