package com.payu.checkoutpro.parser.utils

import com.payu.base.models.PayUAddressDetails
import com.payu.base.models.PayUBeneficiaryAccountType
import com.payu.base.models.PayUBeneficiaryDetail
import com.payu.base.models.PayUBeneficiaryVerificationMode
import com.payu.base.models.PayUBillingCycle
import com.payu.base.models.PayUSIParams
import com.payu.base.models.PayUWealthProducts
import com.payu.base.models.PayuBillingLimit
import com.payu.base.models.PayuBillingRule
import com.payu.base.models.SKU
import com.payu.base.models.SkuDetails
import com.payu.checkoutpro.parser.constants.PayUHybridKeys

object PaymentParamParserUtils {

    /**
     * Method to map/transform Hashmap to PayUSIParams
     */

    internal fun getAddressDetails(map: HashMap<String, Any?>): PayUAddressDetails {
        val addressDetails = PayUAddressDetails.Builder()
        (map[PayUHybridKeys.PaymentParam.AddressDetails.lastName] as? String)?.let {
            addressDetails.setLastName(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.address1] as? String)?.let {
            addressDetails.setAddress1(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.address2] as? String)?.let {
            addressDetails.setAddress2(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.city] as? String)?.let {
            addressDetails.setCity(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.state] as? String)?.let {
            addressDetails.setState(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.country] as? String)?.let {
            addressDetails.setCountry(it)
        }
        (map[PayUHybridKeys.PaymentParam.AddressDetails.zipcode] as? String)?.let {
            addressDetails.setZipcode(it)
        }
        return addressDetails.build()
    }
    internal fun getSIParams(map: HashMap<String, Any?>): PayUSIParams {
        val siParamBuilder: PayUSIParams.Builder = PayUSIParams.Builder()
        if (map.containsKey(PayUHybridKeys.PaymentParam.SIParams.isFreeTrial))
            siParamBuilder.setIsFreeTrial(map[PayUHybridKeys.PaymentParam.SIParams.isFreeTrial] as Boolean)

        (map[PayUHybridKeys.PaymentParam.SIParams.billingAmount] as? String)?.let {
            siParamBuilder.setBillingAmount(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.billingInterval] as? Int)?.let {
            siParamBuilder.setBillingInterval(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.paymentStartDate] as? String)?.let {
            siParamBuilder.setPaymentStartDate(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.paymentEndDate] as? String)?.let {
            siParamBuilder.setPaymentEndDate(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.billingCycle] as? String)?.let {
            (getBillingCycle(it))?.let { billingCycle ->
                siParamBuilder.setBillingCycle(billingCycle)
            }
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.remarks] as? String)?.let {
            siParamBuilder.setRemarks(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.billingCurrency] as? String)?.let {
            siParamBuilder.setBillingCurrency(
                it
            )
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.billingLimit] as? String)?.let {
            siParamBuilder.setBillingLimit(PayuBillingLimit.valueOf(it))
        }

        (map[PayUHybridKeys.PaymentParam.SIParams.billingRule] as? String)?.let {
            siParamBuilder.setBillingRule(PayuBillingRule.valueOf(it))
        }

        if (map.containsKey(PayUHybridKeys.PaymentParam.SIParams.isPreAuthTxn))
            siParamBuilder.setPreAuthTxn(map[PayUHybridKeys.PaymentParam.SIParams.isPreAuthTxn] as Boolean)
        return siParamBuilder.build()
    }

    /**
     * Method to map/transform billingCycle string to PayUBillingCycle enum
     */
    private fun getBillingCycle(billingCycle: String): PayUBillingCycle? {
        when (billingCycle.lowercase()) {
            PayUHybridKeys.Others.daily -> return PayUBillingCycle.DAILY
            PayUHybridKeys.Others.weekly -> return PayUBillingCycle.WEEKLY
            PayUHybridKeys.Others.once -> return PayUBillingCycle.ONCE
            PayUHybridKeys.Others.yearly -> return PayUBillingCycle.YEARLY
            PayUHybridKeys.Others.adhoc -> return PayUBillingCycle.ADHOC
            PayUHybridKeys.Others.monthly -> return PayUBillingCycle.MONTHLY
        }
        return null
    }

    /**
     * Method to map/transform skuDetails
     */
    internal fun getSkuDetails(skus: List<HashMap<String, Any>>): SkuDetails {
        val skuList = ArrayList<SKU>()
        for (skuMap in skus.toList()){
            val quantity = skuMap[PayUHybridKeys.PaymentParam.quantity] as Int
            val skuAmount = skuMap[PayUHybridKeys.PaymentParam.skuAmount] as String
            val skuId = skuMap[PayUHybridKeys.PaymentParam.skuId] as String
            val skuName = skuMap[PayUHybridKeys.PaymentParam.skuName] as String
            val offerKeys = skuMap[PayUHybridKeys.PaymentParam.offerKeys] as? ArrayList<String>?
            val sku = SKU(quantity, skuAmount, skuId, skuName, offerKeys)
            skuList.add(sku)
        }
        return SkuDetails(skuList)
    }

    internal fun getPayuBeneficiaryDetails(beneficiaryMapList: List<HashMap<String, Any>>) : List<PayUBeneficiaryDetail>{
        val payUBeneficiaryDetailList = ArrayList<PayUBeneficiaryDetail>()
        for (beneficiaryMap in beneficiaryMapList) {
            val payUBeneficiaryAccount =
                beneficiaryMap[PayUHybridKeys.PaymentParam.payuBeneficiaryAccount] as? String
            val payuBeneficiaryIfsc =
                beneficiaryMap[PayUHybridKeys.PaymentParam.payuBeneficiaryIfsc] as? String
            val name =
                beneficiaryMap[PayUHybridKeys.PaymentParam.payuBeneficiaryName] as? String
            val accountType =
                beneficiaryMap[PayUHybridKeys.PaymentParam.payuBeneficiaryAccountType] as? String
            val verificationMode =
                beneficiaryMap[PayUHybridKeys.PaymentParam.payuBeneficiaryVerificationMode] as? String
            val payUBeneficiaryDetail = PayUBeneficiaryDetail.Builder()
                .setBeneficiaryAccountNumber(payUBeneficiaryAccount ?: "")
                .setBeneficiaryIfsc(payuBeneficiaryIfsc ?: "")

            name?.let { payUBeneficiaryDetail.setBeneficiaryName(it) }
            accountType?.let { type ->
                PayUBeneficiaryAccountType.typeFrom(type)?.let {
                    payUBeneficiaryDetail.setBeneficiaryAccountType(it)
                }
            }
            verificationMode?.let { mode ->
                PayUBeneficiaryVerificationMode.typeFrom(mode)?.let {
                    payUBeneficiaryDetail.setVerificationMode(it)
                }
            }
            payUBeneficiaryDetailList.add(payUBeneficiaryDetail.build())
        }
        return payUBeneficiaryDetailList
    }

    internal fun getWealthTechDetails(wealthTechProductMapList: List<HashMap<String, Any>>): List<PayUWealthProducts> {
        val wealthTechProductList = ArrayList<PayUWealthProducts>()
        for (wealthMap in wealthTechProductMapList) {
            val type = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_TYPE] as? String) ?: ""
            val amount = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_AMOUNT] as? String) ?: ""
            val receipt = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_RECEIPT] as? String) ?: ""
            val mfMemberId =
                (wealthMap[PayUHybridKeys.PaymentParam.PAYU_MF_MEMBER_ID] as? String) ?: ""
            val mfUserId = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_MF_USER_ID] as? String) ?: ""
            val plan = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_PLAN] as? String) ?: ""
            val folio = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_FOLIO] as? String) ?: ""
            val mfAmcCode =
                (wealthMap[PayUHybridKeys.PaymentParam.PAYU_MF_AMC_CODE] as? String) ?: ""
            val mfInvestmentType =
                (wealthMap[PayUHybridKeys.PaymentParam.PAYU_MF_INVESTMENT_TYPE] as? String) ?: ""
            val mfPartner =
                (wealthMap[PayUHybridKeys.PaymentParam.PAYU_MF_PARTNER] as? String) ?: ""
            val option = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_OPTION] as? String) ?: ""
            val scheme = (wealthMap[PayUHybridKeys.PaymentParam.PAYU_SCHEME] as? String) ?: ""
            val payUWealthProducts =
                PayUWealthProducts.Builder(
                    type,
                    amount,
                    receipt,
                    mfMemberId,
                    mfUserId,
                    mfPartner,
                    mfInvestmentType
                )
                    .setPlan(plan)
                    .setFolio(folio)
                    .setMfAmcCode(mfAmcCode)
                    .setOption(option)
                    .setScheme(scheme).build()
            wealthTechProductList.add(payUWealthProducts)
        }
        return wealthTechProductList
    }

}