package com.payu.checkoutpro.models

import android.app.Activity
import android.util.Log
import com.payu.base.listeners.VerifyServiceListener
import com.payu.base.models.*
import com.payu.checkoutpro.R
import com.payu.checkoutpro.layers.PayUbizApiLayer
import com.payu.checkoutpro.utils.CommonUtils
import com.payu.checkoutpro.utils.ParserUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY_ERROR_MESSAGE
import com.payu.india.Interfaces.CheckoutDetailsListener
import com.payu.india.Model.MerchantWebService
import com.payu.india.Payu.PayuConstants
import com.payu.india.Payu.PayuErrors
import com.payu.india.PostParams.MerchantWebServicePostParams
import com.payu.india.Tasks.GetCheckoutDetailsTask
import com.payu.olamoney.OlaMoney
import com.payu.olamoney.callbacks.OlaMoneyCallback
import com.payu.olamoney.utils.PayUOlaMoneyParams
import com.payu.upisdk.Upi
import com.payu.upisdk.callbacks.PayUUPICallback
import org.json.JSONObject

class WebServiceApiObject(
    val paymentOption: PaymentOption,
    val payUbizApiLayer: PayUbizApiLayer,
    val var1: String?,
    _listener: Any?
) : V1BaseApiObject(payUbizApiLayer.payuBizparams) {

    private val TAG = PayUCheckoutProConstants.WEBSERVICEAPIOBJECT

    internal var context: Activity? = null
    private var hashName: String? = null
    private var emiTenureList: ArrayList<PaymentOption>? = null
    private var bnplList: ArrayList<PaymentOption>? = null
    private var verifyServiceListener: VerifyServiceListener? = null

    init {
        Log.d(TAG, "PaymentType =" + this.paymentOption.paymentType)

        this.verifyServiceListener = _listener as VerifyServiceListener
        when (this.paymentOption.paymentType) {
            PaymentType.WALLET ->
                hashName = PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY
            PaymentType.UPI ->
                hashName = PayUCheckoutProConstants.CP_VPA_VALIDATION
            PaymentType.EMI, PaymentType.BNPL ->
                hashName = PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS
        }
    }

    private var payuUpiCallback = object : PayUUPICallback() {
        override fun onCommandResponse(payuCommandResponse: String?, command: String?) {
            when (command) {
                PayUCheckoutProConstants.CP_VPA_VALIDATION -> {
                    val apiResponse = ParserUtils.parseVpaValidation(payuCommandResponse!!)
                    verifyServiceListener?.eligibilityDetails(apiResponse = apiResponse)
                }
            }
        }
    }

    override fun callApi(map: HashMap<String, String?>) {
        val hash = map[getHashName()]
        when (paymentOption.paymentType) {
            PaymentType.WALLET -> {
                hashName = PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY
                val walletOption = paymentOption as WalletOption
                OlaMoney().checkForPaymentAvailability(
                    context,
                    object : OlaMoneyCallback {
                        override fun onPaymentInitialisationSuccess() {
                            val apiResponse = ApiResponse()
                            apiResponse.status = true
                            apiResponse.successMessage = ""
                            verifyServiceListener?.eligibilityDetails(apiResponse = apiResponse)
                        }

                        override fun onPaymentInitialisationFailure(p0: Int, p1: String?) {
                            val apiResponse = ApiResponse()
                            apiResponse.status = false
                            apiResponse.errorMessage = CP_OLA_MONEY_ELIGIBILITY_ERROR_MESSAGE
                            verifyServiceListener?.eligibilityDetails(apiResponse = apiResponse)
                        }
                    },
                    generateOlaMoneyModel(hash, walletOption)
                )
            }
            PaymentType.UPI -> {
                Upi.getInstance()
                    .getCommandResponse(
                        context,
                        generateUpiPostData(hash),
                        ParserUtils.isProdEnvironment,
                        payuUpiCallback
                    )
            }

            PaymentType.EMI -> {
                emiTenureList = CommonUtils.getEmiTenureListForBank(paymentOption as EMIOption)
                callEligibilityApi(hash)
            }
            PaymentType.BNPL -> {
                bnplList = ParserUtils.bnplList!!
                callEligibilityApi(hash)
            }
        }
    }

    private fun callEligibilityApi(hash: String?) {
        val merchantWebService = MerchantWebService()
        merchantWebService.key = payuBizparams.key
        merchantWebService.command = PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS
        merchantWebService.var1 = var1
        merchantWebService.hash = hash

        val postData =
            MerchantWebServicePostParams(merchantWebService).merchantWebServicePostParams

        if (postData.code == PayuErrors.NO_ERROR) {
            payuConfig.data = postData.result

            val getCheckoutDetailsTask =
                GetCheckoutDetailsTask(checkoutDetailsListener)
            getCheckoutDetailsTask.execute(payuConfig)
        } else sendEligibilityErrorResponse(payUbizApiLayer.context.getString(R.string.payu_emi_not_eligible_error))
    }

    private fun sendEligibilityErrorResponse(error: String) {
        verifyServiceListener?.eligibilityDetails(ApiResponse().apply {
            status = false
            errorMessage = error
        })
    }

    private val checkoutDetailsListener =
        CheckoutDetailsListener { payuResponse ->
            if (payuResponse?.responseStatus?.status.equals(
                    PayuConstants.SUCCESS,
                    ignoreCase = true
                )
            ) {
                val paymentOptionsList =
                    if (paymentOption.paymentType?.equals(PaymentType.BNPL) == true)
                        ParserUtils.getPaymentOptionFilteredList(payuResponse, bnplList)
                    else ParserUtils.getPaymentOptionFilteredList(payuResponse, emiTenureList)
                if (!paymentOptionsList.isNullOrEmpty())
                    verifyServiceListener?.eligibilityDetails(ApiResponse().apply {
                        paymentOptionList = paymentOptionsList
                    })
                else sendEligibilityErrorResponse(payUbizApiLayer.context.getString(R.string.payu_emi_not_eligible_error))
            } else sendEligibilityErrorResponse(payUbizApiLayer.context.getString(R.string.payu_emi_not_eligible_error))
        }

    /**
     * @param hash: Hash Generated for OlaMoney ELigibility check.
     * Using this hash and other values, generating  [PayUOlaMoneyParams] object for eligibility check.
     */
    private fun generateOlaMoneyModel(
        hash: String?,
        walletOption: WalletOption
    ): PayUOlaMoneyParams {
        val payUOlaMoneyParams = PayUOlaMoneyParams()
        payUOlaMoneyParams.mobile = walletOption.phoneNumber
        payUOlaMoneyParams.firstName = payuBizparams.firstName
        payUOlaMoneyParams.txnId = payuBizparams.txnId
        payUOlaMoneyParams.merchantKey = payuBizparams.key
        payUOlaMoneyParams.hash = hash
        payUOlaMoneyParams.amount = payuBizparams.amount
        return payUOlaMoneyParams
    }

    /**
     *@param hash, Hash generated for UPI VPA validation,
     * Generating Postdata to verify the VPA.
     */
    private fun generateUpiPostData(hash: String?): String {
        val upiOption = paymentOption as UPIOption
        return "key=" + payuBizparams.key + "&var1=" + upiOption.vpa + "&var2=" + JSONObject(
            mapOf(
                PayUCheckoutProConstants.CP_VALIDATE_AUTO_PAY_VPA to "1"
            )
        ) +
                "&command=validateVPA&" + "hash=" + hash
    }

    override fun getHashName(): String {
        return hashName!!
    }


}