package com.payu.checkoutpro.layers

import android.app.Activity
import android.os.Handler
import android.os.Looper
import android.util.Log
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.graphics.drawable.toBitmap
import androidx.fragment.app.Fragment
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.payu.assetprovider.AssetManager
import com.payu.assetprovider.BitmapCallback
import com.payu.assetprovider.model.ImageDetails
import com.payu.base.listeners.BaseApiListener
import com.payu.base.listeners.BaseTransactionListener
import com.payu.base.listeners.EmiCalculationListener
import com.payu.base.listeners.OnDeleteSavedOptionListener
import com.payu.base.listeners.OnEmiDetailsListener
import com.payu.base.listeners.OnFetchGaidListener
import com.payu.base.listeners.OnFetchImageListener
import com.payu.base.listeners.OnGVQuickPayListener
import com.payu.base.listeners.OnIFSCDetailsListener
import com.payu.base.listeners.OnLookupApiListener
import com.payu.base.listeners.OnValidateOfferListener
import com.payu.base.listeners.PaymentVerificationListener
import com.payu.base.listeners.VerifyServiceListener
import com.payu.base.models.*
import com.payu.base.models.calculateEmi.CalculateEmiRequest
import com.payu.checkoutpro.handlers.NavigationHandler
import com.payu.checkoutpro.listeners.NavigationHandlerListener
import com.payu.checkoutpro.models.BaseApiObject
import com.payu.checkoutpro.models.BinBaseDetailsApiObject
import com.payu.checkoutpro.models.CalculateEmiApiObject
import com.payu.checkoutpro.models.CheckBalanceApiObject
import com.payu.checkoutpro.models.ConfigApiObject
import com.payu.checkoutpro.models.DeleteSavedOptionsApiObject
import com.payu.checkoutpro.models.DeviceIdApiObject
import com.payu.checkoutpro.models.EligibleBinsForEMIApiObject
import com.payu.checkoutpro.models.EnforcementManager
import com.payu.checkoutpro.models.FetchConvFeeApiObject
import com.payu.checkoutpro.models.FetchIFSCDetailsApiObject
import com.payu.checkoutpro.models.FetchOfferDetailsApiObject
import com.payu.checkoutpro.models.GetCheckoutDetailsApiObject
import com.payu.checkoutpro.models.GlobalVaultResendOTPObject
import com.payu.checkoutpro.models.GlobalVaultSendOTPObject
import com.payu.checkoutpro.models.GlobalVaultVerifyOTPObject
import com.payu.checkoutpro.models.LookupApiObject
import com.payu.checkoutpro.models.MakePaymentApiObject
import com.payu.checkoutpro.models.PayUCheckoutProConfig
import com.payu.checkoutpro.models.PaymentRenderer
import com.payu.checkoutpro.models.QuickPayApiObject
import com.payu.checkoutpro.models.SendOlwOtpApiObject
import com.payu.checkoutpro.models.SetMPinApiObject
import com.payu.checkoutpro.models.V1AuthlessBaseApiObject
import com.payu.checkoutpro.models.V1BaseApiObject
import com.payu.checkoutpro.models.V2BaseApiObject
import com.payu.checkoutpro.models.ValidateOfferDetailsApiObject
import com.payu.checkoutpro.models.VerifyMPinApiObject
import com.payu.checkoutpro.models.VerifyPaymentApiObject
import com.payu.checkoutpro.models.WebServiceApiObject
import com.payu.checkoutpro.utils.AndroidUtils
import com.payu.checkoutpro.utils.ApiResponseRepo
import com.payu.checkoutpro.utils.CommonUtils
import com.payu.checkoutpro.utils.DataConstant
import com.payu.checkoutpro.utils.ParserUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_BANK_CODE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_BENEFICIARY_DETAILS
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_ERROR_INVALID_BENEFICIARY_DETAILS_MESSAGE
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_IS_L1_OPTION
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_NULL
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_UPI_APP_NAME
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.SODEXO
import com.payu.checkoutpro.utils.PaymentOptionStateManager
import com.payu.custombrowser.CBFragment
import com.payu.india.Model.CheckoutFilter
import com.payu.india.Model.CustomerDetails
import com.payu.india.Model.GetCheckoutDetailsRequest
import com.payu.india.Model.GetTransactionDetails
import com.payu.india.Model.PayuResponse
import com.payu.india.Model.Usecase
import com.payu.india.Payu.Payu
import com.payu.otpassist.OtpAssistFragment
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.listeners.PayUCheckoutProListener
import com.payu.ui.model.listeners.PayUHashGenerationListener
import com.payu.ui.model.managers.OfferFilterManager
import com.payu.ui.model.utils.Utils
import com.payu.upisdk.upiintent.PaymentResponseUpiSdkFragment
import org.json.JSONObject
import java.util.Locale
import java.util.concurrent.Executors

class PayUbizApiLayer(
    var context: Activity,
    payUPaymentParams: PayUPaymentParams,
    var payUCheckoutProConfig: PayUCheckoutProConfig,
    var checkoutListener: PayUCheckoutProListener
) : BaseApiLayer(payUPaymentParams, payUCheckoutProConfig) {

    private val TAG = this.javaClass.simpleName
    internal val payuBizparams: com.payu.paymentparamhelper.PaymentParams
    internal var baseTransactionListener: BaseTransactionListener? = null
    private var eligibleBinsForEMIApiObject: EligibleBinsForEMIApiObject? = null
    internal var fetchOfferDetailsApiObject: FetchOfferDetailsApiObject? = null
    internal var isEligibleBinsApiResponseReceived: Boolean = false
    internal var isEmiDetailsApiInProgress: Boolean = false

    //below flags are used for MCP
    internal var isLookupApiInProgress: Boolean = false
    internal var isMakePaymentCalled: Boolean = false
    internal var mcpPaymentModel: PaymentModel? = null
    internal var mcpToolbar: PayuToolbar? = null
    internal var checkoutAPICompleted: Boolean = false
    internal var configAPICompleted: Boolean = false
    private var baseApiObject: BaseApiObject? = null

    private var quickPayApiObject: QuickPayApiObject? = null

    constructor(
        context: Activity,
        payUPaymentParams: PayUPaymentParams,
        checkoutListener: PayUCheckoutProListener
    ) : this(
        context,
        payUPaymentParams,
        PayUCheckoutProConfig(),
        checkoutListener
    )

    init {
        Payu.setInstance(context) //Initialize PayUbiz SDK
        // SI Txn only if it is non OTM. because we pass "payUPaymentParams.payUSIParams" for OTM also.
        ParserUtils.isSIMode = payUPaymentParams.payUSIParams != null
        if (!payUCheckoutProConfig.enforcePaymentList.isNullOrEmpty()) InternalConfig.isEnforced =
            true
        ParserUtils.enforcePaymentList = payUCheckoutProConfig.enforcePaymentList
        this.payuBizparams = CommonUtils.preparePayUbizPaymentParams(
            payUPaymentParams,
            context,
            payUCheckoutProConfig.isQrScan
        )
    }

    /**
     * Function to attach listener of SDKUI with Biz layer for routing biz layer events to SDKUI
     * @param baseTransactionListener handler to interact with SDKUI for transaction success/ failure
     * */

    override fun connectListener(
        context: Activity,
        baseTransactionListener: BaseTransactionListener
    ) {
        this.baseTransactionListener = baseTransactionListener
        this.context = context
    }

    private fun fetchMoreOptions(
        baseApiListener: BaseApiListener,
        listener: (ArrayList<PaymentMode>?) -> Unit?
    ) {
        val usecase =
            Usecase.Builder().setCheckCustomerEligibility(true).shouldCheckDownStatus(true)
                .shouldGetAdditionalCharges(true).shouldGetExtendedPaymentDetails(true)
                .shouldGetTaxSpecification(true)
                .shouldGetMerchantDetails(true).shouldGetPaymentDetailsWithExtraFields(true)
                .shouldGetSdkDetails(true).build()
        val transactionDetails =
            GetTransactionDetails.Builder().setAmount(payuBizparams.amount.toDouble())
                .setAdditionalCharges(payuBizparams.additionalCharges)
                .setPercentageAdditionalCharges(payuBizparams.percentageAdditionalCharges)
                .setPreAuthorize(payUPaymentParams.payUSIParams?.isPreAuthTxn ?: false)
                .build()
        val customerDetails = CustomerDetails.Builder().setMobile(payUPaymentParams.phone)
        val ifscCodes = CommonUtils.getIfscCodesList(payUPaymentParams)
        if (!payUPaymentParams.beneficiaryDetailList.isNullOrEmpty() && ifscCodes.isNullOrEmpty()) {
            baseApiListener.showProgressDialog(false)
            val errorResponse = ErrorResponse()
            errorResponse.errorCode = CP_ERROR_INVALID_BENEFICIARY_DETAILS
            errorResponse.errorMessage = CP_ERROR_INVALID_BENEFICIARY_DETAILS_MESSAGE
            baseApiListener.onError(errorResponse)
            return
        } else customerDetails.setIfscCodes(
            ifscCodes
        )
        val var1 = GetCheckoutDetailsRequest.Builder()
            .setUsecase(usecase)
            .setCustomerDetails(customerDetails.build())
            .setTransactionDetails(transactionDetails)
            .setRequestId(payuBizparams.txnId + System.currentTimeMillis())
            .build().prepareJSON()
        payUPaymentParams.additionalParams?.put(
            PayUCheckoutProConstants.CP_GET_CHECKOUT_DETAILS_VAR1,
            var1
        )
        val getCheckoutDetailsApiObject =
            GetCheckoutDetailsApiObject(
                this,
                payuBizparams,
                var1,
                baseApiListener,
                listener
            )

        initApiCall(getCheckoutDetailsApiObject)
    }

    override fun reset() {
        isEligibleBinsApiResponseReceived = false
        isEmiDetailsApiInProgress = false
        ParserUtils.isNBOfferFetched = false
        ParserUtils.quickOptionsList = null
        ParserUtils.moreOptionsList = null
        ParserUtils.convenienceFeeResponse = null
        ParserUtils.sodexoSourceId = null
        ParserUtils.isRetryTxn = false
        ParserUtils.retryCount = -1
        ParserUtils.emiList = null
        ParserUtils.paymentOptionMap = HashMap<PaymentType, HashMap<String, Any>>()
        ParserUtils.enforcePaymentList = null
        ParserUtils.payuBizParams = null
        ParserUtils.issuingBankDownList = null
        ParserUtils.isProdEnvironment = true
        ParserUtils.isSIMode = false
        ParserUtils.bnplList = null
        InternalConfig.selectedOfferInfo = null
        InternalConfig.emiOfferInfo = null
        InternalConfig.noCostEmi = null
        InternalConfig.interestCharged = null
        fetchOfferDetailsApiObject = null
        InternalConfig.isOfferEnabled = false
        InternalConfig.userSelectedOfferInfo = null
        InternalConfig.outletId = null
        InternalConfig.offerBankListEmi = null
        InternalConfig.isPaymentOptionSelected = false
        resetMcpFlags()
        ParserUtils.resetApiRepoResponse()
        InternalConfig.isEnforced = false
        InternalConfig.isRetryTxn = false
        baseTransactionListener = null
        InternalConfig.couponsAvailable = false
        InternalConfig.appliedCouponOfferKey = null
        InternalConfig.isUserPersonalizedOffersAvailable = false
        InternalConfig.isUserConsentAvailableForPersonalisedOffers = false
        OfferFilterManager.clearCouponCache()
    }

    override fun deleteSavedOption(
        paymentOption: PaymentOption,
        onDeleteSavedOptionListener: OnDeleteSavedOptionListener
    ) {
        getDeviceId {
            val deleteSavedOptionsApiObject =
                DeleteSavedOptionsApiObject(
                    payuBizparams,
                    payUPaymentParams,
                    it ?: "",
                    paymentOption,
                    onDeleteSavedOptionListener
                )
            initApiCall(deleteSavedOptionsApiObject)
        }
    }


    /**
     * This API provides with a list of currency conversion enabled on merchant access key
     * */
    override fun callLookupApi(cardOption: CardOption, onLookupApiListener: OnLookupApiListener) {
        var additionalParams: HashMap<String, Any?>? = payUPaymentParams.additionalParams
        if (additionalParams.isNullOrEmpty()) additionalParams = HashMap()

        val baseCurrency = PayUCheckoutProConstants.CP_BASE_CURRENCY_VALUE  //INR is supported
        val merchantAccessKey = CommonUtils.getParamValue(
            additionalParams,
            PayUCheckoutProConstants.CP_MERCHANT_ACCESS_KEY
        )
        resetMcpFlags()

        //call Lookup API only when merchantAccessKey and baseCurrency is not empty
        if (baseCurrency.isNotEmpty() && merchantAccessKey.isNotEmpty()) {
            var transactionAmount =
                SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
            transactionAmount = transactionAmount?.plus(cardOption.cardBinInfo?.additionalCharge!!)
            val merchantOrderId = System.currentTimeMillis().toString()
            payUPaymentParams.additionalParams?.put(
                PayUCheckoutProConstants.CP_LOOKUP_API_POST_DATA,
                "$baseCurrency$merchantOrderId$transactionAmount"
            )

            val lookupApiObject = LookupApiObject(
                payuBizparams,
                baseTransactionListener,
                this,
                merchantOrderId,
                cardOption,
                merchantAccessKey,
                baseCurrency,
                onLookupApiListener
            )
//            initApiCall(lookupApiObject)
        } else {
            onLookupApiListener.showProgressDialog(false)
            onLookupApiListener.onLookupApiCalled()
        }
    }

    override fun fetchIFSCDetails(ifscCode: String, onIFSCDetailsListener: OnIFSCDetailsListener) {
        initApiCall(
            FetchIFSCDetailsApiObject(payuBizparams, payUPaymentParams, ifscCode, onIFSCDetailsListener)
        )
    }

    /**
     * Function to fetch all the payment options available for merchant
     * @param baseApiListener handler to pass parsed response to the SDKUI
     * */
    override fun fetchPaymentOptions(
        baseApiListener: BaseApiListener,
        listener: (ArrayList<PaymentMode>?) -> Unit
    ) {
        if (ApiResponseRepo.moreOptionsResponse?.response.isNullOrEmpty()) {
            fetchMoreOptionsApi(baseApiListener, listener)
        } else {
            ApiResponseRepo.getMoreOptionsResponse(listener)
        }
    }

    private fun fetchMoreOptionsApi(
        baseApiListener: BaseApiListener,
        listener: (ArrayList<PaymentMode>?) -> Unit
    ) {
        if (ApiResponseRepo.moreOptionsResponse?.inProgress != true) {
            fetchMoreOptions(baseApiListener, listener)
        }
        val payUApiResponse = PayUApiResponse(listener, true, arrayListOf())
        ApiResponseRepo.moreOptionsResponse = payUApiResponse
    }

    /**
     * Function to get payment State via Biz SDK
     * @param paymentModel a bean containing the pg, bankcode and other crucial payment instrument specific details
     *
     * @return  [PaymentFlowState]
     * */
    override fun getPaymentState(paymentModel: PaymentModel): PaymentFlowState? {
        return PaymentOptionStateManager.getNextState(paymentModel)
    }

    override fun onBackPressed(fragment: Fragment) {
        if (CommonUtils.isSdkAvailable(PayUCheckoutProConstants.CP_UPI_PACKAGE_NAME) && fragment is PaymentResponseUpiSdkFragment) {
            fragment.onBackButtonClicked()
        } else if (CommonUtils.isSdkAvailable(PayUCheckoutProConstants.CP_OTP_ASSIST_PACKAGE_NAME) && fragment is OtpAssistFragment) {
            fragment.onBackButtonClicked()
        } else if (CommonUtils.isSdkAvailable(PayUCheckoutProConstants.CP_CBPACKAGE_NAME) && fragment is CBFragment) {
            fragment.onBackButtonClicked()
        }
    }


    /**
     * Function to make payment via Biz SDK
     * @param paymentModel a bean containing the pg, bankcode and other crucial payment instrument specific details
     *
     * On Payment completion, the response will be passed via baseTransactionListener
     * */
    override fun updatePaymentState(
        paymentModel: PaymentModel,
        toolbar: PayuToolbar?
    ) {
        isMakePaymentCalled = true
        if (!isLookupApiInProgress) {
            val nextState = getPaymentState(paymentModel)
            if (nextState?.paymentState?.name == null || DataConstant.supportedRedirectionPaymentState()
                    .contains(nextState.paymentState?.name)
            ) {
                makePayment(paymentModel, toolbar)
            } else {
                paymentModel.paymentFlowState = nextState
                baseTransactionListener?.loadNextState(paymentModel)
            }
        } else {
            baseTransactionListener?.showProgressDialog(true)
            mcpPaymentModel = paymentModel
            mcpToolbar = toolbar
        }
    }

    fun resetMcpFlags() {
        ParserUtils.lookupDetails = null
        isLookupApiInProgress = false
        isMakePaymentCalled = false
        mcpPaymentModel = null
        mcpToolbar = null
    }
    override fun getBalanceFromSodexo(
        sodexoBalanceListener: (QuickOptionsModel?) -> Unit?
    ) {
        val additionalParams = payUPaymentParams.additionalParams

        if (ParserUtils.convenienceFeeResponse?.isSodexoAvailable == true && additionalParams?.containsKey(
                PayUCheckoutProConstants.SODEXO_SOURCE_ID
            ) == true
        ) {
            val sodexoSourceIdJson = JSONObject()
            sodexoSourceIdJson.put(
                "sodexoSourceId",
                payUPaymentParams.additionalParams?.get(PayUCheckoutProConstants.SODEXO_SOURCE_ID)
                    .toString()
            )
            val var1 = sodexoSourceIdJson.toString()
            payUPaymentParams.additionalParams?.put(
                PayUCheckoutProConstants.CP_CHECK_BALANCE_VAR1,
                var1
            )

            val response: (PayuResponse?) -> Unit = {
                if (!ParserUtils.quickOptionsList.isNullOrEmpty()) {
                    for ((index, item) in ParserUtils.quickOptionsList!!.withIndex()) {
                        if (item.type == PaymentType.SODEXO) {
                            ParserUtils.quickOptionsList?.set(index, ParserUtils.getSodexoMode(it))
                            val quickOptionsModel = QuickOptionsModel()
                            quickOptionsModel.quickOptionsList = ParserUtils.quickOptionsList
                            val payUApiResponse =
                                PayUApiResponse(sodexoBalanceListener, false, quickOptionsModel)
                            ApiResponseRepo.setGvResponseForClwAndSodexo(payUApiResponse)
                            break
                        }
                    }
                }
            }

            val checkBalanceApiObject =
                CheckBalanceApiObject(
                    payuBizparams,
                    response,
                    var1
                )
            initApiCall(checkBalanceApiObject)
        }
    }

    override fun getBalanceForClosedLoopWallet(
        walletIdentifier: String?,
        mobile: String,
        onGVQuickPayListener: (QuickOptionsModel?) -> Unit?
    ) {
        if (null != walletIdentifier && "null" != walletIdentifier && ParserUtils.convenienceFeeResponse?.isClosedLoopWalletPayAvailable == true) {
            val ibiboCode =
                ParserUtils.convenienceFeeResponse?.closedLoopWallet?.get(0)?.bankCode ?: ""
            val clwJsonForVar1 = JSONObject()
            clwJsonForVar1.put(PayUCheckoutProConstants.WALLET_IDENTIFIER, walletIdentifier)
            clwJsonForVar1.put(PayUCheckoutProConstants.MOBILE, mobile)
            clwJsonForVar1.put(PayUCheckoutProConstants.IBIBO_CODE, ibiboCode)
            val var1 = clwJsonForVar1.toString()

            payUPaymentParams.additionalParams?.put(
                PayUCheckoutProConstants.CP_CHECK_BALANCE_VAR1,
                var1
            )

            val response: (PayuResponse?) -> Unit = {
                if (!ParserUtils.quickOptionsList.isNullOrEmpty()) {
                    for ((index, item) in ParserUtils.quickOptionsList!!.withIndex()) {
                        if (item.type == PaymentType.CLOSED_LOOP_WALLET) {
                            ParserUtils.quickOptionsList!![index] =
                                ParserUtils.updateClosedLoopWallet(item, it)

                            val quickOptionsModel = QuickOptionsModel()
                            quickOptionsModel.quickOptionsList = ParserUtils.quickOptionsList

                            val payUApiResponse =
                                PayUApiResponse(onGVQuickPayListener, false, quickOptionsModel)
                            ApiResponseRepo.setGvResponseForClwAndSodexo(payUApiResponse)
                            break
                        }
                    }
                }
            }

            val checkBalanceApiObject =
                CheckBalanceApiObject(
                    payuBizparams,
                    response,
                    var1
                )
            initApiCall(checkBalanceApiObject)
        }
    }

    override fun fetchGaid(
        onFetchGaidListener: OnFetchGaidListener
    ) {

        val response: (GoogleAdIdInfo?) -> Unit = {
            onFetchGaidListener.onFetchGaidResponse(it?.info?.id.toString())
        }

        AndroidUtils.determineAdvertisingInfo(context, response)
    }

    override fun callDeviceInfoApi(payuId: String, gaid: String) {
        DeviceIdApiObject(
            DeviceIdRequestModel(payuId, gaid),
            payUPaymentParams,
            payuBizparams,
            null
        ).callApi()
    }

    override fun calculateEmiApi(
        calculateEmiRequest: CalculateEmiRequest,
        emiCalculationListener: EmiCalculationListener
    ) {
        val calculateEmiApiObject =
            CalculateEmiApiObject(
                payuBizparams,
                payUPaymentParams,
                calculateEmiRequest,
                emiCalculationListener
            )
        initApiCall(calculateEmiApiObject)
    }

    /**
     * Function to fetch offers
     * */
    override fun fetchOfferDetails(resetCache: Boolean?, loggedInPhoneNumber: String?, offerListener: (FetchOfferDetails?) -> Unit?) {
        if (ApiResponseRepo.payuApiFetchOffer?.response == null || resetCache == true) {
            fetchOfferApiCall(loggedInPhoneNumber, offerListener)
        } else {
            ApiResponseRepo.getPayUApiResponse(offerListener)
        }
    }

    override fun verifyPayment(paymentVerificationListener: PaymentVerificationListener) {
        val verificationListener = VerifyPaymentApiObject(
            context,
            payuBizparams,
            paymentVerificationListener
        )
        initApiCall(verificationListener)
    }

    override fun getRetryCount(): Int {
        return ParserUtils.retryCount
    }

    override fun setRetryCount(retryCount: Int) {
        ParserUtils.retryCount = retryCount
    }

    /**
     * Function to handled the events of GlobalVault and Recommendation Engine  APIs
     */
    override fun fetchQuickPay(
        quickPayEnum: GlobalVaultAPIsCommand,
        resultMap: HashMap<String, String>,
        onGVQuickPayListener: OnGVQuickPayListener
    ) {

        Log.d(TAG, "fetchQuickPay function call enum ${quickPayEnum.name}")

        when (quickPayEnum) {
            GlobalVaultAPIsCommand.SEND_OTP -> {
                val globalVaultSendOTPObject = GlobalVaultSendOTPObject(
                    payuBizparams,
                    payUPaymentParams,
                    resultMap,
                    onGVQuickPayListener
                )
                initApiCall(globalVaultSendOTPObject)
            }

            GlobalVaultAPIsCommand.RESEND_OTP -> {
                val globalVaultReSendOTPObject = GlobalVaultResendOTPObject(
                    payuBizparams,
                    payUPaymentParams,
                    resultMap,
                    onGVQuickPayListener
                )
                initApiCall(globalVaultReSendOTPObject)
            }

            GlobalVaultAPIsCommand.VERIFY_OTP -> {
                val globalVaultVerifyOTPObject = GlobalVaultVerifyOTPObject(
                    payuBizparams,
                    payUPaymentParams,
                    resultMap,
                    onGVQuickPayListener
                )
                initApiCall(globalVaultVerifyOTPObject)
            }
        }
    }

    /**
     * Function to validate offer
     * */
    override fun validateOfferDetails(
        category: String,
        cardNumber: String?,
        paymentCode: String?,
        cardToken: String?,
        loggedInPhoneNumber: String?,
        validateOfferListener: OnValidateOfferListener
    ) {
        if (InternalConfig.isOfferEnabled.not() && DataConstant.offerNotSupportedList()
                .contains(category.lowercase()) || !cardToken.isNullOrEmpty() && payUPaymentParams.beneficiaryDetailList.isNullOrEmpty()
                .not()
        ) {
            validateOfferListener.onValidateOfferResponse(
                null
            )
        } else {
            payuBizparams.category = category
            payuBizparams.cardNumber = cardNumber
            payuBizparams.paymentCode = paymentCode
            payuBizparams.cardToken = cardToken
            payuBizparams.loggedInPhoneNumber = loggedInPhoneNumber

            val validateOfferDetailsApiObject = ValidateOfferDetailsApiObject(
                payuBizparams,
                payUPaymentParams,
                validateOfferListener
            )
            initApiCall(validateOfferDetailsApiObject)
        }
    }

    //Function to get/fetch/download images URL where default image is not available
    override fun getBitmapImageFormURL(
        imageURL: String,
        onFetchImageListener: OnFetchImageListener
    ) {
        AssetManager.getInstance(context)
            .getBitmapFromURL(imageURL, object : BitmapCallback {
                override fun onBitmapReceived(types: String, data: ImageDetails) {
                    if (data.image != null && data.drawableType != null)
                        onFetchImageListener.onImageGenerated(
                            ImageDetails(
                                CommonUtils.drawableTypeMapping(
                                    data.drawableType!!
                                ), data.image!!
                            )
                        )
                }
            })
    }


    /**
     * Function to verify eligibility based on PaymentTypes
     */

    override fun verifyEligibilityAPI(
        paymentOption: PaymentOption,
        verifyServiceListener: VerifyServiceListener
    ) {

        var var1: String? = null

        Log.d(TAG, "PaymentType =" + paymentOption.paymentType)

        when (paymentOption.paymentType) {
            PaymentType.WALLET -> {
                val walletOption = paymentOption as WalletOption
                payuBizparams.phone = walletOption.phoneNumber

                payUPaymentParams.additionalParams?.put(
                    PayUCheckoutProConstants.CP_OLA_MONEY_ELIGIBILITY,
                    CommonUtils.generateVasForOlaMoney(payuBizparams)
                )
            }

            PaymentType.UPI -> {
                val upiOption = paymentOption as UPIOption

                if (!CommonUtils.isSupportedVpaForSI(upiOption.vpa) && ParserUtils.isSIMode) {
                    verifyServiceListener.eligibilityDetails(apiResponse = ApiResponse().apply {
                        status = false
                    })
                    return
                } else {
                    payUPaymentParams.additionalParams?.put(
                        PayUCheckoutProConstants.CP_VPA_VALIDATION,
                        upiOption.vpa
                    )
                }
            }

            PaymentType.EMI -> {
                val emiOption = paymentOption as EMIOption
                val subType = CommonUtils.getValueFromPaymentOption<String>(
                    PayUCheckoutProConstants.CP_KEY_SUB_TYPE,
                    emiOption.otherParams as? HashMap<String, Any?>
                )
                val emiCode = CommonUtils.getValueFromPaymentOption<String>(
                    PayUCheckoutProConstants.CP_KEY_EMI_CODE,
                    emiOption.otherParams as? HashMap<String, Any?>
                )
                val usecase = Usecase.Builder()
                    .shouldGetTaxSpecification(true)
                    .setCheckCustomerEligibility(true).build()
                val customerDetails =
                    CustomerDetails.Builder().setMobile(emiOption.phoneNumber).build()
                val checkoutFilter = CheckoutFilter.Builder()
                    .setPaymentOptionName(PayUCheckoutProConstants.CP_EMI)
                    .setPaymentOptionType(subType)
                    .setPaymentOptionValue(emiCode).build()
                val transactionDetails =
                    GetTransactionDetails.Builder().setAmount(payuBizparams.amount.toDouble())
                        .build()
                var1 = GetCheckoutDetailsRequest.Builder()
                    .setUsecase(usecase)
                    .setCustomerDetails(customerDetails)
                    .setRequestId(payuBizparams.txnId+System.currentTimeMillis())
                    .setCheckoutFilter(checkoutFilter)
                    .setTransactionDetails(transactionDetails)
                    .build().prepareJSON()
                payUPaymentParams.additionalParams?.put(
                    PayUCheckoutProConstants.CP_CHECK_ELIGIBILITY_VAR1,
                    var1
                )
            }

            PaymentType.BNPL -> {
                val bnplOption = paymentOption
                val bankCode = CommonUtils.getValueFromPaymentOption<String>(
                    PayUCheckoutProConstants.CP_BANK_CODE,
                    bnplOption.otherParams as? HashMap<String, Any?>
                )
                val usecase = Usecase.Builder()
                    .shouldGetTaxSpecification(true)
                    .setCheckCustomerEligibility(true).build()
                val customerDetails =
                    CustomerDetails.Builder().setMobile(bnplOption.phoneNumber).build()
                val checkoutFilter = CheckoutFilter.Builder()
                    .setPaymentOptionName(PayUCheckoutProConstants.CP_BNPL_CODE)
                    .setPaymentOptionType(bankCode).build()
                val transactionDetails =
                    GetTransactionDetails.Builder().setAmount(payuBizparams.amount.toDouble())
                        .build()
                var1 = GetCheckoutDetailsRequest.Builder()
                    .setUsecase(usecase)
                    .setCustomerDetails(customerDetails)
                    .setCheckoutFilter(checkoutFilter)
                    .setTransactionDetails(transactionDetails)
                    .build().prepareJSON()
                payUPaymentParams.additionalParams?.put(
                    PayUCheckoutProConstants.CP_CHECK_ELIGIBILITY_VAR1,
                    var1
                )
            }
        }

        val userVerifyServiceApiObject =
            WebServiceApiObject(
                paymentOption,
                this,
                var1,
                verifyServiceListener
            )
        userVerifyServiceApiObject.context = context
        CommonUtils.generateHash(
            context,
            payUPaymentParams,
            checkoutListener,
            userVerifyServiceApiObject
        )

    }

    override fun emiDetails(emiDetailsListener: OnEmiDetailsListener?) {
        when {
            isEligibleBinsApiResponseReceived -> ParserUtils.emiList?.let {
                emiDetailsListener?.emiDetailsReceived(
                    it
                )
            }

            isEmiDetailsApiInProgress -> {
                emiDetailsListener?.showProgressDialog(true)
                eligibleBinsForEMIApiObject?.emiDetailsListener = emiDetailsListener
            }

            else -> callEmiApis(emiDetailsListener)
        }
    }

    private fun callEmiApis(emiDetailsListener: OnEmiDetailsListener?) {

        if (!isEligibleBinsApiResponseReceived) {
            eligibleBinsForEMIApiObject = EligibleBinsForEMIApiObject(
                payuBizparams,
                emiDetailsListener,
                this
            )
            initApiCall(eligibleBinsForEMIApiObject!!)
        }
    }

    // get image for paymentOptions from asset lib with image url
    override fun getImageForPaymentOption(
        imageParam: ImageParam,
        onFetchImageListener: OnFetchImageListener
    ) {
        when {
            imageParam.imageKey != null -> {
                if (!imageParam.paymentOption.imageURL.isNullOrEmpty())
                    getBitmapFromURL(imageParam, onFetchImageListener)
                else {
                    setPaymentInstrumentImage(
                        imageParam.imageKey!!,
                        imageParam,
                        onFetchImageListener
                    )
                }
            }
            //saved cards details
            imageParam.isCardScheme -> {
                val bankName = imageParam.paymentOption.bankName
                if (bankName.isNotEmpty() && !bankName.equals(
                        CP_NULL
                    )
                ) {
                    setPaymentInstrumentImage(
                        bankName,
                        imageParam,
                        onFetchImageListener
                    )
                } else {
                    val scheme =
                        (imageParam.paymentOption as? CardOption)?.cardBinInfo?.cardScheme?.name
                    scheme
                        ?.let { setPaymentInstrumentImage(it, imageParam, onFetchImageListener) }
                }
            }

            else -> {
                when (imageParam.paymentOption.paymentType) {
                    PaymentType.UPI_INTENT, PaymentType.UPI -> {
                        if (context == null)
                            return

                        val otherParamsMap =
                            imageParam.paymentOption.otherParams as? HashMap<String, Any?>

                        if (otherParamsMap != null
                            && otherParamsMap.containsKey(CP_IS_L1_OPTION)
                            && otherParamsMap.containsKey(CP_UPI_APP_NAME)
                            && otherParamsMap[CP_UPI_APP_NAME] as? String != null
                            && !imageParam.paymentOption.imageURL.isNullOrEmpty()
                        ) {
                            getBitmapFromURL(imageParam, onFetchImageListener)
                        } else if (imageParam.paymentOption.bankCode != null) {
                            setPaymentInstrumentImage(
                                imageParam.paymentOption.bankCode!!,
                                imageParam,
                                onFetchImageListener
                            )
                        } else {
                            try {

                                val upiOption = imageParam.paymentOption as UPIOption
                                val bitmap = AndroidUtils.getUpiAppBitmap(
                                    context.applicationContext,
                                    upiOption?.packageName ?: ""
                                )

                                if (bitmap != null) onFetchImageListener.onImageGenerated(
                                    ImageDetails(
                                        DrawableType.Bitmap,
                                        bitmap
                                    )
                                )
                            } catch (e: Exception) {
                                if (otherParamsMap?.get(CP_BANK_CODE) != null){
                                    setPaymentInstrumentImage(
                                        otherParamsMap[CP_BANK_CODE].toString(),
                                        imageParam,
                                        onFetchImageListener
                                    )
                                } else {
                                    AppCompatResources.getDrawable(
                                        context,
                                        imageParam.defaultDrawable
                                    )
                                        ?.toBitmap()
                                }
                            }
                        }
                    }

                    PaymentType.EMI -> {
                        val emiOption = imageParam.paymentOption as? EMIOption
                        if (emiOption?.imageURL.isNullOrEmpty())
                            emiOption?.bankShortName
                                ?.let {
                                    setPaymentInstrumentImage(
                                        it,
                                        imageParam,
                                        onFetchImageListener
                                    )
                                }
                        else getBitmapFromURL(imageParam, onFetchImageListener)

                    }

                    PaymentType.SODEXO -> {
                        val sodexoCardOption = imageParam.paymentOption as? SodexoCardOption
                        setPaymentInstrumentImage(
                            sodexoCardOption?.bankName ?: "",
                            imageParam,
                            onFetchImageListener
                        )
                    }

                    else -> {
                        val hashMap = imageParam.paymentOption.otherParams as? HashMap<String, Any?>

                        if (hashMap != null
                            && hashMap.containsKey(PayUCheckoutProConstants.CP_BANK_CODE)
                            && (hashMap[PayUCheckoutProConstants.CP_BANK_CODE] as? String) != null
                        ) {

                            val bankCode: String =
                                hashMap[PayUCheckoutProConstants.CP_BANK_CODE] as? String ?: ""
                            if (!imageParam.paymentOption.imageURL.isNullOrEmpty())
                                getBitmapFromURL(imageParam, onFetchImageListener)
                            else setPaymentInstrumentImage(
                                bankCode,
                                imageParam,
                                onFetchImageListener
                            )
                        }
                    }
                }
            }
        }

    }

    private fun setPaymentInstrumentImage(
        bankName: String,
        imageParam: ImageParam,
        onFetchImageListener: OnFetchImageListener
    ) {
        AssetManager.getInstance(context)
            .get(
                (bankName.lowercase(Locale.getDefault())),
                imageParam.defaultDrawable,
                callback = object :
                    BitmapCallback {
                    override fun onBitmapReceived(types: String, data: ImageDetails) {
                        onFetchImageListener.onImageGenerated(
                            ImageDetails(
                                CommonUtils.drawableTypeMapping(
                                    data.drawableType!!
                                ), data.image!!
                            )
                        )
                    }
                })
    }


    private fun initApiCall(baseApiObject: V1BaseApiObject) {
        this.baseApiObject = baseApiObject
        CommonUtils.generateHash(
            context,
            payUPaymentParams,
            checkoutListener,
            baseApiObject
        )
    }

    private fun initRendering(renderer: PaymentRenderer, paymentOption: PaymentOption?) {
        //check if hash is already available
        CommonUtils.generateHash(
            paymentOption,
            payUPaymentParams,
            checkoutListener,
            renderer,
            object : PayUHashGenerationListener {
                override fun onHashGenerated(map: HashMap<String, String?>) {
                    NavigationHandler(object : NavigationHandlerListener<Fragment?> {
                        override fun onResultReceived(result: Fragment?) {
                            baseTransactionListener?.loadNextState(result)
                        }
                    }).executeTask { renderer.render(map) }
                }

            }
        )
    }

    private fun initApiCall(baseApiObject: V2BaseApiObject) {
        CommonUtils.checkoutProListener = checkoutListener
        Handler(Looper.getMainLooper()).post{
            baseApiObject.callApi()
        }

    }


    private fun initApiCall(baseApiObject: V1AuthlessBaseApiObject) {
        baseApiObject.callApi()
    }


    override fun getEnforcedState(): PaymentState? {
        var enforcementManager: EnforcementManager? = null
        if (!payUCheckoutProConfig.enforcePaymentList.isNullOrEmpty()) {
            enforcementManager = EnforcementManager(payUCheckoutProConfig.enforcePaymentList!!)
        }
        return enforcementManager?.getNextStateFromManager()
    }

    override fun fetchConfig() {
        initApiCall(ConfigApiObject(payuBizparams, this))
    }

    //Function to get/fetch/download images URL where default image is available
    private fun getBitmapFromURL(
        imageParam: ImageParam,
        onFetchImageListener: OnFetchImageListener
    ) {
        val executor = Executors.newFixedThreadPool(5)
        val worker = Runnable {
            imageParam.paymentOption.imageURL?.let {
                AssetManager.getInstance(context)
                    .getBitmapFromURL(
                        it,
                        imageParam.paymentOption.imageUpdatedOn ?: 0,
                        imageParam.defaultDrawable,
                        callback = object :
                            BitmapCallback {
                            override fun onBitmapReceived(
                                types: String,
                                data: ImageDetails
                            ) {
                                Handler(Looper.getMainLooper()).post {
                                    onFetchImageListener.onImageGenerated(
                                        ImageDetails(
                                            CommonUtils.drawableTypeMapping(
                                                data.drawableType!!
                                            ), data.image!!
                                        )
                                    )
                                }
                            }
                        })
            }
        }
        executor.execute(worker)
    }


    override fun cancelQuickPayTask() {
        quickPayApiObject?.cancelQuickPayTask()
    }

    override fun fetchGvQuickPay(
        resultMap: HashMap<String, String>,
        isNeedToApiCall: Boolean,
        onGvQuickPayListener: (QuickOptionsModel?) -> Unit?
    ) {
        when {
            (payUPaymentParams.beneficiaryDetailList.isNullOrEmpty()
                .not()) -> {
                onGvQuickPayListener.invoke(null)
            }

            (ApiResponseRepo.gvOptionsResponse == null || isNeedToApiCall) -> {
                fetchGvInitApiCall(resultMap, onGvQuickPayListener)

            }

            else -> {
                ApiResponseRepo.getGvResponse(onGvQuickPayListener)
            }
        }
    }

    private fun sendHashError(hashName: String?) {
        val errorResponse = ErrorResponse()
        errorResponse.errorMessage =
            "$hashName ${PayUCheckoutProConstants.CP_HASH_NOT_NULL}"
        checkoutListener.onError(errorResponse)
    }


    private fun getDeviceId(response: (String?) -> Unit) {
        Thread {
            try {
                val deviceIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(context)
                if (deviceIdInfo.isLimitAdTrackingEnabled) return@Thread response(null)
                else return@Thread response(deviceIdInfo.id.toString())
            } catch (e: Exception) {
                e.message?.let { Log.d(TAG, it) }
                return@Thread response(
                    Utils.getAndroidId(context)
                )
            } catch (t: Throwable) {
                t.message?.let { Log.d(TAG, it) }
                return@Thread response(
                    Utils.getAndroidId(context)
                )
            }
        }.start()
    }

    private fun makePayment(paymentModel: PaymentModel, toolbar: PayuToolbar?) {
        val makePaymentApiObject =
            MakePaymentApiObject(
                payuBizparams,
                baseTransactionListener,
                this,
                checkoutListener
            )
        makePaymentApiObject.context = context
        makePaymentApiObject.paymentType = paymentModel.paymentOption?.paymentType
        makePaymentApiObject.paymentOption = paymentModel.paymentOption

        //should show toolbar
        toolbar?.let {
            if (payUCheckoutProConfig.showCbToolbar)
                makePaymentApiObject.toolbar = it
        }


        if (payUPaymentParams.payUSIParams != null &&
                (paymentModel.paymentOption?.paymentType == PaymentType.UPI ||
                        paymentModel.paymentOption?.paymentType == PaymentType.UPI_INTENT)
            ) {
            payUPaymentParams.additionalParams?.put(
                PayUCheckoutProConstants.CP_IS_UPISI,
                true
            )
        }
        if (payUPaymentParams.payUSIParams != null && paymentModel.paymentOption?.paymentType == PaymentType.NB) {
            payuBizparams.amount = "0.0"
            payUPaymentParams.additionalParams?.put(
                PayUCheckoutProConstants.CP_ENACH_AMOUNT,
                payuBizparams.amount
            )
        }
        initRendering(makePaymentApiObject, paymentModel.paymentOption)
    }

    override fun resetApiResponseRepo() {
        ParserUtils.resetApiRepoResponse()
        CommonUtils.clearReferences()
        quickPayApiObject = null
        baseApiObject = null
        mcpToolbar = null
        mcpPaymentModel = null
        fetchOfferDetailsApiObject = null
        eligibleBinsForEMIApiObject = null
        baseTransactionListener = null
    }

    override fun fetchRetryOptions(
        quickOptionsListener: (QuickOptionsModel?) -> Unit?,
        listener: (ArrayList<PaymentMode>?) -> Unit?
    ) {
        ParserUtils.getMoreOptionsForRetry(
            this,
            listener,
            quickOptionsListener
        )
    }

    private fun fetchOfferApiInit(loggedInPhoneNumber: String?, offerListener: (FetchOfferDetails?) -> Unit?) {
        if (ApiResponseRepo.payuApiFetchOffer?.inProgress != true) {
            payuBizparams.loggedInPhoneNumber = loggedInPhoneNumber
            val fetchOfferDetailsApiObject = FetchOfferDetailsApiObject(
                payuBizParams = payuBizparams,
                payUPaymentParams = payUPaymentParams,
                offerListener = offerListener
            )
            this.fetchOfferDetailsApiObject = fetchOfferDetailsApiObject
            initApiCall(fetchOfferDetailsApiObject)

        }
        val payUApiResponse =
            PayUApiResponse<FetchOfferDetails>(offerListener, true, null)
        ApiResponseRepo.payuApiFetchOffer = payUApiResponse
    }

    private fun fetchOfferApiCall(loggedInPhoneNumber: String?, offerListener: (FetchOfferDetails?) -> Unit?) {
        if (isOfferApiNeedToCall()) {
            fetchOfferApiInit(loggedInPhoneNumber, offerListener)
        } else {
            offerListener.invoke(null)
        }
    }

    private fun isOfferApiNeedToCall(): Boolean {
        return !CommonUtils.isSITxn(
            payUPaymentParams
        ) && payUPaymentParams.beneficiaryDetailList.isNullOrEmpty()
    }

    private fun fetchGvInitApiCall(
        resultMap: HashMap<String, String>,
        onGvQuickPayListener: (QuickOptionsModel?) -> Unit?
    ) {
        ParserUtils.sodexoSourceId =
            ParserUtils.getSodexoSourceId(payUPaymentParams.additionalParams)
        getDeviceId { deviceId ->
            resultMap[PayUCheckoutProConstants.CP_DEVICE_ID] = deviceId ?: ""
            payuBizparams.deviceDetails = deviceId ?: ""
            quickPayApiObject = QuickPayApiObject(
                context,
                payuBizparams,
                payUPaymentParams,
                resultMap,
                onGvQuickPayListener
            )
            quickPayApiObject?.let { quickPayApiObject -> initApiCall(quickPayApiObject) }
        }

    }
    override fun fetchConvenienceFee(
        mode: String,
        variableDynamicList: ArrayList<VariableDynamics>,
        onFetchConvFeeResponse: (HashMap<String, List<ConvenienceFeeCharges>>?) -> Unit
    ) {
        val fetchConvFeeApiObject = FetchConvFeeApiObject(
            payuBizparams,
            payUPaymentParams,
            mode,
            variableDynamicList,
            onFetchConvFeeResponse
        )
        initApiCall(fetchConvFeeApiObject)
    }

    override fun fetchBinBaseDetails(
        cardNumber: String,
        cardToken: String?,
        category: String?,
        paymentCode: String?,
        onBinBaseDetailsResponse: (BinBaseDetails?) -> Unit
    ) {
        if (CommonUtils.isSodexoCard(cardNumber)) {
            val binInfo = CardBinInfo().apply {
                cardScheme = CardScheme.SODEXO
                ParserUtils.convenienceFeeResponse?.mealCard?.let {
                    additionalCharge = CommonUtils.getAdditionalChargesForBankCode(
                        SODEXO, it
                    )
                }
                ParserUtils.convenienceFeeResponse?.taxSpecification?.mealCardTaxValue?.toDoubleOrNull()
                    ?.let { gstPercentageValue = it }
                isBankDown = ParserUtils.isSodexoBankDown(ParserUtils.convenienceFeeResponse)
            }
            val binBaseDetails = BinBaseDetails(binInfo)
            onBinBaseDetailsResponse.invoke(binBaseDetails)
        } else {

            val binBaseDetailsApiObject = BinBaseDetailsApiObject(
                context,
                cardNumber,
                cardToken,
                category,
                paymentCode,
                payuBizparams,
                payUPaymentParams,
                onBinBaseDetailsResponse
            )
            initApiCall(binBaseDetailsApiObject)
        }
    }

    override fun sendOlwOtp(
        walletIdentifier: String,
        bankCode: String,
        mobile: String,
        v2ApiResponseListener: (OlwAPIResponse?) -> Unit?
    ) {
        getDeviceId {
            InternalConfig.deviceId = it
        }
        val sendOtpApiObject = SendOlwOtpApiObject(
            walletIdentifier,
            bankCode,
            mobile,
            payuBizparams,
            payUPaymentParams,
            v2ApiResponseListener
        )
        initApiCall(sendOtpApiObject)
    }

    override fun setMPin(
        otp: String,
        mPin: String,
        mobile: String,
        walletIdentifier: String,
        bankCode: String,
        action: String,
        v2ApiResponseListener: (SetMPinResponse?) -> Unit?
    ) {
        InternalConfig.deviceId?.let {
            val setMPinApiObject = SetMPinApiObject(
                otp,
                it,
                mPin,
                mobile,
                action,
                walletIdentifier,
                bankCode,
                payuBizparams,
                payUPaymentParams,
                v2ApiResponseListener
            )
            initApiCall(setMPinApiObject)
        }
    }

    override fun verifyMPin(
        mPin: String,
        walletIdentifier: String,
        bankCode: String,
        mobile: String,
        v2ApiResponseListener: (OlwAPIResponse?) -> Unit?
    ) {
        getDeviceId {
            InternalConfig.deviceId = it
            val verifyMPinApiObject = VerifyMPinApiObject(
                it ?: "",
                mPin,
                walletIdentifier,
                bankCode,
                mobile,
                payuBizparams,
                payUPaymentParams,
                v2ApiResponseListener
            )
            initApiCall(verifyMPinApiObject)
        }
    }

    override fun fetchBalanceForOlw(
        olwPaymentOption: OlwPaymentOption?,
        mobile: String,
        bankCode: String,
        v2ApiResponseListener: (OlwPaymentOption?) -> Unit?
    ) {
        val olwJsonForVar1 = JSONObject()
        olwJsonForVar1.put(PayUCheckoutProConstants.WALLET_IDENTIFIER, olwPaymentOption?.category?.lowercase())
        olwJsonForVar1.put(PayUCheckoutProConstants.MOBILE, mobile)
        olwJsonForVar1.put(PayUCheckoutProConstants.IBIBO_CODE, bankCode)
        val var1 = olwJsonForVar1.toString()
        payUPaymentParams.additionalParams?.put(
            PayUCheckoutProConstants.CP_CHECK_BALANCE_VAR1,
            var1
        )

        val response: (PayuResponse?) -> Unit = {
            val sodexoCardInfo = it?.sodexoCardInfo
            olwPaymentOption?.isMPinSet = sodexoCardInfo?.olwSDKInfo?.isMPinSet ?: false
            olwPaymentOption?.customerRegistered =
                sodexoCardInfo?.olwSDKInfo?.isCustomerRegistered ?: false
            olwPaymentOption?.balance = sodexoCardInfo?.cardBalance ?: ""
            if (sodexoCardInfo?.olwSDKInfo != null)
                olwPaymentOption?.fetchedStatus = 1
            else {
                olwPaymentOption?.fetchedStatus = 0
                olwPaymentOption?.errorMessage = sodexoCardInfo?.msg
            }

            v2ApiResponseListener.invoke(olwPaymentOption)
        }
        val checkBalanceApiObject =
            CheckBalanceApiObject(
                payuBizparams,
                response,
                var1
            )
        initApiCall(checkBalanceApiObject)
    }
}
