package com.payu.checkoutpro

import android.app.Activity
import android.os.Looper
import com.payu.base.models.BaseApiLayer
import com.payu.base.models.ErrorResponse
import com.payu.base.models.PayUPaymentParams
import com.payu.checkoutpro.layers.PayUbizApiLayer
import com.payu.checkoutpro.models.PayUCheckoutProConfig
import com.payu.checkoutpro.models.PayuSdk
import com.payu.checkoutpro.parser.HybridParameterMapParser
import com.payu.checkoutpro.utils.CPAnalyticsUtils
import com.payu.checkoutpro.utils.CommonUtils
import com.payu.checkoutpro.utils.PayUCheckoutProConstants
import com.payu.checkoutpro.utils.PayUCheckoutProConstants.CP_QR_SCANNER_ERROR_MESSAGE
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.listeners.PayUCheckoutProListener
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.CPCallbackType

object PayUCheckoutPro {

    @JvmStatic
    fun open(
        activity: Activity,
        payUPaymentParams: PayUPaymentParams,
        checkoutProListener: PayUCheckoutProListener
    ) {
        val action = {
            try {
                callUi(
                    activity,
                    PayUbizApiLayer(activity, payUPaymentParams, checkoutProListener),
                    checkoutProListener
                )
            } catch (e: Exception) {
                sendErrorResponse(activity, e.message, checkoutProListener)
            }
        }
        if(Looper.myLooper() == Looper.getMainLooper()){
            action.invoke()
        } else {
            activity.runOnUiThread {
                action.invoke()
            }
        }
    }

    @JvmStatic
    fun open(
        activity: Activity,
        payUPaymentParams: PayUPaymentParams,
        payUCheckoutProConfig: PayUCheckoutProConfig,
        checkoutProListener: PayUCheckoutProListener
    ) {
        CPAnalyticsUtils.logSdkInitialization(activity.applicationContext, payUPaymentParams.convertToJson().toString(), payUCheckoutProConfig.convertToJson().toString())
        val action = {
            try {
                callUi(
                    activity,
                    PayUbizApiLayer(
                        activity,
                        payUPaymentParams,
                        payUCheckoutProConfig,
                        checkoutProListener
                    ),
                    checkoutProListener
                )
            } catch (e: Exception) {
                sendErrorResponse(activity, e.message, checkoutProListener)
            }
        }
        if(Looper.myLooper() == Looper.getMainLooper()){
            action.invoke()
        } else {
            activity.runOnUiThread {
                action.invoke()
            }
        }
    }

    @JvmStatic
    fun open(
        activity: Activity,
        apiLayer: BaseApiLayer,
        checkoutProListener: PayUCheckoutProListener
    ) {
        val action = {
            try {
                callUi(
                    activity,
                    apiLayer,
                    checkoutProListener
                )
            } catch (e: Exception) {
                sendErrorResponse(activity, e.message, checkoutProListener)
            }
        }
        if(Looper.myLooper() == Looper.getMainLooper()){
            action.invoke()
        } else {
            activity.runOnUiThread {
                action.invoke()
            }
        }
    }

    private fun sendErrorResponse(
        activity: Activity,
        errorMessage: String?,
        checkoutProListener: PayUCheckoutProListener
    ) {
        val errorResponse = ErrorResponse()
        errorResponse.errorMessage = errorMessage
        AnalyticsUtils.logCPCallbackEventKibana(activity, CPCallbackType.Error)
        checkoutProListener.onError(errorResponse)
    }

    private fun callUi(
        activity: Activity,
        apiLayer: BaseApiLayer,
        checkoutProListener: PayUCheckoutProListener
    ) {
        if (apiLayer.config.isQrScan && !CommonUtils.isSdkAvailable(PayuSdk.QR_SCANNER.getClassName()))
            sendErrorResponse(activity, CP_QR_SCANNER_ERROR_MESSAGE, checkoutProListener)
        else
            SdkUiInitializer.startPayment(activity, apiLayer, checkoutProListener)
    }

    @JvmStatic
    fun open(
        activity: Activity,
        params: HashMap<String, Any>,
        checkoutProListener: PayUCheckoutProListener
    ) {
        val action = {
            try {
                val bizApiLayer =
                    HybridParameterMapParser.extractAndCreateAPILayer(
                        activity,
                        params,
                        checkoutProListener
                    )
                if (bizApiLayer == null)
                    sendErrorResponse(
                        activity,
                        PayUCheckoutProConstants.PAYMENT_PARAM_MISSING,
                        checkoutProListener
                    )
                else callUi(
                    activity,
                    bizApiLayer,
                    checkoutProListener
                )
            } catch (e: Exception) {
                sendErrorResponse(activity, e.message, checkoutProListener)
            }
        }
        if(Looper.myLooper() == Looper.getMainLooper()){
            action.invoke()
        } else {
            activity.runOnUiThread {
                action.invoke()
            }
        }
    }
}