package com.payu.ui.viewmodel

import android.app.Application
import android.content.Context
import android.webkit.WebView
import androidx.fragment.app.Fragment
import androidx.lifecycle.MutableLiveData
import com.payu.base.listeners.BaseTransactionListener
import com.payu.base.listeners.VerifyServiceListener
import com.payu.base.models.ApiResponse
import com.payu.base.models.BnplOption
import com.payu.base.models.ErrorResponse
import com.payu.base.models.PaymentMode
import com.payu.base.models.PaymentModel
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentType
import com.payu.base.models.VariableDynamics
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.managers.BottomSheetManager
import com.payu.ui.model.models.FragmentModel
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.CPCallbackType
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils
import com.payu.ui.model.utils.ViewUtils
import java.util.Stack

class QuickPayViewModel(application: Application) : BaseViewModel(application),
    BaseTransactionListener {

    val applicationContext = application

    internal var openCheckout: MutableLiveData<Event<Boolean>> = MutableLiveData()
    internal val showBottomSheet: MutableLiveData<Pair<Int, BottomSheetManager?>> = MutableLiveData()
    val hideExitDialogSheet: MutableLiveData<Boolean> = MutableLiveData()

    val bnplEligibilityLiveData: MutableLiveData<Event<Boolean>> = MutableLiveData()

    internal fun validateOffer(category: String?, bankCode: String?) {
        //todo: implement offer validation api call when it is planned for quick-pay flow
    }

    override fun onPaymentSuccess(response: Any) {
        flowComplete.value = true
        SdkUiInitializer.apiLayer?.reset()
        AnalyticsUtils.logCPCallbackEventKibana(applicationContext, CPCallbackType.Success)
        SdkUiInitializer.checkoutProListener?.onPaymentSuccess(
            response
        )
    }

    override fun onPaymentFailure(response: Any) {
        flowComplete.value = true
        SdkUiInitializer.apiLayer?.reset()
        AnalyticsUtils.logCPCallbackEventKibana(applicationContext, CPCallbackType.Failure)
        SdkUiInitializer.checkoutProListener?.onPaymentFailure(
            response
        )

    }

    override fun onPaymentCancel() {
        flowComplete.value = true
        SdkUiInitializer.apiLayer?.reset()
        AnalyticsUtils.logCPCallbackEventKibana(applicationContext, CPCallbackType.Cancel)
        SdkUiInitializer.checkoutProListener?.onPaymentCancel(true)
    }

    override fun loadNextState(paymentModel: PaymentModel) {
    }

    override fun loadNextState(fragment: Fragment?) {
        fragment?.let {
            loadFragment(it, SdkUiConstants.TAG_DYNAMIC_FRAGMENT)
        }
    }

    override fun setWebViewProperties(webView: WebView?, bank: Any?) {
        SdkUiInitializer.checkoutProListener?.setWebViewProperties(webView, bank)

    }

    override fun loadRetryPaymentOption() {
        flowComplete.value = true
        openCheckout.value = Event(true)
    }

    internal fun openCheckout() {
        openCheckout.value = Event(true)
    }

    override fun showLoaderScreen(showLoader: Boolean) {
    }

    override fun onError(errorResponse: ErrorResponse) {
        flowComplete.postValue(true)
        SdkUiInitializer.apiLayer?.reset()
        AnalyticsUtils.logCPCallbackEventKibana(application.applicationContext, CPCallbackType.Error)
        SdkUiInitializer.checkoutProListener?.onError(errorResponse)
    }

    override fun showProgressDialog(showDialog: Boolean) {

    }

    private fun loadFragment(fragment: Fragment, tag: String?) {
        ViewUtils.dismissSnackBar()

        val fragmentModel = FragmentModel()
        fragmentModel.fragment = fragment
        fragmentModel.tag = tag
        loadFragment.value = fragmentModel
    }

    internal fun showExitConfirmation(fragmentStack: Stack<FragmentModel>?){
        if (false == SdkUiInitializer.apiLayer?.config?.showExitConfirmationOnCheckoutScreen) {
            exitDialogYesClicked(fragmentStack)
            return
        }
        showBottomSheet.value = Pair(R.layout.payu_cancel_payment_confirmation, null)
    }

    fun exitDialogYesClicked(fragmentStack: Stack<FragmentModel>?) {
        fragmentStack?.let {
            if (it.isNotEmpty()) {
                it.pop()
            }
        }
        SdkUiInitializer.apiLayer?.reset()
        flowComplete.value = true
        hideBottomSheet.value = true
        hideExitDialogSheet.value = true
        AnalyticsUtils.logCPCallbackEventKibana(applicationContext, CPCallbackType.Cancel)
        SdkUiInitializer.checkoutProListener?.onPaymentCancel(false)
    }

    fun exitDialogNoClicked() {
        hideExitDialogSheet.value = true
    }

    internal fun makePayment(paymentOption: PaymentOption, context: Context) {
        paymentOption.isQuickPayTxn = true
        SdkUiInitializer.apiLayer?.updatePaymentState(
            Utils.getPaymentModel(
                paymentOption,
                null
            ),
            ViewUtils.getToolbar(
                context,
                paymentOption.additionalCharge
            )
        )
    }

    fun checkEligibilityForBNPL(quickPayList: ArrayList<PaymentMode>) {
        val bnplItems = quickPayList.filter { it.type == PaymentType.BNPL}

        for (item in bnplItems) {
            SdkUiInitializer.apiLayer?.verifyEligibilityAPI(item.optionDetail!![0], object :
                VerifyServiceListener {
                override fun eligibilityDetails(apiResponse: ApiResponse) {
                    val resourceId: Int? = Utils.getEligibilityDetails(apiResponse, item.type!!)
                    updateItem(item, resourceId)
                }
            })
        }
    }

    private fun updateItem(item: PaymentMode, resourceId: Int?) {
        item.optionDetail?.let {
            (it[0] as BnplOption).isEligible = resourceId == null
            bnplEligibilityLiveData.value = Event(true)
        }
    }

    internal fun showOfferAppliedView(){
        showOfferAppliedDialog.value = Event(true)
    }
}