package com.payu.ui.view.customViews

import android.animation.ValueAnimator
import android.animation.ValueAnimator.INFINITE
import android.animation.ValueAnimator.RESTART
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import android.view.animation.LinearInterpolator
import com.payu.ui.R

class PayUCircularProgressBar(context: Context, attrs: AttributeSet? = null) : View(context, attrs) {

    private var progress = 0
    private var maxProgress = 100
    private var progressBarColor = Color.BLUE
    private var progressBackgroundColor = Color.LTGRAY
    private var strokeWidth = 10f
    private var isIndeterminate = false // New flag

    private val paint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        style = Paint.Style.STROKE
        strokeCap = Paint.Cap.ROUND
    }

    private val backgroundPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        style = Paint.Style.STROKE
        strokeCap = Paint.Cap.ROUND
        color = progressBackgroundColor
        strokeWidth = this@PayUCircularProgressBar.strokeWidth
    }

    private val rectF = RectF()
    private var animator: ValueAnimator? = null // Animator

    init {
        attrs?.let {
            val typedArray = context.obtainStyledAttributes(it, R.styleable.PayUCircularProgressBar, 0, 0)
            progress = typedArray.getInt(R.styleable.PayUCircularProgressBar_progress, progress)
            maxProgress = typedArray.getInt(R.styleable.PayUCircularProgressBar_maxProgress, maxProgress)
            progressBarColor = typedArray.getColor(R.styleable.PayUCircularProgressBar_progressBarColor, progressBarColor)
            progressBackgroundColor = typedArray.getColor(R.styleable.PayUCircularProgressBar_progressBackgroundColor, progressBackgroundColor)
            strokeWidth = typedArray.getDimension(R.styleable.PayUCircularProgressBar_strokeWidth, strokeWidth)
            isIndeterminate = typedArray.getBoolean(R.styleable.PayUCircularProgressBar_isIndeterminate, false) // Get indeterminate flag
            typedArray.recycle()

            paint.strokeWidth = strokeWidth
            backgroundPaint.strokeWidth = strokeWidth
            if (isIndeterminate) {
                startIndeterminateAnimation()
            }
        }
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        val padding = strokeWidth / 2
        rectF.set(padding, padding, w - padding, h - padding)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        // Draw background circle
        canvas.drawArc(rectF, 0f, 360f, false, backgroundPaint)

        // Draw progress arc
        paint.color = progressBarColor
        val angle = 360f * progress / maxProgress
        canvas.drawArc(rectF, -90f, angle, false, paint)
    }

    fun setProgress(progress: Int) {
        this.progress = progress.coerceIn(0, maxProgress)
        invalidate()
    }

    fun setMaxProgress(maxProgress: Int) {
        this.maxProgress = maxProgress
        invalidate()
    }

    fun setProgressBarColor(color: Int) {
        this.progressBarColor = color
        invalidate()
    }

    fun setProgressBackgroundColor(color: Int) {
        this.progressBackgroundColor = color
        backgroundPaint.color = color
        invalidate()
    }

    fun setStrokeWidth(width: Float) {
        this.strokeWidth = width
        paint.strokeWidth = width
        backgroundPaint.strokeWidth = width
        invalidate()
    }

    fun setIndeterminate(isIndeterminate: Boolean) { // New method
        this.isIndeterminate = isIndeterminate
        if (isIndeterminate) {
            startIndeterminateAnimation()
        } else {
            stopIndeterminateAnimation()
        }
    }

    private fun startIndeterminateAnimation() { // New method
        animator = ValueAnimator.ofInt(0, maxProgress).apply {
            addUpdateListener { valueAnimator ->
                progress = valueAnimator.animatedValue as Int
                invalidate()
            }
            repeatMode = RESTART
            repeatCount = INFINITE
            duration = 1000 // Adjust duration as needed
            interpolator = LinearInterpolator()
            start()
        }
    }

    private fun stopIndeterminateAnimation() { // New method
        animator?.cancel()
        animator = null
        progress = 0
        invalidate()
    }

    override fun onDetachedFromWindow() { // Important: Stop animation when view is detached
        super.onDetachedFromWindow()
        stopIndeterminateAnimation()
    }
}