package com.payu.ui.model.utils

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64
import android.util.Log
import android.view.View
import android.view.inputmethod.InputMethodManager
import com.payu.base.models.ApiResponse
import com.payu.base.models.BnplOption
import com.payu.base.models.CardScheme
import com.payu.base.models.CardType
import com.payu.base.models.CustomNote
import com.payu.base.models.EMIOption
import com.payu.base.models.EmiType
import com.payu.base.models.InternalConfig
import com.payu.base.models.OfferInfo
import com.payu.base.models.PayUBeneficiaryAccountType
import com.payu.base.models.PayUBillingCycle
import com.payu.base.models.PayUPaymentParams
import com.payu.base.models.PayUSIParams
import com.payu.base.models.PaymentFlowState
import com.payu.base.models.PaymentMode
import com.payu.base.models.PaymentModel
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentOptionOfferinfo
import com.payu.base.models.PaymentType
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.managers.OfferFilterManager
import com.payu.ui.model.utils.SdkUiConstants.CP_ANDROID_ID_PREF
import com.payu.ui.model.utils.SdkUiConstants.CP_ANDROID_ID_PREF_FILE
import com.payu.ui.model.utils.SdkUiConstants.GAID_TOKEN_PREF
import com.payu.ui.model.utils.SdkUiConstants.GAID_TOKEN_PREF_FILE
import com.payu.ui.model.utils.SdkUiConstants.GLOBAL_VAULT_USER_TOKEN_PREF
import com.payu.ui.model.utils.SdkUiConstants.GLOBAL_VAULT_USER_TOKEN_PREF_FILE
import java.lang.reflect.Field
import java.security.KeyStore
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale
import java.util.UUID
import java.util.concurrent.TimeUnit
import java.util.regex.Pattern
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.SecretKey
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.IvParameterSpec

object Utils {
    private var CARD_EXPIRE_REGEX: String = "^(0[1-9]|1[0-2])[/][0-9]{2}$"
    private const val VALIDATE_MOBILE_NUMBER_REGEX = "[6789][0-9]{9}?"
    private const val VALIDATE_VPA_REGEX = "^[a-zA-Z0-9.-]+@[a-zA-Z0-9.-]+\$"
    private const val AMEX_CARD_REGEX: String = "^3[47][\\d]+"
    private const val NAME_ON_CARD_REGEX = "^[a-zA-Z0-9. ]+\$"
    private const val VALIDATE_IFSC_REGEX = "^[A-Z0-9]{11}\$"
    private const val VALIDATE_BAJAJ_CARD_REGEX = "^203040\\d{10}\$"
    private const val NUMBER_REGEX = "[0-9]+"
    internal var customNote: String? = null
    private var offerInfo: OfferInfo? = null
    private const val VALIDATE_PAN_NUMBER_REGEX = "^[A-Za-z]{5}[0-9]{4}[A-Za-z]\$"
    private val topBankList = ArrayList<PaymentOption>()
    private var otherBanksList = ArrayList<PaymentOption>()
    private var upiIntentList = ArrayList<PaymentOption>()
    private var prefMap: HashMap<String, SharedPreferences>? = null

    private const val CIPHER_TRANSFORMATION = SdkUiConstants.PAYU_AES_GCM_NO_PADDING

    private const val CIPHER_TRANSFORMATION1 = "AES/"
    private const val CIPHER_TRANSFORMATION2 = "CBC/"
    private const val CIPHER_TRANSFORMATION3 = "PKCS7Padding"
    private const val ENCRYPTION_ALGORITHM = "AES"
    private const val KEYSTORE_PROVIDER_ANDROID_KEYSTORE: String = "AndroidKeyStore"
    private const val mAlias: String = "PayUKeyAlias"
    private const val PAYU_ENCRYPTION_KEY_GCM = "payu_encryption_key_gcm"

    private val TAG: String = SdkUiConstants.UTILS

    const val DATE_FORMAT_yyyy_MM_dd = "yyyy-MM-dd"
    const val DATE_FORMAT_dd_MMM_yyyy = "dd MMM yyyy"
    const val DATE_FORMAT_dd_MM_yyyy = "dd/MM/yyyy"
    const val DATE_FORMAT_MMM_yyyy = "MMM yyyy"
    const val DATE_FORMAT_DD = "dd"

    /**
     * Returns Scheme icon.
     */
    fun getCardIconId(cardScheme: CardScheme?): Int = when (cardScheme) {

        CardScheme.MAST -> R.drawable.payu_master_card
        CardScheme.VISA -> R.drawable.payu_visa
        CardScheme.AMEX -> R.drawable.payu_amex
        CardScheme.MAES, CardScheme.SMAE -> R.drawable.payu_maestro
        CardScheme.RUPAY, CardScheme.RUPAYCC -> R.drawable.payu_rupay
        CardScheme.JCB -> R.drawable.payu_jcb
        CardScheme.DINR -> R.drawable.payu_dinersclub
        CardScheme.SODEXO -> R.drawable.payu_sodexo
        else -> R.drawable.payu_cards_placeholder
    }

    /*
        Returns the maximum length of input edit text based on Formatting
        and valid card length of scheme
     */
    fun getCardInputMaxLength(cardScheme: CardScheme?): Int {
        return when (cardScheme) {
            CardScheme.MAES, CardScheme.SMAE, CardScheme.VISA -> 23
            CardScheme.AMEX -> 17
            CardScheme.DINR -> 17
            CardScheme.JCB, CardScheme.RUPAY, CardScheme.RUPAYCC, CardScheme.MAST, CardScheme.DISCOVER -> 19
            else -> 23
        }
    }

    /**
     * Will tell if the provided classname
     * is available in the current SDK
     * classname must be full qualified name with package name
     *
     * @param className String i.e: [com.payu.ui.model.utils.Utils]
     *
     */
    fun isSdkAvailable(className: String?): Boolean {
        try {
            if (!className.isNullOrEmpty()) {
                val wrapperClassLoader =
                    Utils::class.java.classLoader
                val wrapperClass =
                    wrapperClassLoader?.loadClass(className)
                return true
            }
        } catch (e: Exception) {
        }
        return false
    }


    /*
        Returns whether inserted card length matches with the scheme limit
     */
    fun isValidCardLength(cardScheme: CardScheme?, length: Int): Boolean {
        return when (cardScheme) {
            CardScheme.MAES, CardScheme.SMAE, CardScheme.VISA -> length in (13..19)
            CardScheme.JCB, CardScheme.RUPAY, CardScheme.RUPAYCC, CardScheme.MAST, CardScheme.DISCOVER, CardScheme.SODEXO -> length == 16
            CardScheme.AMEX -> length == 15
            CardScheme.DINR -> length == 14
            else -> length in (13..19)
        }
    }

    /*
        Checks whether CVV is required for this scheme or not.
     */
    fun isCvvLessCard(cardScheme: CardScheme?): Boolean {
        return cardScheme == CardScheme.SMAE
    }

    fun getCvvInputLength(cardScheme: CardScheme?): Int = when (cardScheme) {
        CardScheme.AMEX -> 4
        else -> 3
    }

    /**
     * Returns true if cvv length is valid as per the Scheme
     */
    fun isValidCvvLength(scheme: CardScheme?, length: Int): Boolean {
        return when (scheme) {
            CardScheme.AMEX -> length == 4
            CardScheme.UNKNOWN, null -> length in intArrayOf(0, 3, 4)
            else -> length == 3
        }
    }

    fun isValidNumberFormat(number: String): Boolean {
        return if (number.isNotEmpty())
            Regex(NUMBER_REGEX).matches(number)
        else false
    }

    /**
     * Checks whether cvv is correct number and check the length
     */
    fun isValidCvv(number: String, scheme: CardScheme?): Boolean {
        return isValidNumberFormat(number) && isValidCvvLength(scheme, number.length)
    }

    /**
     * Checks whether inserted date is valid or not.
     */
    fun isValidExpiry(date: String): Boolean {
        if (date.matches(Regex(CARD_EXPIRE_REGEX))) {
            val sdf = SimpleDateFormat("MM/yy", Locale.getDefault())
            val enteredDate = sdf.parse(date)
            val calendar = Calendar.getInstance()
            val currentDate = calendar.time
            val currentDateValue = sdf.format(currentDate)
            return enteredDate.after(currentDate) || currentDateValue.equals(date)
        } else {
            return false
        }
    }

    /**
     * Checks if the card number is of AMEX type
     */
    fun isAmexCard(cardNumber: String): Boolean {
        return cardNumber.matches(Regex(AMEX_CARD_REGEX))
    }

    /**
     * Checks if the name on card is valid
     * @return true, if name is valid else false
     */
    fun isValidNameOnCard(nameOnCard: String): Boolean = if (nameOnCard.isEmpty()) false
    else nameOnCard.matches(Regex(NAME_ON_CARD_REGEX))

    /**
     * Returns year in (yyyy) format from inserted (yy) format
     */
    fun getExpiryYear(currentDate: String): String {
        try {
            var currentDateFormat = SimpleDateFormat("yy", Locale.getDefault())
            var desiredDateFormat = SimpleDateFormat("yyyy", Locale.getDefault())
            var currentYear = currentDateFormat.parse(currentDate)
            return desiredDateFormat.format(currentYear)
        } catch (e: ParseException) {
            return "00"
        }
    }

    /**
     * Luhn's algorithm to check card number is valid or not.
     */
    fun luhn(cardNumber: String): Boolean {
        var sum = 0
        var alternate = false
        for (i in cardNumber.length - 1 downTo 0) {
            var n = cardNumber.substring(i, i + 1).toInt()
            if (alternate) {
                n *= 2
                if (n > 9) {
                    n = n % 10 + 1
                }
            }
            sum += n
            alternate = !alternate
        }
        return sum % 10 == 0
    }

    fun getPaymentOptionList(
        paymentModesList: ArrayList<PaymentMode>,
        paymentType: PaymentType
    ): ArrayList<PaymentOption>? {
        var paymentOptionList: ArrayList<PaymentOption>? = null
        for (paymentMode in paymentModesList) {
            when (paymentMode.type) {
                paymentType -> paymentOptionList = paymentMode.optionDetail
            }
        }
        return paymentOptionList
    }

    fun getSodexoPaymentOptionList(paymentModesList: ArrayList<PaymentMode>): ArrayList<PaymentMode>? {
        val sodexoPaymentModeList = ArrayList<PaymentMode>()
        for (paymentMode in paymentModesList) {
            if (paymentMode.type == PaymentType.SODEXO)
                sodexoPaymentModeList.add(paymentMode)
        }
        return sodexoPaymentModeList
    }

    fun getCardPaymentOptionList(paymentModesList: ArrayList<PaymentMode>): ArrayList<PaymentMode> {
        val cardPaymentModeList = ArrayList<PaymentMode>()
        for (paymentMode in paymentModesList) {
            if (paymentMode.type == PaymentType.CARD)
                cardPaymentModeList.add(paymentMode)
        }
        return cardPaymentModeList
    }

    fun getPaymentOptionFromModeList(
        paymentModesList: ArrayList<PaymentMode>,
        paymentType: PaymentType
    ): PaymentOption? {
        for (paymentMode in paymentModesList) {
            if (paymentMode.type == paymentType && !paymentMode.optionDetail.isNullOrEmpty()) {
                return paymentMode.optionDetail?.get(0)
            }
        }
        return null
    }

    fun getSavedOptionsList(
        paymentModesList: ArrayList<PaymentMode>?,
        paymentType: PaymentType
    ): ArrayList<PaymentMode>? {
        var savedOptionsList: ArrayList<PaymentMode>? = null
        if (paymentModesList.isNullOrEmpty()) return savedOptionsList

        for (paymentMode in paymentModesList) {
            when (paymentMode.type) {
                paymentType -> {
                    savedOptionsList = ArrayList()
                    savedOptionsList.add(paymentMode)
                }
            }
        }
        return savedOptionsList
    }

    fun getFormattedString(cardNumber: String?, cardScheme: CardScheme?): String? {
        if (cardScheme == null)
            return cardNumber

        var formattedString: String? = null
        when (cardScheme) {
            //Amex cards are formatted as xxxx xxxxxx xxxxx
            CardScheme.AMEX -> formattedString = "${cardNumber?.substring(0, 4)} ${
                cardNumber?.substring(
                    4,
                    10
                )
            } ${cardNumber?.substring(10)}"

            //Other cards are formatted as xxxx xxxx xxxx xxxx
            else -> formattedString = cardNumber?.replace("....(?!$)".toRegex(), "$0 ")
        }

        return formattedString
    }

    fun getPaymentModel(
        paymentOption: PaymentOption,
        paymentFlowState: PaymentFlowState?
    ): PaymentModel {
        val paymentModel = PaymentModel()
        paymentModel.paymentOption = paymentOption
        paymentModel.paymentFlowState = paymentFlowState
        return paymentModel
    }

    /**
     * Checks if the phone number is valid or not
     *
     * @param phone phone number
     * @return true, if phone number is valid else false
     */
    fun isValidPhoneNumber(phone: String): Boolean {
        return isValidRegexValue(phone, VALIDATE_MOBILE_NUMBER_REGEX)
    }

    fun isValidNumber(number: String): Boolean {
        return isValidRegexValue(number, NUMBER_REGEX)
    }

    fun isValidBajajcardNumber(cardNumber: String): Boolean {
        return isValidRegexValue(cardNumber, VALIDATE_BAJAJ_CARD_REGEX)
    }


    /**
     * Checks if the ifsc code is valid or not
     *
     * @param ifscCode ifsc code
     * @return true, if ifsc is valid else false
     */
    fun isValidIfsc(ifscCode: String): Boolean {
        return isValidRegexValue(ifscCode, VALIDATE_IFSC_REGEX)
    }

    fun isValidPanNumber(panNumber: String): Boolean {
        return isValidRegexValue(panNumber, VALIDATE_PAN_NUMBER_REGEX)
    }

    /**
     * Checks if VPA is in valid format or not.
     * @param vpa VPA Address
     * @return true ,if vpa is in valid format else false
     */
    fun isValidVPA(vpa: String): Boolean {
        if (vpa.isNotEmpty() && vpa.length >= 3 && vpa.length <= 120) {
            return isValidRegexValue(vpa, VALIDATE_VPA_REGEX)
        } else return false
    }

    private fun isValidRegexValue(regexValue: String, regEx: String): Boolean {
        val pattern = Pattern.compile(regEx)
        val matcher = pattern.matcher(regexValue)
        return matcher.matches()
    }

    fun isPaymentTypeAvailable(list: ArrayList<PaymentOption>, paymentType: PaymentType): Boolean {
        var isEnabled = false

        for (paymentOption in list) {
            when (paymentOption.paymentType) {
                paymentType -> isEnabled = true
            }
        }
        return isEnabled
    }

    fun isPaymentOptionAvailable(
        list: ArrayList<PaymentOption>,
        paymentOptionName: String
    ): Boolean {
        for (paymentOption in list) {
            val bankcode = getValueFromPaymentOption<String>(
                SdkUiConstants.CP_BANK_CODE,
                paymentOption.otherParams as HashMap<String, Any?>?
            )
            if (bankcode?.equals(paymentOptionName, ignoreCase = true) == true) return true
        }
        return false
    }


    fun getIntentAppsList(list: ArrayList<PaymentOption>): ArrayList<PaymentOption>? {
        var appsList: ArrayList<PaymentOption>? = null
        for (paymentOption in list) {
            when (paymentOption.paymentType) {
                PaymentType.UPI_INTENT -> appsList = paymentOption.optionList
            }
        }
        upiIntentList = appsList ?: arrayListOf()

        return appsList
    }

    fun getUpiCollectPaymentOption(list: ArrayList<PaymentOption>): PaymentOption? {
        var upiCollectOption: PaymentOption? = null
        for (paymentOption in list) {
            when (paymentOption.paymentType) {
                PaymentType.UPI -> upiCollectOption = paymentOption
            }
        }

        return upiCollectOption
    }

    fun isUpiIntentAvailable(list: ArrayList<PaymentOption>): Boolean {
        val intentAppsList = getIntentAppsList(list)
        return isPaymentTypeAvailable(
            list,
            PaymentType.UPI_INTENT
        ) && intentAppsList != null && intentAppsList.size > 0
    }

    fun getDefaultDrawable(paymentType: PaymentType?, emiType: EmiType? = null) =
        when (paymentType) {
            PaymentType.UPI_INTENT, PaymentType.UPI -> R.drawable.payu_upi
            PaymentType.WALLET, PaymentType.OPEN_LOOP_WALLET -> R.drawable.payu_wallet
            PaymentType.EMI -> {
                when (emiType) {
                    EmiType.CC -> R.drawable.payu_cc_emi
                    EmiType.DC -> R.drawable.payu_dc_emi
                    EmiType.CARD_LESS -> R.drawable.payu_cardless_emi
                    else -> R.drawable.payu_emi
                }
            }

            PaymentType.CARD -> R.drawable.payu_cards_placeholder
            PaymentType.NEFTRTGS -> R.drawable.payu_neft_rtgs
            PaymentType.BNPL -> R.drawable.payu_bnpl
            else -> R.drawable.payu_netbanking
        }

    fun formatDate(
        date: String?,
        currentFormat: String = DATE_FORMAT_yyyy_MM_dd,
        newFormat: String = DATE_FORMAT_dd_MMM_yyyy,
        addDaySuffix: Boolean = false
    ): String {
        date?.let {
            try {
                val format1 = SimpleDateFormat(currentFormat, Locale.getDefault())
                val date1 = format1.parse(it)
                val format2 = SimpleDateFormat(newFormat, Locale.getDefault())

                return if (addDaySuffix) {
                    val formatDateOfMonth = SimpleDateFormat(DATE_FORMAT_DD, Locale.getDefault())
                    val day: Int = formatDateOfMonth.format(date1).toInt()
                    val dayStr: String = day.toString() + getDayOfMonthSuffix(day)
                    dayStr + " " + format2.format(date1)
                } else {
                    format2.format(date1)
                }
            } catch (ex: Exception) {
                return it
            }
        } ?: return ""
    }

    fun getDayOfMonthSuffix(n: Int): String {
        return if (n in 11..13) {
            "th"
        } else when (n % 10) {
            1 -> "st"
            2 -> "nd"
            3 -> "rd"
            else -> "th"
        }
    }

    fun getAccountType(accountType: String) = when (accountType) {
        "Savings account" -> PayUBeneficiaryAccountType.SAVINGS
        "Current account" -> PayUBeneficiaryAccountType.CURRENT
        else -> PayUBeneficiaryAccountType.SAVINGS
    }

    fun convertBillingCycle(billingCycle: PayUBillingCycle?): String? {
        var billingCycleValue: String? = null
        when {
            PayUBillingCycle.YEARLY == billingCycle -> {
                billingCycleValue = "year"
            }

            PayUBillingCycle.MONTHLY == billingCycle -> {
                billingCycleValue = "month"
            }

            PayUBillingCycle.DAILY == billingCycle -> {
                billingCycleValue = "day"
            }

            PayUBillingCycle.WEEKLY == billingCycle -> {
                billingCycleValue = "week"
            }
        }
        return billingCycleValue
    }


    internal fun isEligibleForEMI(emiTenureList: ArrayList<PaymentOption>): Boolean {
        if (emiTenureList.isEmpty()) return false
        for (item in emiTenureList) {
            (item as? EMIOption)?.let {
                if (it.isEligible) return true
            }
        }
        return false
    }

    internal fun getEligibleEmiTenuresList(emiTenureList: ArrayList<PaymentOption>?): ArrayList<PaymentOption>? {
        if (emiTenureList.isNullOrEmpty()) return null
        val eligibleList: ArrayList<PaymentOption> = ArrayList()
        for (item in emiTenureList) {
            (item as? EMIOption)?.let {
                if (it.isEligible)
                    eligibleList.add(it)
            }
        }
        return eligibleList
    }

    internal fun getNoCostEmiEligibleList(
        emiTenureList: ArrayList<PaymentOption>?,
        noCostEmiList: ArrayList<PaymentOption>? = null
    ): ArrayList<PaymentOption>? {
        if (emiTenureList.isNullOrEmpty()) return null
        val eligibleList: ArrayList<PaymentOption> = ArrayList()
        if (noCostEmiList.isNullOrEmpty()) {
            for (item in emiTenureList) {
                (item as? EMIOption)?.let {
                    if (it.isEligible)
                        eligibleList.add(it)
                }
            }
        } else {
            for (item in emiTenureList) {
                for (noCostItem in noCostEmiList) {
                    (item as? EMIOption)?.let {
                        if (it.isEligible && it == noCostItem)
                            eligibleList.add(noCostItem)
                    }
                }
            }
        }
        return eligibleList
    }

    /**
     * This function format the amount to max 2 decimal places and returns formatted amount string with currency symbol
     * @param amount txn amount
     * @param context Application Context
     * @return formatted string
     * */
    internal fun getFormattedAmount(
        amount: Double?,
        context: Context,
        resId: Int? = R.string.payu_amount_with_rupee_symbol
    ): String? {
        val splitAmt = getFormattedAmountValue(amount)
        return resId?.let { context.getString(it, splitAmt) }
    }

    internal fun getFormattedAmountValue(amount: Double?): String {
        val formatAmount = formatAmount(
            String.format(
                "%.2f",
                amount
            )
        )
        val splitAmount = formatAmount.split(".")
        //TODO handle null value of amount
        return if (splitAmount.size > 1 && splitAmount[1] == "00")
            splitAmount[0]
        else formatAmount
    }

    /**
     * This function will format the amount as below -
     * 1000 -> 1,000
     * 100000 -> 1,00,000
     * */
    internal fun formatAmount(amount: String?): String {
        if (amount.isNullOrEmpty()) return ""
        val stringBuilder = StringBuilder()
        var amountString: String = amount
        var decimalString = ""
        if (amount.contains(".")) {
            amountString = amount.split(".")[0]
            decimalString = amount.split(".")[1]
        }
        val amountArray: CharArray = amountString.toCharArray()
        var a = 0
        var b = 0
        for (i in amountArray.indices.reversed()) {
            if (a < 3) {
                stringBuilder.append(amountArray[i])
                a++
            } else if (b < 2) {
                if (b == 0) {
                    stringBuilder.append(",")
                    stringBuilder.append(amountArray[i])
                    b++
                } else {
                    stringBuilder.append(amountArray[i])
                    b = 0
                }
            }
        }
        return if (decimalString.isEmpty()) stringBuilder.reverse()
            .toString() else stringBuilder.reverse().toString() + ".$decimalString"
    }

    internal fun isEnachPayment(paymentType: PaymentType?): Boolean {
        if (paymentType == null) return false
        return paymentType == PaymentType.NB && SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams != null
    }


    internal fun getCustomeNoteDetails(
        paymentType: PaymentType?,
        customeNoteDetails: ArrayList<CustomNote>?
    ): String? {
        if (paymentType == null || customeNoteDetails?.isEmpty() == true) return null
        var customNote: String = ""
        if (customeNoteDetails != null) {
            for (customeNotelist in customeNoteDetails) {
                if (customeNotelist.custom_note_category != null && customeNotelist.custom_note_category?.contains(
                        paymentType
                    ) == true
                )
                    customNote = customeNotelist.custom_note
            }
        }
        return customNote
    }

    internal fun isCustomeNoteCategoryNull(customeNoteDetails: ArrayList<CustomNote>?): Boolean {
        var isCustomeNoteCategoryNull = false
        if (customeNoteDetails != null) {
            for (customeNotelist in customeNoteDetails) {
                if (customeNotelist.custom_note_category == null || customeNotelist.custom_note_category?.isEmpty() == true)
                    isCustomeNoteCategoryNull = true
                customNote = customeNotelist.custom_note
            }
        }
        return isCustomeNoteCategoryNull
    }

    fun getSIParams(): PayUSIParams? {
        return SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams
    }

    //TODO: This api need to be optimized
    internal fun isNonSodexoCardPresent(quickOptionsList: ArrayList<PaymentMode>?): Boolean {
        if (!quickOptionsList.isNullOrEmpty()) {
            val sizeOfCardList = getCardPaymentOptionList(quickOptionsList).size
            if (sizeOfCardList != null && sizeOfCardList > 0)
                return true
        }
        return false
    }

    internal fun isListHasItemsWithoutSodexoAndClw(quickOptionsList: ArrayList<PaymentMode>?): Boolean {
        val filteredList =
            quickOptionsList?.filter { it.type != PaymentType.SODEXO && it.type != PaymentType.CLOSED_LOOP_WALLET }
        return !filteredList.isNullOrEmpty()
    }

    internal fun isOfferAvailable(bankCode: String, paymentType: PaymentType?): Boolean {
        if (InternalConfig.userSelectedOfferInfo?.isAllPaymentMethodsAvailable == true)
            return true
        when (paymentType) {
            PaymentType.NB -> {
                return isOfferAvailableForPaymentCode(
                    InternalConfig.userSelectedOfferInfo?.nbOptionOfferInfoList,
                    bankCode
                )
            }

            PaymentType.UPI -> {
                return isOfferAvailableForPaymentCode(
                    InternalConfig.userSelectedOfferInfo?.upiOptionOfferInfoList,
                    bankCode
                )
            }

            PaymentType.WALLET -> {
                return isOfferAvailableForPaymentCode(
                    InternalConfig.userSelectedOfferInfo?.walletOptionOfferInfoList,
                    bankCode
                )
            }

            PaymentType.BNPL -> {
                return isOfferAvailableForPaymentCode(
                    InternalConfig.userSelectedOfferInfo?.bnplOptionOfferInfoList,
                    bankCode
                )
            }

            PaymentType.EMI -> {
                InternalConfig.userSelectedOfferInfo?.emiOfferInfo?.emiCCOfferList?.forEach { paymentOptionInfo ->
                    if (paymentOptionInfo.paymentCode == bankCode)
                        return true
                }
                InternalConfig.userSelectedOfferInfo?.emiOfferInfo?.emiDCOfferList?.forEach { paymentOptionInfo ->
                    if (paymentOptionInfo.paymentCode == bankCode)
                        return true
                }
                InternalConfig.userSelectedOfferInfo?.emiOfferInfo?.emiCardLessOfferList?.forEach { paymentOptionInfo ->
                    if (paymentOptionInfo.paymentCode == bankCode)
                        return true
                }
            }

            PaymentType.CLOSED_LOOP_WALLET -> {
                return isOfferAvailableForPaymentCode(
                    InternalConfig.userSelectedOfferInfo?.clwOptionOfferInfoList,
                    bankCode
                )
            }

            else -> return false
        }
        return false
    }


    internal fun isOfferSelected(): Boolean {
        return InternalConfig.selectedOfferInfo != null
    }

    internal fun isOfferAvailableForCards(
        bankCode: String?,
        scheme: String,
        cardType: CardType?,
    ): Boolean {
        //TODO will rewrite when offer is supported in saved card

//        var bankFound = false
//        var networkFound = false
//        var isBankAvailable = false
//        var isNetworkAvailable = false
//
//        val banksInfoForCards = offerInfo?.bankOfferInfoCards
//        val nwInfoForCards = offerInfo?.networkOfferInfoCards
//        if (cardType == CardType.CC) {
//            isBankAvailable = !banksInfoForCards?.banksListForCCCards.isNullOrEmpty()
//            isNetworkAvailable = !nwInfoForCards?.networkListForCCCards.isNullOrEmpty()
//            bankFound = isOfferAvaialbleinList(banksInfoForCards?.banksListForCCCards, bankCode)
//            networkFound = isOfferAvaialbleinList(nwInfoForCards?.networkListForCCCards, scheme)
//        } else {
//            isBankAvailable = !banksInfoForCards?.banksListForDCCards.isNullOrEmpty()
//            isNetworkAvailable = !nwInfoForCards?.networkListForDCCards.isNullOrEmpty()
//            bankFound = isOfferAvaialbleinList(banksInfoForCards?.banksListForDCCards, bankCode)
//            networkFound = isOfferAvaialbleinList(nwInfoForCards?.networkListForDCCards, scheme)
//        }
//
//        when (Pair(isBankAvailable,isNetworkAvailable)) {
//            Pair(true,true) -> {
//                return bankFound && networkFound
//            }
//            Pair(true,false) -> {
//            return bankFound
//            }
//            Pair(false,true) -> {
//            return networkFound
//            }
//            Pair(false,false) -> {
//            return false
//            }
//        }
//
        return false

    }

    private fun isOfferAvailableInList(
        list: ArrayList<PaymentOptionOfferinfo>?,
        bankCode: String?
    ): Boolean {
        list?.forEach {
            if (bankCode.equals(it.paymentCode))
                return true
        }
        return false
    }

    internal fun <T> getValueFromPaymentOption(key: String, map: HashMap<String, Any?>?): T? {
        var value: T? = null
        if (map != null
            && map.containsKey(key)
            && (map[key] as? T) != null
        ) {
            value = map[key] as T
        }
        return value
    }

    fun storeGlobalVaultUserToken(
        context: Context,
        userTokenGlobalVault: String,
        vaultVerifiedPhoneNumber: String
    ) {
        addStringToSharedPreference(
            context,
            GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
            SdkUiConstants.USER_TOKEN,
            userTokenGlobalVault
        )
        addStringToSharedPreference(
            context,
            GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
            SdkUiConstants.MOBILE_NUMBER,
            vaultVerifiedPhoneNumber
        )
    }

    fun getGlobalVaultStoredUserToken(context: Context): Pair<String?, String?> {
        try {
            return Pair(
                getStringFromSharedPreference(
                    context,
                    GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
                    SdkUiConstants.USER_TOKEN,
                    "",
                    GLOBAL_VAULT_USER_TOKEN_PREF
                ),
                getStringFromSharedPreference(
                    context,
                    GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
                    SdkUiConstants.MOBILE_NUMBER,
                    "",
                    GLOBAL_VAULT_USER_TOKEN_PREF
                )
            )
        } catch (e: Exception) {
            clearSharedPref(context, GLOBAL_VAULT_USER_TOKEN_PREF_FILE)
        }
        return Pair("", "")
    }

    fun storeUserConsent(
        context: Context,
        userConsent: Boolean
    ) {
        InternalConfig.isUserConsentAvailableForPersonalisedOffers = userConsent
        addStringToSharedPreference(
            context,
            GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
            SdkUiConstants.USER_CONSENT,
            userConsent.toString()
        )
    }

    fun getUserConsent(context: Context): Boolean? {
        try {
            val value = getStringFromSharedPreference(
                context,
                GLOBAL_VAULT_USER_TOKEN_PREF_FILE,
                SdkUiConstants.USER_CONSENT,
                "",
                GLOBAL_VAULT_USER_TOKEN_PREF
            )
            return if(value?.isNotEmpty() == true) value.toBoolean() else null
        } catch (e: Exception) {
            clearSharedPref(context, GLOBAL_VAULT_USER_TOKEN_PREF_FILE)
        }
        return null
    }

    fun clearSharedPref(context: Context, prefKey: String) {
        val sharedPreference = getSharedPref(context, prefKey)
        sharedPreference?.edit()?.clear()?.apply()
        if (prefMap?.contains(prefKey) == true)
            prefMap?.remove(prefKey)
    }


    /**
     * Function to create or fetch a Master Key from the Android Keystore,
     * and use it to initialize/open a EncryptedSharedPreferences instance
     */

    private fun getSharedPref(
        context: Context,
        fileName: String
    ): SharedPreferences? {
        getKeys(context)
        if (prefMap == null)
            prefMap = HashMap()
        if (prefMap?.contains(fileName) == true)
            return prefMap?.get(fileName)
        return try {
            context.getSharedPreferences(fileName, MODE_PRIVATE).also {
                prefMap?.put(fileName, it)
            }
        } catch (e: Exception) {
            Log.d(TAG, "getEncryptedSharedPreference Exception = $e")
            AnalyticsUtils.logEventNameForKibana(
                context,
                AnalyticsConstant.CP_GV_CREDENTIAL_EXCEPTION,
                e.message
            );
            null
        }
    }

    private fun addStringToSharedPreference(
        context: Context,
        spFileName: String,
        spKey: String,
        spValue: String
    ) {
        val sharedPref: SharedPreferences? = getSharedPref(context, spFileName)
        val editor = sharedPref?.edit()
        editor?.putString(
            spKey, encrypt(
                context,
                spValue
            )
        )
        editor?.apply()
        editor?.commit()
    }

    private fun getStringFromSharedPreference(
        context: Context,
        spFileName: String,
        spKey: String,
        defaultValue: String,
        spFileNameOld: String
    ): String? {
        try {
            var decodedString: String? = null
            val sharedPref: SharedPreferences? =
                getSharedPref(context, spFileName)
            val encryptedString = sharedPref?.getString(spKey, defaultValue)
            decodedString = encryptedString?.let {
                decrypt(
                    context,
                    it
                )
            }
            if (decodedString.isNullOrBlank()) {
                val sharedPreference: SharedPreferences? =
                    getSharedPref(context, spFileNameOld)
                val encryptedValue = sharedPreference?.getString(spKey, defaultValue)
                decodedString = encryptedValue?.let {
                    decryptData(
                        context,
                        it
                    )
                }
            }
            if (decodedString != null) {
                addStringToSharedPreference(context, spFileName, spKey, decodedString)
            }
            return decodedString ?: defaultValue
        } catch (e: java.lang.Exception) {
            clearSharedPref(context, spFileName)
            return defaultValue
        }
    }

    fun storeGaidToken(
        context: Context,
        gaidToken: String
    ) {
        addStringToSharedPreference(context, GAID_TOKEN_PREF, SdkUiConstants.GAID, gaidToken)
    }

    fun getGaidToken(context: Context): String? {
        try {
            return getStringFromSharedPreference(context, GAID_TOKEN_PREF_FILE, SdkUiConstants.GAID, "", GAID_TOKEN_PREF)
        } catch (e: Exception) {
            clearSharedPref(context, GAID_TOKEN_PREF)
        }
        return null
    }


    fun getCategoryForOffer(category: String?): String? {
        return when (category?.uppercase()) {
            SdkUiConstants.CP_CC -> SdkUiConstants.CP_CREDITCARD
            SdkUiConstants.CP_DC -> SdkUiConstants.CP_DEBITCARD
            SdkUiConstants.CP_NB -> SdkUiConstants.CP_NET_BANKING
            SdkUiConstants.CP_WALLET.uppercase(Locale.ROOT), SdkUiConstants.CP_CASH, SdkUiConstants.CP_CASH_CARD -> SdkUiConstants.CP_WALLET.uppercase(
                Locale.ROOT
            )

            SdkUiConstants.CP_UPI, SdkUiConstants.CP_UPI_INTENT, SdkUiConstants.CP_EMI, SdkUiConstants.CP_BNPL -> category.uppercase(
                Locale.ROOT
            )

            else -> category
        }
    }

    fun getTimeDifferenceInSeconds(apiCalledTime: Long, currentTime: Long) = "${
        TimeUnit.MILLISECONDS.toSeconds(
            currentTime - apiCalledTime
        )
    }"

    /**
     * Funtion which returns Phone Number label based on PaymentType
     */
    fun phoneNumberLabel(paymentType: PaymentType?): String {
        Log.d(TAG, "PaymentType =" + paymentType)
        return when (paymentType) {
            PaymentType.BNPL -> SdkUiConstants.MOBILE_NUMBER_REGISTERED
            else -> SdkUiConstants.PHONE_NUMBER
        }
    }

    /**
     * Function will hide the keyboard
     * @param view EditText
     */
    fun hideKeyboard(view: View) {
        view.clearFocus()
        val imm: InputMethodManager = view.context
            .getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
        imm.hideSoftInputFromWindow(view.windowToken, 0)
    }

    /**
     * Function will show the keyboard and focus on editText
     * @param view EditText
     */
    fun displayKeyboard(view: View) {
        view.requestFocus()
        val imm = view.context
            .getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
        imm.showSoftInput(view, InputMethodManager.SHOW_IMPLICIT)
    }

    /**
     * Function will return the string id of error if detail was invalid
     * @param apiResponse [ApiResponse]
     * @param paymentType [PaymentType]
     *
     */
    fun getEligibilityDetails(apiResponse: ApiResponse, paymentType: PaymentType?): Int? {
        when (paymentType) {
            PaymentType.EMI -> {
                if (!(apiResponse.paymentOptionList?.get(0) as EMIOption).isEligible) {
                    return R.string.payu_mobile_not_eligible_error
                }
            }

            PaymentType.UPI -> {
                if (apiResponse.status == true) {
                    return R.string.payu_vpa_supported_text
                }
            }

            PaymentType.WALLET -> {
                if (apiResponse.status == false) {
                    return -1
                }
            }

            PaymentType.BNPL -> {
                val bnplOption = apiResponse.paymentOptionList?.get(0) as? BnplOption
                if (bnplOption?.isEligible == false) {
                    return R.string.payu_mobile_not_eligible_error
                }
            }

            else -> return null
        }
        return null
    }


    fun getTopBanks(): ArrayList<String> {
        return arrayListOf(
            SdkUiConstants.CP_SBIB, SdkUiConstants.CP_SBINENCC,
            SdkUiConstants.CP_HDFB, SdkUiConstants.CP_HDFCENCC,
            SdkUiConstants.CP_ICIB, SdkUiConstants.CP_ICICENCC,
            SdkUiConstants.CP_AXIB, SdkUiConstants.CP_UTIBENCC,
            SdkUiConstants.CP_PNBB, SdkUiConstants.CP_PUNBENCC,
            SdkUiConstants.CP_162B, SdkUiConstants.CP_KKBKENCC,
            SdkUiConstants.CP_AIRNB, SdkUiConstants.CP_AIRPENCC,
            SdkUiConstants.CP_BBRB, SdkUiConstants.CP_BARBENCC
        )
    }

    fun getBankNameByBankCode(bankCode: String): String {
        return when (bankCode) {
            SdkUiConstants.CP_SBIB, SdkUiConstants.CP_SBINENCC -> SdkUiConstants.UI_SBI
            SdkUiConstants.CP_HDFB, SdkUiConstants.CP_HDFCENCC -> SdkUiConstants.UI_HDFC
            SdkUiConstants.CP_ICIB, SdkUiConstants.CP_ICICENCC -> SdkUiConstants.UI_ICICI
            SdkUiConstants.CP_AXIB, SdkUiConstants.CP_UTIBENCC -> SdkUiConstants.UI_AXIS
            SdkUiConstants.CP_PNBB, SdkUiConstants.CP_PUNBENCC -> SdkUiConstants.UI_PNB
            SdkUiConstants.CP_162B, SdkUiConstants.CP_KKBKENCC -> SdkUiConstants.UI_KOTAK
            SdkUiConstants.CP_AIRNB, SdkUiConstants.CP_AIRPENCC -> SdkUiConstants.UI_AIRNB
            SdkUiConstants.CP_BBRB, SdkUiConstants.CP_BARBENCC -> SdkUiConstants.UI_BBRB
            else -> bankCode
        }
    }

    fun isOffersInEmi(emiOption: EMIOption): Boolean {
        return InternalConfig.userSelectedOfferInfo != null && emiOption.isOfferAvailable && InternalConfig.userSelectedOfferInfo?.isNoCostEmi == false
    }

    fun getUpiAnalyticsString(context: Context): String {
        val deviceTotalMemory = AnalyticsUtils.getTotalMemoryInfo(context)
        val availMemory = AnalyticsUtils.getAvailableMemoryInfo(context)
        val deviceName = Build.MANUFACTURER + " " + Build.MODEL + "/"

        var analyticsString = SdkUiConstants.ANDROID + "/"
        analyticsString += "" + Build.VERSION.SDK_INT + "/"
        analyticsString += deviceName
        analyticsString += SdkUiConstants.TOTAL + " $deviceTotalMemory " + SdkUiConstants.MB + "/"
        analyticsString += SdkUiConstants.FREE + " $availMemory " + SdkUiConstants.MB + "/"

        if (isAppNameInEnglish(context.getString(R.string.app_name))) {
            analyticsString += SdkUiConstants.MERCHANT_APP_NAME + ": " + context.getString(R.string.app_name) + "/"
        }

        analyticsString += getAppVersion(context)
        return analyticsString
    }

    private fun isAppNameInEnglish(appName: String?): Boolean {
        val appNameRegex = "^[ a-zA-Z0-9_-]*$".toRegex()
        return (!appName.isNullOrEmpty() && appName.matches(appNameRegex))
    }

    internal fun getAppVersion(
        context: Context,
    ): String? {
        return try {
            val packageManager = context.packageManager
            return packageManager.getPackageInfo(context.packageName, 0).versionName
        } catch (e: Exception) {
            ""
        }
    }

    internal fun upiTopAppsDescending(): MutableList<String> {
        val upiTopAppsList = mutableListOf<String>()
        upiTopAppsList.add(SdkUiConstants.CRED)
        upiTopAppsList.add(SdkUiConstants.AMAZON)
        upiTopAppsList.add(SdkUiConstants.CP_PHONE_PE)
        upiTopAppsList.add(SdkUiConstants.CP_PAYTM)
        upiTopAppsList.add(SdkUiConstants.GPAY)
        return upiTopAppsList
    }

    /**
     *  Function to prepare list of top priority upi apps from existing upi apps in user's list
     *  intentUpiAppList is the existing upi intent app list for user
     */
    fun prepareTopUpiAppList(intentUpiAppList: ArrayList<PaymentOption>?): ArrayList<PaymentOption> {

        val topUpiAppsList = ArrayList<PaymentOption>()

        intentUpiAppList?.forEach {
            if (topUpiAppsList.size < 3) {
                topUpiAppsList.add(it)
            }
        }

        val otherPaymentOption = PaymentOption()
        otherPaymentOption.bankName = SdkUiConstants.OTHERS
        otherPaymentOption.paymentType = PaymentType.UPI

        topUpiAppsList.add(otherPaymentOption)
        return topUpiAppsList
    }

    internal fun getTopBankList(): ArrayList<PaymentOption> {
        return OfferFilterManager.filterPaymentOption(PaymentType.NB, topBankList) ?: arrayListOf()
    }

    internal fun getOtherBanksList(): ArrayList<PaymentOption> {
        return OfferFilterManager.filterPaymentOption(PaymentType.NB, otherBanksList)
            ?: arrayListOf()
    }

    internal fun getUpiIntentList(): ArrayList<PaymentOption> {
        return OfferFilterManager.filterPaymentOption(PaymentType.UPI, upiIntentList)
            ?: arrayListOf()
    }

    internal fun filterBankList(banksFilteredList: ArrayList<PaymentOption>) {
        otherBanksList.clear()
        topBankList.clear()

        for (item in banksFilteredList) {
            if (!getTopBanks().contains(
                    (item.otherParams as? HashMap<*, *>)?.get(SdkUiConstants.CP_BANK_CODE)
                ) || topBankList.size == 6
            ) {
                otherBanksList.add(item)
            } else {
                topBankList.add(item)
            }
        }
    }

    internal fun isSiTxn(): Boolean {
        return SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams != null
    }

    private fun isOfferAvailableForPaymentCode(
        paymentInfoList: ArrayList<PaymentOptionOfferinfo>?,
        bankCode: String
    ): Boolean {
        if (!paymentInfoList.isNullOrEmpty()) {
            for (list in paymentInfoList) {
                if (bankCode.lowercase(Locale.ROOT) == list.paymentCode.lowercase(Locale.ROOT)
                )
                    return true
            }
        }
        return false
    }

    internal fun clonePayUPaymentParams(params: PayUPaymentParams): PayUPaymentParams.Builder {
        val paramClass: Class<*> = params.javaClass
        val fields: Array<Field> = paramClass.declaredFields
        val builder = PayUPaymentParams.Builder()
        val clonedClass: Class<*> = builder.javaClass
        for (field in fields) {
            if (!(field.isSynthetic)) {
                try {
                    field.isAccessible = true
                    val value: Any? = field.get(params)

                    val newField = clonedClass.getDeclaredField(field.name)
                    newField.isAccessible = true
                    newField.set(builder, value)

                    field.isAccessible = false
                    newField.isAccessible = false

                } catch (e1: IllegalArgumentException) {
                    Log.d(Utils.javaClass.simpleName, "IllegalArgumentEx " + e1.toString())
                } catch (e1: IllegalAccessException) {
                    Log.d(Utils.javaClass.simpleName, "IllegalAccessEx " + e1.toString())
                }
            }
        }
        return builder
    }

    internal fun isOfferAvailableInPaymentType(
        paymentType: PaymentType,
        isNewCard: Boolean = true
    ): Boolean {
        return when (paymentType) {
            PaymentType.CARD -> InternalConfig.userSelectedOfferInfo?.cardsOfferInfo != null && isNewCard
            PaymentType.NB -> !InternalConfig.userSelectedOfferInfo?.nbOptionOfferInfoList.isNullOrEmpty()
            PaymentType.WALLET, PaymentType.L1_OPTION -> !InternalConfig.userSelectedOfferInfo?.walletOptionOfferInfoList.isNullOrEmpty()
            PaymentType.BNPL -> !InternalConfig.userSelectedOfferInfo?.bnplOptionOfferInfoList.isNullOrEmpty()
            PaymentType.EMI -> InternalConfig.userSelectedOfferInfo?.emiOfferInfo != null && InternalConfig.userSelectedOfferInfo?.isNoCostEmi == false
            PaymentType.UPI, PaymentType.UPI_INTENT -> !InternalConfig.userSelectedOfferInfo?.upiOptionOfferInfoList.isNullOrEmpty()
            else -> false
        }
    }

    fun splitCardNumber(cardNumber: String): String {
        // From xxxxxxxxxxxx9528  To   xxxx xxxx xxxx 9528
        return cardNumber.chunked(4).joinToString(" ")
    }

    fun getWizRocketID(): Triple<String, String, String> {
        return Triple(SdkUiConstants.CTID1, SdkUiConstants.CTID2, SdkUiConstants.CTID3)
    }

    fun getWizRocketPass(): Triple<String, String, String> {
        return Triple(SdkUiConstants.CTPASS1, SdkUiConstants.CTPASS2, SdkUiConstants.CTPASS3)
    }

    private fun createKeys(context: Context?): SecretKey? {
        try {
            val kpGenerator = KeyGenerator
                .getInstance(
                    ENCRYPTION_ALGORITHM,
                    KEYSTORE_PROVIDER_ANDROID_KEYSTORE
                )

            val spec = if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
                if (context == null) {
                    return null
                }
                KeyPairGeneratorSpec.Builder(context)
                    .setAlias(mAlias)
                    .setKeySize(2048)
                    .build()
            } else {
                KeyGenParameterSpec.Builder(
                    mAlias,
                    KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
                )
                    .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                    .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_PKCS7)
                    .setKeySize(256)
                    .build()
            }
            kpGenerator.init(spec)
            return kpGenerator.generateKey()
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "createKey:Exception" + e.localizedMessage)
        }
        return null
    }

    private fun getKeys(context: Context?): SecretKey? {
        return try {
            val ks = KeyStore.getInstance(KEYSTORE_PROVIDER_ANDROID_KEYSTORE)
            ks.load(null)
            if (ks.containsAlias(mAlias)) {
                // Key exists, retrieve it
                val existingKey = ks.getEntry(mAlias, null) as? KeyStore.SecretKeyEntry
                if (existingKey?.secretKey != null) {
                    Log.i(this.javaClass.simpleName, "Using existing key")
                    existingKey.secretKey
                } else {
                    // Entry exists but key is null (rare case)
                    Log.w(this.javaClass.simpleName, "Key entry exists but key is null, creating new key")
                    createKeys(context)
                }
            } else {
                Log.i(this.javaClass.simpleName, "Key doesn't exist, creating new key")
                createKeys(context)
            }
        } catch (e: Exception) {
            createKeys(context)
        }
    }

    private const val GCM_IV_LENGTH = 12 // 12 bytes is recommended for GCM
    private const val GCM_TAG_LENGTH = 128 // 128-bit authentication tag

    private fun getSecretKey(context: Context): SecretKey? {
        try {
            val keyStore = KeyStore.getInstance(KEYSTORE_PROVIDER_ANDROID_KEYSTORE)
            keyStore.load(null)
            
            // Check if key already exists
            if (keyStore.containsAlias(PAYU_ENCRYPTION_KEY_GCM)) {
                val entry = keyStore.getEntry(PAYU_ENCRYPTION_KEY_GCM, null) as? KeyStore.SecretKeyEntry
                if (entry != null) {
                    return entry.secretKey
                }
            }
            
            // Generate a new key if it doesn't exist
            val keyGenerator = KeyGenerator.getInstance(
                KeyProperties.KEY_ALGORITHM_AES,
                KEYSTORE_PROVIDER_ANDROID_KEYSTORE
            )
            
            val builder = KeyGenParameterSpec.Builder(
                "payu_encryption_key_gcm",
                KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
            )
            
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                builder.apply {
                    setBlockModes(KeyProperties.BLOCK_MODE_GCM)
                    setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
                    setRandomizedEncryptionRequired(true)
                    setKeySize(256)
                }
                
                keyGenerator.init(builder.build())
                return keyGenerator.generateKey()
            } else {
                // For API < 23, we'll use a different approach since GCM requires API 23+
                Log.d("EncryptionUtils", "GCM mode requires API 23+, falling back to legacy encryption")
                return getKeys(context)
            }
        } catch (e: Exception) {
            Log.d("EncryptionUtils", "Error getting secret key", e)
            return null
        }
    }

    private fun encrypt(context: Context, data: String): String? {
        return try {
            getSecretKey(context)?.let { secretKey ->
                // Initialize cipher with GCM mode
                val cipher = Cipher.getInstance(CIPHER_TRANSFORMATION)
                
                // Initialize without IV - let the cipher generate a secure random one
                cipher.init(Cipher.ENCRYPT_MODE, secretKey)
                
                // Get the IV that was automatically generated
                val iv = cipher.iv
                
                // Encrypt the data
                val encrypted = cipher.doFinal(data.toByteArray(Charsets.UTF_8))
                
                // Combine IV + encrypted data (which includes the authentication tag)
                val combined = ByteArray(iv.size + encrypted.size)
                System.arraycopy(iv, 0, combined, 0, iv.size)
                System.arraycopy(encrypted, 0, combined, iv.size, encrypted.size)
                
                // Return as Base64 encoded string
                Base64.encodeToString(combined, Base64.NO_WRAP)
            } ?: run {
                Log.e("EncryptionUtils", "Failed to get secret key")
                null
            }
        } catch (e: Exception) {
            Log.e("EncryptionUtils", "Encryption error", e)
            null
        }
    }

    private fun decrypt(context: Context, encryptedData: String): String? {
        if (encryptedData.isBlank()) {
            Log.d("EncryptionUtils", "Empty encrypted data")
            return null
        }
        Log.d("EncryptionUtils", "encrypted data $encryptedData")
        return try {
            getSecretKey(context)?.let { secretKey ->
                // Decode the Base64 string
                val encryptedBytes = Base64.decode(encryptedData, Base64.NO_WRAP)
                
                // For GCM, we need at least 12 bytes IV + 16 bytes tag + some encrypted data
                if (encryptedBytes.size <= GCM_IV_LENGTH + 16) {
                    Log.d("EncryptionUtils", "Invalid encrypted data format: too short")
                    return null
                }
                
                // Extract the IV (first 12 bytes for GCM)
                val iv = encryptedBytes.copyOfRange(0, GCM_IV_LENGTH)
                
                // The rest is the actual encrypted data + authentication tag
                val encrypted = encryptedBytes.copyOfRange(GCM_IV_LENGTH, encryptedBytes.size)
                
                // Initialize the cipher for decryption with GCM mode
                val cipher = Cipher.getInstance("AES/GCM/NoPadding")
                val spec = GCMParameterSpec(GCM_TAG_LENGTH, iv)
                cipher.init(Cipher.DECRYPT_MODE, secretKey, spec)
                
                // Decrypt the data (which includes the authentication tag)
                val decrypted = cipher.doFinal(encrypted)
                String(decrypted, Charsets.UTF_8)
            } ?: run {
                Log.d("EncryptionUtils", "Failed to get secret key for decryption")
                null
            }
        } catch (e: Exception) {
            Log.e("EncryptionUtils", "Decryption error: ${e.message}", e)
            null
        }
    }

    private fun decryptData(context: Context, encryptedData: String): String? {
        return try {
            getKeys(context)?.let {
                val ex = encryptedData.replace("\n", "");
                val cipherText: ByteArray = Base64.decode(ex, Base64.DEFAULT)
                val cipher =
                    Cipher.getInstance(CIPHER_TRANSFORMATION1 + CIPHER_TRANSFORMATION2 + CIPHER_TRANSFORMATION3)
                val iv = cipherText.copyOfRange(0, cipher.blockSize)
                cipher.init(Cipher.DECRYPT_MODE, it, IvParameterSpec(iv))
                val data = cipherText.copyOfRange(cipher.blockSize, cipherText.size)
                String(cipher.doFinal(data))
            }
        } catch (e: java.lang.Exception) {
            Log.d(this.javaClass.simpleName, "decrypt:Exception" + e.localizedMessage)
            null
        }
    }


    internal fun getMaskedAccountNoForTpv(
        resId: Int,
        context: Context,
        accountNumber: String
    ): String {
        return if (accountNumber.isEmpty()) "" else context.getString(
            resId,
            getLastFourDigitFromAccount(accountNumber)
        )

    }

    internal fun getLastFourDigitFromAccount(accountNumber: String): String {
        return accountNumber.takeLast(4).ifEmpty { accountNumber }
    }

    internal fun String.removeRupeeSymbolFromPrefix(): String{
        return this.replace("₹", "")
    }

    fun getBankCodeFromMap(map: HashMap<String, Any>): String {
        return map[SdkUiConstants.CP_BANK_CODE] as? String ?: ""
    }

    internal fun isValidMPin(mPin: String): Boolean {
        return (mPin.matches(Regex(SdkUiConstants.CP_MPIN_REGEX)))
    }

    internal fun getMaskedMobileNumber(
        context: Context,
        mobileNumber: String
    ): String {
        return if (mobileNumber.isEmpty()) "" else context.getString(
            R.string.payu_masked_mobile_string,
            getLastTwoDigitFromNumber(mobileNumber)
        )
    }

    internal fun getLastTwoDigitFromNumber(mobileNumber: String): String {
        return mobileNumber.takeLast(2).ifEmpty { mobileNumber }
    }
    internal fun String?.addRupeeSymbolInPrefix(): String?{
        return this?.let { "₹ $it" }
    }

    internal fun String.toDoubleOrZero(): Double {
        return try {
            this.toDouble()
        } catch (e: NumberFormatException) {
            0.0
        }
    }

    internal fun Double.toStringFormatted(): String {
        return String.format("%.2f", this) // Formats to 2 decimal places
    }

    private fun saveAndGetAndroidId(context: Context): String {
        // Generate a UUID
        val uniqueID = UUID.randomUUID().toString()
      // Store it securely using EncryptedSharedPreferences
        addStringToSharedPreference(
            context,
            CP_ANDROID_ID_PREF,
            SdkUiConstants.CP_ANDROID_ID,
            uniqueID
        )
        return uniqueID;
    }

    fun getAndroidId(context: Context): String {
        val androidId = getStringFromSharedPreference(
            context,
            CP_ANDROID_ID_PREF_FILE,
            SdkUiConstants.CP_ANDROID_ID,
            "",
            CP_ANDROID_ID_PREF
        )
        return if (androidId.isNullOrBlank().not()) androidId
            ?: saveAndGetAndroidId(context) else saveAndGetAndroidId(context)
    }

}