package com.payu.ui.model.adapters.viewholders

import android.view.View
import android.widget.TextView
import androidx.core.content.ContextCompat
import com.payu.base.listeners.OnFetchImageListener
import com.payu.base.models.ImageDetails
import com.payu.base.models.ImageParam
import com.payu.base.models.InternalConfig
import com.payu.base.models.PaymentMode
import com.payu.base.models.PaymentType
import com.payu.base.models.SodexoCardOption
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.utils.ImageViewUtils
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils
import com.payu.ui.model.utils.ViewUtils

class ClosedLoopWalletViewHolder(itemView: View) : QuickOptionViewHolder(itemView) {
    private val tvWalletBalance: TextView = itemView.findViewById(R.id.tvWalletBalance)
    private val tvErrorButton: TextView = itemView.findViewById(R.id.tvErrorButton)
    private val tvLowBalance: TextView = itemView.findViewById(R.id.tvLowBalance)
    private val tvCvvText: TextView = itemView.findViewById(R.id.tv_cvv_text)

    fun bind(paymentMode: PaymentMode, selectedPosition: Int) {
        val clwCardOption = paymentMode.optionDetail?.get(0) as? SodexoCardOption
        tvPaymentOptionName.text = paymentMode.name
        tvCvvText.visibility = View.GONE
        setPaymentIcon()
        updateSelectedItem(paymentMode, selectedPosition)
        val bankCode = Utils.getValueFromPaymentOption<String>(
            SdkUiConstants.CP_BANK_CODE,
            clwCardOption?.otherParams as? HashMap<String, Any?>
        )

        when {
            paymentMode.isBankDown -> showBankDownViewForClosedLoopWallet()
            paymentMode.isOfferValid && InternalConfig.selectedOfferInfo != null -> {
                showOfferView(true)
            }

            bankCode?.let {
                paymentMode.type?.let { paymentType ->
                    Utils.isOfferAvailable(
                        it,
                        paymentType
                    )
                }
            } == true && InternalConfig.selectedOfferInfo != null -> {
                showOfferView()
            }

            else -> showDefaultView()
        }

        clwCardOption?.let { clwOption ->
            updateFetchStatusBasedText(clwOption.fetchedStatus, clwOption)
            if (paymentMode.type == PaymentType.SODEXO) {
                val param = ImageParam(clwOption, false, R.drawable.payu_cards_placeholder)
                SdkUiInitializer.apiLayer?.getImageForPaymentOption(param,
                    object : OnFetchImageListener {
                        override fun onImageGenerated(result: ImageDetails) {
                            ImageViewUtils.setImage(ivPaymentOptionIcon, result)
                        }
                    })
            }
        }

        rlQuickOptions.setOnClickListener {
            if (clwCardOption?.fetchedStatus != 1) showFetchingTextForClw(tvWalletBalance)
            onItemClickListener?.invoke(
                paymentMode,
                absoluteAdapterPosition,
                clwCardOption?.fetchedStatus == 1
            )
        }

        btnProceedToPay.setOnClickListener {
            clwCardOption?.let { clwOption -> makePayment(btnProceedToPay.context, clwOption) }
        }
    }

    private fun showBankDownViewForClosedLoopWallet() {
        ViewUtils.disableViews(listOf(ivPaymentOptionIcon, btnProceedToPay))
    }

    private fun showFetchingTextForClw(
        tvPaymentOptionDetail: TextView,
    ) {
        tvPaymentOptionDetail.setTextColor(
            ContextCompat.getColor(
                tvPaymentOptionDetail.context,
                R.color.payu_color_7b7b7b
            )
        )
        tvPaymentOptionDetail.text =
            tvPaymentOptionDetail.context?.getString(R.string.payu_fetching_details)
    }

    private fun updateFetchStatusBasedText(fetchedStatus: Int, sodexoCardOption: SodexoCardOption) {
        tvWalletBalance.visibility = View.VISIBLE
        tvWalletBalance.setTextColor(
            ContextCompat.getColor(
                tvWalletBalance.context,
                R.color.payu_color_7b7b7b
            )
        )
        when (fetchedStatus) {
            -1 -> {
                tvWalletBalance.text =
                    tvWalletBalance.context.getString(R.string.payu_fetching_details)
            }

            0 -> {
                tvWalletBalance.text =
                    tvWalletBalance.context?.getString(R.string.payu_tap_to_try_again_clw)
            }

            else -> updateBalanceView(sodexoCardOption)
        }
    }

    private fun lowBalanceView() {
        tvLowBalance.visibility = View.VISIBLE
        ViewUtils.disableViews(
            listOf(
                rlQuickOptions,
                ivPaymentOptionIcon,
                tvPaymentOptionName
            )
        )
    }

    private fun updateBalanceView(sodexoCardOption: SodexoCardOption) {
        val amount = SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDouble() ?: 0.0
        val balance = sodexoCardOption.balance.toDoubleOrNull() ?: 0.0
        when (sodexoCardOption.paymentType) {
            PaymentType.CLOSED_LOOP_WALLET -> {
                if (balance <= amount) {
                    tvErrorButton.visibility = View.VISIBLE
                    btnProceedToPay.text =
                        btnProceedToPay.context?.getString(R.string.payu_recharge_and_pay)
                } else btnProceedToPay.text =
                    btnProceedToPay.context?.getString(R.string.payu_pay_now)
            }

            else -> {
                if (balance <= amount) {
                    lowBalanceView()
                }
            }
        }
        tvWalletBalance.text = tvWalletBalance.context.getString(R.string.payu_balance, balance)
    }

    private fun setPaymentIcon() {
        if (!InternalConfig.logo.isNullOrEmpty()) {
            SdkUiInitializer.apiLayer?.getBitmapImageFormURL(InternalConfig.logo!!, object :
                OnFetchImageListener {
                override fun onImageGenerated(result: ImageDetails) {
                    ImageViewUtils.setImage(ivPaymentOptionIcon, result)
                }
            })
        } else {
            SdkUiInitializer.apiLayer?.config?.merchantLogo?.let { merchantLogo ->
                ivPaymentOptionIcon.context?.let {
                    ivPaymentOptionIcon.setImageDrawable(
                        ContextCompat.getDrawable(
                            it, merchantLogo
                        )
                    )
                }
            }
        }
    }

}