package com.payu.ui.viewmodel

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import com.payu.base.listeners.OnFetchImageListener
import com.payu.base.listeners.OnIFSCDetailsListener
import com.payu.base.models.*
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.utils.*

class NBDetailsViewModel(application: Application, mParam: Map<String, Any>) :
    AndroidViewModel(application), OnIFSCDetailsListener {

    var paymentOption: PaymentOption? = null
    var paymentType: PaymentType? = null
    var paymentFlowState: PaymentFlowState? = null
    val bankImage: MutableLiveData<ImageDetails> = MutableLiveData()
    val showKeyboard: MutableLiveData<Boolean> = MutableLiveData()
    val showLoaderInEditText: MutableLiveData<Boolean> = MutableLiveData()
    val showAccountTypeBottomSheet: MutableLiveData<Int> = MutableLiveData()
    val showVerificationModeBottomSheet: MutableLiveData<Int> = MutableLiveData()
    var applicationContext = application
    val enablePayButton: MutableLiveData<Boolean> = MutableLiveData()
    val hideBottomSheet: MutableLiveData<Boolean> = MutableLiveData()
    val ifscCodeError: MutableLiveData<String?> = MutableLiveData()
    val retryIfscDetails: MutableLiveData<Event<Boolean>> = MutableLiveData()
    val ifscDetailsText: MutableLiveData<String> = MutableLiveData()
    val isSIMode: MutableLiveData<Boolean> = MutableLiveData()
    val shouldShowVerificationModeOption: MutableLiveData<Boolean> = MutableLiveData()
    val accountNumberLabelText: MutableLiveData<String> = MutableLiveData()
    var siHeaderSummary: MutableLiveData<String> = MutableLiveData()
    val verificationModeLabelText: MutableLiveData<String> = MutableLiveData()
    lateinit var verificationModeList: List<PayUBeneficiaryVerificationMode>
    val disableVerificationMode: MutableLiveData<Event<Boolean>> = MutableLiveData()

    var isValidIfsc: Boolean = false
    var isValidName: Boolean = false
    var isValidAccountType: Boolean = false
    var isValidVerificationMode: Boolean = false
    var isValidAccountNumber: Boolean = false
    var ifscCode: String = ""

    val accountNumberMinLengthError: MutableLiveData<Boolean> = MutableLiveData()
    val accountNumberFieldColor: MutableLiveData<Any> = MutableLiveData()
    val accountNameFieldColor: MutableLiveData<Any> = MutableLiveData()
    val ifscFieldColor: MutableLiveData<Any> = MutableLiveData()
    val accountTypeFieldColor: MutableLiveData<Any> = MutableLiveData()
    val modeOfVerificationTypeFieldColor: MutableLiveData<Any> = MutableLiveData()
    var verificationModeSelected: PayUBeneficiaryVerificationMode? = null

    init {
        val paymentModel = mParam[SdkUiConstants.CP_PAYMENT_MODEL] as PaymentModel
        paymentOption = paymentModel.paymentOption
        paymentType = paymentOption?.paymentType
        paymentFlowState = paymentModel.paymentFlowState
        initLabelText(paymentOption)
        getBitmapByBankCode()
        showConsent()
        siHeaderSummary.value = Utils.getCustomeNoteDetails(
            PaymentType.NB,
            SdkUiInitializer.apiLayer?.config?.customNoteDetails
        )
        shouldShowVerificationModeOption(paymentOption)
    }

    fun shouldShowVerificationModeOption(paymentOption: PaymentOption?) {

        if (paymentOption?.verificationModeList.isNullOrEmpty()) {
            shouldShowVerificationModeOption.value = false
        } else {
            shouldShowVerificationModeOption.value = true
            verificationModeList = paymentOption?.verificationModeList!!
        }
    }

    private fun initLabelText(paymentOption: PaymentOption?) {
        paymentOption?.let {
            accountNumberLabelText.value =
                applicationContext.getString(R.string.payu_bank_account_number, it.bankName)
        }
    }

    fun showConsent() {
        val payUSIParams = getSIParams()
        isSIMode.value = payUSIParams != null
    }

    private fun getBitmapByBankCode() {
        val param = ImageParam(
            paymentOption!!,
            false,
            Utils.getDefaultDrawable(paymentOption?.paymentType)
        )
        SdkUiInitializer.apiLayer?.getImageForPaymentOption(
            param,
            onFetchImageListener = object : OnFetchImageListener {
                override fun onImageGenerated(result: ImageDetails) {
                    bankImage.value = result
                }
            })
    }

    fun accountNumberFocused(accountNumberFocused: Boolean) {
        if (accountNumberFocused) {
            showKeyboard.value = true
            accountNumberMinLengthError.value = false
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                accountNumberFieldColor.value = R.color.one_payu_colorPrimary
            else accountNumberFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
        } else {
            showKeyboard.value = false
            accountNumberFieldColor.value = R.color.payu_color_338f9dbd
        }
    }

    fun verificationModeSelection(
        mode: PayUBeneficiaryVerificationMode,
        modeOfVerificationFocused: Boolean = true
    ) {
        isValidVerificationMode = true
        modeOfVerificationTypeFocused(modeOfVerificationFocused)
        verificationModeSelected = mode
        verificationModeLabelText.value = mode.modeName
        (paymentOption as? NBOptions)?.beneficiaryDetails?.verificationMode = mode
        validateForm()
        bottomSheetCloseClicked()
    }

    fun showAccountType() {
        showKeyboard.value = false
        showAccountTypeBottomSheet.value = R.layout.account_type_bottom_sheet_layout
    }

    fun showModeOfVerificationType() {
        showKeyboard.value = false
        showVerificationModeBottomSheet.value =
            R.layout.mode_of_verification_type_bottom_sheet_layout
    }

    fun validateName(name: String?) {
        isValidName = if (name.isNullOrEmpty()) {
            false
        } else (name.trim().isNotEmpty())
    }

    fun validateAccountType(accountType: String?) {
        if (accountType?.isEmpty()!!) {
            isValidAccountType = false
        } else if (accountType.trim().isEmpty()) {
            isValidAccountType = false
        } else isValidAccountType =
            accountType.equals(
                PayUBeneficiaryAccountType.SAVINGS.getTypeName(),
                ignoreCase = true
            ) ||
                    accountType.equals(
                        PayUBeneficiaryAccountType.CURRENT.getTypeName(),
                        ignoreCase = true
                    )
    }

    fun validateForm() {
        if (!shouldShowVerificationModeOption.value!!) {
            isValidVerificationMode = true
        }
        enablePayButton.value =
            isValidName && isValidIfsc && isValidAccountType && isValidAccountNumber && isValidVerificationMode
    }

    fun makePayment(name: String, accountNumber: String, ifscCode: String, accountType: String) {

        val payUSIParams = getSIParams()
        val beneficiaryAccountType = Utils.getAccountType(accountType)

        if (payUSIParams != null) {
            val beneficiaryDetailBuilder =
                PayUBeneficiaryDetail.Builder().setBeneficiaryName(name.trim())
                    .setBeneficiaryAccountNumber(accountNumber.trim())
                    .setBeneficiaryIfsc(ifscCode.trim())
                    .setBeneficiaryAccountType(beneficiaryAccountType)
                    .setVerificationMode(verificationModeSelected)

            val beneficiaryDetail = beneficiaryDetailBuilder.build()
            payUSIParams.beneficiaryDetail = beneficiaryDetail

        }

        AnalyticsUtils.logMakePaymentEvent(
            applicationContext,
            paymentOption!!,
            verificationModeSelected?.modeName ?: ""
        )
        SdkUiInitializer.apiLayer?.updatePaymentState(
            Utils.getPaymentModel(
                paymentOption!!,
                paymentFlowState
            ),
            ViewUtils.getToolbar(
                applicationContext,
                paymentOption!!.additionalCharge,
                paymentOption?.paymentType
            )
        )
    }

    fun getSIParams(): PayUSIParams? {
        return SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams
    }

    fun bottomSheetCloseClicked() {
        hideBottomSheet.value = true
    }

    fun validateAccountNumberMinLength(accountNumber: String?, hasFocus: Boolean) {
        if (accountNumber.isNullOrEmpty()) {
            isValidAccountNumber = false
            if (!hasFocus) accountNumberFieldColor.value = R.color.payu_color_de350b
        } else if (accountNumber.length < 8) {
            isValidAccountNumber = false

            if (!hasFocus) {
                accountNumberMinLengthError.value = true
                isValidAccountNumber = false
                accountNumberFieldColor.value = R.color.payu_color_de350b
            }
        } else {
            accountNumberMinLengthError.value = false
            isValidAccountNumber = true
            if (hasFocus) {
                if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                    accountNumberFieldColor.value = R.color.one_payu_colorPrimary
                else accountNumberFieldColor.value =
                    SdkUiInitializer.apiLayer?.config?.primaryColor
            } else accountNumberFieldColor.value = R.color.payu_color_338f9dbd
        }
    }

    fun accountNameFocused(accountNameFocused: Boolean) {
        if (accountNameFocused) {
            showKeyboard.value = true
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                accountNameFieldColor.value = R.color.one_payu_colorPrimary
            else accountNameFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
        } else {
            showKeyboard.value = false
            if (isValidName) accountNameFieldColor.value = R.color.payu_color_338f9dbd
            else accountNameFieldColor.value = R.color.payu_color_de350b
        }
    }

    fun accountTypeFocused(accountTypeFocused: Boolean) {
        if (accountTypeFocused) {
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                accountTypeFieldColor.value = R.color.one_payu_colorPrimary
            else accountTypeFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
        } else {
            accountTypeFieldColor.value = R.color.payu_color_338f9dbd
        }
    }

    fun modeOfVerificationTypeFocused(modeOfVerificationFocused: Boolean) {
        if (modeOfVerificationFocused) {
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                modeOfVerificationTypeFieldColor.value = R.color.one_payu_colorPrimary
            else modeOfVerificationTypeFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
        } else {
            modeOfVerificationTypeFieldColor.value = R.color.payu_color_338f9dbd
        }
    }

    fun validateIfsc(ifscCode: String, isTpvFlow: Boolean = false) {
        this.retryIfscDetails.value = Event(false)
        this.ifscCode = ifscCode.trim()

        if (this.ifscCode.length < 11) {
            /**
             * If IFSC Code is being entered then the box should be highlighted and no error should be displayed.
             * If user entered 11 characters but were not valid then on deleting characters again the field should be
             * highlighted and error should not be displayed
             * */
            isValidIfsc = false
            if (isTpvFlow) {
                ifscCodeError.value = applicationContext.getString(R.string.payu_invalid_ifsc_code)
                ifscFieldColor.value = R.color.payu_color_de350b
            } else {
                ifscCodeError.value = null
                if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                    ifscFieldColor.value = R.color.one_payu_colorPrimary
                else ifscFieldColor.value =
                    SdkUiInitializer.apiLayer?.config?.primaryColor
            }
        } else if (Utils.isValidIfsc(this.ifscCode)) {
            ifscCodeError.value = null
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                ifscFieldColor.value = R.color.one_payu_colorPrimary
            else ifscFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
            SdkUiInitializer.apiLayer?.fetchIFSCDetails(ifscCode, this)
        } else {
            isValidIfsc = false
            ifscCodeError.value = applicationContext.getString(R.string.payu_invalid_ifsc_code)
            ifscFieldColor.value = R.color.payu_color_de350b
        }
    }

    override fun onIFSCDetailsReceived(ifscDetails: IFSCDetails) {
        if (this.ifscCode.equals(ifscDetails.ifsc, ignoreCase = true)) {
            isValidIfsc = true

            //Removes any field error for IFSC if displayed
            if (ifscFieldColor.value == R.color.payu_color_de350b)
                ifscFieldColor.value = R.color.payu_color_338f9dbd

            ifscDetailsText.value = "${ifscDetails.branch}, ${ifscDetails.city}"
            validateForm()
        }
    }

    override fun onError(errorResponse: ErrorResponse) {
        isValidIfsc = false
        ifscCodeError.value = errorResponse.errorMessage
        ifscFieldColor.value = R.color.payu_color_de350b
        retryIfscDetails.value = Event(errorResponse.errorCode != 400)
    }

    override fun showProgressDialog(showDialog: Boolean) {
        showLoaderInEditText.value = showDialog
    }
    
    fun ifscFocused(hasFocus: Boolean) {
        if (hasFocus) {
            if (!isValidIfsc) ifscCodeError.value = null
            if (SdkUiInitializer.apiLayer?.config?.primaryColor.isNullOrEmpty())
                ifscFieldColor.value = R.color.one_payu_colorPrimary
            else ifscFieldColor.value =
                SdkUiInitializer.apiLayer?.config?.primaryColor
        } else {
            if (isValidIfsc) ifscFieldColor.value = R.color.payu_color_338f9dbd
            else {
                ifscCodeError.value = applicationContext.getString(R.string.payu_invalid_ifsc_code)
                ifscFieldColor.value = R.color.payu_color_de350b
            }
        }
    }
}
