package com.payu.ui.model.adapters

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.isVisible
import androidx.fragment.app.FragmentManager
import androidx.recyclerview.widget.RecyclerView
import com.payu.base.listeners.OnFetchImageListener
import com.payu.base.models.BottomSheetType
import com.payu.base.models.ImageDetails
import com.payu.base.models.ImageParam
import com.payu.base.models.InternalConfig
import com.payu.base.models.PaymentFlowState
import com.payu.base.models.PaymentModel
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentState
import com.payu.base.models.PaymentType
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.listeners.BottomSheetValidateApiListener
import com.payu.ui.model.listeners.BottomSheetValidateResultListener
import com.payu.ui.model.listeners.OfferApplyListener
import com.payu.ui.model.listeners.OnBankAdapterListener
import com.payu.ui.model.listeners.ValidateOfferResultListener
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.ImageViewUtils
import com.payu.ui.model.utils.MultipleClickHandler
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils
import com.payu.ui.model.utils.ViewUtils
import com.payu.ui.view.customViews.BNPLAndUPIBottomSheet
import com.payu.ui.view.customViews.BankDownBottomSheet
import java.lang.ref.WeakReference

class TopBankAdapter(
    val paymentType: PaymentType,
    var banksList: ArrayList<PaymentOption>,
    var paymentState: PaymentState? = null,
    var childFragmentManager: FragmentManager,
    val offerApplyListener: OfferApplyListener,
    val onBankAdapterListener: OnBankAdapterListener?
) : RecyclerView.Adapter<TopBankAdapter.ViewHolder>() {

    private var TAG = javaClass.simpleName
    private var selectedPosition: Int = -1
    private var banksFilteredList: ArrayList<PaymentOption>
    private var isOfferValid: Boolean = false
    private var bankCode: String? = null
    // Storing in a weak reference to prevent leaking the FragmentManager
    private var fragmentManagerRef: WeakReference<FragmentManager> = WeakReference(childFragmentManager)
    // Tracking viewHolders for cleanup
    private val viewHolders = ArrayList<WeakReference<ViewHolder>>()

    init {
        banksFilteredList = banksList
    }

    override fun onCreateViewHolder(
        parent: ViewGroup,
        viewType: Int
    ): ViewHolder {
        val inflater = LayoutInflater.from(parent.context)
        val view: View = inflater.inflate(R.layout.top_bank_list_item, parent, false)
        val holder = ViewHolder(view)
        viewHolders.add(WeakReference(holder))
        return holder
    }

    override fun onBindViewHolder(holder: TopBankAdapter.ViewHolder, position: Int) {
        holder.bind(position)
    }

    override fun getItemCount(): Int {
        return banksFilteredList.size
    }

    inner class ViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView),
        View.OnClickListener, BottomSheetValidateApiListener {
        val ivPaymentOptionIcon: ImageView = itemView.findViewById(R.id.ivPaymentOptionIcon)
        val tvUseOffer: TextView = itemView.findViewById(R.id.tvUseOffer)
        val tvPaymentOptionName: TextView = itemView.findViewById(R.id.tvPaymentOptionName)
        val ivBankDown: ImageView = itemView.findViewById(R.id.ivBankDown)
        val clTopBankDetail: ConstraintLayout = itemView.findViewById(R.id.clTopBankDetail)
        // Don't initialize until needed
        var bottomSheet: BNPLAndUPIBottomSheet.Builder? = null
        var nbBottomSheet: BNPLAndUPIBottomSheet? = null
        // Track validate listener for cleanup
        private var validateOfferResultListener: ValidateOfferResultListener? = null

        init {
            ivBankDown.setOnClickListener {
                if (MultipleClickHandler.isSafeOnClickListener()) {
                    fragmentManagerRef.get()?.let { fm ->
                        val bottomSheet = BankDownBottomSheet.Builder(fm, TAG)
                        bottomSheet.build().show()
                    }
                }
            }

            clTopBankDetail.setOnClickListener {
                if (ViewUtils.isInternetAvailable(it.context)) {
                    if (MultipleClickHandler.isSafeOnClickListener()) {
                        if (!ivBankDown.isVisible) {
                            selectedPosition = bindingAdapterPosition
                            proceedToPayClicked(it)
                        } else {
                            fragmentManagerRef.get()?.let { fm ->
                                val bottomSheet = BankDownBottomSheet.Builder(fm, TAG)
                                bottomSheet.build().show()
                            }
                        }
                    }
                } else {
                    ViewUtils.showNetworkError(it.context)
                }
            }
        }

        fun bind(position: Int) {
            bankCode = Utils.getValueFromPaymentOption<String>(
                SdkUiConstants.CP_BANK_CODE,
                banksFilteredList[position].otherParams as? HashMap<String, Any?>
            ).toString()

            tvPaymentOptionName.text = Utils.getBankNameByBankCode(bankCode.toString())
            when {
                banksFilteredList[position].isBankDown -> showBankDownView(
                    this
                )

                Utils.isOfferAvailable(
                    bankCode!!,
                    paymentType
                ) && InternalConfig.userSelectedOfferInfo != null && paymentType != PaymentType.EMI -> showOfferView(
                    this
                )
            }

            var param: ImageParam? = null
            when (paymentType) {
                PaymentType.NB -> {
                    param = ImageParam(
                        banksFilteredList[position],
                        false,
                        Utils.getDefaultDrawable(banksFilteredList[position].paymentType)
                    )
                }
            }
            param?.let {
                SdkUiInitializer.apiLayer?.getImageForPaymentOption(
                    it,
                    onFetchImageListener = object : OnFetchImageListener {
                        override fun onImageGenerated(result: ImageDetails) {
                            // Check if view is still attached before updating
                            if (ivPaymentOptionIcon.isAttachedToWindow) {
                                ImageViewUtils.setImage(ivPaymentOptionIcon, result)
                            }
                        }
                    })
            }
        }

        private fun proceedToPayClicked(view: View) {
            InternalConfig.isPaymentOptionSelected = true
            onBankAdapterListener?.itemSelected(banksFilteredList[bindingAdapterPosition], false, true)

            fragmentManagerRef.get()?.let { fm ->
                // Create builder only when needed
                bottomSheet = BNPLAndUPIBottomSheet.Builder(fm, TAG)
                    .setTitle(
                        SdkUiConstants.NB_TEXT +
                                banksFilteredList[bindingAdapterPosition].bankName +
                                " " + SdkUiConstants.PAGE
                    )

                val imageParam = ImageParam(
                    banksFilteredList[bindingAdapterPosition],
                    false,
                    R.drawable.payu_netbanking
                )

                bottomSheet?.setMessage("")?.setIconParam(imageParam)?.setValidateApiListener(this)
                    ?.setPaymentOption(banksFilteredList[selectedPosition])?.setProceedListener {
                        makeNbPayment(view)
                    }?.setOnDetachCallBack {
                        bottomSheet = null
                    }

                nbBottomSheet = bottomSheet?.setBottomSheetType(BottomSheetType.L2_TOP_BANKS)
                    ?.setOfferApplyListener(offerApplyListener)
                    ?.setOnDetachCallBack {
                        onBankAdapterListener?.itemUnSelected(true)
                        // Clean up when bottom sheet is detached
                        nbBottomSheet = null
                        validateOfferResultListener = null
                    }?.build()

                nbBottomSheet?.show()
            }
        }

        private fun makeNbPayment(view: View) {
            InternalConfig.removeOfferIfNeeded = false
            nbBottomSheet?.close()
            isOfferValid = false
            if (onBankAdapterListener?.isActivityAlive() == true) {
                if (ViewUtils.isInternetAvailable(view.context)) {
                    ViewUtils.dismissSnackBar()
                    val banksFilteredListPosition = banksFilteredList[selectedPosition]
                    AnalyticsUtils.logMakePaymentEvent(
                        view.context,
                        banksFilteredListPosition
                    )
                    makePayment(view)
                } else {
                    ViewUtils.showNetworkError(view.context)
                }
            }
            // Cleanup after payment is made
            cleanupBottomSheetReferences()
        }

        override fun validate(
            paymentOption: PaymentOption?,
            bsResultListener: BottomSheetValidateResultListener
        ) {
            // Store reference to listener to prevent memory leaks
            validateOfferResultListener = object : ValidateOfferResultListener {
                override fun onValidateOfferResponse(isOfferValid: Boolean) {
                    // Check if still active before calling
                    if (bindingAdapterPosition != RecyclerView.NO_POSITION) {
                        bsResultListener.onValidateResultListener(isOfferValid)
                    }
                }
            }
            validateOfferResultListener?.let {
                onBankAdapterListener?.validateOffer(
                    banksFilteredList[selectedPosition],
                    it
                )
            }
        }

        // Method to clean up all references
        fun cleanupBottomSheetReferences() {
            bottomSheet = null
            nbBottomSheet = null
            validateOfferResultListener = null
        }

        override fun onClick(v: View?) {

        }
    }

    internal fun makePayment(itemView: View, paymentModel: PaymentModel? = null) {
        SdkUiInitializer.apiLayer?.updatePaymentState(
            Utils.getPaymentModel(
                banksFilteredList[selectedPosition],
                PaymentFlowState().apply {
                    this.paymentState = paymentModel?.paymentFlowState?.paymentState
                }),
            ViewUtils.getToolbar(
                itemView.context,
                banksFilteredList[selectedPosition].additionalCharge
            )
        )
    }

    private fun showOfferView(holder: ViewHolder) {
        holder.tvUseOffer.visibility = View.VISIBLE
        ViewUtils.updateBackgroundColor(
            holder.itemView.context,
            holder.tvUseOffer,
            SdkUiInitializer.apiLayer?.config?.primaryColor,
            R.color.one_payu_colorPrimary
        )
    }

    private fun showBankDownView(holder: ViewHolder) {
        holder.ivBankDown.visibility = View.VISIBLE
        holder.tvUseOffer.visibility = View.GONE
        ViewUtils.disableView(holder.ivPaymentOptionIcon)
        ViewUtils.disableView(holder.tvUseOffer)
        ViewUtils.disableView(holder.tvPaymentOptionName)
    }

    override fun onDetachedFromRecyclerView(recyclerView: RecyclerView) {
        super.onDetachedFromRecyclerView(recyclerView)

        // Clean up all ViewHolder references
        for (holderRef in viewHolders) {
            holderRef.get()?.cleanupBottomSheetReferences()
        }
        viewHolders.clear()

        // Clear the FragmentManager reference
        fragmentManagerRef.clear()
    }

    override fun onViewRecycled(holder: ViewHolder) {
        super.onViewRecycled(holder)
        holder.cleanupBottomSheetReferences()
    }
}
