package com.payu.ui.viewmodel

import android.app.Activity
import android.app.Application
import android.content.Context
import android.os.CountDownTimer
import android.os.Handler
import android.os.Looper
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import com.payu.base.listeners.OnValidateOfferListener
import com.payu.base.models.ErrorResponse
import com.payu.base.models.InternalConfig
import com.payu.base.models.MPinAction
import com.payu.base.models.OfferInfo
import com.payu.base.models.OlwAPIResponse
import com.payu.base.models.OlwPaymentOption
import com.payu.base.models.PaymentOption
import com.payu.base.models.SelectedOfferInfo
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.managers.OfferFilterManager
import com.payu.ui.model.models.BottomSheetLoaderData
import com.payu.ui.model.models.EnterMPinData
import com.payu.ui.model.models.FragmentModel
import com.payu.ui.model.models.SetMPinData
import com.payu.ui.model.models.SimpleStatusBottomSheetData
import com.payu.ui.model.models.SnackBarModel
import com.payu.ui.model.models.WalletCardData
import com.payu.ui.model.utils.AnalyticsUtils
import com.payu.ui.model.utils.ErrorConstants
import com.payu.ui.model.utils.EventStatus
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils
import com.payu.ui.model.utils.ViewUtils
import com.payu.ui.view.fragments.olw.BottomSheetLoaderFragment
import com.payu.ui.view.fragments.olw.CardBalanceFragment
import com.payu.ui.view.fragments.olw.EnterMPinFragment
import com.payu.ui.view.fragments.olw.MPinSetStatusFragment
import com.payu.ui.view.fragments.olw.SetMPinFragment
import org.json.JSONObject

open class OLWViewModel(application: Application) :
    BaseViewModel(application) {

    // LiveData for UI to observe
    internal val resendOtpTimerRemainingTime = MutableLiveData<Int>()
    internal val isResendOtpTimerRunning = MutableLiveData<Boolean>()
    internal val updateErrorTextOnMPin: MutableLiveData<String> = MutableLiveData()
    internal val updateErrorTextOnSetMPin: MutableLiveData<String> = MutableLiveData()
    internal val updateErrorTextForOtpOnSetMPin: MutableLiveData<String> = MutableLiveData()
    internal val updateBankItem: MutableLiveData<PaymentOption> = MutableLiveData()
    private var countDownTimer: CountDownTimer? = null
    internal val showOlwBottomSheet: MutableLiveData<BottomSheetDialogFragment> =
        MutableLiveData()
    internal val hideOlwBottomSheet: MutableLiveData<Boolean> = MutableLiveData()


    internal fun showSetMPinBottomSheet(
        context: Context,
        setMPinAction: MPinAction,
        olwPaymentOption: OlwPaymentOption
    ) {
        AnalyticsUtils.logKibanaInfoEvent(
            context = application.applicationContext,
            eventKey = SdkUiConstants.CP_OLW_SET_MPIN_UI_SHOWN,
            eventValue = JSONObject().apply {
                put(SdkUiConstants.CP_KEY_WALLET_NAME, olwPaymentOption.bankName)
                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
            }
        )
        val olwBalance = olwPaymentOption.balance.toDoubleOrNull() ?: 0.0
        sendOlwOtp(context, olwPaymentOption)
        val data = SetMPinData(
            actionText = context.getString(R.string.payu_submit),
            bankName = if (setMPinAction == MPinAction.TOKENEXPIRED) application.getString(
                R.string.verification_required,
                olwPaymentOption.bankName
            ) else application.getString(
                R.string.setup_mpin_for_with_bankname,
                olwPaymentOption.bankName
            ),
            subHeader = if (setMPinAction == MPinAction.TOKENEXPIRED) context.getString(R.string.verify_using_otp) else "",
            currentBalance = olwBalance,
            otpLength = 6,
            mPinLength = 6,
            reActionDescText = context.getString(R.string.didn_t_receive),
            reActionText = context.getString(R.string.payu_resend_otp),
            setMPinText = if (setMPinAction == MPinAction.TOKENEXPIRED) application.getString(R.string.enter_mpin) else application.getString(
                R.string.enter_new_mpin
            ),
            setMPinAction = setMPinAction,
            mobileNumber = getLoggedInPhoneNumber() ?: "",
            paymentOption = olwPaymentOption,
            postSuccessNavigation = {
                val txnAmount =
                    SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
                        ?: 0.0
                if (txnAmount < olwBalance)
                    showEnterMPinBottomSheet(olwPaymentOption)
                else
                    showTopUpConfirmationBottomSheet(olwPaymentOption, showSecondaryButton = false)
            }
        )
        val frag = SetMPinFragment.newInstance(data)
        showOlwBottomSheet.value = frag
        updateErrorTextOnSetMPin.value = ""
        updateErrorTextForOtpOnSetMPin.value = ""
        cancelCountdown()
    }

    internal fun showEnterMPinBottomSheet(paymentOption: OlwPaymentOption) {
        val txnAmount =
            SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull() ?: 0.0
        val olwBalance = paymentOption.balance.toDoubleOrNull() ?: 0.0

        AnalyticsUtils.logKibanaInfoEvent(
            context = application.applicationContext,
            eventKey = SdkUiConstants.CP_OLW_ENTER_MPIN_UI_SHOWN,
            eventValue = JSONObject().apply {
                put(SdkUiConstants.CP_KEY_WALLET_NAME, paymentOption.bankName)
                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
                put(SdkUiConstants.CP_KEY_AMOUNT, txnAmount)
            }
        )

        val data = EnterMPinData(
            bankName = paymentOption.bankName,
            currentBalance = olwBalance,
            bankIcon = null,
            mPinHint = application.getString(R.string.enter_mpin),
            mPinLength = 6,
            reActionDescText = application.getString(R.string.forgot_mpin),
            reActionText = application.getString(R.string.reset_now),
            actionText = application.getString(R.string.payu_proceed),
            showInsufficientTag = txnAmount > olwBalance,
            topUpAmountNeeded = if (txnAmount > olwBalance) txnAmount - olwBalance else 0.00,
            showSecondaryButton = false,
            showTopUpAmountConfirmation = false,
            paymentOption = paymentOption
        )
        val frag = EnterMPinFragment.newInstance(data)
        showOlwBottomSheet.value = frag
        updateErrorTextOnMPin.value = ""
    }

    internal fun showTopUpCardFragment(paymentOption: OlwPaymentOption) {
        AnalyticsUtils.logKibanaInfoEvent(
            context = application.applicationContext,
            eventKey = SdkUiConstants.CP_TOP_UP_SCREEN_UI_SHOWN,
            eventValue = JSONObject().apply {
                put(SdkUiConstants.CP_KEY_WALLET_NAME, paymentOption.bankName)
                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
            }
        )

        val txnAmount =
            SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull() ?: 0.0
        val olwBalance = paymentOption.balance.toDoubleOrNull() ?: 0.0

        val data = WalletCardData(
            walletName = paymentOption.bankName,
            currentBalance = olwBalance,
            walletIcon = null,
            showInsufficientTag = txnAmount > olwBalance,
            showTopUpCard = true,
            minTopUpAmountNeeded = if (txnAmount > olwBalance) txnAmount - olwBalance else 0.00,
            actionText = application.getString(R.string.payu_continue),
            paymentOption = paymentOption
        )
        val frag = CardBalanceFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }

    internal fun sendOlwOtp(context: Context, olwPaymentOption: OlwPaymentOption) {
        selectedPaymentOption?.category?.let {
            SdkUiInitializer.apiLayer?.sendOlwOtp(
                walletIdentifier = it,
                bankCode = Utils.getBankCodeFromMap(
                    (selectedPaymentOption?.otherParams as? HashMap<String, Any>) ?: hashMapOf()
                ),
                mobile = getLoggedInPhoneNumber() ?: ""
            ) { olwApiResponse ->
                when (olwApiResponse?.responseCode) {
                    SdkUiConstants.CP_OLW_API_SUCCESS_CODE -> {
                        startCountdown(60)
                    }
                    ErrorConstants.CP_USER_BLOCKED_CODE, ErrorConstants.CP_OTP_LIMIT_REACHED_CODE -> {
                        showErrorScreen(
                            olwPaymentOption,
                            title = olwApiResponse.responseMessage
                                ?: context.getString(R.string.payu_please_try_again),
                            message = context.getString(R.string.closing_this_tab_please_wait)
                        )
                    }
                    else -> {
                        updateErrorTextForOtpOnSetMPin.value =
                            context.getString(R.string.payu_please_try_again)
                    }
                }

            }
        }
    }


    fun showTopUpConfirmationBottomSheet(
        paymentOption: OlwPaymentOption,
        showSecondaryButton: Boolean = true
    ) {
        val txnAmount =
            SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull() ?: 0.0
        val olwBalance = paymentOption.balance.toDoubleOrNull() ?: 0.0

        AnalyticsUtils.logKibanaInfoEvent(
            context = application,
            eventKey = SdkUiConstants.CP_OLW_LOAD_BALANCE_CONFIRMATION_UI_SHOWN,
            eventValue = JSONObject().apply {
                put(SdkUiConstants.CP_KEY_WALLET_NAME, paymentOption.bankName)
                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
                put(SdkUiConstants.CP_KEY_TOP_UP_AMOUNT, paymentOption.topUpAmount ?: (txnAmount - olwBalance))
            }
        )

        val data = EnterMPinData(
            bankName = paymentOption.bankName,
            currentBalance = olwBalance,
            bankIcon = null,
            mPinHint = application.getString(R.string.enter_mpin),
            mPinLength = 6,
            reActionDescText = application.getString(R.string.forgot_mpin),
            reActionText = application.getString(R.string.reset_now),
            actionText = application.getString(R.string.load_and_pay),
            showInsufficientTag = txnAmount > olwBalance,
            topUpAmountNeeded = paymentOption.loadAmount?.toDoubleOrNull() ?: (txnAmount - olwBalance),
            showSecondaryButton = showSecondaryButton,
            showTopUpAmountConfirmation = true,
            paymentOption = paymentOption
        )
        val frag = EnterMPinFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }

    internal fun startCountdown(seconds: Int) {
        countDownTimer?.cancel()

        isResendOtpTimerRunning.value = true
        countDownTimer = object : CountDownTimer(seconds * 1000L, 1000) {
            override fun onTick(millisUntilFinished: Long) {
                resendOtpTimerRemainingTime.value = (millisUntilFinished / 1000).toInt()
            }

            override fun onFinish() {
                resendOtpTimerRemainingTime.value = 0
                isResendOtpTimerRunning.value = false
            }
        }.start()
    }

    /**
     * Cancel current countdown
     */
    internal fun cancelCountdown() {
        countDownTimer?.cancel()
        isResendOtpTimerRunning.value = false
    }

    override fun onCleared() {
        super.onCleared()
        countDownTimer?.cancel()
    }

    internal fun performDelayedAction(delayMillis: Long, action: () -> Unit): Handler {
        val handler = Handler(Looper.getMainLooper())
        handler.postDelayed(action, delayMillis)
        return handler
    }



    internal fun showErrorScreen(
        olwPaymentOption: OlwPaymentOption,
        title: String,
        message: String,
        postDelayAction: (() -> Unit)? = null
    ) {
        val data = SimpleStatusBottomSheetData(
            title = title,
            message = message,
            statusIcon = R.drawable.ic_hexagonal_warning_red,
            bgColor = R.color.payu_color_F7E8E8,
            dismissInterval = 3000,
            isSetMPinSuccess = false,
            paymentOption = olwPaymentOption,
            postDelayAction = postDelayAction
        )
        val frag = MPinSetStatusFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }

    internal fun verifyMPin(context: Activity, olwPaymentOption: OlwPaymentOption, mPin: String) {
        showBottomSheetDialog.value = true
        selectedPaymentOption?.category?.let { category ->
            SdkUiInitializer.apiLayer?.verifyMPin(
                mPin,
                category,
                Utils.getBankCodeFromMap(
                    (selectedPaymentOption?.otherParams as? HashMap<String, Any>) ?: hashMapOf()
                ),
                mobile = getLoggedInPhoneNumber() ?: ""
            ) { olwResponse ->
                showBottomSheetDialog.value = false
                updateErrorTextOnMPin.value = ""
                when (olwResponse?.responseCode) {
                    SdkUiConstants.CP_OLW_API_SUCCESS_CODE -> {
                        hideOlwBottomSheet.value = true
                        val olwPaymentOption = selectedPaymentOption as? OlwPaymentOption
                        olwPaymentOption?.mPin = mPin
                        olwPaymentOption?.token = olwResponse.token
                        makePaymentApiCall(paymentOption = selectedPaymentOption ?: PaymentOption())
                    }

                    ErrorConstants.CP_INVALID_TOKEN, ErrorConstants.CP_TOKEN_EXPIRED_CODE -> {
                        val olwPaymentOption = selectedPaymentOption as? OlwPaymentOption
                        olwPaymentOption?.mPin = mPin
                        olwPaymentOption?.token = olwResponse.token
                        if (olwPaymentOption != null) {
                            showSetMPinBottomSheet(
                                context,
                                MPinAction.TOKENEXPIRED,
                                olwPaymentOption = olwPaymentOption
                            )
                        }
                    }

                    ErrorConstants.CP_MPIN_EXPIRED_CODE, ErrorConstants.CP_DEVICE_VERIFICATION_FAILED_CODE -> {
                        showSetMPinBottomSheet(
                            context,
                            MPinAction.SETMPIN,
                            olwPaymentOption = olwPaymentOption
                        )
                    }

                    ErrorConstants.CP_USER_BLOCKED_CODE, ErrorConstants.CP_MPIN_LIMIT_REACHED_CODE, ErrorConstants.CP_OTP_LIMIT_REACHED_CODE -> {
                        showErrorScreen(
                            olwPaymentOption,
                            title = olwResponse.responseMessage ?: application.getString(
                                R.string.payu_please_try_again
                            ),
                            message = application.getString(R.string.closing_this_tab_please_wait)
                        )
                    }

                    else -> {
                        updateErrorTextOnMPin.value = olwResponse?.responseMessage
                            ?: context.getString(R.string.payu_please_try_again)
                    }
                }
            }
        }
    }

    internal fun setMPIn(
        otp: String,
        mPin: String,
        mobile: String,
        action: String,
        olwPaymentOption: OlwPaymentOption,
        onResponseCallBack: (() -> Unit)? = null,
        postSuccessNavigation: (() -> Unit)? = null,
        postFailureNavigation: (() -> Unit)? = null
    ) {
        showBottomSheetDialog.value = true

        olwPaymentOption.category?.let { category ->
            SdkUiInitializer.apiLayer?.setMPin(
                otp,
                mPin,
                mobile,
                category,
                Utils.getBankCodeFromMap(
                    (selectedPaymentOption?.otherParams as? HashMap<String, Any>) ?: hashMapOf()
                ),
                action
            ) { setMpinResponse ->
                showBottomSheetDialog.value = false
                updateErrorTextForOtpOnSetMPin.value = ""
                updateErrorTextOnSetMPin.value = ""
                val olwAPIResponse = setMpinResponse?.olwAPIResponse
                when (olwAPIResponse?.responseCode) {
                    SdkUiConstants.CP_OLW_API_SUCCESS_CODE -> {
                        onResponseCallBack?.invoke()
                        AnalyticsUtils.logKibanaInfoEvent(
                            context = application,
                            eventKey = SdkUiConstants.CP_OLW_MPIN_UPDATED,
                            eventValue = JSONObject().apply {
                                put(SdkUiConstants.CP_KEY_WALLET_NAME, olwPaymentOption.bankName)
                                put(SdkUiConstants.CP_KEY_MOBILE,getLoggedInPhoneNumber())
                                put(SdkUiConstants.CP_KEY_FLOW, action)
                            },
                            eventStatus = EventStatus.Success.name
                        )
                        showMPinSetSuccessStatusScreen(
                            olwPaymentOption,
                            olwAPIResponse,
                            postDelayAction = postSuccessNavigation
                        )
                    }
                    ErrorConstants.CP_OTP_EXPIRED_CODE, ErrorConstants.CP_MPIN_OR_OTP_DID_NOT_MATCH, ErrorConstants.CP_INVALID_OTP, ErrorConstants.CP_OTP_INVALID_CODE, ErrorConstants.CP_INVALID_OTP_LENGTH, ErrorConstants.CP_INVALID_MPIN_LENTH_CODE -> {
                        updateErrorTextForOtpOnSetMPin.value =
                            setMpinResponse.olwAPIResponse?.responseMessage
                                ?: application.getString(R.string.payu_please_try_again)
                    }
                    else -> {
                        AnalyticsUtils.logKibanaInfoEvent(
                            context = application,
                            eventKey = SdkUiConstants.CP_OLW_MPIN_UPDATED,
                            eventValue = JSONObject().apply {
                                put(SdkUiConstants.CP_KEY_WALLET_NAME, olwPaymentOption.bankName)
                                put(SdkUiConstants.CP_KEY_MOBILE,getLoggedInPhoneNumber())
                                put(SdkUiConstants.CP_KEY_FLOW, action)
                            },
                            eventStatus = EventStatus.Failure.name
                        )
                        showMPinSetFailureStatusScreen(
                            olwPaymentOption,
                            postDelayAction = postFailureNavigation
                        )
                    }
                }
            }
        }
    }

    internal fun fetchBalanceForOlw(context: Context) {
        val bottomSheetLoaderData =
            BottomSheetLoaderData(message = context.getString(R.string.payu_loader_processing))
        showBottomSheetLoader(bottomSheetLoaderData)
        getLoggedInPhoneNumber()?.let { mobile ->
            SdkUiInitializer.apiLayer?.fetchBalanceForOlw(
                selectedPaymentOption as? OlwPaymentOption,
                mobile,
                Utils.getBankCodeFromMap(
                    (selectedPaymentOption?.otherParams as? HashMap<String, Any>) ?: hashMapOf()
                )
            ) { olwPaymentOption ->
                hideOlwBottomSheet.value = true
                olwPaymentOption?.bankName = selectedPaymentOption?.bankName ?: ""
                olwPaymentOption?.paymentType = selectedPaymentOption?.paymentType
                if (olwPaymentOption?.fetchedStatus == 1) {

                    val txnAmount =
                        SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
                            ?: 0.0
                    val olwBalance = olwPaymentOption.balance.toDoubleOrNull() ?: 0.0
                    val userHasLowBalance = txnAmount > olwBalance

                    if (olwPaymentOption.customerRegistered.not()) {
                        val walletName = olwPaymentOption.bankName
                        AnalyticsUtils.logKibanaInfoEvent(
                            context = application.applicationContext,
                            eventKey = SdkUiConstants.CP_OLW_UNREGISTERED_USER,
                            eventValue = JSONObject().apply {
                                put(SdkUiConstants.CP_KEY_WALLET_NAME, walletName)
                                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
                            }
                        )
                        updateBankItem.value = selectedPaymentOption?.apply {
                            error = application.getString(
                                R.string.number_not_registered_with_bank,
                                selectedPaymentOption?.bankName
                            )
                        }
                    } else if (olwPaymentOption.isMPinSet.not() && userHasLowBalance.not()) {
                        showSetMPinBottomSheet(
                            context,
                            MPinAction.SETMPIN,
                            olwPaymentOption = olwPaymentOption
                        )
                    } else if (olwPaymentOption.isMPinSet.not() && userHasLowBalance) {
                        showTopUpCardFragment(paymentOption = olwPaymentOption)
                    } else {
                        if (txnAmount < olwBalance)
                            showEnterMPinBottomSheet(paymentOption = olwPaymentOption)
                        else
                            showTopUpCardFragment(paymentOption = olwPaymentOption)
                    }
                } else if (olwPaymentOption?.errorMessage == null && olwPaymentOption?.customerRegistered?.not() == true) {
                    updateBankItem.value = selectedPaymentOption?.apply {
                        error = application.getString(
                            R.string.number_not_registered_with_bank,
                            selectedPaymentOption?.bankName
                        )
                    }
                } else {
                    showSnackBar.value = SnackBarModel(
                        (context.getString(R.string.payu_please_try_again)),
                        R.drawable.ic_hexagonal_warning_red
                    )
                }
            }
        }
    }

    internal fun showBottomSheetLoader(bottomSheetLoaderData: BottomSheetLoaderData) {
        val frag =
            BottomSheetLoaderFragment.newInstance(bottomSheetLoaderData = bottomSheetLoaderData)
        showOlwBottomSheet.value = frag
    }

    internal fun showMPinOrTopUpBtSheet(data: SimpleStatusBottomSheetData) {
        if (data.postDelayAction == null) {
            if (data.isSetMPinSuccess) {
                val txnAmount =
                    SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
                        ?: 0.0
                val olwBalance = data.paymentOption.balance.toDoubleOrNull() ?: 0.0

                if (txnAmount < olwBalance)
                    showEnterMPinBottomSheet(data.paymentOption)
                else
                    showTopUpConfirmationBottomSheet(data.paymentOption)
            }
        } else data.postDelayAction?.invoke()
    }
    fun showReSetMPinBottomSheet(olwPaymentOption: OlwPaymentOption) {
        val txnAmount =
            SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull() ?: 0.0
        val olwBalance = olwPaymentOption.balance.toDoubleOrNull() ?: 0.0
        sendOlwOtp(application, olwPaymentOption)
        AnalyticsUtils.logKibanaInfoEvent(
            context = application,
            eventKey = SdkUiConstants.CP_OLW_RESET_MPIN_UI_SHOWN,
            eventValue = JSONObject().apply {
                put(SdkUiConstants.CP_KEY_WALLET_NAME, olwPaymentOption.bankName)
                put(SdkUiConstants.CP_KEY_MOBILE, getLoggedInPhoneNumber())
                put(SdkUiConstants.CP_KEY_TOP_UP_AMOUNT, olwPaymentOption.topUpAmount)
                put(SdkUiConstants.CP_KEY_AMOUNT, txnAmount + olwBalance)
            }
        )

        val data = SetMPinData(
            actionText = application.getString(R.string.payu_submit),
            bankName = application.getString(
                R.string.reset_mpin_for,
                olwPaymentOption.bankName
            ),
            currentBalance = olwBalance,
            otpLength = 6,
            mPinLength = 6,
            reActionDescText = application.getString(R.string.didn_t_receive),
            reActionText = application.getString(R.string.payu_resend_otp),
            setMPinAction = MPinAction.FORGOTMPIN,
            mobileNumber = getLoggedInPhoneNumber() ?: "",
            setMPinText = application.getString(R.string.enter_mpin),
            paymentOption = olwPaymentOption,
            postSuccessNavigation = {
                val txnAmount =
                    SdkUiInitializer.apiLayer?.payUPaymentParams?.amount?.toDoubleOrNull()
                        ?: 0.0
                if (txnAmount < olwBalance)
                    showEnterMPinBottomSheet(olwPaymentOption)
                else
                    showTopUpConfirmationBottomSheet(olwPaymentOption, showSecondaryButton = false)
            }
        )
        val frag = SetMPinFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }
    internal fun makeOlwPayment(mPin: String?, otp: String? = null, token: String? = null) {
        hideOlwBottomSheet.value = true
        val olwPaymentOption = selectedPaymentOption as? OlwPaymentOption
        olwPaymentOption?.mPin = mPin
        token?.let {
            olwPaymentOption?.token = token
        }
        otp?.let {
            olwPaymentOption?.otp = it
        }
        olwPaymentOption?.let { makePaymentApiCall(it) }
    }

    private fun showMPinSetSuccessStatusScreen(
        olwPaymentOption: OlwPaymentOption,
        olwAPIResponse: OlwAPIResponse,
        postDelayAction: (() -> Unit)? = null
    ) {
        val data = SimpleStatusBottomSheetData(
            title = olwAPIResponse.responseMessage
                ?: application.getString(R.string.mpin_is_successfully_set),
            message = application.getString(R.string.closing_this_tab_please_wait),
            statusIcon = R.drawable.payu_success_icon,
            bgColor = R.color.payu_color_E8EDF7,
            dismissInterval = 3000,
            isSetMPinSuccess = true,
            paymentOption = olwPaymentOption,
            postDelayAction = postDelayAction
        )
        val frag = MPinSetStatusFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }

    private fun showMPinSetFailureStatusScreen(
        olwPaymentOption: OlwPaymentOption,
        postDelayAction: (() -> Unit)? = null
    ) {
        val data = SimpleStatusBottomSheetData(
            title = application.getString(R.string.facing_issue_while_setting_up_mpin),
            message = application.getString(R.string.close_this_tab_and_retry),
            statusIcon = R.drawable.ic_hexagonal_warning_red,
            bgColor = R.color.payu_color_F7E8E8,
            dismissInterval = 3000,
            isSetMPinSuccess = false,
            paymentOption = olwPaymentOption,
            postDelayAction = postDelayAction
        )
        val frag = MPinSetStatusFragment.newInstance(data)
        showOlwBottomSheet.value = frag
    }

    internal fun resetBankItem(){
        updateBankItem.value = selectedPaymentOption?.apply {
            error = ""
        }
    }
}