package com.payu.ui.viewmodel

import android.app.Application
import android.util.Log
import androidx.annotation.DrawableRes
import androidx.lifecycle.MutableLiveData
import com.payu.base.models.EMIOption
import com.payu.base.models.EmiType
import com.payu.base.models.InternalConfig
import com.payu.base.models.OfferInfo
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentState
import com.payu.base.models.PaymentType
import com.payu.base.models.UPIOption
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.managers.OfferFilterManager
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.Utils

class BankViewModel(
    application: Application, mParam: Map<String, Any>
) :
    BaseViewModel(application) {

    val showMCPHeader: MutableLiveData<Boolean> = MutableLiveData()
    val resetBankSelection: MutableLiveData<Boolean> = MutableLiveData()
    val hideSearchView: MutableLiveData<Boolean> = MutableLiveData()
    val hideHeaderView: MutableLiveData<Boolean> = MutableLiveData()
    val allBanksText: MutableLiveData<String> = MutableLiveData()
    val allBanksIcon: MutableLiveData<Int?> = MutableLiveData()
    val showUpiCollectView: MutableLiveData<Boolean> = MutableLiveData()
    val showTezOmniView: MutableLiveData<Boolean> = MutableLiveData()
    val showOfferViewInUPICOllect: MutableLiveData<Boolean> = MutableLiveData()
    val adapterList: MutableLiveData<ArrayList<PaymentOption>> = MutableLiveData()
    val offersListLiveData: MutableLiveData<ArrayList<OfferInfo>> = MutableLiveData()
    var siHeaderSummary: MutableLiveData<String> = MutableLiveData()
    var removeAnyAppliedOffer: MutableLiveData<Boolean> = MutableLiveData()
    var allBanksList: ArrayList<PaymentOption>? = null
    var offersList: ArrayList<OfferInfo>? = null
    val applicationContext = application
    var paymentState: PaymentState? = null
    private var paymentType: PaymentType? = null
    var isEmiOption: Boolean = false
    var isSkuOffer: Boolean = false
    var displayTopBanks: MutableLiveData<Boolean> = MutableLiveData()
    val skuListLiveData: MutableLiveData<ArrayList<OfferInfo>> = MutableLiveData()
    val otmMessageView: MutableLiveData<Boolean> = MutableLiveData()

    private val TAG = javaClass.simpleName

    init {

        Log.d(TAG, "BankViewModel initiated")
        offersList = mParam[SdkUiConstants.CP_OFFERS_LIST] as? ArrayList<OfferInfo>
        displayTopBanks.value = false
        if (offersList.isNullOrEmpty()) {
            paymentState = mParam[SdkUiConstants.CP_PAYMENT_STATE] as? PaymentState
            paymentType = mParam[SdkUiConstants.CP_PAYMENT_TYPE] as? PaymentType
            allBanksList = mParam[SdkUiConstants.CP_ALL_BANKS_LIST] as? ArrayList<PaymentOption>
            if (paymentType == PaymentType.NB)
                Utils.filterBankList(allBanksList!!)
            if (paymentState != null && paymentState == PaymentState.MCP)
                triggerInitViewLogicForMCP()
            else
                triggerInitViewLogic(paymentType)
        } else {
            isSkuOffer = (mParam[SdkUiConstants.CP_IS_SKU_OFFER] ?: false) as Boolean
            if (!isSkuOffer) {
                allBanksText.value = applicationContext.getString(R.string.payu_available_offers)
                offersListLiveData.value = offersList
                hideSearchView.value = true
                hideHeaderView.value = true
            } else {
                skuListLiveData.value = offersList
                allBanksText.value = null
            }
        }
        // To show OTM Message View
        if (SdkUiInitializer.apiLayer?.payUPaymentParams?.payUSIParams?.isPreAuthTxn == true) {
            otmMessageView.value = true
        }
    }


    private fun triggerInitViewLogicForMCP() {
        hideSearchView.value = true
        hideHeaderView.value = false
        showMCPHeader.value = true
        adapterList.value = allBanksList
        allBanksText.value = applicationContext.getString(R.string.payu_available_currencies)
    }

    private fun updateEMIOptionsForNoCostEmi(emiList: ArrayList<PaymentOption>): ArrayList<PaymentOption>{
        for (emiOption in emiList) {
            emiOption.optionList?.forEach { emi ->
                if (!emi.optionList.isNullOrEmpty()) {
                    emi.optionList?.forEach { tenure ->
                        val bankCode =
                            (tenure.otherParams as HashMap<String, String>)[SdkUiConstants.CP_BANK_CODE]
                        if (InternalConfig.noCostEmi?.contains(bankCode) == true) {
                            (emiOption as EMIOption).isNoCostEmi = true
                            (emi as EMIOption).isNoCostEmi = true
                            (tenure as EMIOption).isNoCostEmi = true
                        }
                        if (InternalConfig.offerBankListEmi?.contains(bankCode) == true) {
                            (emiOption as EMIOption).isOfferAvailable = true
                            (emi as EMIOption).isOfferAvailable = true
                        }
                    }
                } else {
                    val bankCode =
                        (emi.otherParams as HashMap<String, String>)[SdkUiConstants.CP_BANK_CODE]
                    if (InternalConfig.noCostEmi?.contains(bankCode) == true) {
                        (emiOption as EMIOption).isNoCostEmi = true
                        (emi as EMIOption).isNoCostEmi = true
                    }
                    if (InternalConfig.offerBankListEmi?.contains(bankCode) == true) {
                        (emiOption as EMIOption).isOfferAvailable = true
                        (emi as EMIOption).isOfferAvailable = true
                    }
                }

            }
        }
        return emiList
    }

    fun updateListRefreshedForOffer(paymentType: PaymentType?){
        if(paymentType == PaymentType.EMI) {
            allBanksList?.let { this.allBanksList = updateEMIOptionsForNoCostEmi(it) }
            triggerInitViewLogic(paymentType)
        }
    }

    private fun triggerInitViewLogic(paymentType: PaymentType?) {
        hideSearchView.value = InternalConfig.userAccountInfoList.isNullOrEmpty().not()
        hideHeaderView.value = false
        updateAllBanksText(paymentType)
        updateAllBanksIcon(paymentType)

        when (paymentType) {
            PaymentType.UPI -> {
                //Check if collect flow is enabled
                showUpiCollectView.value =
                    Utils.isPaymentTypeAvailable(allBanksList!!, PaymentType.UPI)
                showTezOmniView.value =
                    Utils.isPaymentOptionAvailable(allBanksList!!, SdkUiConstants.CP_TEZOMNI)
//                showOfferViewInUPICOllect.value = Utils.isOfferSelected() && Utils.isOfferAvailable(
//                    SdkUiConstants.CP_UPI,
//                    PaymentType.UPI
//                )
                adapterList.value = Utils.getIntentAppsList(allBanksList!!)
                siHeaderSummary.value = Utils.getCustomeNoteDetails(
                    PaymentType.UPI,
                    SdkUiInitializer.apiLayer?.config?.customNoteDetails
                )

            }

            PaymentType.EMI -> {
                val uniqueList = ArrayList<PaymentOption>()
                //Remove duplicates from the list
                uniqueList.addAll(allBanksList!!.distinctBy { it.bankName })
                adapterList.value = uniqueList
                if (!uniqueList.isNullOrEmpty() && (uniqueList[0].bankName.equals(SdkUiConstants.CP_CREDIT_CARD) || uniqueList[0].bankName.equals(
                        SdkUiConstants.CP_DEBIT_CARD
                    ) || uniqueList[0].bankName.equals(SdkUiConstants.CP_CARDLESS_EMI))
                ) {
                    allBanksText.value = applicationContext.getString(R.string.payu_emi_options)
                    hideSearchView.value = true
                    isEmiOption = false
                } else {
                    var emiOption = uniqueList[0] as EMIOption
                    if (emiOption.emiType == EmiType.CARD_LESS)
                        allBanksText.value =
                            applicationContext.getString(R.string.payu_cardless_emi_options)
                    hideSearchView.value = false
                    isEmiOption = true
                }
                siHeaderSummary.value = Utils.getCustomeNoteDetails(
                    PaymentType.EMI,
                    SdkUiInitializer.apiLayer?.config?.customNoteDetails
                )

            }

            PaymentType.NB -> {
                allBanksList?.let {
                    displayTopBanks.value = !Utils.getTopBankList().isNullOrEmpty()

                    hideHeaderView.value = Utils.getOtherBanksList().isNullOrEmpty() && InternalConfig.userAccountInfoList.isNullOrEmpty()
                }
                initNonNbView()
            }

            else -> {
                initNonNbView()
            }
        }
    }

    private fun initNonNbView() {
        if (!allBanksList.isNullOrEmpty())
            adapterList.value = allBanksList
        siHeaderSummary.value = Utils.getCustomeNoteDetails(
            paymentType,
            SdkUiInitializer.apiLayer?.config?.customNoteDetails
        )
    }

    private fun updateAllBanksText(paymentType: PaymentType?) {
        allBanksText.value = when (paymentType) {
            PaymentType.WALLET -> applicationContext.getString(R.string.payu_all_wallets)
            PaymentType.UPI -> applicationContext.getString(R.string.payu_pay_using_any_upi_app)
            PaymentType.BNPL -> applicationContext.getString(R.string.payu_pay_later_options)
            PaymentType.NB -> if (InternalConfig.userAccountInfoList.isNullOrEmpty()
                    .not()
            ) applicationContext.getString(R.string.payu_net_banking) else applicationContext.getString(
                R.string.payu_all_banks
            )
            else -> applicationContext.getString(R.string.payu_all_banks)
        }
    }

    private fun updateAllBanksIcon(paymentType: PaymentType?) {
        allBanksIcon.value = when(paymentType) {
            PaymentType.UPI -> R.drawable.payu_upi_icon
            else -> null
        }
    }

    fun getAllBanksList(paymentType: PaymentType) {
        when (paymentType) {
            PaymentType.UPI -> {
                allBanksList?.let {
                    showUpiCollectView.value =
                        Utils.isPaymentTypeAvailable(it, PaymentType.UPI)
                    showTezOmniView.value =
                        Utils.isPaymentOptionAvailable(it, SdkUiConstants.CP_TEZOMNI)
                    //TODO will uncomment when offer is applicable in UPI
//                showOfferViewInUPICOllect.value = Utils.isOfferSelected() && Utils.isOfferAvailable(
//                    SdkUiConstants.CP_UPI,
//                    PaymentType.UPI
//                )
//                checkForAvailablePaymentModeAndUpdateList(Utils.getUpiIntentList())
                    adapterList.value = Utils.getIntentAppsList(it)
                }
            }

            else -> {
                checkForAvailablePaymentModeAndUpdateList(OfferFilterManager.filterPaymentOption(paymentType, allBanksList))
            }
        }
    }

    private fun checkForAvailablePaymentModeAndUpdateList(paymentOptionsList: ArrayList<PaymentOption>?){
        if (!paymentOptionsList.isNullOrEmpty()) {
            adapterList.value = paymentOptionsList
            if(paymentType == PaymentType.NB){
                hideHeaderView.value = Utils.getOtherBanksList().isEmpty() && InternalConfig.userAccountInfoList.isNullOrEmpty()
            }
        } else {
            showOfferError()
            removeAnyAppliedOffer.value = true
        }
    }

    internal fun validateOffer() {
        if(shouldCallValidateOfferApi().not())
            return

        showOfferAppliedTag.value = Event(false)
        if (selectedPaymentOption != null && selectedPaymentOption?.paymentType != PaymentType.EMI && !Utils.isSiTxn()) {
            showAndroidLoader.value = Event(true)
            var bankCode = Utils.getValueFromPaymentOption<String>(
                SdkUiConstants.CP_BANK_CODE,
                selectedPaymentOption?.otherParams as? HashMap<String, Any?>
            ).toString()
            var category = Utils.getCategoryForOffer(selectedPaymentOption?.paymentType?.name)

            val upiOption = selectedPaymentOption as? UPIOption
            if (upiOption?.category.equals(SdkUiConstants.INTENT)) {
                bankCode = upiOption?.appName!!
                category = SdkUiConstants.CP_UPI
            }

            category?.let {
                SdkUiInitializer.apiLayer?.validateOfferDetails(
                    it, null, bankCode, null, loggedInPhoneNumber = getLoggedInPhoneNumber(), this
                )
            }
        }
    }

    internal fun updateSelectedPaymentOption(paymentOption: PaymentOption?) {
        this.selectedPaymentOption = paymentOption
    }

    var isTopBankSelected = false
    var isBankListItemSelected = false

    internal fun checkIfHeaderNeedToBeReset(): Boolean{
        return !isTopBankSelected && !isBankListItemSelected
    }

    override fun removeOffer(isAutoApply: Boolean) {
        resetBankSelection.value = true
        showOfferApplied.value = Event(!isAutoApply)

    }

}