package com.payu.ui.model.utils

import android.app.ActivityManager
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.Build.VERSION
import android.util.Log
import com.payu.base.models.PaymentOption
import com.payu.base.models.PaymentType
import com.payu.ui.BuildConfig
import com.payu.ui.SdkUiInitializer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import com.payu.ui.model.utils.SdkUiConstants.CP_KEY_SCREEN_NAME
import com.payu.ui.model.utils.SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI
import org.json.JSONObject
import java.util.concurrent.TimeUnit
import java.util.concurrent.locks.ReentrantReadWriteLock


object AnalyticsUtils {
    var time: Long = 0
    var isSavedCardAvailable: Boolean = false

    private val logList = mutableListOf<JSONObject>()

    private val readWriteLock = ReentrantReadWriteLock()

    fun addLogInLogListWithoutContext(logMessage: JSONObject) {
        readWriteLock.writeLock().lock()
        try {
            logList.add(logMessage)
        } finally {
            readWriteLock.writeLock().unlock()
        }
    }

    private fun clearLogList() {
        logList.clear()
        readWriteLock.writeLock().unlock()
    }

    fun logL2ClickEvents(
        context: Context,
        paymentType: PaymentType?,
        ctaName: String,
        isOfferApplied: Boolean = false
    ) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L2 $paymentType"
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_IS_OFFER_APPLIED] = isOfferApplied
        logData(context, SdkUiConstants.CP_L2_PROCEED_CLICKED, eventData)
        time = System.currentTimeMillis()
    }


    fun logCardsL2ClickEvents(context: Context, ctaName: String, isOfferApplied: Boolean) {
        if (ctaName.equals(SdkUiConstants.CP_ADD_CARD)) isSavedCardAvailable = true
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L2" + PaymentType.CARD
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_IS_OFFER_APPLIED] = isOfferApplied
        logData(context, SdkUiConstants.CP_L2_PROCEED_CLICKED, eventData)
        time = System.currentTimeMillis()
    }

    fun logMakePaymentEvent(
        context: Context,
        paymentOption: PaymentOption,
        ctaPage: String? = null,
        ctaName: String? = ""
    ) {
        when (paymentOption.paymentType) {
            PaymentType.CARD -> {
                val eventData = HashMap<String, Any>()
                val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
                eventData[SdkUiConstants.CP_TIME] = timevalue
                eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
                if (ctaName.isNullOrEmpty()) eventData[SdkUiConstants.CP_CTA_NAME] =
                    SdkUiConstants.CP_NEW_CARD
                else eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
                if (isSavedCardAvailable) eventData[SdkUiConstants.CP_CTA_PAGE] =
                    "L3 " + PaymentType.CARD
                else eventData[SdkUiConstants.CP_CTA_PAGE] = "L2 " + PaymentType.CARD
                eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
                logData(context, SdkUiConstants.CP_L3_PROCEED_CLICKED, eventData)
            }
            PaymentType.SODEXO -> {
                val eventData = HashMap<String, Any>()
                val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
                eventData[SdkUiConstants.CP_TIME] = timevalue
                eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
                eventData[SdkUiConstants.CP_CTA_NAME] = SdkUiConstants.CP_NEW_CARD
                eventData[SdkUiConstants.CP_CTA_PAGE] = "L3 " + PaymentType.SODEXO
                eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
                logData(context, SdkUiConstants.CP_L3_PROCEED_CLICKED, eventData)
            }
            PaymentType.EMI -> {
                val eventData = HashMap<String, Any>()
                val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
                eventData[SdkUiConstants.CP_TIME] = timevalue
                eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
                eventData[SdkUiConstants.CP_CTA_NAME] = SdkUiConstants.CP_NEW_CARD
                eventData[SdkUiConstants.CP_CTA_PAGE] = "L4 " + PaymentType.EMI
                eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
                logData(context, SdkUiConstants.CP_L4_PROCEED_CLICKED, eventData)
            }
            else -> {
                val eventData = HashMap<String, Any>()
                val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
                eventData[SdkUiConstants.CP_TIME] = timevalue
                eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
                eventData[SdkUiConstants.CP_CTA_NAME] = ctaName!!
                if (ctaPage != null) {
                    eventData[SdkUiConstants.CP_CTA_PAGE] = ctaPage
                }
                eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
                logData(context, SdkUiConstants.CP_L2_PROCEED_CLICKED, eventData)

            }
        }

        time = System.currentTimeMillis()
    }

    fun logL3CTAClickEvent(context: Context, ctaName: String, paymentType: PaymentType?) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L3 $paymentType"
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        logData(context, SdkUiConstants.CP_L3_PROCEED_CLICKED, eventData)
        time = System.currentTimeMillis()
    }


    fun logBackButtonClickEvent(
        context: Context,
        ctaName: String,
        ctaPage: String,
        sdkClosed: Boolean = false
    ) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[SdkUiConstants.CP_CTA_PAGE] = ctaPage
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_SECTION_NAME_C] = SdkUiConstants.CP_LEFT_SIDEBAR
        if (sdkClosed) eventData[SdkUiConstants.SDK_CLOSED] = SdkUiConstants.CP_YES
        else eventData[SdkUiConstants.SDK_CLOSED] = SdkUiConstants.CP_NO
        logData(context, SdkUiConstants.CP_BACK_BUTTON, eventData)
    }

    fun logCheckoutScreenLoadedEvent(context: Context, apiCalledTime: Long?) {
        apiCalledTime?.let {
            val eventData = HashMap<String, Any>()
            eventData[SdkUiConstants.CP_TIME] =
                Utils.getTimeDifferenceInSeconds(apiCalledTime, System.currentTimeMillis())
            eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] =
                TimeUnit.MILLISECONDS.toMillis(System.currentTimeMillis())
            eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_VIEW
            logData(context, SdkUiConstants.CP_EVENT_CHECKOUT_SCREEN_LOADED, eventData)
            time = System.currentTimeMillis()
        }

    }

    fun logCheckoutL1CTAClickEvent(
        context: Context,
        ctaName: String,
        sectionName: String = "",
        isOfferApplied: Boolean = false
    ) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = SdkUiConstants.CP_SCREEN_CHECKOUT_L1
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[SdkUiConstants.CP_IS_OFFER_TAG_SHOWN] = isOfferApplied
        eventData[SdkUiConstants.CP_SECTION_NAME_C] = sectionName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_CTA_L1_CLICKED, eventData)
        time = System.currentTimeMillis()
    }

    fun logCheckoutViewDetailsCTAClickEvent(context: Context, ctaName: String, ctaPage: String) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = ctaPage
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_VIEW_DETAILS, eventData)
    }

    fun logCheckoutOfferDetailsCTAClickEvent(context: Context, ctaName: String, ctaPage: String) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = ctaPage
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_OFFERS, eventData)
    }

    fun logEmiTenureClicked(context: Context, ctaName: String, paymentType: PaymentType) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_VIEW
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L4 $paymentType"
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_TENURE_EMI, eventData)
    }

    fun logVerifyVpa(context: Context, ctaName: String, isVpaVerified: Boolean) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = SdkUiConstants.CP_L3_UPI
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        if (isVpaVerified) eventData[SdkUiConstants.CP_VPA_VERIFIED] = "Yes"
        else eventData[SdkUiConstants.CP_VPA_VERIFIED] = "No"
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_VERIFY_VPA, eventData)
        time = System.currentTimeMillis()
    }

    fun logVerify(
        context: Context,
        ctaName: String,
        paymentType: PaymentType,
        bankCode: String,
        isEligible: Boolean
    ) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L3 $paymentType"
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName + " $paymentType"
        eventData[SdkUiConstants.CP_BANK_CODE] = bankCode
        if (isEligible) eventData[SdkUiConstants.CP_IS_ELIGIBLE] = "Yes"
        else eventData[SdkUiConstants.CP_IS_ELIGIBLE] = "No"
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_VERIFY, eventData)
        time = System.currentTimeMillis()
    }

    fun logManageCard(context: Context, ctaName: String, paymentType: PaymentType) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = "L2 $paymentType"
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_MANAGE_CARD, eventData)
        time = System.currentTimeMillis()
    }

    fun logSaveCard(context: Context, ctaPage: String) {
        val eventData = HashMap<String, Any>()
        val timevalue = Utils.getTimeDifferenceInSeconds(time, System.currentTimeMillis())
        eventData[SdkUiConstants.CP_TIME] = timevalue
        eventData[SdkUiConstants.CP_TIMEIN_MILISECONDS] = timevalue.toInt() * 1000
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_PAGE] = ctaPage
        eventData[SdkUiConstants.CP_CTA_NAME] = SdkUiConstants.CP_SAVE_CARD
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_SAVE_CARD, eventData)
        time = System.currentTimeMillis()
    }

    fun logGVProceedToPayClicked(context: Context, ctaName: String, orderNo: Int) {
        val eventData = HashMap<String, Any>()
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[AnalyticsConstant.CP_ORDER_NO] = orderNo
        AnalyticsHandler.logAnalytics(
            context,
            AnalyticsConstant.CP_GV_PROCEED_TO_PAY_CLICKED,
            eventData
        )
    }

    internal fun logOtpSubmission(context: Context, eventName: String, status: String) {
        val eventData = HashMap<String, Any>()
        eventData[AnalyticsConstant.CP_GV_STATUS] = status
        AnalyticsHandler.logAnalytics(context, eventName, eventData)
    }

    internal fun logGvDataShown(
        context: Context,
        eventName: String,
        tilesCount: Int,
        loggedIn: Boolean,
        unlockedOptionShown: Boolean,
        savedOptionCount: Int
    ) {
        val eventData = HashMap<String, Any>()
        eventData[AnalyticsConstant.CP_TILES_SHOWN_COUNT] = tilesCount
        if (loggedIn) {
            eventData[AnalyticsConstant.CP_SAVED_OPTION_COUNT] = savedOptionCount
            eventData[AnalyticsConstant.CP_LOGGED_IN] = AnalyticsConstant.CP_YES
        } else {
            eventData[AnalyticsConstant.CP_UNLOCKED_OPTION_SHOWN] = AnalyticsConstant.CP_YES
            eventData[AnalyticsConstant.CP_LOGGED_IN] = AnalyticsConstant.CP_NO
        }
        eventData[AnalyticsConstant.CP_LOGGED_IN] = loggedIn
        AnalyticsHandler.logAnalytics(context, eventName, eventData)
    }

    internal fun logTilesData(
        context: Context,
        ctaName: String,
        eventName: String,
        loggedIn: Boolean,
        orderNo: Int
    ) {
        val eventData = HashMap<String, Any>()
        eventData[SdkUiConstants.CP_CTA_NAME] = ctaName
        eventData[AnalyticsConstant.CP_LOGGED_IN] = loggedIn
        eventData[AnalyticsConstant.CP_ORDER_NO] = orderNo
        AnalyticsHandler.logAnalytics(context, eventName, eventData)
    }

    internal fun logData(context: Context, eventName: String, eventData: HashMap<String, Any>) {
        AnalyticsHandler.logAnalytics(context, eventName, eventData)
    }

    internal fun getCTANameValueForPaymentType(paymentType: PaymentType): String? {
        return when (paymentType) {
            PaymentType.CARD -> SdkUiConstants.CP_CARDS
            PaymentType.NB -> SdkUiConstants.CP_NET_BANKING
            PaymentType.UPI -> SdkUiConstants.CP_UPI
            PaymentType.WALLET -> SdkUiConstants.CP_WALLET
            else -> paymentType.name
        }
    }

    //kibana loggin funtion

    fun logTilesDataForKibana(
        context: Context,
        ctaName: String,
        eventName: String,
        orderNo: Int,
        loggedIn: Boolean = false
    ) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(SdkUiConstants.CP_CTA_NAME, ctaName)
            analyticsJson.put(AnalyticsConstant.CP_LOGGED_IN, loggedIn)
            analyticsJson.put(
                AnalyticsConstant.EVENT_VALUE,
                SdkUiConstants.CP_EVENT_CHECKOUT_L1_CTA_CLICK
            )
            analyticsJson.put(AnalyticsConstant.CP_ORDER_NO, orderNo)
            analyticsJson.put(SdkUiConstants.CP_EVENT_NAME, eventName)
            logKibanaData(context, analyticsJson)

        } catch (e: Exception) {
            Log.d(javaClass.simpleName, e.message.toString())
        }
    }

    @Synchronized
    private fun logKibanaData(context: Context, analyticsJson: JSONObject) {
        AnalyticsHandler.logKibanaData(context, analyticsJson)
        CoroutineScope(Dispatchers.IO).launch {
            readWriteLock.writeLock().lock()
            try {
                for (log in logList) {
                    AnalyticsHandler.logKibanaData(context, log)
                }
                logList.clear()
            } finally {
                readWriteLock.writeLock().unlock()
            }
        }
    }
    fun logInitCpForKibana(
        context: Context,
        eventName: String
    ) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(
                SdkUiConstants.CP_ENFORCED_PAYMETNS,
                SdkUiInitializer.apiLayer?.config?.enforcePaymentList
            )
            analyticsJson.put(SdkUiConstants.CP_EVENT_NAME, eventName)
            logKibanaData(context, analyticsJson)

        } catch (e: Exception) {
            Log.d(javaClass.simpleName, e.message.toString())
        }
    }
    fun logResetCpForKibana(
        context: Context,
        eventName: String
    ) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(
                SdkUiConstants.CP_RESET_OPTIONS,
                true
            )
            analyticsJson.put(SdkUiConstants.CP_EVENT_NAME, eventName)
            logKibanaData(context, analyticsJson)

        } catch (e: Exception) {
            Log.d(javaClass.simpleName, e.message.toString())
        }
    }

    fun logGVDataForKibana(
        context: Context,
        ctaName: String,
        tilesCount: Int,
        savedOptionCount: Int,
        unlockedOptionShown: Boolean,
        loggedIn: Boolean = false
    ) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(SdkUiConstants.CP_CTA_NAME, ctaName)
            analyticsJson.put(AnalyticsConstant.CP_TILES_SHOWN_COUNT, tilesCount)
            analyticsJson.put(AnalyticsConstant.CP_LOGGED_IN, loggedIn)
            analyticsJson.put(AnalyticsConstant.CP_SAVED_OPTION_COUNT, savedOptionCount)
            analyticsJson.put(AnalyticsConstant.CP_UNLOCKED_OPTION_SHOWN, unlockedOptionShown)
            logKibanaData(context, analyticsJson)
        } catch (e: Exception) {
            Log.d(javaClass.simpleName, e.message.toString())
        }
    }

    fun getAnalyticsJsonWithBasicProperties(context: Context?, eventKey: String, loggingSDK: String, sdkName: String, sdkVersion: String): JSONObject {
        val analyticsJson = JSONObject()

        analyticsJson.put(
            SdkUiConstants.CP_EVENT_KEY,
            eventKey
        )
        analyticsJson.put(SdkUiConstants.CP_TIMESTAMP, ViewUtils.getSystemCurrentTime())
        analyticsJson.put(SdkUiConstants.CP_SDK_VERSION, sdkVersion)
        analyticsJson.put(
            AnalyticsConstant.CP_LOGGING_SDK,
            loggingSDK
        )
        analyticsJson.put(
            SdkUiConstants.CP_TXN_ID,
            SdkUiInitializer.apiLayer?.payUPaymentParams?.transactionId
        )
        analyticsJson.put(
            SdkUiConstants.CP_MERCHANT_KEY,
            SdkUiInitializer.apiLayer?.payUPaymentParams?.key
        )
        analyticsJson.put(
            SdkUiConstants.CP_PLATFORM,
            SdkUiConstants.CP_ANDROID
        )
        analyticsJson.put(SdkUiConstants.CP_AMOUNT, SdkUiInitializer.apiLayer?.payUPaymentParams?.amount)
        analyticsJson.put(SdkUiConstants.CP_SDK_NAME, sdkName)
        val deviceName = Build.MANUFACTURER + " " + Build.MODEL
        val deviceOsVersion = VERSION.SDK_INT

        if (context != null) {
            val deviceTotalMemory = getTotalMemoryInfo(context)
            val availMemory = getAvailableMemoryInfo(context)
            val memoryInfo = availMemory * 100 / deviceTotalMemory

            analyticsJson.put(
                AnalyticsConstant.CP_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
            )
            analyticsJson.put(
                AnalyticsConstant.CP_APPLICATION_PACKAGE,
                context.applicationContext?.packageName?:""
            )
            analyticsJson.put(
                AnalyticsConstant.CP_APPLICATION_VERSION,
                getAppVersion(context)
            )
        } else {
            analyticsJson.put(
                AnalyticsConstant.CP_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion
            )
        }
        return analyticsJson
    }

    private fun getAppVersion(context: Context): String {
        try {
            return context.applicationContext.packageManager.getPackageInfo(context.applicationContext.packageName, 0).versionName
        } catch (ex: PackageManager.NameNotFoundException) {
            ex.printStackTrace()
            return "UnKnown"
        }
    }

    internal fun getTotalMemoryInfo(context: Context): Long {
        var activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        var memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        var totalRam = memoryInfo.totalMem / (1024 * 1024)
        return totalRam
    }

    internal fun getAvailableMemoryInfo(context: Context): Long {
        var activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        var memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        var availmem = memoryInfo.availMem / (1024 * 1024)
        return availmem
    }

    internal fun logOtpSubmissionForKibana(context: Context, eventName: String, status: String) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(SdkUiConstants.CP_EVENT_NAME, eventName)
            analyticsJson.put(AnalyticsConstant.CP_GV_STATUS, status)
            logKibanaData(context, analyticsJson)
        } catch (e: Exception) {

        }
    }

    fun logCPCallbackEventKibana(context: Context, callbackType: CPCallbackType) {
        val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
        analyticsJson.put(SdkUiConstants.CP_EVENT_KEY, SdkUiConstants.CP_CALLBACK)
        val value = JSONObject()
        value.put(SdkUiConstants.CP_TYPE, callbackType.method)
        analyticsJson.put(AnalyticsConstant.EVENT_VALUE, value.toString())
        analyticsJson.put(AnalyticsConstant.EVENT_TYPE, callbackType.eventType)
        analyticsJson.put(AnalyticsConstant.EVENT_STATUS, callbackType.status)
        logKibanaData(context, analyticsJson)
    }

    internal fun logEventNameForKibana(context: Context, eventName: String, eventData: String? = null) {
        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context, SdkUiConstants.CP_EVENT_KEY_CHECKOUT_TRANSACTION, BuildConfig.LIBRARY_PACKAGE_NAME, SdkUiConstants.SDK_NAME_CHECKOUT_PRO_UI, BuildConfig.VERSION_NAME)
            analyticsJson.put(SdkUiConstants.CP_EVENT_NAME, eventName)
            eventData?.let { analyticsJson.put(SdkUiConstants.CP_EVENT_DATA, eventData) }
            logKibanaData(context, analyticsJson)
        } catch (e: Exception) {

        }
    }

    internal fun logGetQuickPayOptionsEvent(context: Context) {
        val eventData = HashMap<String, Any>()
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_GET_QUICK_PAY_OPTIONS, eventData)
    }

    internal fun logQuickPayOptionsFetched(
        context: Context,
        savedOptionCount: Int,
        paymentOptionCategory: String,
        paymentOptionName: String
    ) {
        val unlockedSavedOptionShown = Utils.getGlobalVaultStoredUserToken(context).first.isNullOrEmpty()
        val eventData = HashMap<String, Any>()
        eventData[SdkUiConstants.CP_SAVED_OPTION_COUNT] = savedOptionCount
        eventData[SdkUiConstants.CP_UNLOCKED_SAVED_OPTION_SHOWN] = if (unlockedSavedOptionShown) "Yes" else "No"
        eventData[SdkUiConstants.CP_CATEGORY] = paymentOptionCategory
        eventData[SdkUiConstants.CP_NAME] = paymentOptionName
        eventData[SdkUiConstants.CP_LOGGED_IN] = if (unlockedSavedOptionShown) "No" else "Yes"
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_QUICK_PAY_OPTIONS_FETCHED, eventData)
    }

    internal fun logProceedToPayFromQuickPay(
        context: Context,
        position: Int,
        paymentOptionCategory: String,
        paymentOptionName: String
    ) {
        val eventData = HashMap<String, Any>()
        eventData[SdkUiConstants.CP_POSITION] = position
        eventData[SdkUiConstants.CP_CATEGORY] = paymentOptionCategory
        eventData[SdkUiConstants.CP_NAME] = paymentOptionName
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_PROCEED_TO_PAY_FROM_QUICK_PAY, eventData)
    }

    internal fun logOtherFromQuickPay(context: Context) {
        val eventData = HashMap<String, Any>()
        AnalyticsHandler.logAnalytics(context, SdkUiConstants.CP_OTHER_FROM_QUICK_PAY, eventData)
    }

    fun logNFCImpression(context: Context, eventName: String) {
        val eventData = HashMap<String, Any>()
        eventData[SdkUiConstants.CP_CTA_TYPE] = SdkUiConstants.CP_TYPE_ACTION
        AnalyticsHandler.logAnalytics(context, eventName, eventData)
    }

    /**
     * Generic function to log Kibana info events
     */
    fun logKibanaInfoEvent(
        context: Context,
        eventKey: String,
        eventValue: JSONObject? = null,
        screenName: String? = null,
        eventStatus: String? = null,
        eventSeverity: String? = null
    ) {
        val data = getAnalyticsJsonWithBasicProperties(
            context,
            eventKey,
            BuildConfig.LIBRARY_PACKAGE_NAME,
            SDK_NAME_CHECKOUT_PRO_UI,
            BuildConfig.VERSION_NAME
        )

        eventValue?.let { data.put(AnalyticsConstant.EVENT_VALUE, it.toString()) }
        data.put(AnalyticsConstant.EVENT_TYPE, EventType.Info)
        screenName?.let { data.put(CP_KEY_SCREEN_NAME, it) }
        eventStatus?.let { data.put(AnalyticsConstant.EVENT_STATUS, it) }
        eventSeverity?.let { data.put(AnalyticsConstant.EVENT_SEVERITY, it) }

        AnalyticsHandler.logKibanaData(context, data)
    }
}


enum class CPCallbackType(val method: String, val eventType: String, val status: String) {
    Success("onPaymentSuccess", "Info", "Success"),
    Failure("onPaymentFailure", "Error", "Failure"),
    Cancel("onPaymentCancel", "Cancel", "Failure"),
    Error("onError", "Error", "Failure")
}