package com.payu.ui.model.managers

import android.content.Context
import android.text.Editable
import android.text.TextWatcher
import android.util.Log
import android.view.View
import android.widget.Button
import android.widget.ImageView
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.payu.base.listeners.OnFetchImageListener
import com.payu.base.models.ImageDetails
import com.payu.base.models.InternalConfig
import com.payu.base.models.LoadMoney
import com.payu.ui.R
import com.payu.ui.SdkUiInitializer
import com.payu.ui.model.adapters.LoadMoneyAdapter
import com.payu.ui.model.utils.ImageViewUtils
import com.payu.ui.model.utils.SdkUiConstants
import com.payu.ui.model.utils.ViewUtils
import com.payu.ui.model.widgets.MonitoringEditText


class ClwBottomSheetHandler(
    private val context: Context?,
    private val onPaymentInitiated: (Double) -> Unit,
    private val walletName: String,
    private val walletBalance: String,
    private val amount: String
) : BottomSheetManager(),
    View.OnClickListener {
    private lateinit var ivWalletLogo: ImageView
    private lateinit var tvEnterAmountErrorText: TextView
    private lateinit var etEnterAmount: MonitoringEditText
    private lateinit var ivCloseIcon: ImageView
    private lateinit var rlEnterAmount: View
    private lateinit var btnProceedToPay: Button
    private lateinit var tvWalletName: TextView
    private lateinit var tvWalletBalance: TextView
    private lateinit var rvAddMoneyButton: RecyclerView
    private var loadAmount: Double = 0.0
    override fun init() {
        ivWalletLogo = view.findViewById(R.id.ivWalletLogo)
        tvEnterAmountErrorText = view.findViewById(R.id.tvEnterAmountErrorText)
        etEnterAmount = view.findViewById(R.id.etEnterAmount)
        ivCloseIcon = view.findViewById(R.id.ivCloseIcon)
        rlEnterAmount = view.findViewById(R.id.rlEnterAmount)
        btnProceedToPay = view.findViewById(R.id.btnProceedToPay)
        tvWalletName = view.findViewById(R.id.tvWalletName)
        tvWalletBalance = view.findViewById(R.id.tvWalletBalance)
        rvAddMoneyButton = view.findViewById(R.id.rvAddMoneyButton)

        val onMoneyAdded: (Double) -> Unit = { loadAmount ->
            addAmount(loadAmount)
        }

        ivCloseIcon.setOnClickListener(this)
        btnProceedToPay.setOnClickListener(this)

        setClwLogo()
        loadInitialAmount()

        tvWalletName.text = walletName
        tvWalletBalance.text =
            context?.getString(R.string.payu_balance, walletBalance.toDoubleOrNull() ?: 0.0)

        etEnterAmount.addTextChangedListener(object : TextWatcher {
            override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {

            }

            override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {

            }

            override fun afterTextChanged(p0: Editable?) {
                var amountValue = 0.0
                if (!p0.isNullOrEmpty()) {
                    try {
                        amountValue = p0.toString().toDouble()
                    } catch (e: NumberFormatException) {
                        Log.d(this.javaClass.simpleName, "NumberFormatException" + e.message)
                    }
                }

                loadAmount = amountValue
                handleErrorText()
            }
        })

        handleErrorText()

        rvAddMoneyButton.layoutManager =
            LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false)
        val items = prepareAddMoneyButtons()
        rvAddMoneyButton.adapter = LoadMoneyAdapter(
            items,
            onMoneyAdded
        )

        context?.let {
            ViewUtils.updateBackgroundColor(
                it,
                btnProceedToPay,
                SdkUiInitializer.apiLayer?.config?.primaryColor,
                R.color.one_payu_colorPrimary
            )
        }

        context?.let {
            ViewUtils.updateTextColor(
                it,
                btnProceedToPay,
                SdkUiInitializer.apiLayer?.config?.baseTextColor,
                R.color.one_payu_baseTextColor
            )
        }
    }

    private fun prepareAddMoneyButtons(): ArrayList<String> {
        val addMoneyButtonNameArray = ArrayList<String>()
        for (money in enumValues<LoadMoney>()) {
            addMoneyButtonNameArray.add(parseMoneyName(money.name))
        }
        return addMoneyButtonNameArray
    }

    private fun parseMoneyName(name: String): String {
        val array = name.split(SdkUiConstants.PAY)
        return if (array.size > 1)
            "+" + array[1]
        else
            ""
    }

    private fun loadInitialAmount() {
        var amountToBeLoaded = amount.toDouble() - walletBalance.toDouble()

        if (amountToBeLoaded % 100.0 == 0.0) {
            loadAmount = amountToBeLoaded
            etEnterAmount.setText(amountToBeLoaded.toInt().toString())
        } else {
            amountToBeLoaded = amountToBeLoaded - (amountToBeLoaded % 100.0) + 100
            loadAmount = amountToBeLoaded
            etEnterAmount.setText(loadAmount.toInt().toString())
        }
    }

    private fun setClwLogo() {
        if (!InternalConfig.logo.isNullOrEmpty()) {
            SdkUiInitializer.apiLayer?.getBitmapImageFormURL(InternalConfig.logo!!, object :
                OnFetchImageListener {
                override fun onImageGenerated(result: ImageDetails) {
                    ImageViewUtils.setImage(ivWalletLogo, result)
                }
            })
        } else {
            SdkUiInitializer.apiLayer?.config?.merchantLogo?.let {
                ivWalletLogo.setImageDrawable(
                    context?.getDrawable(
                        it
                    )
                )
            }
        }
    }

    private fun handleErrorText() {
        val amountToBeLoaded = amount.toDouble() - walletBalance.toDouble()
        if (loadAmount < amountToBeLoaded) {
            tvEnterAmountErrorText.text =
                context?.getString(R.string.payu_enter_amount_more_than) + amountToBeLoaded
            ViewUtils.disableView(btnProceedToPay)
            rlEnterAmount.background =
                context?.let { ViewUtils.getDrawable(it, R.drawable.payu_rounded_corner_image_red) }
        } else {
            tvEnterAmountErrorText.text = ""
            ViewUtils.enableView(btnProceedToPay)
            rlEnterAmount.background = context?.let {
                ViewUtils.getDrawable(
                    it,
                    R.drawable.payu_rounded_corner_image_for_edittext_blue
                )
            }
        }
    }

    private fun addAmount(amount: Double) {
        loadAmount += amount
        etEnterAmount.setText(loadAmount.toString())
        handleErrorText()
    }


    override fun handleBottomSheetClose() {
    }

    override fun bottomSheetDetach() {

    }

    override fun bottomSheetAttach() {
    }

    override fun onClick(v: View?) {
        when (v?.id) {
            R.id.ivCloseIcon -> {
                roundedCornerBottomSheet.dismiss()
            }
            R.id.btnProceedToPay -> {
                roundedCornerBottomSheet.dismiss()
                onPaymentInitiated(loadAmount)//closure
            }
        }
    }
}