//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.paymentparamhelper;

import static com.payu.paymentparamhelper.PayuConstants.SI_API_VERSION_VALUE;
import static com.payu.paymentparamhelper.PayuConstants.TPV_API_VERSION_VALUE;
import static com.payu.paymentparamhelper.PayuConstants.TPV_MF_API_VERSION_VALUE;
import static com.payu.paymentparamhelper.PayuErrors.FIRST_NAME_LENGTH_LIMIT;
import static com.payu.paymentparamhelper.PayuErrors.PRODUCT_INFO_LENGTH_LIMIT;
import static com.payu.paymentparamhelper.PayuErrors.TRANSACTION_ID_LENGTH_LIMIT;
import static com.payu.paymentparamhelper.PayuConstants.TPV_MF_API_VERSION_VALUE;

import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import com.payu.paymentparamhelper.siparams.BeneficiaryDetails;
import com.payu.paymentparamhelper.siparams.SIParams;
import com.payu.paymentparamhelper.siparams.enums.BeneficiaryAccountType;
import com.payu.paymentparamhelper.siparams.enums.BillingCycle;
import com.payu.paymentparamhelper.siparams.enums.BillingLimit;
import com.payu.paymentparamhelper.siparams.enums.BillingRule;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by franklin on 27/07/15.
 * Making one common class which takes all the inputs required to make a payment.
 * Inorder to create an object of this kind user needs to provide two inputs.
 */
public class PaymentPostParams extends PayuUtils {

    private PaymentParams mPaymentParams;
    private String mPaymentMode;
    private String TAG = this.getClass().getSimpleName();

    /**
     * JSONException e
     * Not allowing the user to create empty object.
     */
    private PaymentPostParams() {
    }

    /**
     * PaymentPostParams will accept object of {@link PaymentParams} and a payment mode
     *
     * @param paymentParams
     */
    public PaymentPostParams(PaymentParams paymentParams, String paymentMode) {
        this.mPaymentParams = paymentParams;
        this.mPaymentMode = paymentMode;
    }

    public PostData getPaymentPostParams() {
        PostData postData = new PostData();
        StringBuilder post = new StringBuilder();


        // lets validate the pg using payment mode.
        if (!PayuConstants.PG_SET.contains(this.mPaymentMode) && !mPaymentMode.equalsIgnoreCase(PayuConstants.LAZYPAY) && !mPaymentMode.equalsIgnoreCase(PayuConstants.NEFT_RTGS) && !mPaymentMode.equalsIgnoreCase(PayuConstants.SODEXO) && !mPaymentMode.equalsIgnoreCase(PayuConstants.TEZOMNI)) { // invalid pg value.
            return getReturnData(PayuErrors.INVALID_PG);
        }

        // lets set the default Parameters
        // device_type is mandatory, With out this flag SDK or CB can not receive payuPostData (ie: onPayuSuccess or onPayuFailure js callback from payu server)
        post.append(concatParams(PayuConstants.DEVICE_TYPE, "1", true));
        // Adding udid and imei
//        post.append(concatParams(PayuConstants.PAYU_UDID, getUdid()));
//        post.append(concatParams(PayuConstants.PAYU_IMEI, getImei()));
        if (TextUtils.isEmpty(mPaymentParams.getSdkPlatformData())) {
            post.append(concatParams(PayuConstants.SDK_PLATFORM_KEY, PayuUtils.getAnalyticsString(null), false));
        } else {
            post.append(concatParams(PayuConstants.SDK_PLATFORM_KEY, PayuUtils.getAnalyticsString(mPaymentParams.getSdkPlatformData()), false));
        }

        // lets begin with the mandatory default params.
        // TODO apply the validation according to the pg, payment mode!
        for (int i = 0; i < PayuConstants.PAYMENT_PARAMS_ARRAY.length; i++) {
            switch (PayuConstants.PAYMENT_PARAMS_ARRAY[i]) {
//                PayuConstants.KEY, PayuConstants.TXNID, PayuConstants.AMOUNT, PayuConstants.PRODUCT_INFO, PayuConstants.FIRST_NAME, PayuConstants.EMAIL, PayuConstants.SURL, PayuConstants.FURL, PayuConstants.HASH
                case PayuConstants.KEY: // TODO add validation for key
                    if (mPaymentParams.getKey() == null || mPaymentParams.getKey().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_KEY_IS_MISSING);
                    post.append(concatParams(PayuConstants.KEY, mPaymentParams.getKey(), true));
                    break;
                case PayuConstants.TXNID: // TODO add validation for txnid
                    if (!isValidTxnId(mPaymentParams.getTxnId()))
                        return getReturnData(PayuErrors.INVALID_TXN_ID);
                    post.append(concatParams(PayuConstants.TXNID, mPaymentParams.getTxnId(), true));
                    break;
                case PayuConstants.AMOUNT: // validation for amount
                    Double amount = 0.0;
                    try { // this will take care of null check also!
                        amount = mPaymentParams != null ? Double.parseDouble(mPaymentParams.getAmount()) : 0.0;
                    } catch (NumberFormatException e) {
                        return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.INVALID_AMOUNT);
                    } catch (NullPointerException e) {
                        return getReturnData(PayuErrors.INVALID_AMOUNT_EXCEPTION, PayuErrors.INVALID_AMOUNT);
                    }
                    post.append(concatParams(PayuConstants.AMOUNT, mPaymentParams.getAmount(), true));
                    break;
                case PayuConstants.PRODUCT_INFO: // TODO add validation for product info
                    if (mPaymentParams.getProductInfo() == null || mPaymentParams.getProductInfo().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_PRODUCT_INFO_IS_MISSING);
                    if (mPaymentParams.getProductInfo().length() > PRODUCT_INFO_LENGTH_LIMIT)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_PRODUCT_INFO_LENGTH_EXCEEDING);
                    post.append(concatParams(PayuConstants.PRODUCT_INFO, mPaymentParams.getProductInfo(), true));
                    break;
                case PayuConstants.FIRST_NAME: // TODO add validation for first name
                    if (mPaymentParams.getFirstName() == null) // empty string is allowed
                        return getReturnData(PayuErrors.MANDATORY_PARAM_FIRST_NAME_IS_MISSING);
                    if (mPaymentParams.getFirstName().length() > FIRST_NAME_LENGTH_LIMIT)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_FIRST_NAME_LENGTH_EXCEEDING);
                    post.append(concatParams(PayuConstants.FIRST_NAME, mPaymentParams.getFirstName(), true));
                    break;
                case PayuConstants.EMAIL: // TODO add validation for email
                    if (mPaymentParams.getEmail() == null)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_EMAIL_IS_MISSING);
                    post.append(concatParams(PayuConstants.EMAIL, mPaymentParams.getEmail(), true));
                    break;
                case PayuConstants.SURL: // TODO add validation for SURL
                    if (mPaymentParams.getSurl() == null || mPaymentParams.getSurl().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_SURL_IS_MISSING);
                    if (!isValidUrl(mPaymentParams.getSurl()))
                        return getReturnData(PayuConstants.SURL + PayuErrors.INVALID_URL);
                    post.append(concatParams(PayuConstants.SURL, mPaymentParams.getSurl(), true));
                    break;
                case PayuConstants.FURL: // TODO add validation for FURL
                    if (mPaymentParams.getFurl() == null || mPaymentParams.getFurl().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_FURL_IS_MISSING);
                    if (!isValidUrl(mPaymentParams.getFurl()))
                        return getReturnData(PayuConstants.FURL + PayuErrors.INVALID_URL);
                    post.append(concatParams(PayuConstants.FURL, mPaymentParams.getFurl(), true));

                    break;
                case PayuConstants.HASH: // TODO add validation for Hash
                    if (mPaymentParams.getHash() == null || mPaymentParams.getHash().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_HASH_IS_MISSING);
                    post.append(concatParams(PayuConstants.HASH, mPaymentParams.getHash(), true));
                    break;
                case PayuConstants.UDF1:
                    if (mPaymentParams.getUdf1() == null)
                        return getReturnData(PayuErrors.INVALID_UDF1);
                    post.append(concatParams(PayuConstants.UDF1, mPaymentParams.getUdf1(), true));
                    break;
                case PayuConstants.UDF2: // TODO add validation for UDF2
                    if (mPaymentParams.getUdf2() == null)
                        return getReturnData(PayuErrors.INVALID_UDF2);
                    post.append(concatParams(PayuConstants.UDF2, mPaymentParams.getUdf2(), true));
                    break;
                case PayuConstants.UDF3: // TODO add validation for UDF3
                    if (mPaymentParams.getUdf3() == null)
                        return getReturnData(PayuErrors.INVALID_UDF3);
                    post.append(concatParams(PayuConstants.UDF3, mPaymentParams.getUdf3(), true));
                    break;
                case PayuConstants.UDF4: // TODO add validation for UDF4
                    if (mPaymentParams.getUdf4() == null)
                        return getReturnData(PayuErrors.INVALID_UDF4);
                    post.append(concatParams(PayuConstants.UDF4, mPaymentParams.getUdf4(), true));
                    break;
                case PayuConstants.UDF5: // TODO add validation for UDF5
                    if (mPaymentParams.getUdf5() == null)
                        return getReturnData(PayuErrors.INVALID_UDF5);
                    post.append(concatParams(PayuConstants.UDF5, mPaymentParams.getUdf5(), true));
                    break;
                case PayuConstants.RETRY_SDK: // to add retry_sdk for retry txn
                    if (mPaymentParams.getRetryCount() > 0)
                        post.append(PayuConstants.RETRY_SDK);
                    break;
                case PayuConstants.USER_CREDENTIALS: // to add userCredential
                    if (mPaymentParams.getUserCredentials() == null)
                        return getReturnData(PayuErrors.USER_CREDENTIALS_NOT_FOUND);
                    post.append(concatParams(PayuConstants.USER_CREDENTIALS, this.mPaymentParams.getUserCredentials(), true));
                    break;
            }
        }


        if (mPaymentParams.getLoggedInPhoneNumber() != null) { // TODO add phone number validation
            post.append(concatParams(PayuConstants.PHONE, "91"+mPaymentParams.getLoggedInPhoneNumber(), true));
        } else if (mPaymentParams.getPhone() != null) {
            post.append(concatParams(PayuConstants.PHONE, mPaymentParams.getPhone(), true));
        }

        if (mPaymentParams.getPgAppVersion() != null) {
            post.append(String.format("device_context={\"phonePeVersionCode\":\"%s\"}&", mPaymentParams.getPgAppVersion()));
        }

        if (mPaymentParams.getUserToken() != null) {
            post.append(concatParams(PayuConstants.USER_TOKEN_PAYMENT, mPaymentParams.getUserToken(), true));
        }

        if (isValidBeneficiaryAccountNumber()) {
            StringBuilder beneficiarydetail = new StringBuilder();
            beneficiarydetail.append("{");
            beneficiarydetail.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BENEFICIARY_ACCOUNT_NUMBER, mPaymentParams.getBeneficiaryAccountNumber()));
            if (isValidIfscCode())
                beneficiarydetail.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.IFSC_CODE, mPaymentParams.getIfscCode()));
            beneficiarydetail.deleteCharAt(beneficiarydetail.length()-1);
            beneficiarydetail.append("}");
            post.append(concatParams(PayuConstants.BENEFICIARY_DETAILS, beneficiarydetail.toString(), true));
        }

        //validate SI params if its SI payment mode
        if (mPaymentParams.getSiParams() != null) {
            PostData result = validateAndAppendSIParams(post);
            if (result != null)
                return result;
        }
        if (mPaymentParams.getSdkInfo() != null) {
            PostData result = validateAndAppendSDKInfoParams(post);
            if (result != null)
                return result;
        }
        if (mPaymentParams.getProductsList() != null && !mPaymentParams.getProductsList().isEmpty()) {
            String jsonArray = getProductDetailsJsonArray(mPaymentParams);
            post.append(concatParams(PayuConstants.PAYU_PRODUCT, jsonArray, true));
        }
        // optional fields.
        post.append(mPaymentParams.getOfferKey() != null ? concatParams(PayuConstants.OFFER_KEY, mPaymentParams.getOfferKey(), true) : "");
        post.append(mPaymentParams.getLastName() != null ? concatParams(PayuConstants.LASTNAME, mPaymentParams.getLastName(), true) : "");
        post.append(mPaymentParams.getAddress1() != null ? concatParams(PayuConstants.ADDRESS1, mPaymentParams.getAddress1(), true) : "");
        post.append(mPaymentParams.getAddress2() != null ? concatParams(PayuConstants.ADDRESS2, mPaymentParams.getAddress2(), true) : "");
        post.append(mPaymentParams.getCity() != null ? concatParams(PayuConstants.CITY, mPaymentParams.getCity(), true) : "");
        post.append(mPaymentParams.getState() != null ? concatParams(PayuConstants.STATE, mPaymentParams.getState(), true) : "");
        post.append(mPaymentParams.getCountry() != null ? concatParams(PayuConstants.COUNTRY, mPaymentParams.getCountry(), true) : "");
        post.append(mPaymentParams.getZipCode() != null ? concatParams(PayuConstants.ZIPCODE, mPaymentParams.getZipCode(), true) : "");
        post.append(mPaymentParams.getCodUrl() != null ? concatParams(PayuConstants.CODURL, mPaymentParams.getCodUrl(), true) : "");
        post.append(mPaymentParams.getDropCategory() != null ? concatParams(PayuConstants.DROP_CATEGORY, mPaymentParams.getDropCategory(), true) : "");
        post.append(mPaymentParams.getEnforcePayMethod() != null ? concatParams(PayuConstants.ENFORCE_PAYMETHOD, mPaymentParams.getEnforcePayMethod(), true) : "");
        post.append(mPaymentParams.getCustomNote() != null ? concatParams(PayuConstants.CUSTOM_NOTE, mPaymentParams.getCustomNote(), true) : "");
        post.append(mPaymentParams.getNoteCategory() != null ? concatParams(PayuConstants.NOTE_CATEGORY, mPaymentParams.getNoteCategory(), true) : "");
        post.append(mPaymentParams.getShippingFirstName() != null ? concatParams(PayuConstants.SHIPPING_FIRSTNAME, mPaymentParams.getShippingFirstName(), true) : "");
        post.append(mPaymentParams.getShippingLastName() != null ? concatParams(PayuConstants.SHIPPING_LASTNAME, mPaymentParams.getShippingLastName(), true) : "");
        post.append(mPaymentParams.getShippingAddress1() != null ? concatParams(PayuConstants.SHIPPING_ADDRESS1, mPaymentParams.getShippingAddress1(), true) : "");
        post.append(mPaymentParams.getShippingAddress2() != null ? concatParams(PayuConstants.SHIPPING_ADDRESS2, mPaymentParams.getShippingAddress2(), true) : "");
        post.append(mPaymentParams.getShippingCity() != null ? concatParams(PayuConstants.SHIPPING_CITY, mPaymentParams.getShippingCity(), true) : "");
        post.append(mPaymentParams.getShippingState() != null ? concatParams(PayuConstants.SHIPPING_STATE, mPaymentParams.getShippingState(), true) : "");
        post.append(mPaymentParams.getShippingCounty() != null ? concatParams(PayuConstants.SHIPPING_CONTRY, mPaymentParams.getShippingCounty(), true) : "");
        post.append(mPaymentParams.getShippingZipCode() != null ? concatParams(PayuConstants.SHIPPING_ZIPCODE, mPaymentParams.getShippingZipCode(), true) : "");
        post.append(mPaymentParams.getShippingPhone() != null ? concatParams(PayuConstants.SHIPPING_PHONE, mPaymentParams.getShippingPhone(), true) : "");
        post.append(mPaymentParams.getSplitPaymentDetails() != null ? concatParams(PayuConstants.SPLIT_REQUEST, mPaymentParams.getSplitPaymentDetails(), true) : "");
        post.append(mPaymentParams.getOutletId() != null ? concatParams(PayuConstants.OUTLET_ID, mPaymentParams.getOutletId(), true) : "");
        post.append(mPaymentParams.getAdditionalCharges() != null ? concatParams(PayuConstants.ADDITIONAL_CHARGES, mPaymentParams.getAdditionalCharges(), true) : "");
        post.append(mPaymentParams.getPercentageAdditionalCharges() != null ? concatParams(PayuConstants.PERCENTAGE_ADDITIONAL_CHARGES, mPaymentParams.getPercentageAdditionalCharges(), true) : "");
        JSONArray chargeUUidArray = getChargeUuidJsonArray(mPaymentParams.getChargeUUIDs());
        if (chargeUUidArray != null) {
            post.append(concatParams(PayuConstants.PAYU_CHARGE_UUIDS, chargeUUidArray.toString(), true));
            post.append(mPaymentParams.getCombinationUUID() != null ? concatParams(PayuConstants.PAYU_COMBINATION_UUID, mPaymentParams.getCombinationUUID(), true) : "");
            post.append(mPaymentParams.getRequestUUID() != null ? concatParams(PayuConstants.PAYU_REQUEST_UUID, mPaymentParams.getRequestUUID(), true) : "");
        }
        appendApiVersion(post);
        //required for payments using cards tokenized outside PayU
        if (mPaymentParams.getCardTokenType() == 1) {
            post.append(concatParams(PayuConstants.STORED_CARD_TOKEN_TYPE, "" + mPaymentParams.getCardTokenType(), true));

            JSONObject jsonObjectValue = new JSONObject();
            try {
                if (mPaymentParams.getTokenizedCardAdditionalParam() == null)
                    return getReturnData(PayuErrors.MANDATORY_PARAM_ADDITIONAL_INFO_IS_MISSING);

                if (TextUtils.isEmpty(mPaymentParams.getTokenizedCardAdditionalParam().getLast4Digits()))
                    return getReturnData(PayuErrors.MANDATORY_PARAM_LAST4DIGITS_IS_MISSING);
                else
                    jsonObjectValue.put(PayuConstants.TOKENISED_CARD_LAST_4_DIGITS, mPaymentParams.getTokenizedCardAdditionalParam().getLast4Digits());

                if (TextUtils.isEmpty(mPaymentParams.getTokenizedCardAdditionalParam().getTavv()))
                    return getReturnData(PayuErrors.MANDATORY_PARAM_TAVV_IS_MISSING);
                else
                    jsonObjectValue.put(PayuConstants.TOKENISED_CARD_TAVV, mPaymentParams.getTokenizedCardAdditionalParam().getTavv());

                if (TextUtils.isEmpty(mPaymentParams.getTokenizedCardAdditionalParam().getTrid()))
                    return getReturnData(PayuErrors.MANDATORY_PARAM_TRID_IS_MISSING);
                else
                    jsonObjectValue.put(PayuConstants.TOKENISED_CARD_TRID, mPaymentParams.getTokenizedCardAdditionalParam().getTrid());

                if (TextUtils.isEmpty(mPaymentParams.getTokenizedCardAdditionalParam().getTokenRefNo()))
                    return getReturnData(PayuErrors.MANDATORY_PARAM_TOKEN_REF_NO_IS_MISSING);
                else
                    jsonObjectValue.put(PayuConstants.TOKENISED_CARD_TOKENREFNO, mPaymentParams.getTokenizedCardAdditionalParam().getTokenRefNo());

                post.append(concatParams(PayuConstants.STORED_CARD_ADDITIONAL_INFO, jsonObjectValue.toString(), true));
            } catch (JSONException e) {
                return getReturnData(PayuErrors.INVALID_DATA, PayuErrors.INVALID_TOKENIZED_CARD_DETAILS);
            }
        }

        if (mPaymentParams.getSkuCartDetails() != null) {
            post.append(concatParams(PayuConstants.P_CART_DETAILS, mPaymentParams.getSkuCartDetails(), false));
        }
        // Lets  make the analytics call here.
        // lets setup the user inputs.


        // Save store card consent for All modes
        post.append(this.mPaymentParams.getStoreCard() == 1 ? concatParams(PayuConstants.STORED_CARD, "" + this.mPaymentParams.getStoreCard(), true) : "");

        if (mPaymentParams.getDeviceDetails() != null)
            post.append(concatParams(PayuConstants.P_DEVICE_DETAILS, this.mPaymentParams.getDeviceDetails(), false));



        Log.d(TAG,"PaymenMode ="+mPaymentMode);
        switch (mPaymentMode) {
            case PayuConstants.CC:  //credit/debit/stored card/one click card
                if (mPaymentParams.getSiParams() == null) {
                    post.append(concatParams(PayuConstants.PG, PayuConstants.CC, true));
                    String bankCode = PayuConstants.CC;
                    if (mPaymentParams.getBankCode() != null && !mPaymentParams.getBankCode().isEmpty())
                        bankCode = mPaymentParams.getBankCode();
                    post.append(concatParams(PayuConstants.BANK_CODE, bankCode, true));
                }

                if (mPaymentParams.getLookupId() != null && !mPaymentParams.getLookupId().isEmpty())
                    post.append(concatParams(PayuConstants.LOOKUPID, mPaymentParams.getLookupId(), true));

                if (null != this.mPaymentParams.getCardNumber() && validateCardNumber(this.mPaymentParams.getCardNumber())) { // card payment
                    // okay its a valid card number
                    post.append(concatParams(PayuConstants.CC_NUM, this.mPaymentParams.getCardNumber(), true));
                    // if card number is not smae then validate cvv and expiry.
                    if (!getIssuer(this.mPaymentParams.getCardNumber()).contentEquals(PayuConstants.SMAE)) {
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv(), true));
                        } else {
                            return getReturnData(PayuErrors.INVALID_CVV_EXCEPTION, PayuErrors.INVALID_CVV);
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear(), true));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth(), true));
                            } else {
                                return getReturnData(PayuErrors.CARD_EXPIRED_EXCEPTION, PayuErrors.CARD_EXPIRED);
                            }
                        } catch (NumberFormatException e) {
                            return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.CARD_EXPIRED); // todo wrong expiry format
                        } catch (Exception e) {
                            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.CARD_EXPIRED);
                        }
                    } else { // some smae might have cvv, make sure that we have have added them in our post params.
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv(), true));
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear(), true));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth(), true));
                            }
                        } catch (Exception e) {
                            return getReturnData(PayuErrors.INVALID_PARAMETER, PayuErrors.INVALID_EXPIRY_DATE);
                        }
                    }

                    // if name on card is not given use default name on card as "PayuUser"
                    String nameOnCard = null != this.mPaymentParams.getNameOnCard() && this.mPaymentParams.getNameOnCard().trim().length() > 0 ? this.mPaymentParams.getNameOnCard() : "PayuUser";
                    // if card name is not given use name on card instead
                    String cardName = null != this.mPaymentParams.getCardName() ? this.mPaymentParams.getCardName() : nameOnCard;
                    post.append(concatParams(PayuConstants.CC_NAME, nameOnCard, true));
                    if (this.mPaymentParams.getStoreCard() == 1) {
                        post.append(concatParams(PayuConstants.CARD_NAME, cardName, true));
//                            post.append(this.mPaymentParams.getEnableOneClickPayment() == 1 ? concatParams(PayuConstants.ONE_CLICK_CHECKOUT, "" + this.mPaymentParams.getEnableOneClickPayment()) : "");
                    }

                    // TODO add validation for store_card and user_credentials
                    // thats it we can return post Data
                    return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                } else if (null != this.mPaymentParams.getCardToken()) {
                    // its stored card payment! we gotta verify user credentials
                    post.append(concatParams(PayuConstants.STORE_CARD_TOKEN, this.mPaymentParams.getCardToken(), true));
                    if (this.mPaymentParams.getCardBin() != null) {
                        // here we have the card bin we can validate cvv, expiry
                        if (!getIssuer(this.mPaymentParams.getCardBin()).contentEquals(PayuConstants.SMAE)) {
                            if (this.mPaymentParams.getCvv() == null) {
                                return getReturnData(PayuErrors.INVALID_CVV);
                            }
                            try {
                                if (!validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                    return getReturnData(PayuErrors.CARD_EXPIRED);
                                }
                            } catch (Exception e) {
                                return getReturnData(PayuErrors.INVALID_EXPIRY_DATE);
                            }

                        }
                    }
                    if (this.mPaymentParams.getCvv() != null &&  !this.mPaymentParams.getCvv().isBlank() )
                        post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv(), true)); // its not necessary that all the stored cards should have a cvv && we dont have card number so no validation.
                    if (this.mPaymentParams.getExpiryMonth() != null  && !this.mPaymentParams.getExpiryMonth().isBlank())
                        post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth(), true));
                    if (this.mPaymentParams.getExpiryYear() != null  && !this.mPaymentParams.getExpiryYear().isBlank())
                        post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear(), true));

                    post.append(this.mPaymentParams.getNameOnCard() == null ? concatParams(PayuConstants.CC_NAME, "PayuUser", true) : concatParams(PayuConstants.CC_NAME, mPaymentParams.getNameOnCard(), true));

//                        post.append(this.mPaymentParams.getEnableOneClickPayment() == 1 ? concatParams(PayuConstants.ONE_CLICK_CHECKOUT, "" + this.mPaymentParams.getEnableOneClickPayment()) : "");

                    // okey we have data
                    return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                } else {
                    return getReturnData(PayuErrors.INVALID_CARD_NUMBER_EXCEPTION, PayuErrors.INVALID_CARD_NUMBER);
                }
            case PayuConstants.NB: // netbanking
                if (this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // assuming we have a valid bank code now.
                    if (mPaymentParams.getSiParams() != null)
                        post.append(concatParams(PayuConstants.PG, PayuConstants.ENACH, true));
                    else
                        post.append(concatParams(PayuConstants.PG, PayuConstants.NB, true));
                    post.append(concatParams(PayuConstants.BANK_CODE, mPaymentParams.getBankCode(), true));

                    if (!isDataValidForTPV())
                        return getReturnData(PayuErrors.MANDATORY_PARAM_BENEFICIARY_ACCOUNT_NUMBER_IS_MISSING);

                } else {
                    return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.EMI: // emi
                if (this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // TODO: add proper validation for bankcode.
                    post.append(concatParams(PayuConstants.PG, PayuConstants.EMI, true));
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode(), true));
                    // Pan number in case of hdfc and icici cardless emi
                    if (mPaymentParams.getPanNumber() != null && !mPaymentParams.getPanNumber().isEmpty())
                        post.append(concatParams(PayuConstants.PANNUMBER, this.mPaymentParams.getPanNumber(), true));

                    // Subvention amount check
                    if (null != mPaymentParams.getSubventionAmount() && !mPaymentParams.getSubventionAmount().isEmpty()) {
                        // number validation
                        Double subventionAmount = 0.0;
                        try { // this will take care of null, empty check also!
                            subventionAmount = Double.parseDouble(mPaymentParams.getSubventionAmount());
                        } catch (NumberFormatException e) {
                            return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_1);
                        } catch (NullPointerException e) {
                            return getReturnData(PayuErrors.INVALID_SUBVENTION_AMOUNT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_1);
                        }

                        // subvention amount validation formula:  subvention_amount <= amount - offer_amount
                        // sdk does not deal with offer amount
                        // subvention amount should not be negative and should not be greater than amount.
                        if (subventionAmount >= 0 && subventionAmount <= Double.parseDouble(mPaymentParams.getAmount())) {
                            post.append(concatParams(PayuConstants.SUBVENTION_AMOUNT, mPaymentParams.getSubventionAmount(), true));
                        } else {
                            return getReturnData(PayuErrors.INVALID_SUBVENTION_AMOUNT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_2);
                        }
                    }
                    if (PayuUtils.supportedBankCodesForCardlessEmi(this.mPaymentParams.getBankCode())) {
                        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
                    }

                    if (PayuUtils.supportedBankCodesForDCEmi(this.mPaymentParams.getBankCode()) || this.mPaymentParams.getBankCode().startsWith(PayuConstants.BAJFIN)) {
                        if (validateCardNumber("" + this.mPaymentParams.getCardNumber())) {
                            // okay its a valid card number
                            post.append(concatParams(PayuConstants.CC_NUM, "" + this.mPaymentParams.getCardNumber(), true));
                            return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
                        } else {
                            return getReturnData(PayuErrors.INVALID_CARD_NUMBER_EXCEPTION, PayuErrors.INVALID_CARD_NUMBER);
                        }
                    }
                    // lets validate card number
                    else if (validateCardNumber("" + this.mPaymentParams.getCardNumber())) {
                        // okay its a valid card number
                        post.append(concatParams(PayuConstants.CC_NUM, "" + this.mPaymentParams.getCardNumber(), true));
                        // if card number is not smae then validate cvv and expiry.
                        if (!getIssuer("" + this.mPaymentParams.getCardNumber()).contentEquals(PayuConstants.SMAE)) {
                            if (validateCvv("" + this.mPaymentParams.getCardNumber(), "" + this.mPaymentParams.getCvv())) {
                                post.append(concatParams(PayuConstants.C_CVV, "" + this.mPaymentParams.getCvv(), true));
                            } else {
                                return getReturnData(PayuErrors.INVALID_CVV_EXCEPTION, PayuErrors.INVALID_CVV);
                            }
                            try {
                                if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                    post.append(concatParams(PayuConstants.CC_EXP_YR, "" + this.mPaymentParams.getExpiryYear(), true));
                                    post.append(concatParams(PayuConstants.CC_EXP_MON, "" + this.mPaymentParams.getExpiryMonth(), true));
                                } else {
                                    return getReturnData(PayuErrors.CARD_EXPIRED_EXCEPTION, PayuErrors.CARD_EXPIRED);
                                }
                            } catch (NumberFormatException e) {
                                return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.CARD_EXPIRED); // TODO add proper message cast exception
                            }
                        }

                        post.append(this.mPaymentParams.getNameOnCard() == null ? concatParams(PayuConstants.CC_NAME, "PayuUser", true) : concatParams(PayuConstants.CC_NAME, mPaymentParams.getNameOnCard(), true));
                        if (this.mPaymentParams.getStoreCard() == 1) {
                            post.append(this.mPaymentParams.getCardName() == null ? concatParams(PayuConstants.CARD_NAME, "PayuUser", true) : concatParams(PayuConstants.NAME_ON_CARD, mPaymentParams.getCardName(), true));
                        }
                        // TODO add validation for store_card and user_credentials
                        // thats it we can return post Data
                        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                    } else {
                        return getReturnData(PayuErrors.INVALID_CARD_NUMBER_EXCEPTION, PayuErrors.INVALID_CARD_NUMBER);
                    }
                } else {
                    return getReturnData(PayuErrors.INVALID_EMI_DETAILS);
                }
            case PayuConstants.CASH: // cash
                post.append(concatParams(PayuConstants.PG, PayuConstants.CASH, true)); // cash card
                // lets validate payment bank code
                if (this.mPaymentParams != null && this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // assuming we have a valid bank code now.
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode(), true));
                } else {
                    return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
            case PayuConstants.PAYU_MONEY: // payu money.
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.PAYUW.toLowerCase(), true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.WALLET, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());

            /*case PayuConstants.LAZYPAY: // payu money.
                post.append(concatParams(PayuConstants.BANK_CODE, "", true));
                post.append(concatParams(PayuConstants.PG, "", true));
                try {
                    if (mPaymentParams.getNotifyURL() != null && !mPaymentParams.getNotifyURL().equalsIgnoreCase("")) {
                        post.append(concatParams(PayuConstants.NOTIFY_URL, mPaymentParams.getNotifyURL(), true));
                    }
                } catch (Exception e) {
                    return getReturnData(PayuErrors.INVALID_DATA, PayuErrors.INVALID_LAZYPAY_NOTIFY_URL_DETAILS);
                }
                post.append(concatParams(PayuConstants.ENFORCE_PAYMETHOD, PayuConstants.LAZYPAY.toUpperCase(), true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            */
            case PayuConstants.SAMSUNG_PAY:
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.SAMSUNG_PAY.toLowerCase(), true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.SAMSUNG_PAY, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
            case PayuConstants.NEFT_RTGS:
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.EFT_AXIS, true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.NEFT_RTGS, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.UPI:
                // Virtual address Check (vpa check)
                // 1)Vpa length should be less than or equal to 50
                // 2)It can be alphanumeric and can contain a dot(.).
                // 3)It should contain a @

                // NULL Check for Virtual address

                if (mPaymentParams.getVpa() == null) {
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_NULL);
                } else if (mPaymentParams.getVpa().trim().length() == 0) { // Vpa should not be empty
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_EMPTY);
                } else if (mPaymentParams.getVpa().trim().length() > PayuConstants.MAX_VPA_SIZE) {
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_GREATER); // Vpa size should not be greater than 50
                } else {
                    String userVirtualAddress = mPaymentParams.getVpa().trim();
                    // Pattern pattern = Pattern.compile("^([A-Za-z0-9\\.])+\\@[A-Za-z0-9]+$"); //Vpa name can be alphanumeric and can contain a dot(.) preceded by @. After '@' only alphanumeric is allowed.
                    Pattern pattern = Pattern.compile(".+@.+"); // VPA Check: non empty + @ + non empty
                    Matcher matcher = pattern.matcher(userVirtualAddress);
                    if (!matcher.matches()) {
                        return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.INVALID_VPA_MESSAGE);
                    }
                }

                if (isValidBeneficiaryAccountNumber()) {
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.UPITPV, true));
                } else
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.UPI.toLowerCase(), true));

                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI, true));
                post.append(concatParams(PayuConstants.VPA, mPaymentParams.getVpa().trim(), true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.TEZOMNI:
                // Virtual address Check (vpa check)
                // 1)Vpa length should be less than or equal to 50
                // 2)It can be alphanumeric and can contain a dot(.).
                // 3)It should contain a @

                // NULL Check for Virtual address

                if (isValidBeneficiaryAccountNumber()) {
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.UPITPV, true));
                } else
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.TEZOMNI.toLowerCase(), true));

                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI, true));
                post.append(concatParams(PayuConstants.VPA_PHONE, mPaymentParams.getPhone().trim(), true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.TEZ:
                if (isValidBeneficiaryAccountNumber()) {
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.TEZTPV, true));
                } else
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.TEZ, true));

                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI, true));
                if (!TextUtils.isEmpty(mPaymentParams.getVpa()))
                    post.append(concatParams(PayuConstants.VPA, mPaymentParams.getVpa().trim(), true));

                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.PHONEPE_INTENT:
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.PHONEPE_INTENT, true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.CASH, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.UPI_INTENT:
                if (isValidBeneficiaryAccountNumber()) {
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.INTTPV, true));
                } else
                    post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.UPI_INTENT, true));

                if (this.mPaymentParams != null && this.mPaymentParams.getAppName() != null && !this.mPaymentParams.getAppName().isEmpty()) {
                    post.append(concatParams(PayuConstants.UPI_APP_NAME.toLowerCase(), this.mPaymentParams.getAppName(), false));
                }

                if (this.mPaymentParams != null && this.mPaymentParams.getUserAgent() != null && !this.mPaymentParams.getUserAgent().isEmpty()) {
                    post.append(concatParams(PayuConstants.UPI_USER_AGENT, this.mPaymentParams.getUserAgent(), false));
                }

                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.OLA_MONEY:
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.OLAM, true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.CASH, true));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.PAY_BY_REWARDS:
                if (mPaymentParams.getPhone() == null || mPaymentParams.getPhone().equalsIgnoreCase("")) // empty string is allowed
                    return getReturnData(PayuErrors.MANDATORY_PARAM_PHONE_IS_MISSING);

                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.TWID, true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.CASH, true));
                post.append(concatParams(PayuConstants.SALT_VERSION, "1", true));

                if (mPaymentParams.getTwidCustomerHash() != null && !mPaymentParams.getTwidCustomerHash().equalsIgnoreCase("")) {
                    post.append(concatParams(PayuConstants.TWID_CUSTOMER_HASH, mPaymentParams.getTwidCustomerHash(), true));
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.SODEXO:

                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.SODEXO, true));
                post.append(concatParams(PayuConstants.PG, PayuConstants.MC, true));

                if (mPaymentParams.getSodexoSourceId() != null && !mPaymentParams.getSodexoSourceId().equalsIgnoreCase("")) {
                    post.append(concatParams(PayuConstants.SOURCE_ID, mPaymentParams.getSodexoSourceId(), true));
                } else if (null != this.mPaymentParams.getCardNumber() && validateCardNumber(this.mPaymentParams.getCardNumber())) { // card payment
                    // okay its a valid card number
                    post.append(concatParams(PayuConstants.CC_NUM, this.mPaymentParams.getCardNumber(), true));
                    // if card number is not smae then validate cvv and expiry.
                    if (!getIssuer(this.mPaymentParams.getCardNumber()).contentEquals(PayuConstants.SMAE)) {
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv(), true));
                        } else {
                            return getReturnData(PayuErrors.INVALID_CVV_EXCEPTION, PayuErrors.INVALID_CVV);
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear(), true));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth(), true));
                            } else {
                                return getReturnData(PayuErrors.CARD_EXPIRED_EXCEPTION, PayuErrors.CARD_EXPIRED);
                            }
                        } catch (NumberFormatException e) {
                            return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.CARD_EXPIRED); // todo wrong expiry format
                        } catch (Exception e) {
                            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.CARD_EXPIRED);
                        }
                    } else { // some smae might have cvv, make sure that we have have added them in our post params.
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv(), true));
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear(), true));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth(), true));
                            }
                        } catch (Exception e) {
                            return getReturnData(PayuErrors.INVALID_PARAMETER, PayuErrors.INVALID_EXPIRY_DATE);
                        }
                    }

                    // if name on card is not given use default name on card as "PayuUser"
                    String nameOnCard = null != this.mPaymentParams.getNameOnCard() && this.mPaymentParams.getNameOnCard().trim().length() > 0 ? this.mPaymentParams.getNameOnCard() : "PayuUser";
                    // if card name is not given use name on card instead
                    String cardName = null != this.mPaymentParams.getCardName() ? this.mPaymentParams.getCardName() : nameOnCard;
                    post.append(concatParams(PayuConstants.CC_NAME, nameOnCard, true));
//                    if (this.mPaymentParams.getSaveSodexoCard() == 1) {

                    post.append(concatParams(PayuConstants.CARD_NAME, cardName, true));
                    post.append(this.mPaymentParams.getSaveSodexoCard() == 1 || this.mPaymentParams.getSaveSodexoCard()==0  ? concatParams(PayuConstants.SAVE_SODEXO_CARD, "" + this.mPaymentParams.getSaveSodexoCard(), true) : "");
//                    }
                }


//                    post.append(this.mPaymentParams.getSaveSodexoCard() == 1 ? concatParams(PayuConstants.SAVE_SODEXO_CARD, "" + this.mPaymentParams.getSaveSodexoCard()) : "");
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.BNPL:
                post.append(concatParams(PayuConstants.PG, PayuConstants.BNPL, true)); // cash card
                // lets validate payment bank code
                if (this.mPaymentParams != null && this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // assuming we have a valid bank code now.
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode(), true));
                } else {
                    return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());

            case PayuConstants.CLW:
                return appendClwParams(post, PayuConstants.CLW.toUpperCase());
            case PayuConstants.OLW:
                return appendOlwParams(post);
        }


        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());


    }

    private PostData validateAndAppendSIParams(StringBuilder post) {
        SIParams siParams = mPaymentParams.getSiParams();
        if (!PayuUtils.validateCardStartDate(siParams.getSi_details().getPaymentStartDate()))
            return getReturnData(PayuErrors.INVALID_START_DATE_EXCEPTION, PayuErrors.ERROR_INVALID_START_DATE_EXCEPTION);

        if (!PayuUtils.validateCardEndDate(siParams.getSi_details().getPaymentEndDate()))
            return getReturnData(PayuErrors.INVALID_END_DATE_EXCEPTION, PayuErrors.ERROR_INVALID_END_DATE_EXCEPTION);

        if (!PayuUtils.compareDates(siParams.getSi_details().getPaymentStartDate(), siParams.getSi_details().getPaymentEndDate()))
            return getReturnData(PayuErrors.INVALID_DATE_EXCEPTION, PayuErrors.ERROR_INVALID_START_END_DATE_EXCEPTION);

        StringBuilder siObjectValue = new StringBuilder();
        siObjectValue.append("{");
        siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_START_DATE, siParams.getSi_details().getPaymentStartDate()));
        siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_END_DATE, siParams.getSi_details().getPaymentEndDate()));

        try {
            if (siParams.getSi_details().isPreAuthorize()) {
                post.append(concatParams(PayuConstants.PRE_AUTHORIZE, "" + mPaymentParams.getSiParams().getSi_details().getPreAuthorizeAsInt(), false));
            } else {
                if (siParams.getSi_details().getBillingCycle() == null ||
                        !Arrays.asList(BillingCycle.values()).contains(siParams.getSi_details().getBillingCycle()))
                    return getReturnData(PayuErrors.INVALID_BILLING_CYCLE_PARAMETER_EXCEPTION, PayuErrors.ERROR_INVALID_BILLING_CYCLE_EXCEPTION);
                if (!TextUtils.isEmpty(siParams.getSi_details().getRemarks()) && siParams.getSi_details().getRemarks().length() > 50) {
                    return getReturnData(PayuErrors.INVALID_REMARKS_LENGTH_ERROR, PayuErrors.ERROR_REMARKS_MAX_LENGTH_CROSSED);
                }
                post.append(concatParams(PayuConstants.SI_VALUE, siParams.getSi(), true));                //added "" at end of 1 and 0 to make it string
                post.append(concatParams(PayuConstants.SI_FREE_TRIAL, siParams.isFree_trial() ? 1 + "" : 0 + "", true));
                siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_AMOUNT, siParams.getSi_details().getBillingAmount()));
                siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_CURRENCY, siParams.getSi_details().getBillingCurrency()));
                siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_CYCLE, siParams.getSi_details().getBillingCycle()));
                siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_INTERVAL, siParams.getSi_details().getBillingInterval()));
                if (!TextUtils.isEmpty(siParams.getSi_details().getRemarks())){
                    siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_REMARKS, siParams.getSi_details().getRemarks()));
                }
                if (mPaymentMode.equalsIgnoreCase(PayuConstants.NB)) {
                    BeneficiaryDetails beneficiarydetail = siParams.getBeneficiarydetail();
                    if (TextUtils.isEmpty(beneficiarydetail.getBeneficiaryAccountNumber()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_ACCOUNT_NUMBER_IS_MISSING);
                    if (beneficiarydetail.getBeneficiaryAccountNumber().length() < 8)
                        return getReturnData(PayuErrors.INVALID_ACCOUNT_NUMBER_EXCEPTION, PayuErrors.ERROR_INVALID_ACCOUNT_NUMBER_EXCEPTION);
                    if (TextUtils.isEmpty(beneficiarydetail.getBeneficiaryName()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_ACCOUNT_HOLDER_NAME_IS_MISSING);
                    if (TextUtils.isEmpty(beneficiarydetail.getBeneficiaryIfsc()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_ACCOUNT_IFSC_IS_MISSING);
                    if (beneficiarydetail.getBeneficiaryAccountType() == null ||
                            !Arrays.asList(BeneficiaryAccountType.values()).contains(beneficiarydetail.getBeneficiaryAccountType()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_ACCOUNT_TYPE_IS_MISSING);

                    StringBuilder beneficiaryObjectValue = new StringBuilder();
                    beneficiaryObjectValue.append("{");
                    beneficiaryObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BENEFICIARY_NAME, beneficiarydetail.getBeneficiaryName()));
                    beneficiaryObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BENEFICIARY_ACCOUNT_NUMBER, beneficiarydetail.getBeneficiaryAccountNumber()));
                    beneficiaryObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.IFSC_CODE, beneficiarydetail.getBeneficiaryIfsc()));
                    beneficiaryObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BENEFICIARY_ACCOUNT_TYPE, beneficiarydetail.getBeneficiaryAccountType()));
                    if (!TextUtils.isEmpty(beneficiarydetail.getVerificationMode())) {
                        beneficiaryObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.VERTIFICATION_MODE, beneficiarydetail.getVerificationMode()));
                    }
                    beneficiaryObjectValue.deleteCharAt(beneficiaryObjectValue.length()-1);
                    beneficiaryObjectValue.append("}");

                    post.append(concatParams(PayuConstants.BENEFICIARY_DETAILS, beneficiaryObjectValue.toString(), true));
                } else if (mPaymentMode.equalsIgnoreCase(PayuConstants.CC)) {
                    if (TextUtils.isEmpty(siParams.getCcCardType()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_BANK_CODE_IS_MISSING);
                    post.append(concatParams(PayuConstants.BANK_CODE, siParams.getCcCardType(), true));
                    if (TextUtils.isEmpty(siParams.getCcCategory()))
                        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_PAYMENT_MODE_IS_MISSING);
                    post.append(concatParams(PayuConstants.PG, siParams.getCcCategory(), true));
                } else if (mPaymentMode.equalsIgnoreCase(PayuConstants.UPI) || mPaymentMode.equals(PayuConstants.UPI_INTENT)) {
                    if (siParams.getSi_details().getBillingLimit() != null && Arrays.asList(BillingLimit.values()).contains(siParams.getSi_details().getBillingLimit())) {
                        siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_LIMIT, siParams.getSi_details().getBillingLimit().name()));
                    }
                    if (siParams.getSi_details().getBillingRule() != null && Arrays.asList(BillingRule.values()).contains(siParams.getSi_details().getBillingRule())) {
                        siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_RULE, siParams.getSi_details().getBillingRule().name()));
                    }
                    if (!TextUtils.isEmpty(siParams.getSi_details().getBillingDate())) {
                        siObjectValue.append(PayuUtils.appendKeyValueToStingBuilder(PayuConstants.BILLING_DATE, siParams.getSi_details().getBillingDate()));
                    }
                }
            }
        } catch (Exception e) {
            return getReturnData(PayuErrors.INVALID_DATA, PayuErrors.INVALID_SI_DETAILS);
        }
        siObjectValue.deleteCharAt(siObjectValue.length()-1);
        siObjectValue.append("}");
        post.append(concatParams(PayuConstants.SI_DETAILS, siObjectValue.toString(), true));
        return null;
    }

    private boolean isDataValidForTPV() {
        String bankCode = this.mPaymentParams.getBankCode();
        if (bankCode != null && bankCode.endsWith(PayuConstants.TPV) && !isValidBeneficiaryAccountNumber())
            return false;
        else
            return true;
    }

    private boolean isValidBeneficiaryAccountNumber() {
        return this.mPaymentParams.getBeneficiaryAccountNumber() != null && !this.mPaymentParams.getBeneficiaryAccountNumber().isEmpty();
    }

    private boolean isValidIfscCode() {
        return this.mPaymentParams.getIfscCode() != null && !this.mPaymentParams.getIfscCode().isEmpty();
    }

    private PostData validateAndAppendSDKInfoParams(StringBuilder post) {
        JSONObject sdkInfo = new JSONObject();
        JSONObject ephemInfo = new JSONObject();
        JSONObject threeDSRequest = new JSONObject();
        JSONObject sdkDeviceRenderOptionDetails = new JSONObject();

        if (TextUtils.isEmpty(mPaymentParams.getSdkInfo().getSdkEncData()))
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_ENC_DATA_IS_MISSING);
        if (TextUtils.isEmpty(mPaymentParams.getSdkInfo().getSdkAppID()))
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_APP_ID_IS_MISSING);
        if (TextUtils.isEmpty(mPaymentParams.getSdkInfo().getSdkReferenceNumber()))
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_REF_NUM_IS_MISSING);
        if (TextUtils.isEmpty(mPaymentParams.getSdkInfo().getSdkTransID()))
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_TRANS_ID_IS_MISSING);
        if (mPaymentParams.getSdkInfo().getSdkEphemPubKey() == null)
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_EPHEM_IS_MISSING);
        if (mPaymentParams.getSdkInfo().getSdkEphemPubKey().getCrv() == null)
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_EPHEM_CRV_IS_MISSING);
        if (mPaymentParams.getSdkInfo().getSdkEphemPubKey().getKty() == null)
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_EPHEM_KTY_IS_MISSING);
        if (mPaymentParams.getSdkInfo().getSdkEphemPubKey().getX() == null)
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_EPHEM_X_IS_MISSING);
        if (mPaymentParams.getSdkInfo().getSdkEphemPubKey().getKty() == null)
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.MANDATORY_PARAM_SDK_EPHEM_Y_IS_MISSING);


        try {
            sdkInfo.put(PayuConstants.THREEDS2_SDK_ENC_DATA, mPaymentParams.getSdkInfo().getSdkEncData());
            sdkInfo.put(PayuConstants.THREEDS2_SDK_APP_ID, mPaymentParams.getSdkInfo().getSdkAppID());
            sdkInfo.put(PayuConstants.THREEDS2_SDK_REFERENCE_NUMBER, mPaymentParams.getSdkInfo().getSdkReferenceNumber());
            sdkInfo.put(PayuConstants.THREEDS2_SDK_TRANS_ID, mPaymentParams.getSdkInfo().getSdkTransID());
            sdkInfo.put(PayuConstants.THREEDS2_SDK_MAX_TIMEOUT, mPaymentParams.getSdkInfo().getSdkMaxTimeout());


            sdkDeviceRenderOptionDetails.put(PayuConstants.THREEDS2_SDK_INTERFACE, mPaymentParams.getSdkInfo().getSdkDeviceRenderOptions().getSdkInterface());
            sdkDeviceRenderOptionDetails.put(PayuConstants.THREEDS2_SDK_UI_TYPE, new JSONArray(mPaymentParams.getSdkInfo().getSdkDeviceRenderOptions().getSdkUiType()));
            sdkInfo.put(PayuConstants.THREEDS2_SDK_DEVICE_RENDER_OPTIONS, sdkDeviceRenderOptionDetails);

            ephemInfo.put(PayuConstants.THREEDS2_SDK_EMPHEM_KEY_CRV, mPaymentParams.getSdkInfo().getSdkEphemPubKey().getCrv());
            ephemInfo.put(PayuConstants.THREEDS2_SDK_EMPHEM_KEY_KTY, mPaymentParams.getSdkInfo().getSdkEphemPubKey().getKty());
            ephemInfo.put(PayuConstants.THREEDS2_SDK_EMPHEM_KEY_X, mPaymentParams.getSdkInfo().getSdkEphemPubKey().getX());
            ephemInfo.put(PayuConstants.THREEDS2_SDK_EMPHEM_KEY_Y, mPaymentParams.getSdkInfo().getSdkEphemPubKey().getY());
            sdkInfo.put(PayuConstants.THREEDS2_SDK_EMPHEM_KEY, ephemInfo);
            threeDSRequest.put(PayuConstants.THREEDS2_SDK_INFO, sdkInfo);
            threeDSRequest.put(PayuConstants.THREEDS2_SDK_DEVICE_CHANNEL, PayuConstants.THREEDS2_SDK_DEVICE_CHANNEL_VALUE);
            threeDSRequest.put(PayuConstants.THREEDS2_SDK_VERSION, mPaymentParams.getThreeDSVersion());
        } catch (JSONException e) {
            return getReturnData(PayuErrors.INVALID_DATA, PayuErrors.INVALID_SDK_INFO);
        }
        post.append(concatParams(PayuConstants.THREEDS2_SDK_REQUEST_DATA, threeDSRequest.toString(), true));

        return null;
    }

    /**
     * This function validates loadAmount as a double value
     * @param loadAmount
     * @return
     */
    private boolean isValidAmount(String loadAmount) {
        if (loadAmount == null || loadAmount.isEmpty())
            return false;

        try {
            Double.parseDouble(loadAmount);
        } catch (NumberFormatException e) {
            return false;
        }
        return true;
    }

    private void appendApiVersion(StringBuilder post) {
        SIParams siParams = mPaymentParams.getSiParams();
        if (siParams != null && siParams.getApi_vesrion() != null) {
            post.append(concatParams(PayuConstants.SI_API_VERSION, siParams.getApi_vesrion(), true));
        } else if (mPaymentParams.getApiVersion() != null) {
            post.append(concatParams(PayuConstants.SI_API_VERSION, mPaymentParams.getApiVersion(), true));
        } else if (mPaymentParams.getProductsList() != null) {
            post.append(concatParams(PayuConstants.SI_API_VERSION, TPV_MF_API_VERSION_VALUE, true));
        } else if (siParams != null) {
            if (siParams.isFree_trial())
                post.append(concatParams(PayuConstants.SI_API_VERSION, PayuConstants.SI_FREE_TRIAL_API_VERSION, true));
            else
                post.append(concatParams(PayuConstants.SI_API_VERSION, SI_API_VERSION_VALUE, true));
        } else if (isValidBeneficiaryAccountNumber()) {
            post.append(concatParams(PayuConstants.SI_API_VERSION, TPV_API_VERSION_VALUE, true));
        }
    }

    /**
     *
     * @param chargeUuidList
     * @return chargeUuid jsonArray
     */
    private JSONArray getChargeUuidJsonArray(List<String> chargeUuidList) {
        if (chargeUuidList == null || chargeUuidList.isEmpty()) {
            return null;
        }
        JSONArray jsonArray = new JSONArray();
        for (String chargeUUid : chargeUuidList) {
            jsonArray.put(chargeUUid);
        }
        return jsonArray;
    }

    private PostData appendOlwParams(StringBuilder post) {
        OlwParams olwParams = mPaymentParams.getOlwParams();
        if (olwParams != null) {
            try {
                SetMPinEncryptedInfo setMPinEncryptedInfo = EncryptionUtils.encrypt(olwParams.getOtp(), olwParams.getMPin(), PayuConstants.PayUPaymentConstant.subPart1_payment + PayuConstants.PayUPaymentConstant.subPart2_payment + PayuConstants.PayUPaymentConstant.subPart3_payment);
                post.append(concatParams(PayuConstants.PAYU_PPI_INFO, getOlwJson(setMPinEncryptedInfo, olwParams), true));
                post.append(concatParams(PayuConstants.PayUPaymentConstant.PAYU_OLW_KEY, setMPinEncryptedInfo.getEncryptedAesKey(), true));

            } catch (Exception e) {
                Log.d(TAG, Objects.requireNonNull(e.getMessage()));
            }
        }

            post.append(concatParams(PayuConstants.PAYU_DEVICE_ID, olwParams.getDeviceId(), true));
            post.append(concatParams(PayuConstants.PG, PayuConstants.OLW.toUpperCase(), true)); // cash card
            // lets validate payment bank code
            if (this.mPaymentParams != null && this.mPaymentParams.getBankCode() != null && !this.mPaymentParams.getBankCode().isEmpty()) { // assuming we have a valid bank code now.
                post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode(), true));
            } else {
                return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
            }

        return appendLoadAmount(post);
    }

    private PostData appendLoadAmount(StringBuilder post) {
        if (this.mPaymentParams.getLoadAmount() != null) {
            if (isValidAmount(this.mPaymentParams.getLoadAmount())) {
                post.append(concatParams(PayuConstants.LOAD_AMOUNT, this.mPaymentParams.getLoadAmount(), true));
            } else {
                return getReturnData(PayuErrors.INVALID_AMOUNT_EXCEPTION, PayuErrors.INVALID_AMOUNT);
            }
        }
        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
    }

    /**
     * This function is used to prepare post params for Closed/Open Loop Wallet payment method
     * @param post
     * @return
     */
    private PostData appendClwParams(StringBuilder post, String pg) {
        post.append(concatParams(PayuConstants.PG, pg, true)); // cash card
        // lets validate payment bank code
        if (this.mPaymentParams != null && this.mPaymentParams.getBankCode() != null && !this.mPaymentParams.getBankCode().isEmpty()) { // assuming we have a valid bank code now.
            post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode(), true));
        } else {
            return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
        }
        // lets validate payment walletUrn
        if (this.mPaymentParams != null && this.mPaymentParams.getWalletUrn() != null && !this.mPaymentParams.getWalletUrn().isEmpty()) { // assuming we have a valid walletUrn now.
            post.append(concatParams(PayuConstants.WALLET_URN, this.mPaymentParams.getWalletUrn(), true));
        } else {
            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.INVALID_WALLET_URN);
        }
        return appendLoadAmount(post);
    }

    private String getOlwJson(SetMPinEncryptedInfo setMPinEncryptedInfo, OlwParams olwParams) {
        JSONObject olwJson = new JSONObject();
        try {
            olwJson.put(PayuConstants.PAYU_MPIN, setMPinEncryptedInfo.getmPin());
            olwJson.put(PayuConstants.PAYU_OTP, setMPinEncryptedInfo.getOtp());
            olwJson.put(PayuConstants.PAYU_DEVICE_ID, olwParams.getDeviceId());
            olwJson.put(PayuConstants.PAYU_TOKEN, olwParams.getToken());
        } catch (Exception e) {
            Log.d(TAG, Objects.requireNonNull(e.getMessage()));
        }
        return olwJson.toString();
    }

    private boolean isValidUrl(String url) {
        boolean isValid;
        if (url != null && !url.isEmpty()) {
            try {
                new URL(url).toURI();
                isValid = true;
            } catch (Exception e) {
                isValid = false;
            }
        } else {
            isValid = false;
        }
        return isValid;
    }
}
