package com.payu.otpparser

import android.Manifest
import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import com.google.android.gms.auth.api.phone.SmsRetriever
import com.payu.otpparser.Constants.NATIVE_SMS_READER_PERMISSION_ID


class OtpHandler constructor(
    var activity: ComponentActivity,
    var otpHandlerCallback: OtpHandlerCallback
) : DefaultLifecycleObserver,
    SMSReceiver.SmsBroadcastReceiverListener {
    private var smsBroadcastReceiver: SMSReceiver? = null
    private var fragCallback: FragCallback? = null

    private val REQUEST_CODE = 1003
    private var merchantKey = ""
    private var txnId = ""
    fun setBundle(bundle: Bundle) {
        if (null != bundle[Constants.MERCHANT_KEY])
            merchantKey = bundle[Constants.MERCHANT_KEY] as String
        if (null != bundle[Constants.TXNID])
            txnId = bundle[Constants.TXNID] as String

    }

    fun setFragCallback(fragCallback: FragCallback) {
        this.fragCallback = fragCallback
    }

    fun startListening() {
        if (!Utils.isPermissionAvailable(activity)) {
            if (!Utils.checkSmsPermissionAvailable(activity)) {
                L.v("startSmsUserConsent")
                startSmsUserConsent()
            } else {

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    if (Utils.getPermissionDenyingStatusSharedPref(
                            activity,
                            Manifest.permission.RECEIVE_SMS
                        ) && !Utils.checkNeverAskAgainSelectedRationale(
                            activity,
                            Manifest.permission.RECEIVE_SMS
                        )
                    ) {
                        startSmsUserConsent()
                        L.v("startSmsUserConsent after dont ask again")
                    } else {
                        L.v("requestPermissions")
                        ActivityCompat.requestPermissions(
                            activity,
                            arrayOf(Manifest.permission.RECEIVE_SMS), REQUEST_CODE
                        )
                    }
                }


            }
        } else {
            L.v("registerSMSReceiver")
            registerSMSReceiver()
        }
    }

    private fun startSmsUserConsent() {
        if (activity != null && !activity.isFinishing && !activity.isDestroyed) {
            try {
                SmsRetriever.getClient(activity).also {
                    //We can add user phone number or leave it blank
                    it?.startSmsUserConsent(null)
                        ?.addOnSuccessListener {
                            L.v("Consent SmsRetriever success")
                            registerConsentReceiver()
                        }
                        ?.addOnFailureListener {
                            L.v("Consent SmsRetriever failure" + it.message)

                        }
                }
            } catch (e: NoSuchMethodError) {
                L.v("Runtime Exception: NoSuchMethodError ")
            } catch (e: Exception) {
                L.v("Default Exception ")

            }
        }
    }

    fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (resultCode == Activity.RESULT_OK && null != data) {
            val message = data?.getStringExtra(SmsRetriever.EXTRA_SMS_MESSAGE)
            val code = message?.let { Utils.fetchVerificationCode(it) }
            PayUOtpParserConfig.otpCallback?.onOtpReceived(code!!)
        } else if (resultCode == Activity.RESULT_CANCELED) {
            PayUOtpParserConfig.otpCallback?.onUserDenied()

        }
        unregisterReceiver()
    }

    fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        when (requestCode) {
            REQUEST_CODE -> {
                if (grantResults.isNotEmpty() && grantResults[0] ==
                    PackageManager.PERMISSION_GRANTED
                ) {
                    if ((ContextCompat.checkSelfPermission(
                            activity,
                            Manifest.permission.RECEIVE_SMS
                        ) ==
                                PackageManager.PERMISSION_GRANTED)
                    ) {
                        L.v("Runtime Permission Granted")
                        registerSMSReceiver()
                    }
                } else {
                    L.v("Runtime Permission Denyied ")
                    if (null != activity && !activity?.isFinishing)
                        Utils.setPermissionDenyingStatus(activity, Manifest.permission.RECEIVE_SMS)
                    PayUOtpParserConfig.otpCallback?.onUserDenied()
                }

            }
        }

    }

    fun registerSMSReceiver() {
        if (null == smsBroadcastReceiver) {
            smsBroadcastReceiver = SMSReceiver(this)
            val intentFilter = IntentFilter("android.provider.Telephony.SMS_RECEIVED")
            intentFilter.priority = 999
            ContextCompat.registerReceiver(activity, smsBroadcastReceiver,
                intentFilter, ContextCompat.RECEIVER_EXPORTED)
        }
    }

    fun registerConsentReceiver() {
        if (null == smsBroadcastReceiver) {
//            L.v("registerReceiver")
            smsBroadcastReceiver = SMSReceiver(this)
            val intentFilter = IntentFilter(SmsRetriever.SMS_RETRIEVED_ACTION)
            ContextCompat.registerReceiver(activity, smsBroadcastReceiver,
                intentFilter, ContextCompat.RECEIVER_EXPORTED)

        }
    }

    override fun onSuccess(intent: Intent?) {
        L.v("onsuccess")
        try {
            if (Utils.checkifIntentIsValid(intent, activity)) {
                if (fragCallback != null)
                    fragCallback?.onFragCallback(intent!!, NATIVE_SMS_READER_PERMISSION_ID)
                else
                    activity.startActivityForResult(intent!!, NATIVE_SMS_READER_PERMISSION_ID)
            }
        } catch (e: ActivityNotFoundException) {
            e.printStackTrace()
        }
    }

    override fun onSuccess(message: String?) {
        L.v("onSuccess sms permission")
        val code = Utils.fetchVerificationCode(message!!)
        PayUOtpParserConfig.otpCallback?.onOtpReceived(code)
        unregisterReceiver()
    }

    override fun onFailure() {
        L.v("onFailure")
        PayUOtpParserConfig.otpCallback?.onUserDenied()
        unregisterReceiver()
    }

    override fun onDestroy(owner: LifecycleOwner) {
        L.v("onDestroy")
        unregisterReceiver()
        activity.lifecycle.removeObserver(this)
        otpHandlerCallback.lifeCycleOnDestroy()
    }

    fun stopSmsListening() {
        unregisterReceiver()
    }

    private fun unregisterReceiver() {
        if (null != smsBroadcastReceiver) {
            activity.unregisterReceiver(smsBroadcastReceiver)
            smsBroadcastReceiver = null
            L.v("unregisterReceiver")
        }
    }
}