package hu.kole.cleversectionviewadapter.listeners;

import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;

import hu.kole.cleversectionviewadapter.BaseCleverSectionAdapter;

/**
 * Created by koleszargergo on 4/18/16.
 */
public abstract class EndlessScrollListener extends RecyclerView.OnScrollListener {

    // The minimum amount of items to have below your current scroll position
    // before loading more.
    private int visibleThreshold = 3;
    // The current offset index of data you have loaded
    private int currentPage = 0;
    // The total number of items in the dataset after the last load
    private int previousTotalItemCount = 0;
    // True if we are still waiting for the last set of data to load.
    private boolean loading = true;
    // Sets the starting page index
    private int startingPageIndex = 0;

    private LinearLayoutManager mLinearLayoutManager;

    private BaseCleverSectionAdapter mAdapter;

    private boolean isProgressNeeded = false;

    private boolean isEndlessScrollEnabled = true;

    public EndlessScrollListener(LinearLayoutManager layoutManager, BaseCleverSectionAdapter adapter, boolean isProgressNeeded) {
        this.mLinearLayoutManager = layoutManager;
        this.mAdapter = adapter;
        this.isProgressNeeded = isProgressNeeded;
    }

    // This happens many times a second during a scroll, so be wary of the code you place here.
    // We are given a few useful parameters to help us work out if we need to load some more data,
    // but first we check if we are waiting for the previous load to finish.
    @Override
    public void onScrolled(RecyclerView view, int dx, int dy) {
        int firstVisibleItem = mLinearLayoutManager.findFirstVisibleItemPosition();
        int visibleItemCount = view.getChildCount();
        int totalItemCount = mLinearLayoutManager.getItemCount();

        // If the total item count is zero and the previous isn't, assume the
        // list is invalidated and should be reset back to initial state
        if (totalItemCount < previousTotalItemCount) {
            this.currentPage = this.startingPageIndex;
            this.previousTotalItemCount = totalItemCount;
            if (totalItemCount == 0) {
                this.loading = true;
            }
        }
        // If it’s still loading, we check to see if the dataset count has
        // changed, if so we conclude it has finished loading and update the current page
        // number and total item count.
        if (loading && (totalItemCount > previousTotalItemCount)) {
            loading = false;
            previousTotalItemCount = totalItemCount;
        }

        // If it isn’t currently loading, we check to see if we have breached
        // the visibleThreshold and need to reload more data.
        // If we do need to reload some more data, we execute onLoadMore to fetch the data.
        if (!loading && (totalItemCount - visibleItemCount) <= (firstVisibleItem + visibleThreshold) && isEndlessScrollEnabled) {
            currentPage++;
            loading = true;

            if (isProgressNeeded) {
                mAdapter.addShowMoreProgress();
                previousTotalItemCount = mLinearLayoutManager.getItemCount();
            }

            onLoadMore(currentPage, totalItemCount);
        }
    }

    // Defines the process for actually loading more data based on page
    public abstract void onLoadMore(int page, int totalItemsCount);

    public void setEndlessScrollEnabled(boolean isEnabled) {
        isEndlessScrollEnabled = isEnabled;
    }

}
