package ru.noties.spg.processor.writer;

import java.io.IOException;
import java.io.Writer;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.processing.Filer;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Elements;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;

import ru.noties.spg.processor.data.DefItem;
import ru.noties.spg.processor.data.KeyHolder;
import ru.noties.spg.processor.data.KeyType;
import ru.noties.spg.processor.data.PreferenceHolder;

/**
 * Created by Dimitry Ivanov on 15.07.2015.
 */
public class SPGPreferenceWriter implements ru.noties.spg.processor.Logger {

    private static final char SEMICOLON = ';';
    private static final String[] INITIAL_IMPORTS = new String[] {
            "ru.noties.spg.*;",
            "android.content.*;",
            "java.util.*",
            "android.preference.PreferenceManager"
    };
    private static final String GEN_INFO_PATTERN = "// This file is generated by SharedPreferencesGenerator library at %s\n" +
            "// The description for this preference was taken from: %s\n" +
            "// Do not modify this file\n\n";

    private static final String CLASS_STATEMENT_PATTERN = "public class %s implements SPGPreferenceObject%s {\n\n";
    private static final String ENTITY_INTERFACE_PATTERN = ", SPGPreferenceEntity<%s>";

    private static final String CONST_PREF_NAME = "PREFERENCE_NAME";
    private static final String CONST_PREF_MODE = "PREFERENCE_MODE";
    private static final String CONST_KEY = "KEY_";
    private static final String CONST_DEF = "DEF_";
    private static final String CONST_MODIFIERS = "public static final ";

    private final ru.noties.spg.processor.Logger mLogger;
    private final Elements mElements;
    private final Filer mFiler;

    public SPGPreferenceWriter(ru.noties.spg.processor.Logger logger, Elements elements, Filer filer) {
        this.mLogger = logger;
        this.mElements = elements;
        this.mFiler = filer;
    }

    @Override
    public void log(Diagnostic.Kind level, String message, Object... args) {
        mLogger.log(level, message, args);
    }

    public void write(PreferenceHolder preference) {

        final TypeElement element = preference.typeElement;

        final String entityClassName = createQualifiedName(element);
        final String prefPackage     = mElements.getPackageOf(element).toString();
        final String prefClassName   = createClassName(element);

        log(Diagnostic.Kind.NOTE, "Writing @SPGPreference: `%s` to a file: `%s.%s.java`", element, prefPackage, prefClassName);

        final Indent indent = new Indent();
        final StringBuilder builder = new StringBuilder();

        // write package
        builder.append(semicolon(createPackageStatement(prefPackage)))
                .append("\n\n");

        // write imports
        writeImports(builder, preference.imports);

        // write gen info
        writeGenInfo(builder, element);

        // write class statement
        writeClassStatement(builder, prefClassName, preference.toEntity ? entityClassName : null);

        indent.increment();

        // write constants
        writeConstants(builder, indent, preference);

        // write `create` statement
        writeGetInstanceStatement(builder, indent, preference, prefClassName);

        // write local variables
        writeLocalVariables(builder, indent);

        // write constructor
        writeConstructor(builder, indent, prefClassName, preference.defaultName, preference.preferenceMode);

        // write getters
        writeGetters(builder, indent, preference.keys);

        // write setters
        writeSetters(builder, indent, preference);

        // write helper methods
        writeSPGObjectMethods(builder, indent, preference);

        // write Setter
        writeSetterClass(builder, indent, preference);

        writeOnUpdate(builder, indent, preference);

        // close class
        builder.append("}");

        Writer writer = null;
        try {
            final JavaFileObject fileObject = mFiler.createSourceFile(prefPackage + "." + prefClassName);
            writer = fileObject.openWriter();
            writer.write(builder.toString());
        } catch (IOException e) {
            throw new IllegalStateException(e);
        } finally {
            if (writer != null) {
                try {
                    writer.flush();
                    writer.close();
                } catch (IOException e) {}
            }
        }
    }

    private static void writeConstants(StringBuilder builder, Indent indent, PreferenceHolder preference) {

        final String prefName = createConstantStatement(
                KeyType.STRING, CONST_PREF_NAME, "\"" + preference.name + "\""
        );
        final String prefMode = createConstantStatement(KeyType.INT, CONST_PREF_MODE, String.valueOf(preference.preferenceMode));
        builder.append(indent)
                .append(prefName)
                .append("\n")
                .append(indent)
                .append(prefMode)
                .append("\n\n");

        // write defaults
        final Map<KeyType, DefItem> defs = preference.defaults.defs;
        for (Map.Entry<KeyType, DefItem> e: defs.entrySet()) {
            final KeyType keyType = e.getKey();
            final DefItem defItem = e.getValue();
            final String value = (keyType == KeyType.STRING && !defItem.isEvaluation)
                    ? "\"" + defItem.value + "\""
                    : defItem.value;
            builder
                    .append(indent)
                    .append(createConstantStatement(keyType, createDefName(keyType), value))
                    .append("\n");
        }
        builder.append('\n');

        final List<KeyHolder> keys = preference.keys;

        for (ru.noties.spg.processor.data.KeyHolder key: keys) {
            builder.append(indent)
                    .append(
                            createConstantStatement(
                                    KeyType.STRING,
                                    createConstantStatementName(CONST_KEY, key.fieldName),
                                    "\"" + key.name + "\""
                            )
                    )
                    .append('\n');
        }

        builder.append('\n');
    }

    private void writeGetInstanceStatement(StringBuilder builder, Indent indent, PreferenceHolder preference, String className) {
        if (preference.isSingleton) {
            writeSingletonGetInstance(builder, indent, className);
        }
    }

    private static void writeSingletonGetInstance(StringBuilder builder, Indent indent, String className) {

        builder.append(indent)
                .append("private static volatile ")
                .append(className)
                .append(" sInstance = null;\n");

        builder.append(indent)
            .append("public static ")
            .append(className)
            .append(" getInstance() {\n")
                .append(indent.increment())
                .append(className)
                .append(" local = sInstance;\n")
                .append(indent)
                .append("if (local == null) {\n")
                    .append(indent.increment())
                    .append("synchronized (")
                    .append(className)
                    .append(".class) {\n")
                        .append(indent.increment())
                        .append("local = sInstance;\n")
                        .append(indent)
                        .append("if (local == null) {\n")
                            .append(indent.increment())
                            .append("final ContextProvider cp = SPGManager.getContextProvider();\n")
                            .append(indent)
                            .append("local = sInstance = new ")
                            .append(className)
                            .append("(cp.provide());\n")
                            .append(indent.decrement())
                        .append("}\n")
                        .append(indent.decrement())
                    .append("}\n")
                    .append(indent.decrement())
                .append("}\n")
                .append(indent)
                .append("return local;\n")
            .append(indent.decrement())
            .append("}\n\n");
    }

    private static void writeLocalVariables(StringBuilder builder, Indent indent) {
        builder
                .append(indent)
                .append("private final SharedPreferences prefs;\n")
                .append(indent)
                .append("private final SharedPreferences.Editor editor;\n\n");
    }

    private static void writeConstructor(StringBuilder builder, Indent indent, String className, boolean defaultName, int prefMode) {
        builder
                .append(indent)
                .append("public ")
                .append(className)
                .append("(Context context) {\n")
                    .append(indent.increment());

        if (defaultName) {
            builder
                    .append("this.prefs = PreferenceManager.getDefaultSharedPreferences(context);\n");
        } else {
            builder
                    .append("this.prefs = context.getSharedPreferences(")
                    .append(CONST_PREF_NAME)
                    .append(", ")
                    .append(prefMode)
                    .append(");\n");
        }

        builder
                    .append(indent)
                    .append("this.editor = prefs.edit();\n")
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeGetters(StringBuilder builder, Indent indent, List<KeyHolder> keys) {
        for (ru.noties.spg.processor.data.KeyHolder key: keys) {
            final Element e = key.element;
            final TypeMirror typeMirror = e.asType();
            final boolean isBool = key.keyType == KeyType.BOOL;
            builder
                    .append(indent)
                    .append("public ")
                    .append(typeMirror)
                    .append(' ')
                    .append(isBool ? MethodNameUtils.createBooleanGetter(key.fieldName) : MethodNameUtils.createGetter(key.fieldName))
                    .append("() {\n");

            // check whether we have evaluation & serialization

            indent.increment();

            // we have simple type aka natively supported
            final KeyType type = key.keyType;
            if (type != null) {
                final String defValue = createDefValue(type, key.defItem);
                builder
                        .append(indent)
                        .append("return ")
                        .append(createPrefGetStatement(type, key.name, defValue))
                        .append(";\n");

            } else {

                final ru.noties.spg.processor.data.SerializerHolder serializer = key.serializer;
                // check for null...

                final String defValue = createDefValue(serializer.keyType, key.defItem);

                // we have serialization
                builder.append(indent)
                        .append(serializer.name)
                        .append(" s = ")
                        .append("SPGManager.getSerializer(")
                        .append(serializer.name)
                        .append(".class);\n")
                        .append(indent)
                        .append("if (s == null) {\n")
                            .append(indent.increment())
                            .append("s = new ")
                            .append(serializer.name)
                            .append("();\n")
                            .append(indent)
                            .append("SPGManager.addSerializer(s);\n")
                            .append(indent.decrement())
                        .append("}\n")
                        .append(indent)
                        .append("return s.deserialize(")
                        .append(createPrefGetStatement(serializer.keyType, key.name, defValue))
                        .append(");\n");
            }

            builder
                    .append(indent.decrement())
                    .append("}\n\n");
        }
    }

    private static String createDefValue(KeyType type, DefItem item) {
        if (item == null) {
            return createDefName(type);
        }
        if (type == KeyType.STRING && !item.isEvaluation) {
            return "\"" + item.value + "\"";
        } else {
            return item.value;
        }
    }

    private static String createPrefGetStatement(
            KeyType keyType,
            String key,
            String value
    ) {
        switch (keyType) {

            case INT:
                return "prefs.getInt(\"" + key + "\", " + value + ")";

            case LONG:
                return "prefs.getLong(\"" + key + "\", " + value + ")";

            case BOOL:
                return "prefs.getBoolean(\"" + key + "\", " + value + ")";

            case FLOAT:
                return "prefs.getFloat(\"" + key + "\", " + value + ")";

            case STRING:
                return "prefs.getString(\"" + key + "\", " + value + ")";
        }

        return null;
    }

    private static String createClassName(Element element) {
        return element.getSimpleName().toString() + "Preference";
    }

    private static String createQualifiedName(TypeElement element) {
        return element.getQualifiedName().toString();
    }

    private static String createPackageStatement(String p) {
        return "package " + p;
    }

    private static void writeImports(StringBuilder builder, List<String> imports) {
        for (String ii: INITIAL_IMPORTS) {
            builder.append(createImportStatement(ii))
                    .append('\n');
        }

        if (imports != null
                && imports.size() > 0) {
            for (String i: imports) {
                builder.append(createImportStatement(i))
                        .append('\n');
            }
        }
        builder.append('\n');
    }

    private static String createImportStatement(String s) {
        return semicolon("import " + s);
    }

    private static void writeGenInfo(StringBuilder builder, TypeElement element) {
        builder.append(
                String.format(
                        GEN_INFO_PATTERN,
                        new Date(),
                        element.getQualifiedName()
                )
        );
    }

    private static void writeClassStatement(StringBuilder builder, String prefClassName, String entityClassName) {
        builder.append(
                String.format(CLASS_STATEMENT_PATTERN, prefClassName,
                        entityClassName != null ? String.format(ENTITY_INTERFACE_PATTERN, entityClassName) : "")
        );
    }

    private static String createConstantStatement(KeyType type, String name, String value) {
        return semicolon(
                CONST_MODIFIERS
                        + type.getRepr()
                        + " "
                        + name
                        + " = "
                        + value
        );
    }

    private static String createConstantStatementName(String path, String name) {
        final StringBuilder builder = new StringBuilder(path);
        for (int i = 0, size = name.length(); i < size; i++) {
            if (Character.isUpperCase(name.charAt(i))) {
                builder.append('_');
            }
            builder.append(Character.toUpperCase(name.charAt(i)));
        }
        return builder.toString();
    }

    private static String semicolon(String in) {
        if (in.charAt(in.length() - 1) != SEMICOLON) {
            return in + SEMICOLON;
        }
        return in;
    }

    private static String createDefName(KeyType type) {
        return CONST_DEF + type.name();
    }

    private static void writeSetters(StringBuilder builder, Indent indent, PreferenceHolder holder) {
        for (ru.noties.spg.processor.data.KeyHolder key: holder.keys) {
            writeSetter(key, "void", builder, indent, true);
        }
    }

    private static void writeSetter(ru.noties.spg.processor.data.KeyHolder key, String methodReturnType, StringBuilder builder, Indent indent, boolean isApply) {
        final Element element = key.element;
        final TypeMirror type = element.asType();
        builder
                .append(indent)
                .append("public ")
                .append(methodReturnType)
                .append(" ")
                .append(MethodNameUtils.createSetter(key.fieldName))
                .append("(")
                .append(type)
                .append(" value) {\n")
                .append(indent.increment());

        final KeyType keyType = key.keyType;
        if (keyType != null) {
            builder
                    .append(createSimpleSet(keyType, key.name, "value", isApply))
                    .append(";");
        } else {

            final String serializer = key.serializer.name;
            builder
                    .append(serializer)
                    .append(" s = SPGManager.getSerializer(")
                    .append(serializer)
                    .append(".class);\n")
                    .append(indent)
                    .append("if (s == null) {\n")
                    .append(indent.increment())
                    .append("s = new ")
                    .append(serializer)
                    .append("();\n")
                    .append(indent)
                    .append("SPGManager.addSerializer(s);\n")
                    .append(indent.decrement())
                    .append("}\n")
                    .append(indent)
                    .append(createSimpleSet(key.serializer.keyType, key.name, "s.serialize(value)", isApply))
                    .append(";");

        }

        builder
                .append("\n")
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static String createSimpleSet(KeyType type, String key, String value, boolean isApply) {

        final String s;

        switch (type) {

            case INT:
                s = "editor.putInt(\"" + key + "\", " + value + ")";
                break;

            case BOOL:
                s = "editor.putBoolean(\"" + key + "\", " + value + ")";
                break;

            case LONG:
                s = "editor.putLong(\"" + key + "\", " + value + ")";
                break;

            case FLOAT:
                s = "editor.putFloat(\"" + key + "\", " + value + ")";
                break;

            case STRING:
                s = "editor.putString(\"" + key + "\", " + value + ")";
                break;

            default:
                s = null;
                break;

        }

        if (s != null && isApply) {
            return s + ".apply()";
        }

        return s;
    }

    private static void writeSPGObjectMethods(StringBuilder builder, Indent indent, PreferenceHolder preference) {

        // toMap()
        writeToMapMethod(builder, indent, preference);

        // getSharedPreferences()
        writeSimpleGet(builder, indent, "SharedPreferences", "prefs", "sharedPreferences");

        // getEditor()
        writeSimpleGet(builder, indent, "SharedPreferences.Editor", "editor", "editor");

        writeSimpleGet(builder, indent, "String", CONST_PREF_NAME, "sharedPreferencesName");

        writeSimpleGet(builder, indent, "int", CONST_PREF_MODE, "sharedPreferencesMode");

        writeGenericGetMethod(builder, indent, preference);

        // toEntity()
        if (preference.toEntity) {
            writeGenericToEntityMethod(builder, indent, preference);
        }
    }

    private static void writeToMapMethod(StringBuilder builder, Indent indent, PreferenceHolder preference) {
        builder
                .append(indent)
                .append("public Map<String, Object> toMap() {\n")
                    .append(indent.increment())
                    .append("final Map<String, Object> map = new HashMap<String, Object>();\n");

        for (ru.noties.spg.processor.data.KeyHolder key: preference.keys) {
            final boolean isBool = key.keyType == KeyType.BOOL;
            builder
                    .append(indent)
                    .append("map.put(\"")
                    .append(key.name)
                    .append("\", ")
                    .append(isBool ? MethodNameUtils.createBooleanGetter(key.fieldName) : MethodNameUtils.createGetter(key.fieldName))
                    .append("());\n");
        }

        builder
                    .append(indent)
                    .append("return map;\n")
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeSimpleGet(
            StringBuilder builder,
            Indent indent,
            String type,
            String localVarName,
            String reprString
    ) {
        final String method = MethodNameUtils.createGetter(reprString);

        builder
                .append(indent)
                .append("public ")
                .append(type)
                .append(" ")
                .append(method)
                .append("() {\n")
                    .append(indent.increment())
                    .append("return ")
                    .append(localVarName)
                    .append(";\n")
                    .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeGenericToEntityMethod(StringBuilder builder, Indent indent, PreferenceHolder holder){
        String qualifiedName = createQualifiedName(holder.typeElement);

        builder
                .append(indent)
                .append("public ")
                .append(qualifiedName)
                .append(" toEntity() {\n")
                .append(indent.increment())
                .append("return new ")
                .append(qualifiedName)
                .append("(");

        for (Iterator<KeyHolder> iterator = holder.keys.iterator(); iterator.hasNext(); ) {
            KeyHolder key = iterator.next();
            final boolean isBool = key.keyType == KeyType.BOOL;

            builder
                    .append(isBool ? MethodNameUtils.createBooleanGetter(key.fieldName) : MethodNameUtils.createGetter(key.fieldName))
                    .append("()")
                    .append(iterator.hasNext() ? ", " : ");\n");
        }

        builder
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeGenericGetMethod(StringBuilder builder, Indent indent, PreferenceHolder holder) {

        // generic get(String)
        builder
                .append(indent)
                .append("public <T> T get(String key) {\n")
                .append(indent.increment())
                .append("if (key == null) { return null; }\n")
                .append(indent)
                .append("final Object o;\n");
        boolean isFirst = true;
        for (ru.noties.spg.processor.data.KeyHolder key: holder.keys) {
            boolean isBool = key.keyType == KeyType.BOOL;
            if (!isFirst) {
                builder
                        .append(" else ");
            } else {
                builder.append(indent);
                isFirst = false;
            }
            builder
                    .append("if (key.equals(\"")
                    .append(key.name)
                    .append("\")) {\n")
                    .append(indent.increment())
                    .append("o = ")
                    .append(isBool ? MethodNameUtils.createBooleanGetter(key.fieldName) : MethodNameUtils.createGetter(key.fieldName))
                    .append("();\n")
                    .append(indent.decrement())
                    .append("}");
        }

        builder
                .append(" else {\n")
                .append(indent.increment())
                .append("// not in this prefs;\n")
                .append(indent)
                .append("o = null;\n")
                .append(indent.decrement())
                .append("}\n");

        builder
                .append(indent)
                .append("return (T) o;\n")
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeSetterClass(StringBuilder builder, Indent indent, PreferenceHolder holder) {

        builder
                .append(indent)
                .append("public Setter setter() {\n")
                    .append(indent.increment())
                    .append("return new Setter(editor);\n")
                    .append(indent.decrement())
                .append("}\n\n");

        builder
                .append(indent)
                .append("public static final class Setter {\n\n");

        indent.increment();

        builder
                .append(indent)
                .append("private final SharedPreferences.Editor editor;\n\n")
                .append(indent)
                .append("Setter(SharedPreferences.Editor editor) {\n")
                    .append(indent.increment())
                    .append("this.editor = editor;\n")
                    .append(indent.decrement())
                .append("}\n\n");

        for (ru.noties.spg.processor.data.KeyHolder key: holder.keys) {
            writeSetter(key, "Setter", builder, indent, false);
            final int index = builder.lastIndexOf("}");
            final String insert = indent.decrement().toString() + "return this;\n" + indent.increment().toString();
            builder
                    .insert(index, insert);
        }

        builder
                .append(indent)
                .append("public void apply() { editor.apply(); }\n")
                .append(indent.decrement())
                .append("}\n\n");
    }

    private static void writeOnUpdate(StringBuilder builder, Indent indent, PreferenceHolder holder) {
        boolean hasOnUpdate = false;
        for (ru.noties.spg.processor.data.KeyHolder key: holder.keys) {
            if (!key.onUpdate) {
                continue;
            }
            hasOnUpdate = true;
            builder
                    .append(indent)
                    .append("public void ")
                    .append(MethodNameUtils.createSetter(key.fieldName))
                    .append("UpdateListener(OnUpdateListener listener) {\n")
                        .append(indent.increment())
                        .append("updateListeners.put(\"")
                        .append(key.name)
                        .append("\", listener);\n")
                        .append(indent)
                        .append("checkSharedListener();\n")
                        .append(indent.decrement())
                    .append("}\n\n");
        }

        if (hasOnUpdate) {
            //write listeners
            builder
                    .append(indent)
                    .append("private final java.util.Map<String, OnUpdateListener> updateListeners;\n")
                    .append(indent)
                    .append("private final SharedPreferences.OnSharedPreferenceChangeListener sharedListener;\n")
                    .append(indent)
                    .append("private boolean isListenerRegistered;")
                    .append(indent)
                    .append("{\n")
                        .append(indent.increment())
                        .append("updateListeners = new java.util.HashMap<String, OnUpdateListener>();\n")
                        .append(indent)
                        .append("sharedListener = new SharedPreferences.OnSharedPreferenceChangeListener() {\n")
                            .append(indent.increment())
                            .append("public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String key) {\n")
                                .append(indent.increment())
                                .append("final OnUpdateListener listener = updateListeners.get(key);\n")
                                .append(indent)
                                .append("if (listener != null) { listener.onUpdate(); }\n")
                                .append(indent.decrement())
                            .append("}\n")
                            .append(indent.decrement())
                        .append("};\n")
                        .append(indent.decrement())
                    .append("}\n\n");

            // write check sharedListener
            builder
                    .append(indent)
                    .append("private void checkSharedListener() {\n")
                        .append(indent.increment())
                        .append("if (!isListenerRegistered) {\n")
                            .append(indent.increment())
                            .append("prefs.registerOnSharedPreferenceChangeListener(sharedListener);\n")
                            .append(indent)
                            .append("isListenerRegistered = true;\n")
                    .append(indent.decrement())
                        .append("}\n")
//                        .append("if (updateListeners.isEmpty()) {\n")
//                            .append(indent.increment())
//                            .append("if (isListenerRegistered) {\n")
//                                .append(indent.increment())
//                                .append("prefs.unregisterOnSharedPreferenceChangeListener(sharedListener);\n")
//                                .append(indent)
//                                .append("isListenerRegistered = false;\n")
//                                .append(indent.decrement())
//                            .append("}\n")
//                            .append(indent.decrement())
//                        .append("} else {\n")
//                            .append(indent.increment())
//                            .append("if (!isListenerRegistered) {\n")
//                                .append(indent.increment())
//                                .append("prefs.registerOnSharedPreferenceChangeListener(sharedListener);\n")
//                                .append(indent)
//                                .append("isListenerRegistered = true;\n")
//                                .append(indent.decrement())
//                            .append("}\n")
//                            .append(indent.decrement())
//                        .append("}\n")
                    .append(indent.decrement())
                    .append("}\n\n");
        }
    }
}
