package gov.raptor.gradle.plugins.buildsupport

import org.ajoberstar.grgit.Commit
import org.gradle.api.tasks.TaskAction

/**
 * Task definition to create a git tag operation on the current repo.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.3
 */
class TagReleaseTask extends AReleaseTask {

    private static final String[] OPTIONS = RELEASE_OPTIONS + '-Ptag=<tag>	To specify the tag to create'
    private final String overrideTag

    TagReleaseTask() {
        group = 'Release'
        description = makeDescription("Tag the tree with the current base version and given stage (can override with -Ptag=<tag>)", OPTIONS)

        overrideTag = project.findProperty('tag')
    }

    @SuppressWarnings("GroovyUnusedDeclaration")
    @TaskAction
    void doTagRelease() {

        final String buildTag
        final String commitMessage

        if (overrideTag) {
            buildTag = overrideTag
            commitMessage = "Tag $buildTag"
        } else {
            validateStageProperties()

            String buildType = project.ext.finalBuild ? 'FINAL' : project.ext.rcBuild
            String versionString = project.versionFile.text.trim()
            RaptorVersion theVersion = new RaptorVersion(versionString)

            theVersion.stage = project.ext.finalBuild ? '' : project.ext.rcBuild // Update according to build params
            buildTag = 'tag-' + theVersion.toString()
            commitMessage = "Tag $buildTag for $buildType release"
        }

        if (isHeadAlreadyTagged(buildTag)) {
            project.logger.lifecycle("Tag '$buildTag' is already present @ HEAD")
        } else {
            logger.lifecycle("git tag -a '$buildTag' -m '$commitMessage'")

            def tag = null
            if (!project.noExec) tag = project.grgit.tag.add(name: buildTag, annotate: true, message: commitMessage, force: true)

            logger.lifecycle("git push $buildTag ${project.noPush ? '(skipped)' : ''}")
            if (!project.noExec && !project.noPush) project.grgit.push(refsOrSpecs: [tag?.fullName])
        }
    }

    /**
     * Determine if the HEAD revision is already tagged with the given tag name.
     *
     * @param tagName Tag to test
     * @return {@code true} if HEAD and the tag point to the same revision
     */
    private boolean isHeadAlreadyTagged(String tagName) {
        def tagCommit
        try {
            tagCommit = safeResolveCommit(tagName)
        } catch (Exception e) {
            project.logger.debug("No tag 'tagName' found", e)
            return false
        }

        try {
            Commit headCommit = safeResolveCommit('HEAD')

            project.logger.info("Checking tag status:\n" +
                    "TAG  = ${formatCommit(tagCommit)}\n\n" +
                    "HEAD = ${formatCommit(headCommit)}")

            return tagCommit == headCommit
        } catch (Exception e) {
            project.logger.warn("Failed to locate HEAD", e)
        }
        return false
    }
}
