package gov.raptor.gradle.plugins.buildsupport

/**
 * Simple class to handler tri-part version numbers including a stage tag, like 2.6.1-RC1.
 * All parts are expected to be numeric, except the stage, which can be an arbitrary string.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 */
@SuppressWarnings("GroovyUnusedDeclaration")
class RaptorVersion {
    private static final int MAJOR = 0
    private static final int MINOR = 1
    private static final int PATCH = 2

    public static final String SNAPSHOT = 'SNAPSHOT'

    private int[] parts
    private String stage = ''

    /**
     * Construct from the given version string.  The string must be of the form 'X.Y.Z' or 'X.Y.Z-STAGE'.  If
     * X, Y, and Z are not each convertible to an integer, then an exception will be thrown.
     *
     * @param versionString The string to parse, e.g., '2.6.1-RC1'
     */
    RaptorVersion(String versionString) {
        def baseAndStage = versionString.split('-', 2)

        def baseVersion = baseAndStage[0]
        if (baseAndStage.length > 1) stage = baseAndStage[1]

        //noinspection GroovyAssignabilityCheck
        parts = baseVersion.split("\\.")*.toInteger()
    }

    /**
     * @return The major component as an integer
     */
    int getMajor() {
        return parts[MAJOR]
    }

    /**
     * @return The minor component as an integer
     */
    int getMinor() {
        return parts[MINOR]
    }

    /**
     * @return The patch component as an integer
     */
    int getPatch() {
        return parts[PATCH]
    }

    /**
     * Increment the minor component number.
     */
    void bumpMinor() {
        parts[MINOR] += 1
    }

    /**
     * Increment the patch component number.
     */
    void bumpPatch() {
        parts[PATCH] += 1
    }

    /**
     * Set the patch component to the give value.
     *
     * @param patch New patch value
     */
    void setPatch(int patch) {
        parts[PATCH] = patch
    }

    /**
     * @return The stage, if any.  E.g., on a version of '2.6.1-RC1' this returns 'RC1'
     */
    String getStage() {
        return stage
    }

    /**
     * Set the stage to the given value.
     *
     * @param stage New stage value
     */
    void setStage(String stage) {
        this.stage = stage
    }

    /**
     * @return the 'root' version (without the stage), e.g., for '2.6.1-SNAPSHOT' this returns '2.6.1'
     */
    String getRootVersion() {
        return parts.join('.')
    }

    /**
     * Get the major and minor components of the version.  For example, if the full version is 2.6.1-RC1, then this
     * method returns '2.6'.
     *
     * @return The major/minor components of the version
     */
    String getMajorMinor() {
        parts[MAJOR..MINOR].join('.')
    }

    /**
     * @return {@code true} if the stage of this version is SNAPSHOT
     */
    boolean isSnapshot() {
        return stage == SNAPSHOT
    }

    /**
     * @return {@code true} if the stage of this version is empty (final)
     */
    boolean isFinal() {
        return !stage
    }

    /**
     * Return the version with any existing stage replaced with 'SNAPSHOT'.  For example, if this version is
     * 2.6.1-RC1, then this method would return '2.6.1-SNAPSHOT'.
     *
     * @return This version with a stage of SNAPSHOT
     */
    String asSnapshot() {
        return parts.join('.') + "-$SNAPSHOT"
    }

    /**
     * @return The full version including the stage, e.g., '2.6.1-RC1'
     */
    String toString() {
        return parts.join('.') + (stage ? "-$stage" : '')
    }

    boolean equals(o) {
        if (this.is(o)) return true
        if (getClass() != o.class) return false

        RaptorVersion that = (RaptorVersion) o

        if (!Arrays.equals(parts, that.parts)) return false
        if (stage != that.stage) return false

        return true
    }

    int hashCode() {
        int result
        result = (parts != null ? Arrays.hashCode(parts) : 0)
        result = 31 * result + (stage != null ? stage.hashCode() : 0)
        return result
    }
}
