package gov.raptor.gradle.plugins.buildsupport

import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.Task

//@formatter:off
/**
 * Extension for handling RDM Facade configuration and task creation.
 * <p>
 * This extension creates a task named {@code generateRdmFacades} that has no actions.  It is used to tie all the
 * task definitions created from facade generator definitions together into the build lifecycle.  The
 * {@code compileJava} task is set to depend on {@code generateRdmFacades}.
 * <p>
 * This extension is also a named object container for facade generation definitions, using {@link RdmFacadeGeneratorDefinition}
 * objects, to simplify the configuration within the build script.  Here's an example configuration:
 * <pre><code style="line-height: 1.1">
 * buildSupport.configureRdmFacades(project)
 *
 * rdmFacadeGenerators {
 *     acme {
 *         // Just using a single input file
 *         structureXmls = files('src/main/resources/structures/dataStructures.xml')
 *         targetPackage = 'org.acme'
 *         camelCase = true
 *    }
 *
 *    coyote {
 *         // Use all the files in a given directory
 *         structureXmls = files(file('src/main/resources/coyote//structures/master_plans').listFiles())
 *         targetPackage = 'com.my.secret.plans'
 *         camelCase = true
 *         classNamePrefix = 'Stealth'
 *         failBuildOnError = false // Some plans are not working
 *     }
 * }
 * </code></pre>
 * The configuration above would generate two tasks of type {@link GenerateRdmFacadeTask},
 * named {@code generateRdmFacadesForAcme} and {@code generateRdmFacadesForCoyote}.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.4
 * @see GenerateRdmFacadeTask
 */
//@formatter:on
class GenerateRdmFacadesExtension {

    public static final String EXTENSION_NAME = 'generateRdmFacades'
    public static final String GENERATE_FACADES_TASK_NAME = 'generateRdmFacades'
    public static final String GENERATORS_NAME = 'rdmFacadeGenerators'

    final Project project

    GenerateRdmFacadesExtension(Project project) {

        this.project = project

        project.with {
            // Add the generated sources to our java source set
            project.sourceSets.main.java.srcDir "$buildDir/generated-sources"

            tasks.create(name: GENERATE_FACADES_TASK_NAME, type: DefaultTask, group: 'Build', description: 'Generate facade classes for RDMs based on structure definitions')

            tasks.compileJava.dependsOn GENERATE_FACADES_TASK_NAME
        }

        project.extensions."$GENERATORS_NAME" = project.container(RdmFacadeGeneratorDefinition)

        createTaskCreator()
    }

    /**
     * Create a new RDM facade generation task for each RdmFacadeGeneratorDefinition defined.
     */
    private void createTaskCreator() {
        project.extensions."$GENERATORS_NAME".all { RdmFacadeGeneratorDefinition generator ->

            project.with {
                String taskName = "generateRdmFacadesFor${generator.name.capitalize()}"

                // Create new task for this node.
                Task genTask = task(taskName,
                        type: GenerateRdmFacadeTask,
                        group: 'Build',
                        description: "Generate RDM Facades for configuration '${generator.name}'")

                // Delay the configuration of the task until the project is fully evaluated.  At the moment this
                // closure is invoked, the generator has been created but not configured, so all its
                // properties are in a default state.

                project.afterEvaluate {
                    genTask.structureXmls = generator.structureXmls
                    genTask.targetPackage = generator.targetPackage
                    genTask.camelCase = generator.camelCase
                    genTask.classNamePrefix = generator.classNamePrefix
                    genTask.failBuildOnError = generator.failBuildOnError
                }

                tasks[GENERATE_FACADES_TASK_NAME].dependsOn genTask
            }
        }
    }
}
