package gov.raptor.gradle.plugins.buildsupport

import org.ajoberstar.grgit.Grgit
import org.gradle.api.GradleException
import org.gradle.api.InvalidUserDataException
import org.gradle.api.Plugin
import org.gradle.api.internal.project.ProjectInternal
import org.gradle.internal.os.OperatingSystem

/**
 * Plugin to add various support classes, methods and settings for build scripts performing Raptor related builds.
 * <p>
 * The {@link BuildSupportExtension extension} provides a number of useful configuration methods for operations such
 * as validating a branch is ready to perform a release, configuring a 'standard' java project, etc.
 * For most stand-alone plugin projects, {@link BuildSupportExtension#configureStandardPlugin} is all that is needed.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @see BuildSupportExtension
 */
@SuppressWarnings("GroovyUnusedDeclaration")
class BuildSupportPlugin implements Plugin<ProjectInternal> {

    private static final String TAG_RELEASE_TASK_NAME = "tagRelease"
    private static final String UPDATE_RELEASE_VERSION_TASK_NAME = "updateReleaseVersion"
    private static final String ROLL_PATCH_VERSION_TASK_NAME = "rollPatchVersion"
    private static final String ROLL_MINOR_VERSION_TASK_NAME = "rollMinorVersion"
    private static final String BRANCH_REPO_TASK_NAME = "branchRepo"
    private static final String DEFAULT_RAPTOR_PLUGINS_DIR = '../RaptorT/Raptor/bin/plugins'

    private ProjectInternal project

    @Override
    void apply(ProjectInternal project) {

        this.project = project

        // Try to auto-detect that we're running under jenkins so we don't require a command line flag
        // Need to determine 'running under jenkins' before creating tasks and extensions
        boolean haveJenkinsVars = System.getenv('BUILD_ID') != null || System.getenv('JENKINS_HOME') != null

        project.ext.runningUnderJenkins = haveJenkinsVars || project.hasProperty('jenkins')

        // Create our top-level tasks
        def tasks = project.getTasks()

        tasks.create(TAG_RELEASE_TASK_NAME, TagReleaseTask)
        tasks.create(BRANCH_REPO_TASK_NAME, BranchRepoTask)
        tasks.create(UPDATE_RELEASE_VERSION_TASK_NAME, UpdateReleaseVersionTask)
        tasks.create(ROLL_MINOR_VERSION_TASK_NAME, RollMinorVersionTask)
        tasks.create(ROLL_PATCH_VERSION_TASK_NAME, RollPatchVersionTask)

        def pluginManager = project.getPluginManager()

        // Apply common plugins used by all builds
        pluginManager.apply('nu.studer.credentials')
        pluginManager.apply('org.ajoberstar.grgit')

        // Add our extension object to provide our support methods
        def extension = project.extensions.create(BuildSupportExtension.NAME, BuildSupportExtension, project)

        getGitData()
        setupExtVariables()

        // If this is a release build, ensure that the work tree is in a good state
        if (project.ext.isRelease) extension.verifyTreeIsClean()
    }

    /**
     * Ensure that Grgit will properly work on this repo.  Set an ext variable accordingly, and extract the branch
     * name, if Grgit is available.
     */
    private void getGitData() {
        try {
            //noinspection GroovyAssignabilityCheck
            project.ext.grgit = Grgit.open(dir: project.rootProject.rootDir)

        } catch (Exception ignored) {
            // not a git repo or invalid/corrupt
            project.logger.warn 'No git repository found. Build may fail with NPE.'
            project.ext.grgitUsable = false
            return
        }

        project.ext {
            grgitUsable = true
            branchName = 'unknown'

            try {
                // The status() call will throw an exception if grgit can't properly operate on this repo, which
                // happens on a worktree.  Hopefully, a new release of jgit with worktree support will be available soon.
                grgit.status()
                branchName = grgit.branch.current.name

            } catch (Exception ignored) {
                project.logger.error "ERROR: Grgit is not usable on this repo, probably because it's a worktree", ignored
                grgitUsable = false
            }
        }
    }

    /**
     * Setup the variables in the project's ext block.  These values make aspects of the Raptor version and
     * the flags used to invoke the build available to consuming build scripts.
     * <p>
     * Note: <em>If you add/change/remove any values make sure you update the class-level documentation, as that is what
     * is published to Confluence.</em>
     */
    private void setupExtVariables() {
        project.ext {
            // Track the number of test failures so we can prevent publishing if any tests have failed
            testFailureCount = 0

            // If the project has not already defined the rootBranch, then give it the default
            if (!project.hasProperty('rootBranch')) {
                rootBranch = 'develop' // Default to 'develop' as our root branch (used to create release branches)
            }

            // OS specific settings
            os = OperatingSystem.current()
            osFamily = os.isMacOsX() ? 'macosx' : os.familyName // We don't want 'os x' as the family, we want 'macosx'

            def osArch = System.getProperty("os.arch")
            def nativeLibPaths = [(OperatingSystem.WINDOWS): 'native/windows/' + osArch,
                                  (OperatingSystem.MAC_OS) : 'native/macosx',
                                  (OperatingSystem.LINUX)  : 'native/linux/' + osArch
            ]

            nativeLibPath = nativeLibPaths.get(os)
            jvmBitSize = getJvmBitSize()

            project.logger.info "Building on $jvmBitSize-bit $osFamily : $os"

            isRelease = false
            buildVersion = null
            raptorVersion = ''
            RaptorVersion currentBuildVersion = null
            isProjectBuild = false

            raptorVersionFile = project.file("raptorVersion.txt")
            if (raptorVersionFile.exists()) {
                raptorVersion = new RaptorVersion(raptorVersionFile.text.trim())
                currentBuildVersion = raptorVersion

                project.logger.info "Found raptorVersion.txt: '$raptorVersion'"
            }

            // If a projectVersion.txt file, reconfigure to support building a project that depends
            // on Raptor artifacts instead of building Raptor itself

            projectVersionFile = project.file("projectVersion.txt")
            if (projectVersionFile.exists()) {
                isProjectBuild = true
                projectVersion = new RaptorVersion(projectVersionFile.text.trim())
                currentBuildVersion = projectVersion // Use project version instead of Raptor version
                project.logger.info "Found projectVersion.txt: '$projectVersion'"
            }

            // Very special handling for the RaptorPlugins project, where its project version will always be the
            // same as the raptor version, and we don't want to manage 2 files.

            if (project.hasProperty('raptorVersionTracksProjectVersion')) {
                raptorVersion = projectVersion
            }

            if (currentBuildVersion == null) {
                throw new GradleException('You must have a version file to use this plugin')
            } else {
                // Release vs. development build settings
                controllingVersion = currentBuildVersion
                baseVersion = currentBuildVersion.toString()
                rootVersion = currentBuildVersion.getRootVersion()
                versionFile = isProjectBuild ? projectVersionFile : raptorVersionFile

                isRebuildRelease = project.hasProperty('rebuildRelease')

                // If this is a 'rebuildRelease' then we need to pull the final/rc settings from the current version
                // This is because we're rebuilding, and the value in the file will be at the proper stage

                if (isRebuildRelease) {
                    finalBuild = currentBuildVersion.isFinal()
                    rcBuild = currentBuildVersion.stage as String
                    project.logger.info "Performing rebuild of release at version $baseVersion"
                } else {
                    finalBuild = project.hasProperty('final')
                    rcBuild = project.findProperty('rc') as String
                }

                isRelease = project.hasProperty('release') || isRebuildRelease

                // If this is a release build, but not a rebuild, then we need to update the currentBuildVersion with the new stage

                if (isRelease && !isRebuildRelease) {
                    if (finalBuild) {
                        if (rcBuild) throw new InvalidUserDataException("You can't specify both -Pfinal and -Prc")
                    } else {
                        if (!rcBuild) throw new InvalidUserDataException("You must specify one of -Pfinal or -Prc")
                    }

                    currentBuildVersion.stage = finalBuild ? '' : rcBuild // Set version according to build params
                    baseVersion = currentBuildVersion.toString()

                    project.logger.info "Performing 'release' build of version $baseVersion"
                }

                forceSnapshot = project.hasProperty('raptorSnapshot')
                isSnapshot = !isRelease || currentBuildVersion.isSnapshot()
                isForcePublish = project.hasProperty('forcePublish')

                buildVersion = isRelease ? baseVersion : currentBuildVersion.asSnapshot()

                // Handle release builds against Raptor SNAPSHOT versions
                // Only do this if the project has declared a raptor version (in raptorVersion.txt)

                if (isProjectBuild && raptorVersion && isRelease && !isRebuildRelease && raptorVersion.isSnapshot()) {

                    if (finalBuild) {
                        // Prevent a FINAL release build of a project from using a SNAPSHOT raptor version
                        throw new InvalidUserDataException("You can not create a FINAL release based on a SNAPSHOT Raptor version: $raptorVersion")
                    } else {
                        // On non-final release builds, warn the user
                        emitSnapshotBuildWarning()

                        // And then exit unless the user forces it
                        if (!forceSnapshot) {
                            throw new InvalidUserDataException("You should not create a release based on a SNAPSHOT Raptor version: $raptorVersion")
                        }
                    }
                }
            }

            force = project.hasProperty('force')
            noPush = project.hasProperty('nopush') || project.hasProperty('noPush')
            noExec = project.hasProperty('noexec') || project.hasProperty('noExec')

            // Determine the value to be used for the Raptor plugin's directory.
            String propertySetting = project.findProperty('pluginInstallDir')

            // Trim any quotes that may have been included in the setting
            if (propertySetting) {
                if (propertySetting.startsWith('"')) propertySetting = propertySetting.replaceAll(/^"|"$/, '')
                if (propertySetting.startsWith("'")) propertySetting = propertySetting.replaceAll(/^'|'$/, '')
            }

            File installDir = new File(propertySetting ? propertySetting : DEFAULT_RAPTOR_PLUGINS_DIR)

            // Can't just use project.file here because it assumes relative paths.  So, if the path
            // given is an absolute path, then just use it.  Otherwise, assume it is relative to the
            // root project and use project.file().

            raptorPluginsDir = installDir.isAbsolute() ? installDir : project.file(installDir)

            project.logger.info "Raptor plugins dir: $raptorPluginsDir"

            if (isProjectBuild && raptorVersion && (!raptorPluginsDir.exists() || !raptorPluginsDir.isDirectory())) {
                project.logger.warn("Raptor plugins dir is missing or not a directory: $raptorPluginsDir")
            }

            // Make it easier for me to test this
            if (project.hasProperty('safeTest') || project.hasProperty('safetest')) {
                force = true
                noPush = true
                noExec = true
                useTestRepo = true
            }

            // Internal bookkeeping
            grgitFailureReported = false

            if (buildVersion) project.version = buildVersion // And set the project's version accordingly
        }
    }

    /**
     * Print a warning about release builds against Raptor SNAPSHOT versions.
     */
    private void emitSnapshotBuildWarning() {
        project.logger.warn '=============================================================================='
        project.logger.warn '=============================================================================='
        project.logger.warn 'You are performing a release build against a SNAPSHOT version of Raptor!'

        if (project.ext.forceSnapshot) {
            project.logger.warn 'I hope you know what you\'re doing.'
        } else {
            project.logger.warn 'In general, you should NOT do this.'
            project.logger.warn ''
            project.logger.warn 'If you didn\'t intend this, please update the \'raptorVersion.txt\' file.'
            project.logger.warn 'You can override this warning using -PraptorSnapshot'
        }

        project.logger.warn '=============================================================================='
        project.logger.warn '=============================================================================='
    }

    /**
     * Determine the bit size of the executing JVM, either 32 or 64 (as a string).
     *
     * @return 32 or 64
     * @throws org.gradle.api.GradleException if the current architecture can not be interpreted
     */
    private String getJvmBitSize() {
        String arch = System.getProperty('os.arch')
        switch (arch) {
            case ~'^(amd64|ia32e|em64t|x64|x86_64)$': return '64'
            case ~'^(x86|i[3-6]86|ia32|x32)$': return '32'
            default: throw new GradleException("Unknown architecture '$arch'")
        }
    }
}
