package gov.raptor.gradle.plugins.buildsupport

import org.ajoberstar.grgit.operation.BranchChangeOp
import org.gradle.api.tasks.TaskAction

/**
 * Task definition to create a git tag operation on the current repo.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.3
 */
class BranchRepoTask extends ARepoTask {

    private static
    final String[] OPTIONS = REPO_OPTIONS + '-Pbranch=<branchName>	To specify the name of the branch to create'
    private final String overrideBranchName

    BranchRepoTask() {
        group = 'Release'
        description = makeDescription('Create a new release branch based on the current controlling version file', OPTIONS)

        overrideBranchName = project.findProperty('branch')
    }

    @SuppressWarnings("GroovyUnusedDeclaration")
    @TaskAction
    void doBranchRepo() {

        getToRootBranch()

        // Determine the branch name to use
        String branchName = overrideBranchName ? overrideBranchName : project.controllingVersion.getMajorMinor()

        createBranch(branchName)
    }

    /**
     * Create the release branch, rooted at the local dev branch.  Push the new branch, and then set the upstream
     * on the new branch to track the newly create 'remotes/origin/<branch>'.
     *
     * @param branchName The name of the branch to create (e.g., 2.7)
     */
    void createBranch(String branchName) {
        // If the branch already exists, just warn about it
        if (project.grgit.resolve.toBranch(branchName)) {
            logger.warn("Branch '$branchName' already exists!")
            return
        }

        logger.lifecycle "\nCreate branch $branchName"

        logger.lifecycle "git branch $branchName from $project.rootBranch"
        if (!project.noExec) {
            def newBranch = project.grgit.branch.add(name: branchName, startPoint: "$project.rootBranch")

            logger.lifecycle "git push $newBranch.fullName ${project.noPush ? '(skipped)' : ''}"
            if (!project.noPush) {
                project.grgit.push(refsOrSpecs: [newBranch.fullName])

                // Now update the new branch to track the remote
                logger.lifecycle "Set upstream on $branchName to origin/$branchName"
                project.grgit.branch.change(name: branchName, startPoint: "origin/$branchName", mode: BranchChangeOp.Mode.TRACK)
            }
        }
    }
}
