package gov.raptor.gradle.plugins.buildsupport

import org.gradle.api.DefaultTask
import org.gradle.api.InvalidUserDataException
import org.gradle.api.file.FileCollection
import org.gradle.api.tasks.*

import javax.lang.model.SourceVersion

/**
 * Task to generate RDM facade classes from a collection of structure XML files.  Typically, tasks of this type
 * are not directly configured by a build script.  They are generated via the {@link GenerateRdmFacadesExtension},
 * which provides a named object container for facade generation definitions, using {@link RdmFacadeGeneratorDefinition}
 * objects.
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.4
 */
@SuppressWarnings("GroovyUnusedDeclaration")
class GenerateRdmFacadeTask extends DefaultTask {
    /**
     * Collection of structure XML files to process.
     */
    @InputFiles
    FileCollection structureXmls

    /**
     * The package into which to generate the facade classes
     */
    @Input
    String targetPackage

    /**
     * Set {@code true} to convert the message type to camel case when generating the class name.  Otherwise,
     * use the message type "as is".
     *
     * @see #classNamePrefix
     */
    @Input
    boolean camelCase = false

    /**
     * Prefix to apply to all generated class names.  This can be useful when dealing with message types that don't
     * start with a character that can legally start a java identifier (like digits).
     * <p>
     * Defaults to empty.
     */
    @Input
    String classNamePrefix = ''

    /**
     * Set {@code true} to fail the build if any errors are encountered while generating the facade classes (like
     * illegal class or method names being generated from structure definitions).  Set {@code false} to simply
     * emit error messages and delete the class file that was being generated.
     * <p>
     * Default is {@code true}
     *
     */
    @Input
    boolean failBuildOnError = true

    private final File outputDir = new File(getProject().buildDir, '/generated-sources')

    /**
     * The directory into which to generate facade classes.  Read-only, set to {@code buildDir/generated-sources}.
     */
    @OutputDirectory
    File getOutputDir() { return outputDir }

    @Internal
    private File destinationDir

    // Format to generate the start of the class, args:
    //   1. package
    //   2. class name
    //   3. Structure name
    //   4. Date
    //
    private static final String CLASS_START_FMT = '''
// Facade for structure %3$s generated on: %4$s

package %1$s;

import gov.raptor.api.messages.IPrePersistRaptorDataMessage;
import gov.raptor.dataservices.utils.EnhancedFacadeGenerator;

import java.util.function.Function;

@SuppressWarnings("unused")
public class %2$s
{
    private static volatile boolean enhancedGeneratorConstructed = false;
    private static Function<IPrePersistRaptorDataMessage, ? extends %2$s> generator = %2$s::new;

    private final IPrePersistRaptorDataMessage rdm;

    /**
     * Static factory method to produce a facade instance from a given RDM.
     */
    public static %2$s from(IPrePersistRaptorDataMessage rdm) {
        if (!enhancedGeneratorConstructed)
        {
            final Function<IPrePersistRaptorDataMessage, ? extends %2$s> enhancedGenerator = EnhancedFacadeGenerator.createFacadeGenerator(rdm.getStructure(), %2$s.class);
            if (enhancedGenerator != null) generator = enhancedGenerator;
            enhancedGeneratorConstructed = true;
        }
        return generator.apply(rdm);
    }

    /**
     * Construct facade for a given RDM.
     *
     * @param rdm The source RDM on which to create this facade
     */
    protected %2$s(IPrePersistRaptorDataMessage rdm) { this.rdm = rdm; }

    /**
     * @return The source RDM on which this facade was generated
     */
    public IPrePersistRaptorDataMessage rdm() { return rdm; }
'''

    // Format to generate the methods for a single field, args:
    //   1. field name
    //   2. capitalized field name
    //   3. value type
    //   4. RDM method suffix
    //
    private static final String METHODS_FMT = '''
    public %3$s get%2$s() { return rdm.get%4$s("%1$s"); }
    public void set%2$s(%3$s value) { rdm.set%4$s("%1$s", value); }
    public boolean is%2$sNull() { return rdm.isNull("%1$s"); }
    public void set%2$sNull() { rdm.setField("%1$s", null); }
'''

    // Map from XML tag/field type to value type
    private static final Map FIELD_TYPE_TO_VALUE_TYPE_MAP = [
            'BooleanField'     : "boolean",
            'ShortField'       : "short",
            'IntegerField'     : "int",
            'LongField'        : "long",
            'FloatField'       : "float",
            'DoubleField'      : "double",
            'StringField'      : "String",
            'DateTimeField'    : "java.util.Date",
            'BytesField'       : "byte[]",
            'BooleanArrayField': "boolean[]",
            'ShortArrayField'  : "short[]",
            'IntArrayField'    : "int[]",
            'LongArrayField'   : "long[]",
            'FloatArrayField'  : "float[]",
            'DoubleArrayField' : "double[]",
            'StringArrayField' : "String[]",
            'DateArrayField'   : "java.util.Date[]"
    ]

    // Map from XML tag/field type to ARaptorMessage method suffix (e.g., "Int" for getInt and setInt)
    private static final Map FIELD_TYPE_TO_METHOD_SFX = [
            'BooleanField'     : "Boolean",
            'ShortField'       : "Short",
            'IntegerField'     : "Int",
            'LongField'        : "Long",
            'FloatField'       : "Float",
            'DoubleField'      : "Double",
            'StringField'      : "String",
            'DateTimeField'    : "Date",
            'BytesField'       : "ByteArray",
            'BooleanArrayField': "BooleanArray",
            'ShortArrayField'  : "ShortArray",
            'IntArrayField'    : "IntArray",
            'LongArrayField'   : "LongArray",
            'FloatArrayField'  : "FloatArray",
            'DoubleArrayField' : "DoubleArray",
            'StringArrayField' : "StringArray",
            'DateArrayField'   : "DateArray"
    ]

    /**
     * Generate the facade classes for all the configured structures.
     */
    @TaskAction
    void generate() {

        if (!SourceVersion.isName(targetPackage)) throw new InvalidUserDataException("Invalid package name: '$targetPackage'")

        destinationDir = new File(outputDir, targetPackage.replace('.', '/'))

        if (!destinationDir.exists()) destinationDir.mkdirs()

        logger.debug "Input files: ${structureXmls.files}"

        def parser = new XmlParser()

        // Iterate the configured structure XML files, parse them, and generate facades for the defined structures
        structureXmls.forEach { xml ->
            logger.info("Parsing structures from: $xml")

            Node structures = parser.parse(xml)
            structures.Structure.each { Node s -> generateFacade(s) }
        }
    }

    /**
     * Generate the facade class for a single structure.
     *
     * @param structure The XML Node for the structure to process
     */
    protected void generateFacade(Node structure) {
        String messageType = structure.MessageType.text()

        // Prefix the class name as configured
        String className = classNamePrefix + (camelCase ? toCamelCase(messageType) : messageType)

        File classFile = new File(destinationDir, className + ".java")

        if (!SourceVersion.isName(className)) {
            reportError classFile, String.format("Message type [%s] generated illegal class name [%s]", messageType, className)
            return
        }

        logger.info(String.format("Generating class %s.%s from structure %s", targetPackage, className, messageType))

        // Generate the initial class definition
        classFile.text = String.format(CLASS_START_FMT, targetPackage, className, messageType, new Date())

        // And then iterate the fields and generate the methods for each field

        //noinspection GroovyAssignabilityCheck
        Node fields = (Node) structure.Fields[0]

        for (field in fields.children()) {
            String fieldType = field.name()
            String fieldName = field['@name']
            String capFieldName = toCamelCase(fieldName)

            if (!SourceVersion.isIdentifier(className)) {
                reportError classFile, String.format("Message type [%s] field [%s] generated illegal identifier [%s]", messageType, fieldName, capFieldName)
                return
            }

            logger.debug "   $fieldName => $capFieldName"

            String valueType = FIELD_TYPE_TO_VALUE_TYPE_MAP.get(fieldType)
            String methodSfx = FIELD_TYPE_TO_METHOD_SFX.get(fieldType)

            if (!(valueType && methodSfx)) {
                reportError classFile, String.format("Message type [%s] contains unknown field type [%s] for field [%s]", messageType, fieldType, fieldName)
                return
            }

            classFile << String.format(METHODS_FMT, fieldName, capFieldName, valueType, methodSfx)
        }

        classFile << '}\n'
    }

    /**
     * Convert a given string into a camel case string.  Use whitespace, hyphens, underscores, slashes, and periods
     * as word delimiters.  All delimiters are removed from the string and each word start is capitalized.
     *
     * @param text Text to convert to camel case
     * @return camel cased text
     */
    protected String toCamelCase(String text) {
        boolean hasDelimiters = text =~ /[\s_.\-\/]/
        if (hasDelimiters) text = text.toLowerCase() // Avoid smashing to lower case when no delimiters are present

        return text.replaceAll(~'([\\s_.\\-/])([A-Za-z0-9])', { Object[] it -> it[2].capitalize() }).capitalize()
    }

    /**
     * Report an error, and fail the build if {@code failBuildOnError} is {@code true}.
     *
     * @param classFile Current class file being generated
     * @param msg Message to report
     */
    protected void reportError(File classFile, String msg) {
        if (failBuildOnError) {
            throw new InvalidUserDataException(msg)
        } else {
            def path = project.projectDir.toPath().relativize(classFile.toPath())
            logger.error("\n${msg}\n   deleting generated file: $path")
            project.delete classFile
        }
    }
}
