package gov.raptor.gradle.plugins.buildsupport

import org.ajoberstar.grgit.Commit
import org.gradle.api.DefaultTask

/**
 * Base class for tasks that operate on the project's repository (git operations).
 *
 * @author Proprietary information subject to the terms of a Non-Disclosure Agreement
 * @since 0.3
 */
class ARepoTask extends DefaultTask {

    protected static final String[] REPO_OPTIONS = [
            "-Pnopush\tDon't push anything to the remote",
            "-Pnoexec\tDon't actually execute operations, just print what would occur",
    ]

    protected String makeDescription(String desc) {
        return makeDescription(desc, REPO_OPTIONS)
    }

    protected String makeDescription(String desc, String... options) {
        String optionStr = options ? "Options:\n\t${options.join('\n\t')}" : ''
        return "$desc\n$optionStr\n"
    }

    /**
     * Create a short description of a commit.
     */
    protected String formatCommit(Commit commit) {
        return "Commit ${commit.getAbbreviatedId()} by $commit.committer.name on $commit.dateTime msg='$commit.shortMessage'"
    }

    /**
     * Resolve a given tag name to a commit.  Trap any underlying exception (like an NPE which is how jgit responds)
     * and convert it into a RuntimeException with a meaningful error message.
     *
     * @param tagName The tag string to resolve to a Commit
     * @return Commit
     * @throws RuntimeException If the resolve operation fails
     */
    protected Commit safeResolveCommit(String tagName) throws RuntimeException {
        try {
            return project.grgit.resolve.toCommit(tagName)
        } catch (Exception e) {
            throw new RuntimeException("No commit found for revision string: " + tagName, e)
        }
    }

    /**
     * Get to the root branch.
     */
    protected void getToRootBranch() {
        // If the root branch doesn't exist, create it from origin/<rootBranch>.

        String rootBranchName = project.rootBranch

        def rootBranch = project.grgit.resolve.toBranch(rootBranchName)
        if (!rootBranch) {
            // create it from origin/<rootBranchName>

            logger.lifecycle "git co -b $rootBranchName origin/$rootBranchName"
            if (!project.noExec) project.grgit.checkout(branch: rootBranchName, startPoint: "origin/$rootBranchName", createBranch: true)

            if (!project.noExec) {
                rootBranch = project.grgit.resolve.toBranch(rootBranchName)
                // Now that we've created it, resolve the root branch
                def newBranchName = rootBranch.fullName

                def atCommit = project.grgit.resolve.toCommit(newBranchName)
                logger.lifecycle "Added new branch ${rootBranch} at ${atCommit.abbreviatedId}"

                logger.lifecycle "git push $newBranchName ${project.noPush ? '(skipped)' : ''}"
                if (!project.noPush) {
                    project.grgit.push(refsOrSpecs: [newBranchName])
                }
            }
        }

        checkoutBranch(rootBranchName) // And checkout the new branch
    }

    /**
     * Checkout the specified branch.  Perform a hard reset first.
     *
     * @param branchName The name of the branch to checkout
     */
    private void checkoutBranch(String branchName) {
        // We need to be on the desired branch, and fully up to date

        def currentBranch = project.grgit.branch.current().name
        logger.lifecycle "Current branch: $currentBranch"

        if (currentBranch == branchName) {
            // already on the proper branch, do a pull
            logger.lifecycle 'git pull'
            if (!project.noExec) {
                try {
                    project.grgit.pull()
                } catch (Exception e) {
                    // If noPush has been set, then we will ignore a failure trying to pull
                    // This is primarily in support of unit tests
                    if (project.noPush) {
                        logger.warn "Ignored failure on 'pull': $e"
                    } else {
                        throw e
                    }
                }
            }
        } else {
            // Not on branch, so check it out (do a fetch first to be up to date)
            logger.lifecycle 'git fetch'
            if (!project.noExec) project.grgit.fetch()

            logger.lifecycle "git co $branchName"
            if (!project.noExec) project.grgit.checkout(branch: branchName)
        }
    }
}
