package fr.ird.observe.spi;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.dto.form.FormDefinition;
import fr.ird.observe.dto.reference.DataDtoReference;
import fr.ird.observe.dto.reference.DtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.dto.referential.ReferentialLocale;
import fr.ird.observe.spi.context.DataDtoContext;
import fr.ird.observe.spi.context.DataFormContext;
import fr.ird.observe.spi.context.DtoContext;
import fr.ird.observe.spi.context.ReferentialDtoContext;
import fr.ird.observe.spi.context.ReferentialFormContext;
import fr.ird.observe.spi.initializer.DtoFormsInitializerSupport;
import fr.ird.observe.spi.initializer.DtoReferencesInitializerSupport;
import fr.ird.observe.spi.initializer.DtoToMainDtoInitializerSupport;
import fr.ird.observe.spi.map.DtoToDtoClassMap;
import fr.ird.observe.spi.map.ImmutableTypedMap;
import io.ultreia.java4all.application.context.spi.GenerateApplicationComponent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

/**
 * Created by tchemit on 02/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
@GenerateApplicationComponent(name = "Dto Model Helper",
        dependencies = {
                DtoFormsInitializerSupport.class,
                DtoReferencesInitializerSupport.class,
                DtoToMainDtoInitializerSupport.class
        })
public class DtoModelHelper {

    private static final Logger log = LogManager.getLogger(DtoModelHelper.class);

    private static final DtoModelHelper INSTANCE = DtoModelHelperApplicationComponent.value();

    private final DtoReferencesInitializerSupport referencesInitializer;

    private final ImmutableTypedMap<ReferentialDtoContext> referentialDtoContext;
    private final ImmutableTypedMap<ReferentialFormContext> referentialFormContext;
    private final ImmutableTypedMap<DataDtoContext> dataDtoContext;
    private final ImmutableTypedMap<DataFormContext> dataFormContext;
    private final ImmutableSet<Class> referentialClasses;
    private final ImmutableSet<Class> mainDataClasses;
    private final DtoToDtoClassMap dtoToMainDtoClassMapping;

    public DtoModelHelper(DtoFormsInitializerSupport formInitializer,
                          DtoReferencesInitializerSupport referencesInitializer,
                          DtoToMainDtoInitializerSupport dtoToMainDtoInitializer) {
        this.referencesInitializer = referencesInitializer;

        log.info("Dto model helper initialization  (" + this + ").");

//        DtoReferencesInitializerSupport referencesInitializer = getReferencesInitializer();
//        DtoFormsInitializerSupport formInitializer = getFormInitializer();
//        DtoToMainDtoInitializerSupport dtoToMainDtoInitializer = getDtoToMainInitializer();

        ImmutableTypedMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder = ImmutableTypedMap.builder();
        ImmutableTypedMap.Builder<ReferentialFormContext> referentialFormContextBuilder = ImmutableTypedMap.builder();
        ImmutableTypedMap.Builder<DataDtoContext> dataDtoContextBuilder = ImmutableTypedMap.builder();
        ImmutableTypedMap.Builder<DataFormContext> dataFormContextBuilder = ImmutableTypedMap.builder();
        Set<Class> mainDataClassesBuilder = new LinkedHashSet<>();
        Set<Class> referentialClassesBuilder = new LinkedHashSet<>();

        Collection<Class<? extends IdDto>> dtoTypesWithReference = this.referencesInitializer.getReferenceToDtoClassMapping().values();

        int referentialCount = 0;

        for (Class<? extends ReferentialDto> dtoType : this.referencesInitializer.getReferentialTypes()) {

            registerReferential(dtoType, this.referencesInitializer, formInitializer, referentialDtoContextBuilder, referentialFormContextBuilder);
            referentialClassesBuilder.add(dtoType);
            referentialCount++;
        }
        log.info("Load " + referentialCount + " referential definitions.");

        Collection<Class> dtoTypesWithForm = formInitializer.getDtoToFormClassMapping().keys();
        int dataCountWithReference = 0;
        int dataCountWithForm = 0;
        Set<Class<? extends DataDto>> dataTypes = new LinkedHashSet<>(this.referencesInitializer.getDataTypes());
        dataTypes.addAll((Collection) dtoTypesWithForm);

        for (Class<? extends DataDto> dtoType : dataTypes) {

            Class<? extends DataDto> formType = formInitializer.getDtoToFormClassMapping().forData(dtoType);

            if (dtoTypesWithForm.contains(formType)) {

                FormDefinition formDefinition = formInitializer.getFormDefinitions().get(formType);
                if (formDefinition != null) {
                    registerDataForm(dtoType, formType, formDefinition, dataFormContextBuilder);
                    dataCountWithForm++;
                }
            }

            mainDataClassesBuilder.add(dtoType);
            if (this.referencesInitializer.getDtoToReferenceClassMapping().forData(dtoType) == null) {
                continue;
            }

            if (dtoTypesWithReference.contains(dtoType)) {
                registerDataWithReference(dtoType, this.referencesInitializer, dataDtoContextBuilder);
                dataCountWithReference++;
            } else {
                registerData(dtoType, this.referencesInitializer, dataDtoContextBuilder);
            }

        }

        log.info("Load " + dataCountWithReference + " reference definitions.");
        log.info("Load " + dataCountWithForm + " form definitions.");
        log.info("Load " + this.referencesInitializer.getReferentialBinders().size() + " dto referential binders.");
        log.info("Load " + this.referencesInitializer.getDataBinders().size() + " dto data binders.");
        log.info("Load " + dtoToMainDtoInitializer.getDtoToMainDtoClassMapping().size() + " dto to main dto types.");

        this.referentialDtoContext = referentialDtoContextBuilder.build();
        this.referentialFormContext = referentialFormContextBuilder.build();
        this.referentialClasses = ImmutableSet.copyOf(referentialClassesBuilder);
        this.dataDtoContext = dataDtoContextBuilder.build();
        this.dataFormContext = dataFormContextBuilder.build();
        this.mainDataClasses = ImmutableSet.copyOf(mainDataClassesBuilder);
        this.dtoToMainDtoClassMapping = dtoToMainDtoInitializer.getDtoToMainDtoClassMapping();

        log.info("Dto model helper is initialized (" + this + ").");
    }

    public static Set<Class<? extends ReferentialDto>> getReferentialClasses() {
        return (Set) INSTANCE.referentialClasses;
    }

    public static Set<Class<? extends DataDto>> getMainDataClasses() {
        return (Set) INSTANCE.mainDataClasses;
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> DtoContext<D, R> fromDto(Class<D> dtoType) {
        if (isReferential(dtoType)) {
            return INSTANCE.referentialDtoContext.get(dtoType);
        } else {
            return INSTANCE.dataDtoContext.get(dtoType);
        }
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> DtoContext<D, R> fromReference(Class<R> referenceType) {
        if (isReferential(referenceType)) {
            Class<D> dtoType = INSTANCE.referencesInitializer.getReferenceToDtoClassMapping().forReferential((Class) referenceType);
            return INSTANCE.referentialDtoContext.get(dtoType);
        } else {
            Class<D> dtoType = INSTANCE.referencesInitializer.getReferenceToDtoClassMapping().forData((Class) referenceType);
            return INSTANCE.dataDtoContext.get(dtoType);
        }
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> Class<R> getReferenceType(Class<D> dtoType) {
        return DtoModelHelper.<D, R>fromDto(dtoType).toReferenceType();
    }

    public static <D extends ReferentialDto, R extends ReferentialDtoReference<D, R>> ReferentialDtoContext<D, R> fromReferentialDto(Class<D> dtoType) {
        return INSTANCE.referentialDtoContext.get(dtoType);
    }

    public static <D extends ReferentialDto, R extends ReferentialDtoReference<D, R>> ReferentialDtoContext<D, R> fromReferentialReference(Class<R> referenceType) {
        Class<D> dtoType = INSTANCE.referencesInitializer.getReferenceToDtoClassMapping().forReferential(referenceType);
        return INSTANCE.referentialDtoContext.get(dtoType);
    }

    public static <D extends DataDto, R extends DataDtoReference<D, R>> DataDtoContext<D, R> fromDataReference(Class<R> referenceType) {
        Class<D> dtoType = INSTANCE.referencesInitializer.getReferenceToDtoClassMapping().forData(referenceType);
        return INSTANCE.dataDtoContext.get(dtoType);
    }

    public static <D extends DataDto, R extends DataDtoReference<D, R>> DataDtoContext<D, R> fromDataDto(Class<D> dtoType) {
        return INSTANCE.dataDtoContext.get(dtoType);
    }


    public static <D extends IdDto, R extends DtoReference<D, R>> R toReference(ReferentialLocale referentialLocale, D dto) {
        return (R) DtoModelHelper.<D, R>fromDto((Class) dto.getClass()).toReference(referentialLocale, dto);
//
//        if (ReferentialDto.class.isAssignableFrom(dto.getClass())) {
//            return (R) fromReferentialDto((Class) dto.getClass()).toReference(referentialLocale, (ReferentialDto) dto);
//        }
//        return (R) fromDataDto((Class) dto.getClass()).toReference(referentialLocale, (DataDto) dto);
    }

    public static <D extends IdDto, S extends IdDto> Class<S> getMainDtoType(Class<D> dtoType) {
        return INSTANCE.dtoToMainDtoClassMapping.forDto(dtoType);
    }

    public static <D extends IdDto> Optional<FormDefinition<D>> getOptionalFormDefinition(Class<? extends IdDto> dtoType) {
        if (isReferential(dtoType)) {
            Optional<ReferentialFormContext> optional = Optional.ofNullable(DtoModelHelper.fromReferentialForm((Class) dtoType));
            return optional.map((Function<? super ReferentialFormContext, ? extends FormDefinition<D>>) ReferentialFormContext::toFormDefinition);
        }
        Optional<DataFormContext> optional = Optional.ofNullable(DtoModelHelper.fromDataForm((Class) dtoType));
        return optional.map((Function<? super DataFormContext, ? extends FormDefinition<D>>) DataFormContext::toFormDefinition);
    }

    static <D extends ReferentialDto, F extends ReferentialDto> ReferentialFormContext<D, F> fromReferentialForm(Class<D> dtoType) {
        return INSTANCE.referentialFormContext.get(dtoType);
    }

    static <D extends DataDto, F extends DataDto> DataFormContext<D, F> fromDataForm(Class<D> dtoType) {
        return INSTANCE.dataFormContext.get(dtoType);
    }

//    public static DtoReferencesInitializerSupport getReferencesInitializer() {
////        if (REFERENCES_INITIALIZER == null) {
////            REFERENCES_INITIALIZER = ApplicationContext.get().getComponentValue(DtoReferencesInitializerSupport.class);
////        }
//        return INSTANCE.referencesInitializer;
//    }

//    static DtoFormsInitializerSupport getFormInitializer() {
////        if (FORM_INITIALIZER == null) {
////            FORM_INITIALIZER = ApplicationContext.get().getComponentValue(DtoFormsInitializerSupport.class);
////        }
//        return INSTANCE.formInitializer;
//    }

//    static DtoToMainDtoInitializerSupport getDtoToMainInitializer() {
////        if (DTO_TO_MAIN_INITIALIZER == null) {
////            DTO_TO_MAIN_INITIALIZER = ApplicationContext.get().getComponentValue(DtoToMainDtoInitializerSupport.class);
////        }
//        return INSTANCE.dtoToMainDtoInitializer;
//    }

    private static <D extends ReferentialDto> void registerReferential(Class<D> dtoType,
                                                                       DtoReferencesInitializerSupport referencesInitializer,
                                                                       DtoFormsInitializerSupport formInitializer,
                                                                       ImmutableTypedMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder, ImmutableTypedMap.Builder<ReferentialFormContext> referentialFormContextBuilder) {

        referentialDtoContextBuilder.put(dtoType, new ReferentialDtoContext<>(dtoType, referencesInitializer));
        referentialFormContextBuilder.put(dtoType, new ReferentialFormContext(dtoType, dtoType, formInitializer.getFormDefinitions().get(dtoType)));

    }

    private static <D extends DataDto> void registerData(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableTypedMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));

    }

    private static <D extends DataDto, F extends DataDto> void registerDataForm(Class<D> dtoType, Class<F> formType, FormDefinition<F> referencesInitializer, ImmutableTypedMap.Builder<DataFormContext> dataFormContextBuilder) {
        dataFormContextBuilder.put(dtoType, new DataFormContext<>(dtoType, formType, referencesInitializer));
    }

    private static <D extends DataDto> void registerDataWithReference(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableTypedMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));
    }

    public static boolean isReferential(Class type) {
        return ReferentialDto.class.isAssignableFrom(type) || ReferentialDtoReference.class.isAssignableFrom(type);
//        return isReferentialFromPackageName(type.getPackage().getName());
    }

    public static boolean isData(Class type) {
        return DataDto.class.isAssignableFrom(type) || DataDtoReference.class.isAssignableFrom(type);
//        return isDataFromPackageName(type.getPackage().getName());
    }
}
