package fr.ird.observe.navigation.tree;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */


import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.reference.DtoReference;

import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Contract when a node is a container of references.
 * <p>
 * Created by tchemit on 24/09/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
public interface MultipleReferenceContainerNode<D extends IdDto, R extends DtoReference<D, R>> extends MutableTreeNode {

    static <D extends IdDto, R extends DtoReference<D, R>, N extends MultipleReferenceContainerNode<D, R>> N up(Class<R> referenceType, MutableTreeNode node) {
        return (N) upToReferenceContainerNode(referenceType, node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>, N extends MultipleReferenceContainerNode<D, R>> N down(Class<R> referenceType, MutableTreeNode node) {
        return (N) downToReferenceContainerNode(referenceType, node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> MultipleReferenceContainerNode<D, R> upToReferenceContainerNode(Class<R> referenceType, MutableTreeNode node) {
        MutableTreeNode result = Objects.requireNonNull(node);
        while (result != null) {
            if (result instanceof MultipleReferenceContainerNode) {
                MultipleReferenceContainerNode referenceNode = (MultipleReferenceContainerNode) result;
                if (referenceNode.getChildrenReferenceType().equals(referenceType)) {
                    return referenceNode;
                }
            }
            result = (MutableTreeNode) result.getParent();
        }
        throw new IllegalStateException("Can't go up reference container node of type: " + referenceType.getName() + " from node: " + node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> MultipleReferenceContainerNode<D, R> downToReferenceContainerNode(Class<R> referenceType, MutableTreeNode node) {
        if (node instanceof MultipleReferenceContainerNode) {
            MultipleReferenceContainerNode referenceContainerNode = (MultipleReferenceContainerNode) node;
            if (referenceContainerNode.getChildrenReferenceType().equals(referenceType)) {
                return referenceContainerNode;
            }
        }
        Enumeration children = node.children();
        while (children.hasMoreElements()) {
            MutableTreeNode childrenNode = (MutableTreeNode) children.nextElement();
            if (childrenNode instanceof MultipleReferenceContainerNode) {
                MultipleReferenceContainerNode referenceContainerNode = (MultipleReferenceContainerNode) childrenNode;
                if (referenceContainerNode.getChildrenReferenceType().equals(referenceType)) {
                    return referenceContainerNode;
                }
            }
        }
        throw new IllegalStateException("Can't go down to reference container node of type: " + referenceType.getName() + " from node: " + node);
    }

    Class<D> getChildrenDtoType();

    Class<R> getChildrenReferenceType();

    default List<ReferenceNode<D, R>> getChildrenReferenceNodes() {
        Enumeration children = children();
        Class<R> childrenReferenceType = getChildrenReferenceType();
        List<ReferenceNode<D, R>> result = new LinkedList<>();
        while (children.hasMoreElements()) {
            MutableTreeNode o = (MutableTreeNode) children.nextElement();
            if (o instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) o;
                if (childrenReferenceType.isAssignableFrom(referenceNode.getReferenceReferenceType())) {
                    result.add(referenceNode);
                }
            }
        }
        return result;
    }

    default ReferenceNode<D, R> getChildrenReferenceNode(String id) {
        Enumeration children = children();
        Class<R> childrenReferenceType = getChildrenReferenceType();
        while (children.hasMoreElements()) {
            MutableTreeNode o = (MutableTreeNode) children.nextElement();
            if (o instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) o;
                if (childrenReferenceType.isAssignableFrom(referenceNode.getReferenceReferenceType()) && referenceNode.getReference().getId().equals(id)) {
                    return referenceNode;
                }
            }
        }
        return null;
    }

    default List<R> getChildrenReferences() {
        Enumeration children = children();
        Class<R> childrenReferenceType = getChildrenReferenceType();
        List<R> result = new LinkedList<>();
        while (children.hasMoreElements()) {
            MutableTreeNode o = (MutableTreeNode) children.nextElement();
            if (o instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) o;
                if (childrenReferenceType.isAssignableFrom(referenceNode.getReferenceReferenceType())) {
                    result.add((R) referenceNode.getReference());
                }
            }
        }
        return result;
    }

    Comparator<R> getChildrenComparator();

    default int getNodePosition(R reference) {
        return getChildrenPosition(reference);
    }

    default int getChildrenPosition(R reference) {
        Set<R> childrenReferences = new LinkedHashSet<>(getChildrenReferences());
        childrenReferences.remove(reference);
        childrenReferences.add(reference);
        List<R> list = new ArrayList<>(childrenReferences);
        list.sort(getChildrenComparator());
        return list.indexOf(reference);
    }

    ReferenceNode<D, R> newChildNode(R childrenReference);

    ReferenceNode<D, R> newChildNode(String parentId);

    TreeNode[] getPath();
}
