package fr.ird.observe.navigation.model;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import fr.ird.observe.dto.IdDto;
import io.ultreia.java4all.bean.AbstractJavaBean;
import io.ultreia.java4all.lang.Objects2;

import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Created by tchemit on 26/05/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class DtoModelNavigationAggregateModel extends AbstractJavaBean {

    public static final String PROPERTY_ENABLED = "enabled";
    private final ImmutableList<DtoModelNavigationModel<?>> models;
    private transient boolean enabled;

    @SafeVarargs
    public DtoModelNavigationAggregateModel(Class<? extends DtoModelNavigationModel<?>>... models) {
        ImmutableList.Builder<DtoModelNavigationModel<?>> modelsBuilder = ImmutableList.builder();
        for (Class<? extends DtoModelNavigationModel<?>> model : models) {
            modelsBuilder.add(Objects2.newInstance(model));
        }
        this.models = modelsBuilder.build();
        for (DtoModelNavigationModel<?> model : this.models) {
            model.addPropertyChangeListener(DtoModelNavigationModel.PROPERTY_ENABLED, e -> updateEnabled());
        }
        updateEnabled();
    }

    public boolean isEnabled() {
        return enabled;
    }

    @SuppressWarnings("unchecked")
    public <M extends DtoModelNavigationModel<?>> Optional<M> forModelType(Class<M> type) {
        Objects.requireNonNull(type);
        return (Optional<M>) models.stream().filter(e -> Objects.equals(type, e.getClass())).findFirst();
    }

    private void updateEnabled() {
        boolean oldValue = isEnabled();
        enabled = models.stream().anyMatch(DtoModelNavigationModel::isEnabled);
        firePropertyChange(PROPERTY_ENABLED, oldValue, enabled);
    }

    public ImmutableList<DtoModelNavigationModel<?>> getModels() {
        return models;
    }

    public <D extends IdDto> Optional<DtoModelNavigationNode<D>> forDtoType(Class<D> dtoType) {
        Objects.requireNonNull(dtoType);
        for (DtoModelNavigationModel<?> model : models) {
            DtoModelNavigationNode<D> node = model.getDtoNode(dtoType);
            if (node != null) {
                return Optional.of(node);
            }
        }
        return Optional.empty();
    }

    public <D extends IdDto> Optional<DtoModelNavigationNode<D>> forNavigationNode(DtoModelNavigationNode<?> source) {
        Objects.requireNonNull(source);
        for (DtoModelNavigationModel<?> model : models) {
            if (!model.acceptModel(source.getModel())) {
                continue;
            }
            DtoModelNavigationNode<D> node = model.getNavigationNode(source);
            if (node != null) {
                return Optional.of(node);
            }
        }
        return Optional.empty();
    }

    public <N extends DtoModelNavigationNode> Optional<N> forNodeType(Class<N> nodeType) {
        Objects.requireNonNull(nodeType);
        for (DtoModelNavigationModel<?> model : models) {
            N node = model.getNode(nodeType);
            if (node != null) {
                return Optional.of(node);
            }
        }
        return Optional.empty();
    }

    public void load(DtoModelNavigationAggregateModel model) {
        for (DtoModelNavigationModel<?> navigationModel : model.getModels()) {
            for (DtoModelNavigationNode node : navigationModel.getNodesWithIds()) {
                forNodeType(node.getClass()).orElseThrow(IllegalStateException::new).setId(node.getId());
            }
        }
    }

    public ImmutableList<? extends DtoModelNavigationModel<?>> getEnabledModels() {
        return ImmutableList.copyOf(getModels().stream().filter(DtoModelNavigationModel::isEnabled).collect(Collectors.toList()));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DtoModelNavigationAggregateModel that = (DtoModelNavigationAggregateModel) o;
        return Objects.equals(models, that.models);
    }

    @Override
    public int hashCode() {
        return Objects.hash(models);
    }

    public int count() {
        return models.stream().mapToInt(DtoModelNavigationModel::count).sum();
    }

    public void clearModel() {
        models.forEach(DtoModelNavigationModel::clearModel);
    }
}
