package fr.ird.observe.gson;

/*
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.SetMultimap;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import fr.ird.observe.dto.form.Form;
import fr.ird.observe.dto.reference.DataDtoReferenceSet;
import fr.ird.observe.dto.reference.ReferentialDtoReferenceSet;
import fr.ird.observe.gson.reference.DataReferenceSetAdapter;
import fr.ird.observe.gson.reference.ReferentialReferenceSetAdapter;
import fr.ird.observe.navigation.model.DtoModelNavigationAggregateModel;
import fr.ird.observe.navigation.model.DtoModelNavigationModel;
import fr.ird.observe.spi.map.ImmutableSetDtoMap;
import fr.ird.observe.spi.map.ImmutableSetStringMap;
import io.ultreia.java4all.util.ServiceLoaders;
import org.nuiton.version.Version;

import java.util.Date;
import java.util.function.Supplier;

/**
 * @author Tony Chemit - dev@tchemit.fr
 */
public class DtoGsonSupplierSupport implements Supplier<Gson> {

    private final boolean prettyPrint;

    protected GsonBuilder gsonBuilder;

    private Gson gson;

    public DtoGsonSupplierSupport(boolean prettyPrint) {
        this.prettyPrint = prettyPrint;
    }

    @Override
    public Gson get() {
        if (gson == null) {
            gson = getGsonBuilder(prettyPrint).create();
        }
        return gson;
    }

    protected GsonBuilder getGsonBuilder(boolean prettyPrint) {

        if (gsonBuilder == null) {

            // Assign only when initialization is finished to be thread-safe
            gsonBuilder = new GsonBuilder();

            if (prettyPrint) {
                gsonBuilder.setPrettyPrinting();
            }
            // Type adapters : base types
            gsonBuilder.registerTypeAdapter(Class.class, new ClassAdapter());

            DtoModelNavigationModelAdapter dtoModelNavigationModelAdapter = new DtoModelNavigationModelAdapter();
            DtoModelNavigationAggregateModelAdapter dtoModelNavigationAggregateModelAdapter = new DtoModelNavigationAggregateModelAdapter();
            for (Class<DtoModelNavigationModel> model : ServiceLoaders.loadTypes(DtoModelNavigationModel.class)) {
                gsonBuilder.registerTypeAdapter(model, dtoModelNavigationModelAdapter);
            }
            for (Class<DtoModelNavigationAggregateModel> model : ServiceLoaders.loadTypes(DtoModelNavigationAggregateModel.class)) {
                gsonBuilder.registerTypeAdapter(model, dtoModelNavigationAggregateModelAdapter);
            }

            // Type adapters : base types
            gsonBuilder.registerTypeAdapter(Integer.class, new IntegerAdapter());
            gsonBuilder.registerTypeAdapter(Date.class, new DateAdapter());
            gsonBuilder.registerTypeAdapter(java.sql.Timestamp.class, new DateAdapter());
            gsonBuilder.registerTypeAdapter(java.sql.Time.class, new DateAdapter());
            gsonBuilder.registerTypeAdapter(java.sql.Date.class, new DateAdapter());
            gsonBuilder.registerTypeAdapter(Class.class, new ClassAdapter());
            gsonBuilder.registerTypeAdapter(byte[].class, new ByteArrayAdapter());


            gsonBuilder.registerTypeAdapter(ImmutableList.class, new ImmutableListAdapter());
            gsonBuilder.registerTypeAdapter(ImmutableSet.class, new ImmutableSetAdapter());
            gsonBuilder.registerTypeAdapter(ImmutableMap.class, new ImmutableMapAdapter());

            gsonBuilder.registerTypeAdapter(ImmutableMultimap.class, new ImmutableMultimapAdapter());
            gsonBuilder.registerTypeAdapter(SetMultimap.class, new SetMultimapAdapter());
            gsonBuilder.registerTypeAdapter(ArrayListMultimap.class, new ArrayListMultimapAdapter());

            gsonBuilder.registerTypeAdapter(ImmutableSetDtoMap.class, new ImmutableSetDtoMapAdapter());
            gsonBuilder.registerTypeAdapter(ImmutableSetStringMap.class, new ImmutableSetStringMapAdapter());

            gsonBuilder.registerTypeAdapter(DataDtoReferenceSet.class, new DataReferenceSetAdapter());
            gsonBuilder.registerTypeAdapter(ReferentialDtoReferenceSet.class, new ReferentialReferenceSetAdapter());

            gsonBuilder.registerTypeAdapter(Form.class, new FormAdapter());
            gsonBuilder.registerTypeAdapter(Version.class, new VersionAdapter());
            gsonBuilder.enableComplexMapKeySerialization();
        }
        return gsonBuilder;

    }

}
