package fr.ird.observe.dto;

/*
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.MoreObjects;
import fr.ird.observe.dto.reference.DtoReferenceAware;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.Date;
import java.util.Objects;
import java.util.function.Predicate;

public abstract class IdDto extends AbstractObserveDto implements DtoReferenceAware {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_ID = "id";

    public static final String PROPERTY_LAST_UPDATE_DATE = "lastUpdateDate";

    protected String id;

    protected Date lastUpdateDate;
    protected long version;
    protected Date createDate;

    public static <BeanType extends IdDto> BeanType findById(Collection<BeanType> source, String id) {
        return source.stream().filter(newIdPredicate(id)).findFirst().orElse(null);
    }

    public static <BeanType extends IdDto> boolean exists(Collection<BeanType> source, String id) {
        return source.stream().anyMatch(newIdPredicate(id));
    }

    public static <BeanType extends IdDto> Predicate<BeanType> newIdPredicate(String id) {
        return o -> Objects.equals(id, o.getId());
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        String oldValue = getId();
        this.id = id;
        firePropertyChange(PROPERTY_ID, oldValue, id);
    }

    public Date getLastUpdateDate() {
        return lastUpdateDate;
    }

    public void setLastUpdateDate(Date lastUpdateDate) {
        Date oldValue = getLastUpdateDate();
        this.lastUpdateDate = lastUpdateDate;
        firePropertyChange(PROPERTY_LAST_UPDATE_DATE, oldValue, lastUpdateDate);
    }

    public boolean isPersisted() {
        return StringUtils.isNotBlank(id);
    }

    public boolean isNotPersisted() {
        return !isPersisted();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof IdDto)) return false;
        IdDto that = (IdDto) o;
        return !(id == null || that.id == null) && Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add(PROPERTY_ID, id)
                .add(PROPERTY_LAST_UPDATE_DATE, lastUpdateDate)
                .toString();
    }

    @Override
    public final String getTopiaId() {
        return id;
    }

    @Override
    public final long getTopiaVersion() {
        return version;
    }

    @Override
    public final Date getTopiaCreateDate() {
        return createDate;
    }

    public long getVersion() {
        return version;
    }

    public final void setVersion(long version) {
        this.version = version;
    }

    public Date getCreateDate() {
        return createDate;
    }

    public final void setCreateDate(Date createDate) {
        this.createDate = createDate;
    }
}
