package fr.ird.observe.spi.map;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * A special map to store string values using the {@link #key0(Class)} method to produce keys.
 * <p>
 * Created by tchemit on 01/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class ImmutableSetDtoMap<V> extends ImmutableSetMap<V> {

    ImmutableSetDtoMap(ImmutableMap<String, Set<V>> data, ImmutableMap<String, Class> types) {
        super(data, types);
    }

    public static <V> Builder<V> builder() {
        return new Builder<>();
    }

    public static class Builder<V> {

        private final Map<String, Set<V>> data;
        private final ImmutableMap.Builder<String, Class> types;

        public Builder() {
            this.data = new TreeMap<>();
            this.types = ImmutableMap.builder();
        }

        public Builder put(Class type, V value) {
            String key = keyFunction.apply(type);
            data.computeIfAbsent(key, e -> new LinkedHashSet<>()).add(value);
            types.put(key, type);
            return this;
        }

        @SuppressWarnings("unchecked")
        public <VV extends V> Builder put(Class type, Collection<VV> value) {
            String key = keyFunction.apply(type);
            data.computeIfAbsent(key, e -> new LinkedHashSet<>()).addAll(value);
            types.put(key, type);
            return this;
        }

        public ImmutableSetDtoMap<V> build() {
            return new ImmutableSetDtoMap<>(ImmutableMap.copyOf(data), types.build());
        }

    }

}
