package fr.ird.observe.spi.map;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.IdHelper;
import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.dto.reference.DataDtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.referential.ReferentialDto;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * A special map to store string values using the {@link #key0(Class)} method to produce keys.
 * <p>
 * Created by tchemit on 01/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class ImmutableDtoMap<V> {

    private static final transient Function<Class, String> keyFunction = IdHelper::getDtoSimplifiedName;
    private final ImmutableMap<String, V> data;
    private final ImmutableMap<String, Class> types;

    public static <V> Builder<V> builder() {
        return new Builder<>();
    }

    protected ImmutableDtoMap(ImmutableDtoMap<V> dtoMap) {
        this.data = dtoMap.data;
        this.types = dtoMap.types;
    }

    protected ImmutableDtoMap(ImmutableMap<String, V> data, ImmutableMap<String, Class> types) {
        this.data = data;
        this.types= types;
    }

    public int size() {
        return data.size();
    }

    public boolean isEmpty() {
        return data.isEmpty();
    }

    public V get(Class key) {
        return data.get(key0(key));
    }

//    public V get(String key) {
//        return data.get(key0(key));
//    }

    public Set<String> keySet() {
        return data.keySet();
    }

    public Collection<V> values() {
        return data.values();
    }


    @SuppressWarnings("unchecked")
    public Set<Class<? extends ReferentialDtoReference>> referentialReferenceTypes() {
        return (Set) types.values().stream().filter(IdHelper::isReferential).collect(Collectors.toSet());
    }

    @SuppressWarnings("unchecked")
    public Set<Class<? extends ReferentialDto>> referentialTypes() {
        return (Set) types.values().stream().filter(IdHelper::isReferential).collect(Collectors.toSet());
    }

    @SuppressWarnings("unchecked")
    public Set<Class<? extends DataDtoReference>> dataReferenceTypes() {
        return (Set) types.values().stream().filter(IdHelper::isData).collect(Collectors.toSet());
    }
    @SuppressWarnings("unchecked")
    public Set<Class<? extends DataDto>> dataTypes() {
        return (Set) types.values().stream().filter(IdHelper::isData).collect(Collectors.toSet());
    }

    public boolean containsEntry(Class type, V value) {
        String key = key0(type);
        return data.containsKey(key) && data.values().contains(value);
    }

    public static class Builder<V> {

        private final Map<String, V> data;

        private final Map<String, Class> types;
        private Builder() {
            this.data = new LinkedHashMap<>();
            this.types= new LinkedHashMap<>();
        }

        public Builder put(Class type, V value) {
            String key = keyFunction.apply(type);
            data.put(key, value);
            types.put(key, type);
            return this;
        }

        public ImmutableDtoMap<V> build() {
            return new ImmutableDtoMap<>(ImmutableMap.copyOf(data), ImmutableMap.copyOf(types));
        }

        public V get(Class<?> type) {
            return data.get(type);
        }
    }

    private String key0(Class type) {
        return keyFunction.apply(type);
    }
}
