package fr.ird.observe.spi;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.IdHelper;
import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.dto.form.FormDefinition;
import fr.ird.observe.dto.reference.DataDtoReference;
import fr.ird.observe.dto.reference.DtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.dto.referential.ReferentialLocale;
import fr.ird.observe.spi.context.DataDtoContext;
import fr.ird.observe.spi.context.DataFormContext;
import fr.ird.observe.spi.context.DtoContext;
import fr.ird.observe.spi.context.ReferentialDtoContext;
import fr.ird.observe.spi.context.ReferentialFormContext;
import fr.ird.observe.spi.initializer.DtoFormsInitializerSupport;
import fr.ird.observe.spi.initializer.DtoReferencesInitializerSupport;
import fr.ird.observe.spi.initializer.DtoToMainDtoInitializerSupport;
import fr.ird.observe.spi.map.DtoToDtoClassMap;
import fr.ird.observe.spi.map.ImmutableDtoMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Created by tchemit on 02/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
public class DtoModelHelper {

    /** Logger. */
    private static final Logger log = LogManager.getLogger(DtoModelHelper.class);

    private static final DtoModelHelper INSTANCE = new DtoModelHelper();
    private static DtoFormsInitializerSupport FORM_INITIALIZER;
    private static DtoReferencesInitializerSupport REFERENCES_INITIALIZER;
    private static DtoToMainDtoInitializerSupport DTO_TO_MAIN_INITIALIZER;

    private final ImmutableDtoMap<ReferentialDtoContext> referentialDtoContext;
    private final ImmutableDtoMap<ReferentialFormContext> referentialFormContext;
    private final ImmutableDtoMap<DataDtoContext> dataDtoContext;
    private final ImmutableDtoMap<DataFormContext> dataFormContext;
    private final ImmutableSet<Class> referentialClasses;
    private final ImmutableSet<Class> mainDataClasses;
    private final DtoToDtoClassMap dtoToMainDtoClassMapping;

    protected DtoModelHelper() {

        log.info("Dto model helper initialization  (" + this + ").");

        DtoReferencesInitializerSupport referencesInitializer = getReferencesInitializer();
        DtoFormsInitializerSupport formInitializer = getFormInitializer();
        DtoToMainDtoInitializerSupport dtoToMainDtoInitializer = getDtoToMainInitializer();

        ImmutableDtoMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<ReferentialFormContext> referentialFormContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<DataFormContext> dataFormContextBuilder = ImmutableDtoMap.builder();
        Set<Class> mainDataClassesBuilder = new LinkedHashSet<>();
        Set<Class> referentialClassesBuilder = new LinkedHashSet<>();

        Collection<Class<? extends IdDto>> dtoTypesWithReference = referencesInitializer.getReferenceToDtoClassMapping().values();

        int referentialCount = 0;

        for (Class<? extends ReferentialDto> dtoType : referencesInitializer.getReferentialTypes()) {

            registerReferential(dtoType, referencesInitializer, formInitializer, referentialDtoContextBuilder, referentialFormContextBuilder);
            referentialClassesBuilder.add(dtoType);
            referentialCount++;
        }
        log.info("Load " + referentialCount + " referential definitions.");

        Collection<Class> dtoTypesWithForm = formInitializer.getDtoToFormClassMapping().keys();
        int dataCountWithReference = 0;
        int dataCountWithForm = 0;
        Set<Class<? extends DataDto>> dataTypes = new LinkedHashSet<>(referencesInitializer.getDataTypes());
        dataTypes.addAll((Collection) dtoTypesWithForm);

        for (Class<? extends DataDto> dtoType : dataTypes) {

            Class<? extends DataDto> formType = formInitializer.getDtoToFormClassMapping().forData(dtoType);

            if (dtoTypesWithForm.contains(formType)) {

                FormDefinition formDefinition = formInitializer.getFormDefinitions().get(formType);
                if (formDefinition != null) {
                    registerDataForm(dtoType, formType, formDefinition, dataFormContextBuilder);
                    dataCountWithForm++;
                }
            }

            mainDataClassesBuilder.add(dtoType);
            if (referencesInitializer.getDtoToReferenceClassMapping().forData(dtoType) == null) {
                continue;
            }

            if (dtoTypesWithReference.contains(dtoType)) {
                registerDataWithReference(dtoType, referencesInitializer, dataDtoContextBuilder);
                dataCountWithReference++;
            } else {
                registerData(dtoType, referencesInitializer, dataDtoContextBuilder);
            }

        }

        log.info("Load " + dataCountWithReference + " reference definitions.");
        log.info("Load " + dataCountWithForm + " form definitions.");
        log.info("Load " + referencesInitializer.getReferentialBinders().size() + " dto referential binders.");
        log.info("Load " + referencesInitializer.getDataBinders().size() + " dto data binders.");
        log.info("Load " + dtoToMainDtoInitializer.getDtoToMainDtoClassMapping().size() + " dto to main dto types.");

        this.referentialDtoContext = referentialDtoContextBuilder.build();
        this.referentialFormContext = referentialFormContextBuilder.build();
        this.referentialClasses = ImmutableSet.copyOf(referentialClassesBuilder);
        this.dataDtoContext = dataDtoContextBuilder.build();
        this.dataFormContext = dataFormContextBuilder.build();
        this.mainDataClasses = ImmutableSet.copyOf(mainDataClassesBuilder);
        this.dtoToMainDtoClassMapping = dtoToMainDtoInitializer.getDtoToMainDtoClassMapping();

        log.info("Dto model helper is initialized (" + this + ").");
    }

    public static Set<Class<? extends ReferentialDto>> getReferentialClasses() {
        return (Set) INSTANCE.referentialClasses;
    }

    public static Set<Class<? extends DataDto>> getMainDataClasses() {
        return (Set) INSTANCE.mainDataClasses;
    }

    public static Set<Class<? extends ReferentialDto>> filterReferentialDto(Collection<Class<? extends IdDto>> types) {
        return (Set) types.stream().filter(ReferentialDto.class::isAssignableFrom).collect(Collectors.toSet());
    }

    public static Set<Class<? extends DataDto>> filterDataDto(Collection<Class<? extends IdDto>> types) {
        return (Set) types.stream().filter(DataDto.class::isAssignableFrom).collect(Collectors.toSet());
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> DtoContext<D, R> fromDto(Class<D> dtoType) {
        if (IdHelper.isReferential(dtoType)) {
            return INSTANCE.referentialDtoContext.get(dtoType);
        } else {
            return INSTANCE.dataDtoContext.get(dtoType);
        }
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> DtoContext<D, R> fromReference(Class<R> referenceType) {
        if (IdHelper.isReferential(referenceType)) {
            Class<D> dtoType = getReferencesInitializer().getReferenceToDtoClassMapping().forReferential((Class) referenceType);
            return INSTANCE.referentialDtoContext.get(dtoType);
        } else {
            Class<D> dtoType = getReferencesInitializer().getReferenceToDtoClassMapping().forData((Class) referenceType);
            return INSTANCE.dataDtoContext.get(dtoType);
        }
    }

    public static <D extends IdDto, R extends DtoReference<D, R>> Class<R> getReferenceType(Class<D> dtoType) {
        return DtoModelHelper.<D, R>fromDto(dtoType).toReferenceType();
    }

    public static <D extends ReferentialDto, R extends ReferentialDtoReference<D, R>> ReferentialDtoContext<D, R> fromReferentialDto(Class<D> dtoType) {
        return INSTANCE.referentialDtoContext.get(dtoType);
    }

    public static <D extends ReferentialDto, R extends ReferentialDtoReference<D, R>> ReferentialDtoContext<D, R> fromReferentialReference(Class<R> referenceType) {
        Class<D> dtoType = getReferencesInitializer().getReferenceToDtoClassMapping().forReferential(referenceType);
        return INSTANCE.referentialDtoContext.get(dtoType);
    }

    public static <D extends DataDto, R extends DataDtoReference<D, R>> DataDtoContext<D, R> fromDataReference(Class<R> referenceType) {
        Class<D> dtoType = getReferencesInitializer().getReferenceToDtoClassMapping().forData(referenceType);
        return INSTANCE.dataDtoContext.get(dtoType);
    }

    public static <D extends DataDto, R extends DataDtoReference<D, R>> DataDtoContext<D, R> fromDataDto(Class<D> dtoType) {
        return INSTANCE.dataDtoContext.get(dtoType);
    }


    public static <D extends IdDto, R extends DtoReference<D, R>> R toReference(ReferentialLocale referentialLocale, D dto) {
        return (R)DtoModelHelper.<D, R>fromDto((Class)dto.getClass()).toReference(referentialLocale, dto);
//
//        if (ReferentialDto.class.isAssignableFrom(dto.getClass())) {
//            return (R) fromReferentialDto((Class) dto.getClass()).toReference(referentialLocale, (ReferentialDto) dto);
//        }
//        return (R) fromDataDto((Class) dto.getClass()).toReference(referentialLocale, (DataDto) dto);
    }

    public static <D extends IdDto, S extends IdDto> Class<S> getMainDtoType(Class<D> dtoType) {
        return INSTANCE.dtoToMainDtoClassMapping.forDto(dtoType);
    }

    public static <D extends IdDto> Optional<FormDefinition<D>> getOptionalFormDefinition(Class<? extends IdDto> dtoType) {
        if (IdHelper.isReferential(dtoType)) {
            Optional<ReferentialFormContext> optional = Optional.ofNullable(DtoModelHelper.fromReferentialForm((Class) dtoType));
            return optional.map((Function<? super ReferentialFormContext, ? extends FormDefinition<D>>) ReferentialFormContext::toFormDefinition);
        }
        Optional<DataFormContext> optional = Optional.ofNullable(DtoModelHelper.fromDataForm((Class) dtoType));
        return optional.map((Function<? super DataFormContext, ? extends FormDefinition<D>>) DataFormContext::toFormDefinition);
    }

    static <D extends ReferentialDto, F extends ReferentialDto> ReferentialFormContext<D, F> fromReferentialForm(Class<D> dtoType) {
        return INSTANCE.referentialFormContext.get(dtoType);
    }

    static <D extends DataDto, F extends DataDto> DataFormContext<D, F> fromDataForm(Class<D> dtoType) {
        return INSTANCE.dataFormContext.get(dtoType);
    }

    public static DtoReferencesInitializerSupport getReferencesInitializer() {
        if (REFERENCES_INITIALIZER == null) {
            REFERENCES_INITIALIZER = getService(DtoReferencesInitializerSupport.class);
        }
        return REFERENCES_INITIALIZER;
    }

    static DtoFormsInitializerSupport getFormInitializer() {
        if (FORM_INITIALIZER == null) {
            FORM_INITIALIZER = getService(DtoFormsInitializerSupport.class);
        }
        return FORM_INITIALIZER;
    }

    static DtoToMainDtoInitializerSupport getDtoToMainInitializer() {
        if (DTO_TO_MAIN_INITIALIZER == null) {
            DTO_TO_MAIN_INITIALIZER = getService(DtoToMainDtoInitializerSupport.class);
        }
        return DTO_TO_MAIN_INITIALIZER;
    }

    private static <D extends ReferentialDto> void registerReferential(Class<D> dtoType,
                                                                       DtoReferencesInitializerSupport referencesInitializer,
                                                                       DtoFormsInitializerSupport formInitializer,
                                                                       ImmutableDtoMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder, ImmutableDtoMap.Builder<ReferentialFormContext> referentialFormContextBuilder) {

        referentialDtoContextBuilder.put(dtoType, new ReferentialDtoContext<>(dtoType, referencesInitializer));
        referentialFormContextBuilder.put(dtoType, new ReferentialFormContext(dtoType, dtoType, formInitializer.getFormDefinitions().get(dtoType)));

    }

    private static <D extends DataDto> void registerData(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));

    }

    private static <D extends DataDto, F extends DataDto> void registerDataForm(Class<D> dtoType, Class<F> formType, FormDefinition<F> referencesInitializer, ImmutableDtoMap.Builder<DataFormContext> dataFormContextBuilder) {
        dataFormContextBuilder.put(dtoType, new DataFormContext<>(dtoType, formType, referencesInitializer));
    }

    private static <D extends DataDto> void registerDataWithReference(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));
    }

    public static <S> S getService(Class<S> serviceType) {
        Iterator<S> iterator = ServiceLoader.load(serviceType).iterator();
        if (iterator.hasNext()) {

            return iterator.next();
        } else {
            throw new IllegalStateException("No instance found for " + serviceType.getName());
        }
    }


}
