package fr.ird.observe.navigation.tree;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */


import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.reference.DtoReference;

import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import java.util.Enumeration;
import java.util.Objects;

/**
 * Contract when a node is a container of one reference.
 * <p>
 * Created by tchemit on 24/09/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
public interface SingleReferenceContainerNode<D extends IdDto, R extends DtoReference<D, R>> extends MutableTreeNode {

    static <N extends SingleReferenceContainerNode<?, ?>> N up(Class<?> referenceType, MutableTreeNode node) {
        return (N) upToSingleReferenceContainerNode(referenceType, node);
    }

    static <N extends SingleReferenceContainerNode<?, ?>> N down(Class<?> referenceType, MutableTreeNode node) {
        return (N) downToSingleReferenceContainerNode(referenceType, node);
    }

    static SingleReferenceContainerNode<?, ?> upToSingleReferenceContainerNode(Class<?> referenceType, MutableTreeNode node) {
        MutableTreeNode result = Objects.requireNonNull(node);
        while (result != null) {
            if (result instanceof SingleReferenceContainerNode) {
                SingleReferenceContainerNode singleReferenceContainerNode = (SingleReferenceContainerNode) result;
                if (singleReferenceContainerNode.getSingleReferenceType().equals(referenceType)) {
                    return singleReferenceContainerNode;
                }
            }
            result = (MutableTreeNode) result.getParent();
        }
        throw new IllegalStateException("Can't go up to single reference container node of type: " + referenceType.getName() + " from node: " + node);
    }

    static SingleReferenceContainerNode<?, ?> downToSingleReferenceContainerNode(Class<?> referenceType, MutableTreeNode node) {
        if (node instanceof SingleReferenceContainerNode) {
            SingleReferenceContainerNode singleReferenceContainerNode = (SingleReferenceContainerNode) node;
            if (singleReferenceContainerNode.getSingleReferenceType().equals(referenceType)) {
                return singleReferenceContainerNode;
            }
        }
        Enumeration children = node.children();
        while (children.hasMoreElements()) {
            MutableTreeNode childrenNode = (MutableTreeNode) children.nextElement();
            if (childrenNode instanceof SingleReferenceContainerNode) {
                SingleReferenceContainerNode singleReferenceContainerNode = (SingleReferenceContainerNode) childrenNode;
                if (singleReferenceContainerNode.getSingleReferenceType().equals(referenceType)) {
                    return singleReferenceContainerNode;
                }
            }
        }
        throw new IllegalStateException("Can't go down to single reference container node of type: " + referenceType.getName() + " from node: " + node);
    }

    Class<D> getSingleDtoType();

    Class<R> getSingleReferenceType();

    default ReferenceNode<D, R> getSingleReferenceNode() {
        Enumeration children = children();
        Class<R> singleReferenceType = getSingleReferenceType();
        while (children.hasMoreElements()) {
            MutableTreeNode o = (MutableTreeNode) children.nextElement();
            if (o instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) o;
                if (singleReferenceType.isAssignableFrom(referenceNode.getReferenceReferenceType())) {
                    return referenceNode;
                }
            }
        }
        return null;
    }

    default R getSingleReference() {
        Enumeration children = children();
        Class<R> childrenReferenceType = getSingleReferenceType();
        while (children.hasMoreElements()) {
            MutableTreeNode o = (MutableTreeNode) children.nextElement();
            if (o instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) o;
                if (childrenReferenceType.isAssignableFrom(referenceNode.getReferenceReferenceType())) {
                    return (R) referenceNode.getReference();
                }
            }
        }
        return null;
    }

    ReferenceNode<D, R> newSingleChildNode(R childrenReference);

    ReferenceNode<D, R> newSingleChildNode(String parentId);

    TreeNode[] getPath();
}
