package fr.ird.observe.navigation.tree;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.reference.DtoReference;
import fr.ird.observe.navigation.model.DtoModelNavigationNodeSupport;

import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import java.util.Enumeration;
import java.util.Objects;

/**
 * Contract when a node is a reference.
 * <p>
 * Created by tchemit on 24/09/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
public interface ReferenceNode<D extends IdDto, R extends DtoReference<D, R>> extends MutableTreeNode {

    static <D extends IdDto, R extends DtoReference<D, R>, N extends ReferenceNode<D, R>> N up(Class<R> referenceType, MutableTreeNode node) {
        return (N) upToReferenceNode(referenceType, node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>, N extends ReferenceNode<D, R>> N up(MutableTreeNode node) {
        return (N) upToFirstReferenceNode(node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>, N extends ReferenceNode<D, R>> N down(Class<R> referenceType, String id, MutableTreeNode node) {
        return (N) downToReferenceNode(referenceType, id, node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>, N extends ReferenceNode<D, R>> N down(Class<R> referenceType, MutableTreeNode node) {
        return (N) downToFirstReferenceNode(referenceType, node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> ReferenceNode<D, R> upToReferenceNode(Class<R> referenceType, MutableTreeNode node) {
        MutableTreeNode result = Objects.requireNonNull(node);
        while (result != null) {
            if (result instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) result;
                if (referenceNode.getReferenceReferenceType().equals(referenceType)) {
                    //noinspection unchecked
                    return referenceNode;
                }
            }
            result = (MutableTreeNode) result.getParent();
        }
        throw new IllegalStateException("Can't go up reference node of type: " + referenceType.getName() + " from node: " + node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> ReferenceNode<D, R> upToFirstReferenceNode(MutableTreeNode node) {
        MutableTreeNode result = Objects.requireNonNull(node);
        while (result != null) {
            if (result instanceof ReferenceNode) {
                //noinspection unchecked
                return (ReferenceNode) result;
            }
            result = (MutableTreeNode) result.getParent();
        }
        throw new IllegalStateException("Can't go up reference node from node: " + node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> ReferenceNode<D, R> downToFirstReferenceNode(Class<R> referenceType, MutableTreeNode node) {
        if (node instanceof ReferenceNode) {
            ReferenceNode referenceNode = (ReferenceNode) node;
            if (referenceNode.getReferenceReferenceType().equals(referenceType)) {
                //noinspection unchecked
                return referenceNode;
            }
        }
        Enumeration children = Objects.requireNonNull(node).children();
        while (children.hasMoreElements()) {
            MutableTreeNode childrenNode = (MutableTreeNode) children.nextElement();
            if (childrenNode instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) childrenNode;
                if (referenceNode.getReferenceReferenceType().equals(referenceType)) {
                    //noinspection unchecked
                    return referenceNode;
                }
            }
        }
        throw new IllegalStateException("Can't go down reference node of type: " + referenceType.getName() + " from node: " + node);
    }

    static <D extends IdDto, R extends DtoReference<D, R>> ReferenceNode<D, R> downToReferenceNode(Class<R> referenceType, String id, MutableTreeNode node) {
        if (node instanceof ReferenceNode) {
            ReferenceNode referenceNode = (ReferenceNode) node;
            if (referenceNode.getReferenceReferenceType().equals(referenceType)) {
                //noinspection unchecked
                return referenceNode;
            }
        }
        Enumeration children = Objects.requireNonNull(node).children();
        while (children.hasMoreElements()) {
            MutableTreeNode childrenNode = (MutableTreeNode) children.nextElement();
            if (childrenNode instanceof ReferenceNode) {
                ReferenceNode referenceNode = (ReferenceNode) childrenNode;
                if (referenceNode.getReferenceReferenceType().equals(referenceType) && referenceNode.getReference().getId().equals(id)) {
                    //noinspection unchecked
                    return referenceNode;
                }
            }
        }
        throw new IllegalStateException("Can't go down reference node of type: " + referenceType.getName() + " from node: " + node);
    }

    Class<D> getReferenceDtoType();

    Class<R> getReferenceReferenceType();

    R getReference();

    DtoModelNavigationNodeSupport<?> getSelectNode();

    DtoModelNavigationNodeSupport<?> getEditNode();

    TreeNode[] getPath();
}
